\section{Midi}
\label{midi}

Midi (``musical instrument digital interface'') is a standard protocol
adopted by most, if not all, manufacturers of electronic instruments.
At its core is a protocol for communicating {\em musical events} (note
on, note off, key press, etc.) as well as so-called {\em meta events}
(select synthesizer patch, change volume, etc.).  Beyond the logical
protocol, the Midi standard also specifies electrical signal
characteristics and cabling details.  In addition, it specifies what
is known as a {\em standard Midi file} which any Midi-compatible
software package should be able to recognize.

Over the years musicians and manufacturers decided that they also
wanted a standard way to refer to {\em common} or {\em general}
instruments such as ``acoustic grand piano,'' ``electric piano,''
``violin,'' and ``acoustic bass,'' as well as more exotic ones such as
``chorus aahs,'' ``voice oohs,'' ``bird tweet,'' and ``helicopter.''
A simple standard known as {\em General Midi} was developed to fill
this role.  It is nothing more than an agreed-upon list of instrument
names along with a {\em program patch number} for each, a parameter in
the Midi standard that is used to select a Midi instrument's sound.

Most ``sound-blaster''-like sound cards on conventional PC's know
about Midi, as well as General Midi.  However, the sound generated by
such modules, and the sound produced from the typically-scrawny
speakers on most PC's, is often quite poor.  It is best to use an
outboard keyboard or tone generator, which are attached to a computer
via a Midi interface and cables.  It is possible to connect several
Midi instruments to the same computer, with each assigned a different
{\em channel}.  Modern keyboards and tone generators are quite amazing
little beasts.  Not only is the sound quite good (when played on a
good stereo system), but they are also usually {\em multi-timbral},
which means they are able to generate many different sounds
simultaneously, as well as {\em polyphonic}, meaning that simultaneous
instantiations of the same sound are possible.

Note: If you decide to use the General midi features of your
sound-card, you need to know about another set of conventions known as
``Basic Midi'' which is not discussed here.  The most important aspect
of Basic Midi is that Channel 10 is dedicated to {\em percussion}.  A
future release of Haskore should make these distinctions more concrete.

Haskore provides a way to specify a Midi channel number and General
Midi instrument selection for each {\tt IName} in a Haskore
composition.  It also provides a means to generate a Standard Midi
File, which can then be played using any conventional Midi software.
In this section the top-level code needed by the user to invoke this
functionality will be described, along with the gory details.
\begin{verbatim}

> module HaskToMidi (module HaskToMidi, module GeneralMidi, module MidiFile)
>        where
>
> import Basics
> import Performance
> import MidiFile
> import GeneralMidi
> import List(partition)
> import Char(toLower,toUpper)

\end{verbatim} 

Instead of converting a Haskore {\tt Performance} directly into a Midi
file, Haskore first converts it into a datatype that {\em represents}
a Midi file, which is then written to a file in a separate pass.  This
separation of concerns makes the structure of the Midi file clearer,
makes debugging easier, and provides a natural path for extending
Haskore's functionality with direct Midi capability (in fact there is
a version of Haskore that does this under Windows '95, but it is not
described here).

A {\tt UserPatchMap} is a user-supplied table for mapping instrument
names ({\tt IName}'s) to Midi channels and General Midi patch names.
The patch names are by default General Midi names, although the user
can also provide a {\tt PatchMap} for mapping Patch Names to
unconventional Midi Program Change numbers.
\begin{verbatim} 

> type UserPatchMap = [(IName,GenMidiName,MidiChannel)]

\end{verbatim} 

See Appendix \ref{test-functions} for an example of a useful user
patch map.

Given a {\tt UserPatchMap}, a performance is converted to a datatype
representing a Standard Midi File using the {\tt performToMidi}
function.
\begin{verbatim} 

> performToMidi :: Performance -> UserPatchMap -> MidiFile
> performToMidi pf pMap = 
>        MidiFile mfType (Ticks division)
>          (map (performToMEvs pMap) (splitByInst pf))

\end{verbatim} 
A table of General Midi assignments called {\tt genMidiMap} is
imported from {\tt GeneralMidi} in Appendix~\ref{general-midi}.  The
Midi file datatype itself and functions for writing it to files are
imported from the module {\tt MidiFile}, which is described later in
this section.

Now for the Gory Details.

Some preliminaries, otherwise known as constants:
\begin{verbatim} 

> mfType   = 1  :: MFType    -- midi-file type 1 always used
> velocity = 80 :: Velocity  -- default velocity (max 100)
> division = 96 :: Int       -- time-code division: 96 ticks per quarter note

\end{verbatim}

Since we are implementing Type 1 Midi Files, we can associate each
instrument with a separate track.  So first we partition the event list
into separate lists for each instrument.
\begin{verbatim} 

> splitByInst :: Performance ->  [(IName,Performance)]
> splitByInst [] = []
> splitByInst pf = (i,pf1) : splitByInst pf2
>                  where (pf1,pf2) = partition (\e -> getEventInst e == i) pf
>                        i         = getEventInst (head pf)

\end{verbatim} 

The crux of the conversion process is {\tt performToMEvs}, which
converts a {\tt Performance} into a stream of {\tt MEvents}.
\begin{verbatim} 

> performToMEvs :: UserPatchMap -> (IName,Performance) -> [MEvent]
> performToMEvs pMap (inm,perf) =
>   let (midiChan,progNum) = unMap pMap inm
>       setupInst          = MidiEvent 0 (ProgChange midiChan progNum)
>       loop []     = []
>       loop (e:es) = let (mev1,mev2) = mkMEvents midiChan e
>                     in  mev1 : insertMEvent mev2 (loop es)
>   in  setupInst : loop perf

\end{verbatim} 

A source of incompatibilty between Haskore and Midi is that Haskore
represents notes with an onset and a duration, while Midi represents
them as two separate events, an note-on event and a note-off event.
Thus {\tt MkMEvents} turns a Haskore {\tt Event} into two {\tt
MEvents}, a {\tt NoteOn} and a {\tt NoteOff}.
\begin{verbatim} 

> mkMEvents :: MidiChannel -> Event -> (MEvent,MEvent)  
> mkMEvents mChan (Event t i p d v) =
>                     ( MidiEvent (toDelta t)     (NoteOn  mChan p v'),
>                       MidiEvent (toDelta (t+d)) (NoteOff mChan p v') )
>           where v' = min 127 (round (v*1.27))
>
> toDelta t = round (t * 4.0 * float division)

\end{verbatim} 

The final critical function is {\tt insertMEvent}, which inserts an
{\tt MEvent} into an already time-ordered sequence of {\tt MEvents}.
\begin{verbatim} 

> insertMEvent :: MEvent -> [MEvent] -> [MEvent]
> insertMEvent mev1  []         = [mev1]
> insertMEvent mev1@(MidiEvent t1 _) mevs@(mev2@(MidiEvent t2 _):mevs') = 
>       if t1 <= t2 then mev1 : mevs
>                   else mev2 : insertMEvent mev1 mevs'

\end{verbatim} 

The following functions lookup {\tt IName}'s in {\tt UserPatchMaps} to
recover channel and program change numbers.  Note that the function
that does string matching ignores case, and allows substring matches.
For example, {\tt "chur"} matches {\tt "Church Organ"}.  Note also
that the {\em first} match succeeds, so using a substring should be
done with care to be sure that the correct instrument is selected.
\begin{verbatim} 

> unMap :: UserPatchMap -> IName -> (MidiChannel,ProgNum)
> unMap pMap iName = (channel, gmProgNum gmName)
>   where (gmName, channel) = lookup iName pMap
>         lookup x ((y,z,q):ys) = if (x `partialMatch` y) then (z,q) else lookup x ys
>         lookup x []           = error ("Instrument " ++ x ++ " unknown")
>
> gmProgNum :: GenMidiName -> ProgNum
> gmProgNum gmName = lookup gmName genMidiMap
>   where lookup x ((y,z):ys) = if (x `partialMatch` y) then z else lookup x ys
>         lookup x []         = error ("Instrument " ++ x ++ " unknown")
>
> partialMatch       :: String -> String -> Bool
> partialMatch s1 s2 = 
>   let s1' = map toLower s1;  s2' = map toLower s2
>       len = min (length s1) (length s2)
>   in take len s1' == take len s2'

\end{verbatim} 
\ignore{
 mapToChannel :: UserPatchMap -> IName -> MidiChannel
 mapToChannel pMap iname = lookup3 iname pMap
 generalMidi :: UserPatchMap -> IName -> ProgNum
 generalMidi pMap iname = generalMidiNum (lookup2 iname pMap)

 lookup2 x ((y,z,q):ys) = if x==y then z else lookup2 x ys
 lookup2 x []           = error ("Instrument " ++ x ++ " unknown")

 lookup3 x ((y,z,q):ys) = if x==y then q else lookup3 x ys
 lookup3 x []           = error ("Instrument " ++ x ++ " unknown")
}

\ignore{
Rather than having to send a separate program change message for each
event, we send a program change message to each channel at the
beginning of the file.
\begin{verbatim} 
 setupInst :: UserPatchMap -> [MEvent]
 setupInst = map (\(_,g,c) -> MidiEvent 0 (ProgChange c (generalMidiNum g)))
\end{verbatim} 
}
