package biss.jde;

import biss.FileLib;
import biss.ObserverSocket;
import biss.StringLib;
import biss.awt.Awt;
import biss.awt.List;
import biss.awt.Prompter;
import java.awt.Color;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.File;
import java.util.BitSet;
import java.util.Enumeration;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

class AcceptException
  extends Exception
{

AcceptException ( String msg ) {
	super( msg);
}
}

/**
 * logical view for analyzing/modifying CompileUnits (*.java source files)
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class CUBrowser
  implements Mod, Observer, CUObserver
{
	CompileUnit CU;
	Data SelData;
	Import SelImpt;
	Method SelMthd;
	TypeDecl SelType;
	String DFCat;
	BitSet DFMods;
	String MFCat;
	String MFId;
	BitSet MFMods;
	int Mode = 0;
	int PendingChanges = 0;
	boolean ReadOnlyMode = false;
	boolean CommentMode = false;
	int ModifierUsage;
	JavaColorizer Colorizer;
	UsageAnalyzer Searcher;
	ByteCodeCompiler Comp;
	CompileOutputDlg OutputDlg;
	ItemCheckDlg ModDlg;
	CUViewer View = new CUViewer( getOpenRect());
	final static int DEL_DATA = 22;
	final static int DEL_IMPT = 21;
	final static int DEL_MTHD = 23;
	final static int DEL_TYPE = 20;
	final static int REP_DATA = 32;
	final static int REP_IMPT = 31;
	final static int REP_MTHD = 33;
	final static int REP_TYPE = 30;
	final static int SAV_FILE = 99;
	final static int SEL_DATA = 4;
	final static int SEL_IMPT = 3;
	final static int SEL_MTHD = 5;
	final static int SEL_PCKG = 1;
	final static int SEL_TYPE = 2;
	final static int MOD_SET_TYPE = 1;
	final static int MOD_SET_DATA = 2;
	final static int MOD_FLT_DATA = 3;
	final static int MOD_SET_MTHD = 4;
	final static int MOD_FLT_MTHD = 5;
	public static Vector Instances = new Vector( 5);

public CUBrowser ( CompileUnit cu ) {
	this( cu, cu.mainType());
}

public CUBrowser ( CompileUnit cu, TypeDecl td ) {
	CU = cu;

	if ( Jde.ColorizeSyntax )
		Colorizer = new JavaColorizer( View.TxtPane);

	registerObserver();

	updateTitle();
	View.TypePane.setContents( CU.TypeDecls);
	View.ImptPane.setContents( CU.Imports);
	updateFileInfo();
	updateChangeInfo();

	if ( td != null )
		View.TypePane.setSelection( td, true);

	Instances.addElement( this);
}

void accept () {
	String src = View.TxtPane.getStrippedContents();
	AcceptParser parser = new AcceptParser( src);
	parser.storeOutput();

	View.AcceptBtn.setAlert( false);
	CU.setLastChange( 0);

	if ( CommentMode ) {
		if ( acceptComment( src) )
			PendingChanges ++;
		View.TxtPane.setModified( false);
	}
	else {
		try {
			switch ( Mode ) {
			case SEL_PCKG: acceptPackage( parser); break;
			case REP_TYPE: renameType( parser); break;
			case SEL_TYPE: acceptTypeDecl( parser); break;
			case REP_IMPT:
			case SEL_IMPT: acceptImport( parser); break;
			case REP_DATA:
			case SEL_DATA:
			case REP_MTHD:
			case SEL_MTHD: acceptDataOrMethod( parser); break;
			case DEL_TYPE: delTypes(); break;
			case DEL_IMPT: delImports(); break;
			case DEL_DATA: delDatas(); break;
			case DEL_MTHD: delMethods(); break;

			case SAV_FILE: storeOnDisk( false); PendingChanges = -1; Mode = 0; break; 

			default:       acceptAny( parser);
			}

			View.TxtPane.setModified( false);
			PendingChanges ++;
		}
		catch  ( ParseException x ) {
			parser.reportException( x);
			showCompileOutput( parser.getOutBuffer().toString());
		}
		catch ( AcceptException x ) {
			alertInfo( x.getMessage());
			return;
		}
	}

	CU.notifyObservers( this);
	updateChangeInfo();
	updateModeInfo();
}

void acceptAny ( AcceptParser parser ) {
	System.out.println( "not yet supported");
}

boolean acceptComment ( String cmt ) {
	if ( !cmt.startsWith( "/*") )
		cmt = "/**\n" + cmt;
	if ( !cmt.endsWith( "*/") )
		cmt += "\n*/";

	switch ( Mode ) {
	case SEL_MTHD:
		if ( SelMthd != null ) {
			SelMthd.setComment( cmt); return true;
		}
		break;
	case SEL_DATA:
		if ( SelData != null ) {
			SelData.setComment( cmt); return true;
		}
		break;
	case SEL_TYPE:
		if ( SelType != null ) {
			SelType.setComment( cmt); return true;
		}
		break;
	}
	return false;
}

void acceptDataOrMethod ( AcceptParser parser )
	throws ParseException, AcceptException {
	if ( SelType == null && !guessSelType() )
		throw new AcceptException( "no Type selected");

	parser.checkParens();
	parser.reset();
	TypeDecl  d = parser.parseDataOrMethod();

	if ( !d.Datas.isEmpty() )
		acceptDatas( d.Datas);

	if ( !d.Methods.isEmpty() )
		acceptMethods( d.Methods);
}

void acceptDatas ( Vector datas ) {
	BitSet defMods = (SelData != null) ? SelData.Mods : null;

	if ( Mode == REP_DATA ) 
		SelType.delDatas( View.DataPane.getSelections());    

	SelType.mergeDatas( datas, defMods);
	View.DataPane.setContents( getDisplayDatas());
	if ( Mode == SEL_DATA && datas.size() > 0 ){
		SelData = (Data)datas.firstElement();
		View.DataPane.setSelection( SelData, false);
	}

	CU.setLastChange( CompileUnit.CHG_DATA);
}

void acceptImport ( AcceptParser parser ) throws ParseException {
	Vector list = parser.parseImports();

	if ( Mode == REP_IMPT )
		CU.delImports( View.ImptPane.getSelections());

	CU.mergeImports( list);
	View.ImptPane.setContents( CU.Imports);
	if ( list.size() > 0 ) {
		SelImpt = (Import)list.firstElement();
		View.ImptPane.setSelection( SelImpt, false);
	}
}

void acceptMethods ( Vector methods ) {
	BitSet defMods = (SelMthd != null) ? SelMthd.Mods : null;

	if ( Mode == REP_MTHD )
		SelType.delMethods( View.MthdPane.getSelections());

	if ( methods.size() == 0 ) return;

	SelType.mergeMethods( methods, defMods);
	View.TxtPane.setModified( false);  //to avoid checkUnaccepted() via updateContents() selection
	View.MthdPane.updateContents( getDisplayMethods());
	if ( Mode == SEL_MTHD ){
		if ( SelMthd == null || !methods.contains( SelMthd) ) {
			SelMthd = (Method)methods.firstElement();
			View.MthdPane.setSelection( SelMthd, false);
		}
	}

	CU.setLastChange( CompileUnit.CHG_MTHD);
}

void acceptPackage ( AcceptParser parser ) throws ParseException {
	String pckg = null;
	pckg = parser.parsePackage();

	CU.setPackage( pckg);
	updateTitle();
}

void acceptTypeDecl ( AcceptParser parser ) throws ParseException {
	Vector list = parser.parseTypeDecls();
	CU.mergeTypes( list, true);
	View.TypePane.setContents( CU.TypeDecls);
	if ( list.size() > 0 ) {
		SelType = CU.typeWithName( ((TypeDecl) list.firstElement()).Id );
		View.TypePane.setSelection( SelType, false);
		View.DataPane.setContents( SelType.Datas);
		View.MthdPane.setContents( SelType.Methods);
	}
}

String addFilterDescription ( String info ) {
	StringBuffer s = null;

	if ( DFMods != null || DFCat != null ){
		s = new StringBuffer( 30);
		s.append("  {D:");
		if ( DFMods != null )
			appendModKeys( s, DFMods);
		if ( DFCat != null) {
			s.append( ' ');
			s.append( DFCat);
		}
		s.append( '}');
	}

	if ( MFMods != null || MFCat != null || MFId != null ) {
		if ( s == null )
			s = new StringBuffer( 30);
		s.append( "  {M:");
		if ( MFMods != null )
			appendModKeys( s, MFMods);
		if ( MFCat != null) {
			s.append( ' ');
			s.append( MFCat);
		}
		if ( MFId != null ) {
			s.append( ' ');
			s.append( MFId);
		}
		s.append( '}');
	}

	if ( s != null )
		return info + s;
	else
		return info;
}

public void addTemplate ( Object sel ) {
	String t = null;
	if ( SelType == null && !guessSelType()) return;

	if ( sel.equals( "tmpl.main") )
		t = "public static void main ( String[] args ) {\n\n}";
	else if ( sel.equals( "tmpl.ctor") )
		t = "public " + SelType.Id + " () {\n\n}";
	else if ( sel.equals( "tmpl.finalize") )
		t = "protected finalize () {\n\tsuper.finalize();\n}";
	else if ( sel.equals( "tmpl.update") )
		t = "public void update ( Observable obs, Object arg ) {\n\n}";

	setMode( SEL_MTHD, this, t);
}

void alertInfo ( String msg ) {
	View.ModeInfo.setForeground( Color.red );
	View.ModeInfo.setContents( msg);
}

static void appendModKeys ( StringBuffer s, BitSet mods ) {
	for ( int i=0; i<NMods; i++ ) {
		if ( mods.get(i) ){
			s.append( ' ');
			s.append( ModName[i]);
		}
	}
}

void checkIndentation () {
	Thread t = new Thread( new MthdFormatChecker( CU) );
	t.setPriority( Thread.MIN_PRIORITY);
	t.start();
}

void checkModDlg ( int mode ) {
	if ( ModDlg == null || ModifierUsage != mode ) return;
	BitSet mod;

	switch ( mode ) {
	case MOD_SET_TYPE:   if ( SelType != null ) ModDlg.setChecked( SelType.Mods); break;
	case MOD_SET_DATA:   if ( SelData != null ) ModDlg.setChecked( SelData.Mods); break;
	case MOD_SET_MTHD:   if ( SelMthd != null ) ModDlg.setChecked( SelMthd.Mods); break;
	}
}

public boolean checkUnaccepted () {
	if ( View.TxtPane.isModified() ) {
		View.TxtPane.setModified( false);
		alertInfo( View.ModeInfo.getContents()+
		           ", accept changes?");
		View.AcceptBtn.setAlert( true);
		return false;
	}
	else {
		View.AcceptBtn.setAlert( false);      
		return true;
	}
}

void clearDataFilter () {
	DFMods = null;
	DFCat = null;
	View.DataPane.setContents( getDisplayDatas());
	updateModeInfo();
}

void clearMethodFilter () {
	MFMods = null;
	MFCat = null;
	MFId = null;
	View.MthdPane.setContents( getDisplayMethods());
	updateModeInfo();
}

void closeView () {
	if ( !checkUnaccepted() ){
		View.disableDispose();
		return;
	}
	if ( PendingChanges > 0 ) {
		Mode = SAV_FILE;
		View.AcceptBtn.setAlert( true);
		alertInfo( "pending changes, save file ?");
		PendingChanges = 0;
		View.disableDispose();
		return;
	}

	if ( ModDlg != null ) ModDlg.dispose();
	if ( OutputDlg != null ) OutputDlg.dispose();

	if ( Colorizer != null ) Colorizer.stop();

	Instances.removeElement( this);
	unregisterObserver();

	if ( Mode == SAV_FILE ) {          // closed despite of unsaved changes
		if ( CU.ExpandObservers == 0 )   // restore from file next time CU is accessed
			CU.shrink();
	}
}

void compile () {
	if ( !checkUnaccepted() )
		return;

	if ( OutputDlg != null ) 
		OutputDlg.dispose();

	if ( PendingChanges > 0 )
		storeOnDisk( true);
	else
		CU.refreshLines();

	Comp = new ByteCodeCompiler();
	Comp.storeOutput();
	Comp.setTarget( CU.FileName);
	Comp.setTargetPath( Jde.ClassRoot);
	Comp.addObserver( this);
	Comp.startCompile();
	View.ModeInfo.setContents( "Compiling..");
}

void compileReady () {
	if ( !Comp.Result ) {
		showCompileOutput( Comp.LocalOut.toString());
		View.ModeInfo.setContents( "Compile Failed");
	}
	else
		View.ModeInfo.setContents( "Compile Ready");

	Comp = null;
}

void delDatas () {
	if ( SelType != null ) {
		Vector list = View.DataPane.getSelections();
		SelType.delDatas( list);
		SelData = null; SelMthd = null;
		View.DataPane.removeAllElements( list);
		Mode = SEL_DATA;
		updateModeInfo();
	}
}

void delImports () {
	if ( CU != null ) {
		Vector list = View.ImptPane.getSelections();
		CU.delImports( list);
		View.ImptPane.repaint();
		Mode = SEL_IMPT;
		updateModeInfo();
	}
}

void delMethods () {
	if ( SelType != null ) {
		Vector list = View.MthdPane.getSelections();
		SelType.delMethods( list);
		SelMthd = null;
		View.MthdPane.removeAllElements( list);
		Mode = SEL_MTHD;
		updateModeInfo();
	}
}

void delTypes () {
	if ( CU != null ) {
		Vector list = View.TypePane.getSelections();
		CU.delTypeDecls( list);
		SelType = null; SelData = null; SelMthd = null;
		View.TypePane.setContents( CU.TypeDecls);
		View.DataPane.setContents( null);
		View.MthdPane.setContents( null);
		Mode = SEL_TYPE;
		updateModeInfo();
	}
}

boolean expandBytecodeCU () {
	return false;
}

public boolean expandCU ( CompileUnit cu ) {
	if ( CU != cu ) return false;

	if ( cu.SrcFile != null )
		return expandSourceCU();
	else
		return expandBytecodeCU();
}

boolean expandSourceCU () {
	byte[] buf = FileLib.getFileData( CU.SrcFile);
	if ( buf == null ) {
		View.ModeInfo.setContents( "cannot read " + CU.FileName);
		return false;
	}

	Parser p = new Parser( CU.FileName, buf);
	p.storeOutput();

	try {
		View.ModeInfo.setContents( "..parsing " + CU.FileName);
		CompileUnit fullCu = p.parseCompileUnit();
		CU.expandFrom( fullCu);
		return true;
	}
	catch ( ParseException x ) {
		p.reportException( x);
		View.TxtPane.setContents( new String( buf, 0));
		showCompileOutput( p.getOutBuffer().toString());
		return false;      
	}
}

void format () {
	if ( !checkUnaccepted() )
		return;

	View.ModeInfo.setContents( "..formatting");

	CU.format();

	PendingChanges++;
	CU.notifyObservers( null);
	updateChangeInfo();

	View.ModeInfo.setContents( "formatting complete");
}

public void formatMethod () {
	if ( Mode != SEL_MTHD ) return;

	MethodFormatter mf = new MethodFormatter();
	String s = mf.format( View.TxtPane.getContents());
	View.TxtPane.setContents( s);
	View.TxtPane.setModified( true);
}

Vector getDisplayDatas () {
	if ( SelType == null )
		return null;
	else {
		int n = SelType.Datas.size();
		Vector list = new Vector( n);
		for ( int i=0; i<n; i++ ) {
			Data d = (Data)SelType.Datas.elementAt(i);
			if ( DFMods != null && !d.hasAnyMods( DFMods) ) continue;
			if ( DFCat != null && !d.hasCategory( DFCat) ) continue;
			list.addElement( d);
		}
		return list;
	}
}

Vector getDisplayMethods () {
	if ( SelType == null )
		return null;

	int n = SelType.Methods.size();
	Vector list = new Vector( n);

	for ( int i=0; i<n; i++ ) {
		Method m = (Method)SelType.Methods.elementAt(i);
		if ( MFMods != null && !m.hasAnyMods( MFMods) ) continue;
		if ( MFCat != null && !m.hasCategory( MFCat) ) continue;
		if ( MFId != null && !m.usesData( MFId) ) continue;
		list.addElement( m);
	}

	return list;
}

String getDisplayText () {
	String s = null;

	switch ( Mode ) {
	case SEL_PCKG:
		if ( CU == null ) break;
		s = CU.Package;
		break;

	case REP_TYPE:
		s = SelType.definition();
		break;
	case DEL_TYPE:
	case SEL_TYPE:
		if ( SelType == null ) {
			if ( CU.TypeDecls.size() == 0 && !CommentMode )
				s = CU.mainTypeTemplate();
		}
		else {
			if ( CommentMode ) {
				if ( SelType != null) s = SelType.commentOrTemplate();
			}
			else
				s = TypeDecl.allDefinitions( View.TypePane.getSelections(), false, true);
		}
		break;

	case REP_IMPT:
	case DEL_IMPT:
	case SEL_IMPT:
		s = Import.allDefinitions( View.ImptPane.getSelections());
		break;

	case REP_DATA:
	case DEL_DATA:
	case SEL_DATA:
		if ( CommentMode ) {
			if ( SelData != null ) s = SelData.commentOrTemplate();
		}
		else
			s = Data.allDefinitions( View.DataPane.getSelections(), false, (Mode == REP_DATA));     
		break;

	case REP_MTHD:
	case DEL_MTHD:
	case SEL_MTHD:
		if ( CommentMode ) {
			if ( SelMthd != null ) s = SelMthd.commentOrTemplate();
		}
		else
			s = Method.allDefinitions( View.MthdPane.getSelections(), false, (Mode == REP_MTHD));
		break;

	default:
		s = null;
	}

	return s;
}

static Rectangle getOpenRect () {
	Rectangle r = Jde.CUBrowserRect;
	int n = Instances.size();
	if ( n > 0 )
		r.move( 30*n, 30*n);
	return r;
}

boolean guessSelType () {
	if ( CU != null && CU.TypeDecls.size() == 1 ) {
		SelType = (TypeDecl) CU.TypeDecls.firstElement();
		View.TypePane.setSelection( SelType, false);
		View.DataPane.setContents( getDisplayDatas());
		View.MthdPane.setContents( getDisplayMethods());
		return true;
	}
	else
		return false;
}

public static void main ( String[] args ) {
	try {
		byte[] buf = FileLib.getFileData( new File(args[0]) );
		Parser p = new Parser( args[0], buf);
		CompileUnit cu = p.parseCompileUnit();
		new CUBrowser( cu);
	}
	catch ( Throwable x ) { x.printStackTrace(); }
}

public boolean needsExpandedCU ( CompileUnit cu ) {
	return cu == CU;
}

void openMethodBrowser ( Method mthd ) {
	new MethodBrowser( mthd.Body);
}

void openModDlg ( int modUsage ) {
	String title;
	BitSet mods, set = null;

	switch ( modUsage ) {
	case MOD_SET_TYPE:
		mods = TypeDecl.PossibleMods; title = "Set Type Mods";
		set = (SelType == null ) ? null : SelType.Mods;
		break;
	case MOD_SET_DATA:
		mods = Data.PossibleMods; title = "Set Data Mods";
		set = (SelData == null) ? null : SelData.Mods;
		break;
	case MOD_FLT_DATA:
		mods = Data.PossibleMods; title = "Filter Data Mods"; break;
	case MOD_SET_MTHD:
		mods = Method.PossibleMods; title = "Set Method Mods";
		set = (SelMthd == null) ? null : SelMthd.Mods;
		break;
	case MOD_FLT_MTHD: 
		mods = Method.PossibleMods; title = "Filter Method Mods"; break;
	default: return;
	}

	ModifierUsage = modUsage;

	if ( ModDlg == null ) {
		ModDlg = new ItemCheckDlg( title, ModName, mods, set, this);
	}
	else {
		ModDlg.toFront();
		ModDlg.setContents( ModName, mods);
		ModDlg.setChecked( set);
		ModDlg.setTitle( title);
	}
}

void positionOnError ( CompileOutputDlg dlg ) {

	if ( CU == null || dlg.Fname.equals( "<text>") ){
		View.TxtPane.selectLine( dlg.Line);
		return;
	}

	positionOnLine( dlg.Line, dlg.Loc);
}

void positionOnLine ( int line, String pattern ) {
	TypeDecl  t = CU.typeOfLine( line);
	if ( t != null ) {
		if ( t != SelType )
			View.TypePane.setSelection( t, true);

		FieldDecl f = t.fieldOfLine( line);
		if ( f != null ) {
			if ( f instanceof Method ) {
				if ( f != SelMthd ) {
					if ( SelData != null )
						View.DataPane.clearSelections( true);

					View.MthdPane.setSelection( f, true);
				}

				if ( pattern != null ) {
					View.TxtPane.findString( pattern.trim());
					View.TxtPane.selectLine();
				}
				else {
					int d = line - f.Line -1;
					View.TxtPane.selectLine( d);
				}
			}
			else {
				View.DataPane.setSelection( f, true);
			}
		}
	}
}

void print () {
	if ( !checkUnaccepted() )
		return;

	String mi = View.ModeInfo.getContents();

	PrettyPrinter pp = new PrettyPrinter( CU);
	View.ModeInfo.setContents( "Printing to " + pp.PathName + "..");
	View.ModeInfo.sync();
	pp.print();

	View.ModeInfo.setContents( mi);
}

void printText ( String fileName ) {
	if ( fileName == null ) {
		new Prompter( "enter name of output file", "source.ps", "printText", this, View);
		return;	
	}

	if ( !fileName.endsWith( ".ps") )
		fileName += ".ps";

	String mi = View.ModeInfo.getContents();

	PrettyPrinter pp = new PrettyPrinter( View.TxtPane.getContents(), fileName);
	View.ModeInfo.setContents( "Printing to " + pp.PathName + "..");
	View.ModeInfo.sync();
	pp.print();

	View.ModeInfo.setContents( mi);
}

public void registerObserver() {
	CU.addObserver( this);
	View.OsClose.addObserver( this);
	View.BMenu.OsCommand.addObserver( this);

	View.TypePane.OsSelection.addObserver( this);
	View.TypePane.OsCommand.addObserver( this);

	View.ImptPane.OsSelection.addObserver( this);
	View.ImptPane.OsCommand.addObserver( this);

	View.DataPane.OsSelection.addObserver( this);
	View.DataPane.OsCommand.addObserver( this);

	View.MthdPane.OsSelection.addObserver( this);
	View.MthdPane.OsProcess.addObserver( this);
	View.MthdPane.OsCommand.addObserver( this);

	View.AcceptBtn.OsAction.addObserver( this);
	View.TxtPane.OsModified.addObserver( this);
}

void renameType ( AcceptParser parser )
	throws ParseException, AcceptException {
	if ( SelType == null && !guessSelType() )
		throw new AcceptException( "no type selected");


	Vector list = parser.parseTypeDecls();

	if ( list.size() == 1 ) {
		String newName = ((TypeDecl)list.firstElement()).Id;
		if ( CU.renameType( SelType.Id, newName) ) {
			View.TypePane.repaint();
		}
		else
			View.ModeInfo.setContents( "type rename failed");
	}
	else
		View.ModeInfo.setContents( "single type spec required");
}

void save () {
	if ( !checkUnaccepted() )
		return;

	storeOnDisk( false);
}

void setMode ( int newMode, Object reqSel, String newText ) {
	if ( !checkUnaccepted() || reqSel == null ) return;
	Mode = newMode;

	if ( newText == null )
		newText = getDisplayText();
	View.TxtPane.setContents( newText);
	updateModeInfo();
}

void setModifier () {
	if ( ModDlg == null ) return;
	BitSet mods = ModDlg.getChecked();

	switch ( ModifierUsage ) {
	case MOD_SET_TYPE:
		setNewAttrs( View.TypePane, mods); break;
	case MOD_SET_DATA:
		setNewAttrs( View.DataPane, mods); break; 
	case MOD_FLT_DATA:
		DFMods = mods;
		View.DataPane.setContents( getDisplayDatas());
		updateModeInfo();
		break;
	case MOD_SET_MTHD:
		setNewAttrs( View.MthdPane, mods); break;
	case MOD_FLT_MTHD: 
		MFMods = mods;
		View.MthdPane.setContents( getDisplayMethods());
		updateModeInfo();
		break;
	}
}

void setNewAttrs ( List pane, BitSet newMods ) {
	Vector list = pane.getSelections();

	for ( Enumeration e=list.elements(); e.hasMoreElements(); ) {
		Object o = e.nextElement();

		if ( o instanceof TypeDecl ) {
			((TypeDecl)o).Mods = (BitSet) newMods.clone();  // sep changes: must be unique object
		}
		else if ( o instanceof FieldDecl ) {
			((FieldDecl)o).Mods = (BitSet) newMods.clone();
		}
		pane.redrawObject( o);
	}

	PendingChanges++;
	updateChangeInfo();  
}

public void setReadOnly ( boolean readOnly ) {
	ReadOnlyMode = readOnly;
	View.TxtPane.setReadOnly( ReadOnlyMode);
	updateModeInfo();
}

public void setSelection ( List pane ) {
	if ( !checkUnaccepted() ) {
		pane.restoreLastSelection( false);
		return;
	}

	if ( pane == View.TypePane ) {
		Mode = SEL_TYPE;
		SelType = (TypeDecl) pane.getSelection();
		SelMthd = null;  SelData = null;

		View.DataPane.setContents( getDisplayDatas());
		View.MthdPane.setContents( getDisplayMethods());
		checkModDlg( MOD_SET_TYPE);
	}
	else if ( pane == View.ImptPane ) {
		if ( SelData != null ) {
			SelData = null;
			View.DataPane.clearSelections( false);
		}
		if ( SelMthd != null ) {
			SelMthd = null;
			View.MthdPane.clearSelections( false);
		}
		Mode = SEL_IMPT;
		SelImpt = (Import) pane.getSelection();
	}
	else {
		if ( SelImpt != null ) {
			SelImpt = null;
			View.ImptPane.clearSelections( false);
		}
		if ( pane == View.DataPane ) {
			if ( SelType == null && ! guessSelType() ) return;
			Mode = SEL_DATA;
			SelData = (Data) pane.getSelection();
			showDataUserMethods();
			checkModDlg( MOD_SET_DATA);
		}
		else if ( pane == View.MthdPane ) {
			if ( SelType == null && ! guessSelType() ) return;
			Mode = SEL_MTHD;
			SelMthd = (Method) pane.getSelection();
			checkModDlg( MOD_SET_MTHD);
		}
	}

	View.TxtPane.setContents( getDisplayText());
	if ( pane == View.MthdPane && MFId != null )
		View.TxtPane.findStringId( MFId);

	updateModeInfo();
	updateChangeInfo();
}

void showCompileOutput ( String txt ) {
	if ( OutputDlg == null )
		OutputDlg = new CompileOutputDlg( this); 

	OutputDlg.setTitle( "Compiler Output: " + CU.qualifiedClsName( CU.MainTypeName));
	// we don't need a full pathnamd here (dlg is tightly coupled to CUBrowser)
	txt = StringLib.replaceAll( txt, CU.FileName, null);
	OutputDlg.setContents( txt);
}

void showDataUserMethods () {
	if ( SelData == null ){
		if ( MFId != null ) {
			MFId = null;
			View.MthdPane.setContents( getDisplayMethods());
		}
	}
	else {
		MFId = SelData.Id;
		if ( Searcher == null )
			startSearcher( MFId);
	}
}

void startSearcher ( String pattern ) {
	Searcher = new UsageAnalyzer( this, pattern);

	Thread t = new Thread( Searcher );
	t.setPriority( Thread.MIN_PRIORITY);
	t.start();
}

void storeOnDisk ( boolean trackLines ) {
	CU.backup();
	View.ModeInfo.setContents( "..saving file");
	CU.save( trackLines);
	PendingChanges = 0;
	CU.notifyObservers( this);
	updateFileInfo();
	updateChangeInfo();
	View.ModeInfo.setContents( "file saved");
}

void toggleComment () {
	Colorizer.enable( CommentMode);
	CommentMode = !CommentMode;
	View.TxtPane.setContents( getDisplayText());
	updateModeInfo();
}

public void unregisterObserver() {
	CU.deleteObserver( this);
}

public void update ( Observable obs, Object arg ) {
	if ( obs == View.OsClose ){
		closeView();
	}
	else if ( obs == View.TxtPane.OsModified ) {
		updateChangeInfo();
	}
	else if ( obs == CU ) {
		if ( arg != this )
			updatePanes( arg);
	}
	else if ( obs == View.TypePane.OsSelection ) {
		setSelection( View.TypePane);
	}
	else if ( obs == View.ImptPane.OsSelection ) {
		setSelection( View.ImptPane);
	}
	else if ( obs == View.DataPane.OsSelection ) {
		setSelection( View.DataPane);
	}
	else if ( obs == View.MthdPane.OsSelection ) {
		setSelection( View.MthdPane);
	}
	else if ( obs == View.MthdPane.OsProcess ) {
		openMethodBrowser( SelMthd);
	}
	else if ( obs == View.TypePane.OsCommand ) {
		if ( "add".equals(arg) )            setMode( SEL_TYPE, this, TypeDecl.getTemplate());
		else if ( "remove".equals(arg) )     setMode( DEL_TYPE, SelType, null);
		else if ( "attributes".equals(arg) ) openModDlg( MOD_SET_TYPE);
		else if ( "rename".equals(arg) )     setMode( REP_TYPE, SelType, null);
		else if ( "select all".equals( arg)) View.TypePane.selectAll( true);
		else if ( "clear selection".equals(arg)) View.TypePane.clearSelections( true);
	}
	else if ( obs == View.ImptPane.OsCommand ) {
		if ( "add".equals(arg) )	           setMode( SEL_IMPT, this, Import.getTemplate());
		else if ( "remove".equals(arg) )	   setMode( DEL_IMPT, SelImpt, null);
		else if ( "replace".equals(arg) )	   setMode( REP_IMPT, this, null);
		else if ( "select all".equals( arg)) View.ImptPane.selectAll( true);
		else if ( "clear selection".equals(arg)) View.ImptPane.clearSelections( true);
	}
	else if ( obs == View.DataPane.OsCommand ) {
		if ("add".equals(arg))               setMode( SEL_DATA, this, Data.getTemplate());
		else if ("remove".equals(arg))       setMode( DEL_DATA, SelData, null);
		else if ("replace".equals( arg))     setMode( REP_DATA, this, null);
		else if ("attributes".equals(arg))   openModDlg( MOD_SET_DATA);
		else if ("filter mods".equals(arg))  openModDlg( MOD_FLT_DATA);
		else if ("clear filter".equals(arg)) clearDataFilter();
		else if ( "select all".equals( arg)) View.DataPane.selectAll( true);
		else if ( "clear selection".equals(arg)) View.DataPane.clearSelections( true);
	}
	else if ( obs == View.MthdPane.OsCommand ) {
		if ("add".equals(arg))               setMode( SEL_MTHD, this, Method.getTemplate());
		else if ("remove".equals(arg))       setMode( DEL_MTHD, SelMthd, null);
		else if ("replace".equals( arg))     setMode( REP_MTHD, this, null);
		else if ("attributes".equals(arg))   openModDlg( MOD_SET_MTHD);
		else if ("format".equals(arg))       formatMethod();
		else if ("filter mods".equals(arg))  openModDlg( MOD_FLT_MTHD);
		else if ("filter id".equals(arg)) {
			new Prompter( "id to look for", "filterId", this, View);
		}
		else if ("clear filter".equals(arg)) clearMethodFilter();
		else if ( "select all".equals(arg))  View.MthdPane.selectAll( true);
		else if ( "clear selection".equals(arg)) View.MthdPane.clearSelections( true);
		else if ( ((String)arg).startsWith( "tmpl.") ) addTemplate( arg);
	}
	else if ( obs == View.BMenu.OsCommand ) {
		if ("Clone.".equals(arg))            new CUBrowser( CU);
		else if ("Save.".equals(arg))        save();
		else if ("Compile.".equals(arg))     compile();
		else if ("Cmt/Src.".equals(arg))     toggleComment();
		else if ( "Goto..".equals(arg)) {
			new Prompter( "line number to go to", "goto", this, View);
			CU.refreshLines();  // do this while user looks at the prompt dialog
		}
		else if ("Package.".equals(arg))     setMode( SEL_PCKG, this, null);
		else if ("Format".equals(arg))       format();
		else if ("Print".equals(arg))        print();
		else if ("Print Text".equals(arg))   printText( null);
	}
	else if ( obs == View.AcceptBtn.OsAction ) {
		accept();
	}
	else if ( obs == Comp ) {
		compileReady();
	}
	else if ( OutputDlg != null ) {
		if ( obs == OutputDlg.OsClose )      OutputDlg = null;
		else if ( obs == OutputDlg.OsNext )  positionOnError( OutputDlg);
	}
	else if ( ModDlg != null ) {
		if ( obs == ModDlg.OsClose )         ModDlg = null;
		else if ( obs == ModDlg.OsSet )      setModifier();
	}

	else if ( obs instanceof Prompter ) {
		if ( "goto".equals( arg) )
			positionOnLine( ((Prompter)obs).getContentsAsInt(), null);
		else if ( "filterId".equals( arg) ) {
			MFId = ((Prompter)obs).getContents();
			if ( Searcher == null ) startSearcher( MFId);
		}
		else if ( "printText".equals( arg) )
			printText( ((Prompter)obs).getContents());
	}
}

void updateChangeInfo () {
	String info = Integer.toString( PendingChanges);
	if ( View.TxtPane.isModified() )
		info += " +";

	View.ChangeInfo.setContents( info);
}

void updateFileInfo () {
	View.FileInfo.setContents( CU.fileInfoString(false));
}

void updateModeInfo () {
	String info = "";
	String pre  = ReadOnlyMode ? "Show " : "Edit ";
	String post = CommentMode ? " Comment" : " Source";
	//if ( Comp != null )
	if ( false )
		info = "Compiling..";
	else {
		switch ( Mode ) {
		case SEL_PCKG: info = pre+"Package"; break;
		case SEL_TYPE: info = pre+"Type"+post; break;
		case SEL_IMPT: info = pre+"Import"+post; break;
		case SEL_DATA: info = pre+"Variable"+post; break;
		case SEL_MTHD: info = pre+"Method"+post; break;
		case DEL_TYPE: info = "Delete all selected Types"; break;
		case DEL_IMPT: info = "Delete all selected Imports"; break;
		case DEL_DATA: info = "Delete all selected Datas"; break;
		case DEL_MTHD: info = "Delete all selected Methods"; break;
		case REP_TYPE: info = "Rename selected Type"; break;
		case REP_IMPT: info = "Replace all selected Imports"; break;
		case REP_DATA: info = "Replace all selected Datas"; break;
		case REP_MTHD: info = "Replace all selected Methods"; break;
		case SAV_FILE: info = "Save to File"; break;
		}
	}

	info = addFilterDescription( info);
	View.ModeInfo.setForeground( Awt.StaticForeClr );
	View.ModeInfo.setContents( info);
}

void updatePanes ( Object sourceOfChange ) {
	TypeDecl t = SelType;
	Data d = SelData;
	Method m = SelMthd;
	Point p = View.TxtPane.getCursorIdx();

	View.TypePane.setContents( CU.TypeDecls);
	View.ImptPane.setContents( CU.Imports);
	updateTitle();

	View.TypePane.setSelection( t, true);
	View.DataPane.setSelection( d, true);
	if ( m != null ) {
		View.MthdPane.setSelection( m, true);
		View.TxtPane.setCursorIdx( p.x, p.y);
	}

	if ( sourceOfChange instanceof CUBrowser ) {
		PendingChanges = ((CUBrowser)sourceOfChange).PendingChanges;
	}

	updateFileInfo();
	updateChangeInfo();
	checkIndentation();
}

void updateTitle () {
	View.setTitle( CU.qualifiedClsName( CU.MainTypeName));
}
}

class UsageAnalyzer
  implements Runnable
{
	CUBrowser CB;
	String SearchId;
	TypeDecl SelType;

UsageAnalyzer ( CUBrowser cb, String searchId ) {
	CB = cb;
	SearchId = searchId;
}

public void run () {

	if ( CB == null || CB.SelType == null ) {
		CB.Searcher = null;
		return;
	}

	int n = CB.SelType.Methods.size();
	Vector list = new Vector( n);
	Method m;
	SelType = CB.SelType;

	for ( int i=0; i<n; i++ ) {
		Thread.yield();
		if ( CB.SelType == null || CB.MFId == null ) {
			CB.Searcher = null;
			return;
		}
		if ( CB.SelType != SelType || !CB.MFId.equals( SearchId ) ) {
			SelType = CB.SelType;
			SearchId = CB.MFId;
			i = 0;
			n = SelType.Methods.size();
			list = new Vector( n);
			continue;
		}

		m = (Method)SelType.Methods.elementAt(i);
		if ( CB.MFMods != null && !m.hasAnyMods( CB.MFMods) ) continue;
		if ( CB.MFCat != null && !m.hasCategory( CB.MFCat) ) continue;
		if ( !m.usesData( SearchId ) ) continue;

		list.addElement( m);
	}

	CB.View.MthdPane.setContents( list);
	CB.updateModeInfo();
	CB.Searcher = null;
}
}
