// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: menubar.cc,v 1.13 1998/06/21 03:25:04 jgg Exp $
/* ######################################################################

   Menu Bar - Top menu bar
   Menu Popup - Floating vertical menu

   The only quirk in the implementation here is that to get the 
   proper focus effects the popup menu uses the force focus bit. The popup
   must exist outside the clipping region of it's owner in almost all 
   cases so it cannot remain a child widget while visible.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#include <deity/menubar.h>
#include <deity/utils.h>
#include <system.h>
									/*}}}*/

// MenuBar::MenuBar - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
MenuBar::MenuBar(Widget *Parent) : BasicWidget(Parent), LastFocus(0)
{
   LastWidget = this;
   Flag(Region | AutoExtent);
   if (TextGC::GC != 0)
   {
      BorderWidth(0);
      BorderX = 1;
   }
   ExtentMode(ExtentAlways,ExtentAlways);
}
									/*}}}*/
// MenuBar::IdealSize - Computes the best size of the menu bar		/*{{{*/
// ---------------------------------------------------------------------
/* The best size is just large enough to encompass the y direction of the
   children and the maximum width of the parent. */
Point MenuBar::IdealSize()
{
   Point Size = ChildrenExtent(this);
   if (Parent != 0)
      Size.x = Parent->Pos.w;
   return Size;
}
									/*}}}*/
// MenuBar::Layout - Re-position the children widgets			/*{{{*/
// ---------------------------------------------------------------------
/* This is called to position the children inside the widget, we 
   do the layout as a simple horizontal positioning. */
void MenuBar::Layout()
{
   HorizontalLayout(this,1);
   for (iterator I = children(); I.end() == false; I++)
      I->Resize(Rect(I->Pos.x,0,I->Pos.w,Size().h - 2*BorderY - 2*iMargins.y));
}
									/*}}}*/
// MenuBar::Key - Handle keystrokes					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool MenuBar::Key(const KeyEvent &Key,Widget *)
{
   if (Key.Extended == KeyEvent::Left)
   {
      FocusLast();
      return true;
   }
   
   if (Key.Extended == KeyEvent::Right)
   {
      FocusNext();
      return true;
   }   
   
   return false;
}
									/*}}}*/
// MenuBar::FocusGained - Store the previous focus			/*{{{*/
// ---------------------------------------------------------------------
/* */
void MenuBar::FocusGained(Widget *OldFocus)
{
   if (IsFlag(ForceFocus) == false && OldFocus != 0 && OldFocus->Realized == true)
   {
      LastFocus = OldFocus;
      Flag(0,MenuDrop);
   }   
}
									/*}}}*/
// MenuBar::FocusRestore - Return focus to the last focus		/*{{{*/
// ---------------------------------------------------------------------
/* MenuItem widgets will call this when they are activated so as to remove 
   the menu. */
void MenuBar::FocusRestore()
{
   if (LastFocus != 0)
      LastFocus->GiveFocus();
   LastFocus = 0;
}
									/*}}}*/
// MenuBar::Realize - Widget was realized				/*{{{*/
// ---------------------------------------------------------------------
/* We grab the mouse so we can cancel the menu if there is an out of widget
   click. */
void MenuBar::Realize()
{
   Widget::Realize();
   
   GrabMouse(GrabChildren | GrabPermanent | GrabWhenFocused);
}
									/*}}}*/
// MenuBar::Mouse - Mouse handler					/*{{{*/
// ---------------------------------------------------------------------
/* If we get a mouse event then more then we should restore focus, this
   usualy means that someone clicked on an empty section of the menu bar */
void MenuBar::Mouse(const MouseEvent &Event)
{
   if (Event.IsUp() == true)
      FocusRestore();
}
									/*}}}*/

// MenuPopup::MenuPopup - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
MenuPopup::MenuPopup(Widget *Parent) : BasicWidget(Parent)
{
   Flag(DelayRealize | Region);
   LastWidget = this;
   OldParent = 0;

   // Add some more space when in graphics mode
   if (GraphicGC::GC != 0)
      BorderX = BorderY = 3;
}
									/*}}}*/
// MenuPopup::IdealSize - Computes the ideal size of the popup		/*{{{*/
// ---------------------------------------------------------------------
/* This computes the maximum width and then the height for all the widgets
   stacked top to bottom. */
Point MenuPopup::IdealSize()
{   
   Point Size = ChildrenExtent(this);
   Size.y = 2*BorderY + 2*iMargins.y;
   for (iterator I = children(); I.end() == false; I++)
      Size.y += I->Pos.h;
   
   return Size;   
}
									/*}}}*/
// MenuPopup::Layout - Position the children in the popup		/*{{{*/
// ---------------------------------------------------------------------
/* */
void MenuPopup::Layout()
{
   VerticalLayout(this,0);
}
									/*}}}*/
// MenuPopup::Key - Handle keystrokes					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool MenuPopup::Key(const KeyEvent &Key,Widget *Source)
{
   if (Key.Extended == KeyEvent::Up)
   {
      FocusLast();
      return true;
   }
   
   if (Key.Extended == KeyEvent::Down)
   {
      FocusNext();
      return true;
   }   
   
   if (Key.Key == '-')
   {
      UnRealize();
      Parent->FocusLost(this);
      Parent->GiveFocus();
      return true;
   }
   
   /* Reroute keystrokes through the old parent
      We need to be a bit evil here to get correct support for hot keys,
      we invoke keyboard routines on the children and then eat the key
      after passing it up */
   for (Widget *I = Child; I != 0; I = I->Brother)
   {
      if (I->IsFlag(ParentKeys) == true && I != FocusWidget && 
	  I->Realized == true)
	 if (I->Key(Key,Source) == true)
	    return true;
   }      
   
   if (OldParent != 0)
      OldParent->RouteKey(Key);
   
   return true;
}
									/*}}}*/
// MenuPopup::Realize - Reparent the widget				/*{{{*/
// ---------------------------------------------------------------------
/* We parent the widget to the root widget so that it will draw properly. */
void MenuPopup::Realize()
{
   if (Parent == 0 || Realized == true)
      return;
   
   // Reposition us and force focus on the parent widgets
   Widget *I = Parent;
   for (; I->Parent != 0; I = I->Parent)
   {
      Pos.x += I->Pos.x + I->BorderX;
      Pos.y += I->Pos.y + I->BorderY;
      I->Flag(ForceFocus);
   }

   OldParent = Parent;
   ChangeParent(I);

   Widget::Realize();
   if (Child != 0)
      Child->GiveFocus();
   else
      GiveFocus();

   GrabMouse(GrabChildren | GrabPermanent | GrabWhenFocused | 
	     GrabAllGrab,true);
}
									/*}}}*/
// MenuPopup::UnRealize - Parent the widget back			/*{{{*/
// ---------------------------------------------------------------------
/* */
void MenuPopup::UnRealize()
{
   if (Realized == false)
      return;

   // Clear forced focus
   for (Widget *I = OldParent; I->Parent != 0; I = I->Parent)
   {
      I->Flag(0,ForceFocus);
      if (I->IsFlag(ShowFocus) == true)
	 I->Draw();
   }

   Widget::UnRealize();
   
   ChangeParent(OldParent);
   OldParent = 0;
}
									/*}}}*/
// MenuPopup::FocusLost - Unrealize the widget				/*{{{*/
// ---------------------------------------------------------------------
/* */
void MenuPopup::FocusLost(Widget *)
{
   UnRealize();
}
									/*}}}*/
// MenuPopup::FocusRestore - Restores focus				/*{{{*/
// ---------------------------------------------------------------------
/* Just bounces this to the original parent */
void MenuPopup::FocusRestore()
{
   for (Widget *I = OldParent; I != 0; I = I->Parent)
      I->FocusRestore();
}
									/*}}}*/
// MenuPopup::Mouse - Mouse handler					/*{{{*/
// ---------------------------------------------------------------------
/* If there is a mouse event this unrealizes the widget this is part
   of the grab support. */
void MenuPopup::Mouse(const MouseEvent &Event)
{
   if (Event.IsUp() == true)
   {
      UnRealize();
      FocusRestore();
   }
}
									/*}}}*/
