// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: pointer.h,v 1.8 1997/08/18 20:49:34 tom Exp $
/* ######################################################################
   
   Smart Pointer Class - Reference counted pointer implementation

   This is a safe pointer type.  It should eliminate memory leaks because the
   destructor will automatically deallocate memory when no more pointers to
   the allocated memory exist. Be wary of creating loops of smartpointers, 
   if a linked ring is generated (circular linked list) then it is possible 
   to create a memory leak.
   
   Passing of smart pointers by value is fast and very safe. The pointers
   are not copy on write so they closely emulate normal pointers.

   Copying smart-pointers into normal pointers could cause problems.  If doing
   so is necessary, make sure the regular pointer goes out of scope before the
   smart-pointer.

   Additionally, this class will allow up to 16 individual flags to be set on
   each memory space. A "memory modified" flag is defined here, but it is set
   automatically _only_ when a 'Malloc' or 'New' is done! The routine that
   is making the changes must set it manually.
   
   This source is placed in the Public Domain, do with it what you will
   It was originally written by Brian C. White.
   
   ##################################################################### */
									/*}}}*/
// Header section: pkglib
#ifndef PKGLIB_POINTER_H
#define PKGLIB_POINTER_H

#include <stddef.h>
#include <new.h>
#include <defalloc.h>    // STL

/* Pointer flag definitions -- these occupy the flag bits that are
   available under a smart-pointer.  These flags are accessed via the
   smart-pointer member functions 'SetFlag', 'ClrFlag', and 'ChkFlag'. */
#define	PTRF_CHANGED	15		// Standard "memory modified" flag
#define PTRF_INVALID	14		// Memory is now invalid

// Non template base class to help reduce instantiation overhead
class Pointer 
{
   protected:
   void *Ptr;			// pointer (cast appropriately when needed)
   
   // Allocation/Deallocation
   void	arraySize(unsigned int i);
   static void *allocMem(size_t size, bool zero=true);	
   static void freeMem(void *);

   public:
   
   // Comparison operators
   Pointer &operator =(const Pointer& rhs);
   bool	operator ==(const Pointer& rhs) const;
   bool	operator !=(const Pointer& rhs) const;
   bool	operator !(void) const {return Ptr != 0;};

   // Cast to void operators
   operator const void*() const	{return Ptr;};
   operator void*() {return(Ptr);};
   
   // Flags
   bool	isValid(void) const {return Ptr != 0;};
   bool	isNull(void) const {return Ptr == 0;};
   bool	isDirty(void) const {return chkFlag(PTRF_CHANGED);};
   void	clean(void) {clrFlag(PTRF_CHANGED);};
   void	dirty(void) {setFlag(PTRF_CHANGED);};
   void	setFlag(int f);
   void	clrFlag(int f);
   bool	chkFlag(int f) const;

   // Allocation
   unsigned int	arraySize(void) const;
   int	linkCount(void)	const;
   void	malloc(size_t size,const void *memory = 0);
   void free(void);
   void	resize(size_t oldsize,size_t newsize);  

   Pointer(void);
   Pointer(void *mem);
   Pointer(const Pointer &rhs);
   ~Pointer(void);
};

/* Internal Smart-Pointer template.

   This is required because some compilers do not yet implement the new
   template spec so things like operator ->() are not possible with non 
   structure types. */
template <class Type> class IPtr : public Pointer 
{
   // Make these private to prevent accidental use
   void malloc(size_t size) {Pointer::malloc(size);};
   void free(void) {Pointer::free();};
   void	resize(size_t o, size_t n) {Pointer::resize(o,n);};
   
   // Dont allow assignment from void
   IPtr(void*);				
   
   public:
   
   // Access operators, * and []
   const Type &operator *() const {return *(Type*)Ptr;};
   const Type &operator [](unsigned int i) const {return ((Type*)Ptr)[i];};
   Type &operator *() {return *(Type*)Ptr;};
   Type &operator [](unsigned int i) {return ((Type*)Ptr)[i];};

   // Comparison
   bool	operator ==(const IPtr& rhs) const {return Ptr == rhs.Ptr;};
   bool	operator !=(const IPtr& rhs) const {return Ptr != rhs.Ptr;};
   
   // Casting
   operator const Type *() const {return  (Type*)(void*)Ptr;};
   operator Type *() {return (Type*)(void*)Ptr;};
   
   IPtr<Type> &New(unsigned int num = 1);
   void	Delete(void);
   void	resize(size_t newsize);

   IPtr(void) : Pointer() {};
   IPtr(Type* rhs) : Pointer(rhs) {};
   IPtr(const IPtr& rhs) : Pointer(rhs) {};
   ~IPtr(void) {Delete();};
};

/* Smart-Pointer Template

   This overloads -> to allow structure de-references. This smart
   pointer does not handle arrays. */
template <class Type> class SPtr : public Pointer 
{
   // Make these private to prevent accidental use
   void	malloc(size_t size) {Pointer::malloc(size);};
   void free(void) {Pointer::free();};
   void	resize(size_t o, size_t n) {Pointer::resize(o,n);};

   // dont allow assignment from void
//   SPtr(void*); 
   
   public:
   
   // Dereference operators
   const Type& operator *(void) const {return *(Type*)Ptr;};
   const Type* operator->(void) const {return (Type*)Ptr;};
   Type& operator* (void) {return *(Type*)Ptr;};
   Type* operator->(void) {return (Type*)Ptr;};
   
   // Comparison
   bool operator ==(const SPtr& rhs) const {return Ptr == rhs.Ptr;};
   bool operator !=(const SPtr& rhs) const {return Ptr != rhs.Ptr;};

   // Cast
   operator const Type*() const {return (Type*) (void*)Ptr;};
   operator Type*() {return (Type*)(void*)Ptr;};

   // Assignment
   SPtr<Type> &operator =(const SPtr<Type>& rhs);

   // Creation
   SPtr<Type> &New();
   SPtr<Type> &New(const Type &rhs);
//   SPtr<Type> &New(void** buf);       Appears to be missing
   void Delete();
   
   SPtr() : Pointer() {};
   SPtr(Type* rhs) : Pointer(rhs) {};
   SPtr(const SPtr& rhs) : Pointer(rhs) {};
   ~SPtr() {Delete();}
   
   // These are defined, but not implemented by default.  Macros to produce
   // the required code can be found in "Ptrfuncs.h".
   void newSlots(unsigned int number);
   unsigned int slotCount(void);
   void resizeSlots(unsigned int number);
   unsigned int compactSlots(void);
};

/* Constant Smart Pointer
   Forces all dereferences to go through the change() member which sets the
   changed bit */
template <class Type> class CSPtr : public SPtr<Type> 
{  
   // dont allow assignment from void
   CSPtr(void*);

   public:

   // Constant dereference
   const Type &operator *(void) const {return *(Type*)Ptr;};
   const Type *operator->(void) const {return (Type*)Ptr;};
   operator const Type*() const {return (Type*)(void*)Ptr;};

   // Dynamic dereference
   Type* change(void)   
   { 
      setFlag(PTRF_CHANGED);
      return (Type*)Ptr;
   }

   // Construct/destruct
   CSPtr(void) : SPtr<Type>() {}
   CSPtr(Type* rhs) : SPtr<Type>(rhs)  {}
   CSPtr(const SPtr<Type>& rhs) : SPtr<Type>(rhs) {}
};

// Bring in template source for CSPtr and SPtr
#include <pkglib/pointer.icc>

#endif
