// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: mmap.cc,v 1.6 1998/01/16 07:02:30 jgg Exp $
/* ######################################################################
   
   MMap Class - Provides 'real' mmap or a faked mmap using read().

   MMap cover class.

   Some broken versions of glibc2 (libc6) have a broken definition
   of mmap that accepts a char * -- all other systems (and libc5) use
   void *. We can't safely do anything here that would be portable, so
   libc6 generates warnings -- which should be errors, g++ isn't properly
   strict.
   
   The configure test notes that some OS's have broken private mmap's
   so on those OS's we can't use mmap. This means we have to use
   configure to test mmap and can't rely on the POSIX
   _POSIX_MAPPED_FILES test.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#include <sys/mman.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

#include <pkglib/error.h>
#include <pkglib/mmap.h>
   									/*}}}*/

// MMap::MMap - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
MMap::MMap() : Fd(-1), Public(false), ReadOnly(false), iSize(0), 
               CloseFd(false), Base(0)
{
}
									/*}}}*/
// MMap::MMap - Constructor from FD					/*{{{*/
// ---------------------------------------------------------------------
/* */
MMap::MMap(int Fd, bool Public, bool ReadOnly) : Fd(Fd), Public(Public),
               ReadOnly(ReadOnly), iSize(0), CloseFd(false), Base(0)
{
   Map();
}
									/*}}}*/
// MMap::MMap - Constructor from Filename				/*{{{*/
// ---------------------------------------------------------------------
/* */
MMap::MMap(string File, bool Public, bool ReadOnly) : Fd(-1), Public(Public),
               ReadOnly(ReadOnly), iSize(0), CloseFd(false), Base(0)
{
   Open(File,Public,ReadOnly);
}
									/*}}}*/
// MMap::~MMap - Destructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
MMap::~MMap()
{
   Close();
}
									/*}}}*/
// MMap::Map - Perform the mapping					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool MMap::Map()
{
   struct stat StatBuf;
   if (fstat(Fd,&StatBuf) != 0)
      return _error->Errno("fstat","Unable to determine size of file");
   iSize = StatBuf.st_size;
   
   // Set the permissions.
   int Prot = PROT_READ;
   int Map = MAP_SHARED;
   if (ReadOnly == false)
      Prot |= PROT_WRITE;
   if (Public == false)
      Map = MAP_PRIVATE;
   
   // Map it.
   Base = mmap(0,StatBuf.st_size,Prot,Map,Fd,0);
   if (Base == (void *)-1)
      return _error->Errno("mmap","Couldn't mmap ./cache of %u bytes",iSize);

   return true;
}
									/*}}}*/
// MMap::Open - Open from a Fd						/*{{{*/
// ---------------------------------------------------------------------
/* */
bool MMap::Open(int Fd, bool Public, bool ReadOnly)
{
   this->Fd = Fd;
   this->Public = Public;
   this->ReadOnly = ReadOnly;
   return Map();
}
									/*}}}*/
// MMap::Open - Open from a filename					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool MMap::Open(string File,bool Public, bool ReadOnly)
{
   this->Public = Public;
   this->ReadOnly = ReadOnly;

   if (ReadOnly == true)
      Fd = open(File.c_str(),O_RDONLY);
   else
      Fd = open(File.c_str(),O_RDWR);
   if (Fd < 0)
      return _error->Errno("open","Can't open %s",File.c_str());

   if (Map() == false)
      return _error->Warning("Errors apply to the file %s",File.c_str());
   return true;
}
									/*}}}*/
// MMap::Close - Close the map						/*{{{*/
// ---------------------------------------------------------------------
/* */
bool MMap::Close()
{
   if (Fd == -1)
      return true;

   if (Public == true && ReadOnly == false)
   {
      if (msync(Base,iSize,MS_SYNC) != 0)
	 return _error->Error("msync","Unable to write mmap");
   }
   
   if (munmap(Base,iSize) != 0)
      _error->Warning("Unable to munmap");
   
   if (CloseFd == true)
      close(Fd);

   Fd = -1;
   iSize = 0;
   return true;
}
									/*}}}*/
// MMap::Sync - Syncronize the map with the disk			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool MMap::Sync()
{
   if (msync(Base,iSize,MS_SYNC) != 0)
      return _error->Error("msync","Unable to write mmap");
   return true;
}
									/*}}}*/

