/*
 * MAILTO for Arena. Contributed by Gregory L. Kaup <glkaup@kaup.com>.
 */
/* Arena mailto.c using mailto.h and mailto_i.h
 * 
 * Routines to implement mailto: tags for Arena.
 *
 * First things first!
 *   - Edit arena.h.in and set ARENA_MailTo_MAILER_UTILITY 
 *   - Then verify that ARENA_MailTo_MAILER_UTILITY_PATH is correct for your
 *     system.
 *   - When you click a mailto: URL, an html form will appear.  It assumes
 *     your email address, supplies to To: address (which can have more than
 *     one), and supplies the document URL as the subject?  By the way, you
 *     MUST format your own message.  TEXTAREA does not yet help you here.
 *   - Good Luck!
 *
 * Basic outline
 *   - Catch the click (button up) of the tag containing the mailto: 
 *   - Create an input form and open it!
 *   - Then, catch the click of the "submit" button... Do NOT let the standard
 *     form submit happen.
 *   - Fork, execve and waitpid on specified mail handler.
 *     sendmail and mail need the pipe setup to stdin.
 *   - DONE! Well... for now... anyway!
 *
 * History:
 *  05-08-97
 *    -Changed `Cancel' button label to `Reset', as it's more correct.
 *  24-06-97
 *    -Use special form submit method `MAILTO' instead of `action="MaIlTo://"'
 *     to catch `mailto:' form submission.
 *    -Transit to method using memory buffers instead of temporary files.
 *  30-05-97
 *    -Original submitted to Qing Long for looksee!
 *  24-05-97
 *    -Fix the reloading of the "previous" document after sending the mail.
 *    -Control access to the mailto: form with MTState.
 *    -Removed mailto script support.
 *  22-05-97
 *    -Tested pipe for stdin with mail and sendmail and mailto scripts.
 *  20-05-97
 *    -Setup temp file as input to mh post utility.  Worked well, but
 *     seemed unecessary since post only communicated with sendmail
 *     anyway, and sendmail would allow use of pipe for stdin.
 *    -Removed the temp file feeding of "all" mailers, and just setup for
 *     piping via stdout|stdin.
 *  19-05-97
 *    -Original working version of mailto with fork, execve, waitpid
 * 
 * Known bugs: PLENTY!
 *  -- Need to come up with the "correct" email address... I cannot get
 *     mail at <web@ranch.kaup.com> only at <glkaup@kaup.com> or 
 *     <glkaup@teal.csn.net>.  If I'm logged in as <web>, I have to edit
 *     the From: string every time! (pop only account)
 *  -- NEED to work on the formatting of the "Message" string.  There are 
 *     several settings for "wrap" which are ALL ignored at this time?!
 *  -- Reset button does not do nearly enough!
 *     Some of the same problems as the BACK button.
 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <signal.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>

#include "arena.h"
#include "arena_release.h"
#include "bridge.h"
#ifdef ARENA_DEBUG
#  include "debug.h"
#endif
#include "defs.h"
#include "display.h"
#include "forms.h"
#include "html.h"
#include "mailto.h"
#include "mailto_i.h"
#include "main.h"
#include "status.h"
#include "types.h"
#include "util.h"
#include "x11.h"

#include "HTLib.h"
#include "HTInet.h"


/* Interal static data specififications
 */
#define MTS_STATIC 0
#define MTS_FORM   1
#define MTS_SEND   2
#define MTS_CANCEL 4
static int MTState = MTS_STATIC;


static char sAddressFrom[MAX_PATH_LENGTH+1];
static char sAddressTo[MAX_PATH_LENGTH+1];
static char sSubject[MAX_PATH_LENGTH+1];

/* List of form fields that MUST exist, and have defined values.
 * DO NOT rearrange these.  Yes, I expect them in this order!
 */
static char *fn[] = {"From", "To", "Subject", "Message"};
static int  fnt[] = {TEXTFIELD, TEXTFIELD, TEXTFIELD, TEXTAREA};
static char *fv[] = {sAddressFrom, sAddressTo, sSubject, NULL};
static int    nfn = sizeof(fn)/sizeof(fn[0]);

/* List of optional output field
 */
static char *op[] = {"Return-Receipt-To", "X-Priority"};
static int  opt[] = {CHECKBOX, CHECKBOX};
static char *ov[] = {sAddressFrom, "2 (High)"};
static int    nop = sizeof(op)/sizeof(op[0]);

/* Internal declarations
 */
void MTOutputMessage(Form* form, FILE *fp);



/* This routine is called by OpenDoc in html.c instead of libGetDocument
 * whenever a mailto: URL is clicked.  Here, we will create the .html input
 * form, and display it
 */
void MTGetMailtoInputForm(char *to)
{
 char *email;
 char Arena_Mailto_HTML_Form_action_format[] =
                                   ARENA_MailTo_HTML_Form_Action_format;
 char Arena_Mailto_HTML_Format[] = ARENA_MailTo_HTML_Form;
 char* Arena_Mailto_Action_Buffer;
 int Arena_Mailto_Action_Buffer_length =
                            Arena_StrLen(Arena_Mailto_HTML_Form_action_format) + 1 +
                                   ARENA_MailTo_HTML_Form_strings_MAXlength;
 char* Arena_Mailto_Buffer;
 int Arena_Mailto_Buffer_length =  Arena_StrLen(Arena_Mailto_HTML_Format) +
                                   ARENA_MailTo_HTML_Form_strings_MAXlength *
                                  (ARENA_MailTo_HTML_Form_strings_NUMBER - 1) +
                                   Arena_Mailto_Action_Buffer_length;
#ifdef ARENA_DEBUG
 char *Iam = "MTGetMailToInputForm";
#endif


/* See if already busy!
 * Allow procedure if just at the FORM state!
 * Otherwise, we are trying to SEND, and here again?!
 */
 if (MTState > MTS_STATIC)
   {
     if (MTState > MTS_FORM)
       {
	 Warn("mailto: %s", _("Currently busy"));
	 Beep();
	 return;
       }
     MTState = MTS_STATIC;
   }

 ShowBusy();

 Arena_Mailto_Action_Buffer = (char*)Arena_CAlloc(Arena_Mailto_Action_Buffer_length,
						  sizeof(char),
						  False);
 if (Arena_Mailto_Action_Buffer)
   {
#ifdef HAVE_SNPRINTF
    snprintf(Arena_Mailto_Action_Buffer, Arena_Mailto_Action_Buffer_length - 1,
	     Arena_Mailto_HTML_Form_action_format,
	     ARENA_MailTo_HTML_Form_strings_MAXlength, &to[7]);
# else
    sprintf(Arena_Mailto_Action_Buffer,
	    Arena_Mailto_HTML_Form_action_format,
	    ARENA_MailTo_HTML_Form_strings_MAXlength, &to[7]);
#endif

    Arena_Mailto_Buffer = (char*)Arena_CAlloc(Arena_Mailto_Buffer_length,
					      sizeof(char),
					      False);
    if (Arena_Mailto_Buffer)
      {
       Doc* theDoc;

       email = HTUserProfile_email(UserProfile);
#ifdef HAVE_SNPRINTF
       snprintf(Arena_Mailto_Buffer, Arena_Mailto_Buffer_length - 1,
		Arena_Mailto_HTML_Format,
		ARENA_MailTo_HTML_Form_strings_MAXlength, to,
		Arena_Mailto_Action_Buffer_length, Arena_Mailto_Action_Buffer,
		ARENA_MailTo_HTML_Form_strings_MAXlength, email,
		ARENA_MailTo_HTML_Form_strings_MAXlength,
		ARENA_MailTo_HTML_Form_strings_MAXlength, &to[7],
		ARENA_MailTo_HTML_Form_strings_MAXlength,
		ARENA_MailTo_HTML_Form_strings_MAXlength, CurrentDoc->url,
		ARENA_MailTo_HTML_Form_strings_MAXlength);
# else
       sprintf(Arena_Mailto_Buffer,
	       Arena_Mailto_HTML_Format,
	       ARENA_MailTo_HTML_Form_strings_MAXlength, to,
	       Arena_Mailto_Action_Buffer_length, Arena_Mailto_Action_Buffer,
	       ARENA_MailTo_HTML_Form_strings_MAXlength, email,
	       ARENA_MailTo_HTML_Form_strings_MAXlength,
	       ARENA_MailTo_HTML_Form_strings_MAXlength, &to[7],
	       ARENA_MailTo_HTML_Form_strings_MAXlength,
	       ARENA_MailTo_HTML_Form_strings_MAXlength, CurrentDoc->url,
	       ARENA_MailTo_HTML_Form_strings_MAXlength);
#endif
       HideBusy();

       /* glk 17.7.97
	* Changed reload to True... Seems to fix a problem with usage of
	* BACK, FORWARD and History.  If you use one of these and get a
	* failure to load "mailto:blah" and the try to use a regular
	* link to "mailto:blah", we get the dreaded "mailto: not supported"!
	* I'm not sure why this fix works!!!
	*/
       theDoc = pseudoLoadAnchor(to, Arena_StrLen(to),
				 NULL, 0,
				 Arena_Mailto_Buffer,
				 Arena_Mailto_Buffer_length,
				 html_atom,
				 True, False, False, True, False, True,
				 NULL);
       Free(Arena_Mailto_Buffer);    /* pseudoLoadAnchor() has it's own copy */

       if (theDoc)
	 {
	 /* In the doc there should be the only form,
	  * for which we should set access method to `MAILTO'.
	  */
	  Form* theForm;
	  char* theAction;

	  theAction = strstr(theDoc->content_buffer,
			     Arena_Mailto_Action_Buffer);
	  Free(Arena_Mailto_Action_Buffer);

	  if (theAction)
	    {
	     if ((theForm = FindForm(GET, theAction)))
	       {
		theForm->method = MAILTO;
		MTState = MTS_FORM;
		return;
	       }
#ifdef ARENA_DEBUG
	      else
	       {
		if (MAILTO_TRACE)
		  Arena_TracePrint(Iam,
				   " Failed to find the form in the doc.\n");
	       }
#endif
	    }
#ifdef ARENA_DEBUG
	   else
	    {
	     if (MAILTO_TRACE)
	       Arena_TracePrint(Iam,
				" Failed to find the action in the doc.\n");
	    }
#endif
	 }
#ifdef ARENA_DEBUG
        else
	 {
	  if (MAILTO_TRACE)
	    Arena_TracePrint(Iam, " Failed to pseudoload document.\n");
	 }
#endif
      }
#ifdef ARENA_DEBUG
     else
      {
       if (MAILTO_TRACE)
	 Arena_TracePrint(Iam, " Failed allocate document buffer.\n");
      }
#endif
   }
#ifdef ARENA_DEBUG
  else
   {
    if (MAILTO_TRACE)
      Arena_TracePrint(Iam, " Failed allocate form action buffer.\n");
   }
#endif

 Warn("mailto: %s", _("Is not supported"));
 Beep();
 HideBusy();
}


/* Routine to pipe, fork, execve, pump data, and waitpid!
 * To add more mail handlers:
 *  1) modify arena.h.in to flag your new handler.
 *  2) add to the #if #elif below to set up the arguments.
 *  Should work fine... IFF your handler wants its input from stdin.
 *
 *  The mh post or send utilities worked great execpt they HAD to HAVE
 *  a file for input, and I could find no way to get it to be stdin.  I took
 *  out all of the "make a temp file" and feed it to the mail handler stuff!
 *
 *  The mailto program works ok, but PATH must include /bin and /usr/bin.
 *  It is a lot of scripts, and I decided to leave it out.
 */
int MTSendIt(Form* theForm)
{
 int istat, fd[2];
 pid_t childpid;
 FILE *fo;

#ifdef ARENA_DEBUG
 char *Iam = "MTSendIt";
# else
 int nfd;
#endif

#if (ARENA_MailTo_MAILER_UTILITY == ARENA_MailTo_SENDMAIL)
 char *args[] = { ARENA_MailTo_MAILER_UTILITY_PATH,
		  "-t",
		  sAddressTo,
		  NULL
 };
#elif (ARENA_MailTo_MAILER_UTILITY == ARENA_MailTo_MAIL)
 char *args[] = { ARENA_MailTo_MAILER_UTILITY_PATH,
		  "-n",
		  "-N",
		  sAddressTo,
		  NULL
 };
 #endif

/* Give the child NO environment controls what so ever!
 * Must control PATH and IFS completely.
 * Security! Security! OH... wherefore art thou?
 */
 char *envp[] = { "PATH=::", "IFS=' '", NULL };

#ifdef ARENA_DEBUG
 if (MAILTO_TRACE && VERBOSE_TRACE)
   {
    Arena_TracePrint(Iam, " Mailer is %s\n", ARENA_MailTo_MAILER_UTILITY_PATH);
    Arena_TracePrint(Iam, " From: %s\n", sAddressFrom);
    Arena_TracePrint(Iam, " To:   %s\n", sAddressTo);
    Arena_TracePrint(Iam, " Subject: %s\n", sSubject);
   }
#endif

 if (access(ARENA_MailTo_MAILER_UTILITY_PATH, X_OK) == -1)
   {
     Warn("mailto: %s", _("Is not supported"));
     Beep();
     return EXIT_FAILURE;
   }

/* Set up the fork for the child and let it do the work for us!
 * TOTAL control over the arguments AND the environment.
 */ 
#ifdef ARENA_DEBUG
 if (MAILTO_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " Trying to send %s\n", sAddressTo);
#endif

 if (pipe(fd) < 0)
   {
#ifdef ARENA_DEBUG
     if (MAILTO_TRACE)
       Arena_TracePrint(Iam, " Pipe failure sending %s\n", sAddressTo);
#endif
     Warn("mailto: %s", _("Is not supported"));
     Beep();
     return EXIT_FAILURE;
   }

/* Try creating the child process!  Just return with failure if cannot
 * get the process created!
 */
 if ((childpid = fork()) < 0)
   {
#ifdef ARENA_DEBUG
     if (MAILTO_TRACE)
       Arena_TracePrint(Iam, " Fork failure for %s\n", sAddressTo);
#endif
     Warn("mailto: %s", _("Is not supported"));
     Beep();
     close(fd[0]);
     close(fd[1]);
     return EXIT_FAILURE;
   }

/* According to all the comments in other code, HP-UX will send a signal
 * when the child quits that we don't want!
 */
 signal(SIGCHLD, SIG_DFL);

/* See if "we" are the child?
 * If so, send the mail...  We are specifying ALL args, and NO environment.
 * The child will ignore most signals... go ahead and finish the send!
 * Set up stdin of the child to be from our pipe!
 * Make sure that the error stream is ignored!  We get an error, just no mess!
 */
 if (!childpid)
   {
     signal(SIGHUP,  SIG_IGN);
     signal(SIGINT,  SIG_IGN);
     signal(SIGQUIT, SIG_IGN);
     signal(SIGTTIN, SIG_IGN);
     signal(SIGTTOU, SIG_IGN);

     close(fd[1]);
     dup2(fd[0], STDIN_FILENO);
#ifndef ARENA_DEBUG
     nfd = open("/dev/null", O_WRONLY);
     dup2(nfd, STDOUT_FILENO);
#endif
     istat = execve(ARENA_MailTo_MAILER_UTILITY_PATH, args, envp);
     Beep(); Beep(); Beep();
     Warn(_("Can't post (%d) mail to %s"), istat, sAddressTo);
     exit(1);                 /* Child should not return here!              */
   }

/* Parent does this!  We will wait for the child to come back to us.
 * Open our output end of the pipe as a stream, and dump headers and
 * message to the child. Probably should change this so we don't need
 * the stream, just the fildesc... BUT, Later!
 */
 close(fd[0]);
 fo = fdopen(fd[1], "w");
 if (!fo)
   {
#ifdef ARENA_DEBUG
     if (MAILTO_TRACE && VERBOSE_TRACE)
       Arena_TracePrint(Iam, " Unable to stream to pipe %d\n", sAddressTo);
#endif
     Warn(_("Can't post mail to %s"), sAddressTo);
     Beep();
     close(fd[1]);
     return EXIT_FAILURE;
   }

/* Dump the message to the mail handler!
 */
 MTOutputMessage(theForm, fo);
 fprintf(fo, "\n");
 fclose(fo);

 /* WAIT for the child to finish... This does NOT mean that the child
  * has actually moved the message, only that it has returned to us!
  */
 waitpid(childpid, &istat, WUNTRACED);
 if (WIFEXITED(istat))
     if ((istat = WEXITSTATUS(istat)))
	 Warn(_("Can't post (%d) mail to %s"), istat, sAddressTo);

#ifdef ARENA_DEBUG
 if (MAILTO_TRACE && VERBOSE_TRACE)
     Arena_TracePrint(Iam, " Wait on sender returned %d\n", istat);
#endif
 return istat;
}


/* We get here from normal SubmitForm if the form access method is `MAILTO'.
 * Hopefully, this will be obtuse enough not to get used by any other forms.
 * I left this arg list exactly the same as the normal SubmitForm in case
 * any other args are needed later.
 */
void MTSubmitForm(Form* theForm,
		  FormSubmitMethod method, char *action, int alen, int type,
		  char *name, int nlen, char *value, int vlen,
		  Image* image, int dx, int dy, char *bufpos)
{
 int i;
 Field* f;
#ifdef ARENA_DEBUG
 char Iam[] = "MTSubmitForm";
#endif


#ifdef ARENA_DEBUG
/*
 * Check the form submission method.
 */
 if (method != MAILTO)
   {
    if (MAILTO_TRACE)
      Arena_TracePrint(Iam,
		       " ERROR! Invoked with wrong submission method %d.\n",
		       (int)method);

    return;
   }
#endif


/* See if already busy!
 * We will only allow this entry to be completed if no Send
 * is currently active.
 */
 if (MTState != MTS_FORM)
   {
     Warn(_("mailto: Currently posting message to %s"), sAddressTo);
     Beep();
     return;
   }
 MTState = MTS_SEND;

/*
 * Verify that we have the REQUIRED data!
 */
 for (i = 0; i < nfn; i++)
   {
     f = FindField(theForm, fnt[i], fn[i], Arena_StrLen(fn[i]));
     if ( !f || !f->value || !(*f->value))
       {
	 Warn(_("Sorry, you must specify all input fields."));
	 Beep();
	 MTState = MTS_FORM;
	 return;
       }
     if (i <= 2)   /* Save From, To and Subject for later */
       {
	 strncpy(fv[i], f->value, f->buflen);
	 *(fv[i] + f->buflen) = 0;
       }
   }

/* Ok, we have the correct form, and we have enough data to send a mail,
 * so... go create the child and send it!
 */
 Announce(_("mailto: Posting message..."));
 ShowBusy();
 if (MTSendIt(theForm))
   {
    sleep(3);
    Beep();
    MTState = MTS_FORM;
    HideBusy();
   }
  else
   {
    Announce(_("mailto: Message posted!"));
    sleep(3);
    MTState = MTS_STATIC;
    HideBusy();
    BackDoc(DocDispGC, GetDefaultDocViewWindow());
   }
 return;
}


/* MTSendIt will call this routine to pump the mail to the correct output
 * specified stream... Maybe a pipe, maybe a temp file!
 */
void MTOutputMessage(Form* theForm, FILE *fp)
{
 int i;
 Field* f;

/* Standard always output stuff.
 */
 fprintf(fp,
	 "Content-Type: text/plain\n"
	 "MIME-Version: 1.0\n"
	 "X-Mailer: %s [v%s %s/%s mt/%s]\n"
	 "Reply-To: %s\n",
	 HTLib_appName(),
	 HTLib_appVersion(),
	 HTLib_name(),
	 HTLib_version(),
	 ARENA_MailTo_Version,
	 sAddressFrom);

/* Now output the optional settings from the
 * checkboxes for Return receipt and high priority
 */
 for (i = 0; i < nop; i++)
   {
     f = FindFieldFuzzy(theForm, opt[i], op[i], Arena_StrLen(op[i]));
     if (f && (f->flags & CHECKED))
       fprintf(fp, "%.*s: %.*s\n", (int)Arena_StrLen(op[i]), op[i],
	      (int)Arena_StrLen(ov[i]), ov[i]);
   }

/* Now output the To, From, Subject fields.
 * This way they are the last things just before the message!
 */
 for (i = 0; i < nfn-1; i++)
   {
     f = FindFieldFuzzy(theForm, fnt[i], fn[i], Arena_StrLen(fn[i]));
     if (f)
       fprintf(fp, "%.*s: %.*s\n", (int)Arena_StrLen(fn[i]), fn[i],
	       f->buflen, f->value);
   }

/* NOW, we can output the message!
 */
 f = FindFieldFuzzy(theForm, fnt[nfn-1], fn[nfn-1], Arena_StrLen(fn[nfn-1]));
 if (f)
   fprintf(fp, "\n%.*s\n", f->buflen, f->value);
 fclose(fp);
 return;
}
