/*-----------------------------------------------------------------------------
| bitmap.c -- Bitmap converter 
|
|	David Dawson - dawson@pobox.com
|	05 Mar 1997
|
|	Wesc 08 Mar 1997: removed Windows dependencies
|
|   Most of the code bellow was taken verbatim out of pila written by
|   Darrin Massena.
|
|	PBM conversion by Andrew Bulhak (acb@dev.null.org)
|	XBM, PBITM and compression by Hoshi Takanori (hoshi@sra.co.jp)
|
| See pilrc.htm for documentation
-------------------------------------------------------------WESC------------*/
#pragma pack(2)

#include <stdlib.h>
#include <stdio.h>
#include <stddef.h>
#include <memory.h>
#include <string.h>
#include <ctype.h>
#include "pilrc.h"


typedef unsigned char byte;	  /* b */
typedef unsigned short ushort;  /* us */
typedef unsigned long ulong;	  /* ul */

typedef int bool;					  /* f */


ulong LLoadX86(ulong ul)
{
	byte *pb;
	ulong ulOut;

	pb = (byte *) &ul;
	ulOut = (ulong) ((*(pb+3)<<24) | (*(pb+2)<<16) | (*(pb+1)<<8) | (*pb));
	return ulOut;
}

ushort WLoadX86(ushort w)
{
	byte *pb;
	ushort wOut;

	pb = (byte *) &w;
	wOut = (ushort) ((*(pb+1)<<8) | (*pb));
	return wOut;
}



#define SwapBytes(us)	(ushort)((((ushort)(us)) >> 8) | (((ushort)(us)) << 8))



/* Pilot bitmap resource format */
typedef struct _Bitmap
	{ /* bm */
	ushort cx;
	ushort cy;
	ushort cbRow;
	ushort ff;
	ushort ausUnk[4];
	byte abBits[1];
	} Bitmap;

/* Windows bitmap format */
typedef struct tagBITMAPFILEHEADER 
	{ /* bmfh */
	ushort  bfType; 
#ifdef __GNUC__
	ushort  bfSize1;
	ushort  bfSize2;
#else
	ulong   bfSize; 
#endif
	ushort  bfReserved1; 
	ushort  bfReserved2; 
#ifdef __GNUC__
	ushort  bfOffBits1;
	ushort  bfOffBits2;
#else
	ulong   bfOffBits; 
#endif
	} BITMAPFILEHEADER; 



typedef struct tagBITMAPINFOHEADER
	{ /* bmih */
	ulong  biSize; 
	long   biWidth; 
	long   biHeight; 
	ushort biPlanes; 
	ushort biBitCount;
	ulong  biCompression; 
	ulong  biSizeImage; 
	long   biXPelsPerMeter; 
	long   biYPelsPerMeter; 
	ulong  biClrUsed; 
	ulong  biClrImportant; 
	} BITMAPINFOHEADER; 

typedef struct tagRGBQUAD 
	{ /* rgbq */
	byte    rgbBlue; 
	byte    rgbGreen; 
	byte    rgbRed; 
	byte    rgbReserved; 
	} RGBQUAD; 


typedef struct tagBITMAPINFO 
	{ /* bmi */
	BITMAPINFOHEADER bmiHeader; 
	RGBQUAD          bmiColors[1]; 
	} BITMAPINFO; 



int CbRow(int cx, int cbpp, int cBitsAlign)
	{
	return ((cx*cbpp + (cBitsAlign - 1)) & ~(cBitsAlign - 1))/8;
	}


/* Test bit in a 1bpp bitmap */
bool TestBit(int cx, int cy, byte *pb, int x, int y, int cBitsAlign)
{
	int cbRow;

	cbRow = CbRow(cx, 1, cBitsAlign);
	pb += cbRow * y + (x >> 3);
	return (*pb & (1 << (7 - (x & 7))));
}

/* Set bit in a 1bpp bitmap */
void SetBit(int cx, int cy, byte *pb, int x, int y, int cBitsAlign)
{
	int cbRow;

	cbRow = CbRow(cx, 1, cBitsAlign);
	pb += cbRow * y + (x >> 3);
	*pb |= (1 << (7 - (x & 7)));
}


/* gets bits in a 4bpp bitmap */
int GetBits(int cx, int cy, byte *pb, int x, int y, int cBitsAlign)
{
	int cbRow;
	
	cbRow = CbRow(cx, 4, cBitsAlign);
	pb += cbRow * y + (x >> 1);
	if (x & 1)
		return (*pb & 0xf);
	else
		return (*pb & 0xf0) >> 4;
}

/* Set bit in a 2bpp bitmap */
void SetBits(int cx, int cy, byte *pb, int x, int y, int bits, int cBitsAlign)
{
	int msk;
	int cbRow;
	cbRow = CbRow(cx, 2, cBitsAlign);
	pb += cbRow * y + (x >> 2);
	msk = (bits << ((3 - (x & 3))*2));
	*pb |= msk;
	
}


/* Convert from .bmp to Pilot resource data */
void ConvertWindowsBitmap(RCBITMAP *rcbmp, byte *pbResData, int size, BOOL fGray)
{
	byte *pbSrc;
	int cbRow;
	int x, y;
	int cbHeader, dx, dy, cbits;
	int cbitsPel;
	BITMAPINFO *pbmi;
	BITMAPINFOHEADER bmi;
	BOOL fNonGreyWarning;

	pbmi = (BITMAPINFO *)(pbResData + sizeof(BITMAPFILEHEADER)); 

	/* alignment of structure is incorrect for sparc.  Copy
	 structure to known alignment before extracting bitmap parameters
	 */
	memcpy (&bmi, pbmi,sizeof(BITMAPINFOHEADER));
	cbHeader = LLoadX86(bmi.biSize);
	dx = LLoadX86(bmi.biWidth);
	dy = LLoadX86(bmi.biHeight);
	cbits = WLoadX86(bmi.biBitCount);
/*	printf ("bitmap width:%d  dy:%d  size:%d  bitcount:%d\n",
		dx, dy, cbHeader, cbits); */

	pbSrc = ((byte *)pbmi) + cbHeader + sizeof(RGBQUAD) * (1 << cbits);

	/* If image not 1bpp, bail	 */

	if (fGray && cbits != 4)
		{
		ErrorLine ("Bitmap not 16 color");
		}
	if (!fGray && cbits != 1)
		{
		ErrorLine ("Bitmap not monochrome");
		}
	Assert(fGray == 0 || fGray == 1);
	cbitsPel = 1+fGray;

	/* Alloc what we need for image data */
	/* Pilot images are word aligned */
	
	cbRow = ((dx*cbitsPel + 15) & ~15) / 8;
	rcbmp->cbDst = cbRow * dy;
	rcbmp->pbBits = malloc(rcbmp->cbDst);

	/* Image data has been inverted for Macintosh, so invert back */

	memset(rcbmp->pbBits, 0, rcbmp->cbDst);

	/* Convert from source bitmap format (DWORD aligned) to dst format (word 
aligned). */
	
	fNonGreyWarning = fFalse;
	for (y = 0; y < dy; y++)
		{
		for (x = 0; x < dx; x++)
			{
			/* Reverse bits so we get WYSIWYG in msdev (white
				pixels become background (0), black pixels become
				foreground (1)). */

			int yT = y;
			if (dy > 0)
				yT = dy - y - 1;

			if (fGray)
				{
				int w;
				
				w = GetBits(dx, dy, pbSrc, x, yT, 32);
				switch (w)
					{
				default:    /* lt grey */
					if (!fNonGreyWarning)	
						{
						WarningLine("Bitmap has non black,white,dkgray,ltgray pixels");
						fNonGreyWarning = fTrue;
						}
					w = 2;	
					break;
				case 0:	  	/* black (on) */
					w = 3;   
					break;
				case 7:	   /* lt gray (1/3 on) */
					w = 1;
					break;
				case 8:	   /* dk gray (2/3 on) */
					w = 2;
					break;				
				case 15:		/* white  (not on) */
					w = 0;	
					break;
					}
				SetBits(dx, dy, rcbmp->pbBits, x, y, w, 16);
				}
			else
				{
				if (!TestBit(dx, dy,
							 pbSrc, x, yT, 32))
					{
					SetBit(dx, dy,
							 rcbmp->pbBits, x, y, 16);
					}
				}
			}
		}
	rcbmp->cx = (int) dx;
	rcbmp->cy = (int) dy;
	rcbmp->cbRow = (int) cbRow;
}



/* Skip a newline */
int skip_newline(byte *data, int size, int i)
{
	while (i < size && (data[i] != '\n' && data[i] != '\r'))
		{
		++i;
		}
	if (i + 1 < size && data[i] == '\r' && data[i + 1] == '\n')
		{
		i += 2;
		}
	else if (i < size)
		{
		++i;
		}
	return i;
}

/* Convert from .pbitm to Pilot resource data */
void ConvertTextBitmap(RCBITMAP *rcbmp, byte *data, int size)
{
	int i, x, y;

	/* Count width and height of the image. */
	rcbmp->cx = 0;
	rcbmp->cy = 0;
	for (i = 0; i < size; i = skip_newline(data, size, i))
		{
		int j = i;
		while (j < size && (data[j] != '\n' && data[j] != '\r'))
			{
			++j;
			}
		if (rcbmp->cx < j - i)
			{
			rcbmp->cx = j - i;
			}
		++rcbmp->cy;
		}

	/* Allocate image buffer. */
	rcbmp->cbRow = ((rcbmp->cx + 15) & ~15) / 8;
	rcbmp->cbDst = rcbmp->cbRow * rcbmp->cy;
	rcbmp->pbBits = malloc(rcbmp->cbDst);
	memset(rcbmp->pbBits, 0, rcbmp->cbDst);

	/* Convert the image. */
	x = 0;
	y = 0;
	for (i = 0; i < size; )
		{
		if (data[i] == '\n' || data[i] == '\r')
			{
			x = 0;
			++y;
			i = skip_newline(data, size, i);
			}
		else
			{
			if (data[i] != ' ' && data[i] != '-')
				{
				rcbmp->pbBits[y * rcbmp->cbRow + (x >> 3)] |= (1 << (7 - (x & 7)));
				}
			++x;
			++i;
			}
		}
}

/* Convert from .xbm to Pilot resource data */
void ConvertX11Bitmap(RCBITMAP *rcbmp, byte *data, int size)
{
	static byte rev[] = {
		0,	/* 0000 */
		8,	/* 1000 */
		4,	/* 0100 */
		12,	/* 1100 */
		2,	/* 0010 */
		10,	/* 1010 */
		6,	/* 0110 */
		14,	/* 1110 */
		1,	/* 0001 */
		9,	/* 1001 */
		5,	/* 0101 */
		13,	/* 1101 */
		3,	/* 0011 */
		11,	/* 1011 */
		7,	/* 0111 */
		15	/* 1111 */
	};

	char name_and_type[80], *type;
	int i, value, pos;

	/* Read X11 bitmap header. */
	for (i = 0; i < size; i = skip_newline(data, size, i))
		{
		if (sscanf(data + i, "#define %79s %d", name_and_type, &value) == 2)
			{
			type = strrchr(name_and_type, '_');
			type = (type == NULL) ? name_and_type : type + 1;
			if (strcmp(type, "width") == 0)
				{
				rcbmp->cx = value;
				}
			else if (strcmp(type, "height") == 0)
				{
				rcbmp->cy = value;
				}
			}
		if (sscanf(data + i, "static unsigned char %s = {", name_and_type) == 1
			 || sscanf(data + i, "static char %s = {", name_and_type) == 1)
			{
			type = strrchr(name_and_type, '_');
			type = (type == NULL) ? name_and_type : type + 1;
			if (strcmp(type, "bits[]") == 0 && rcbmp->cx > 0 && rcbmp->cy > 0)
				{
				rcbmp->cbRow = ((rcbmp->cx + 15) & ~15) / 8;
				rcbmp->cbDst = rcbmp->cbRow * rcbmp->cy;
				rcbmp->pbBits = malloc(rcbmp->cbDst);
				break;
				}
			}
		}

	if (rcbmp->pbBits == NULL)
		{
		ErrorLine("Invalid X11 bitmap");
		}

	/* Read X11 bitmap data. */
	memset(rcbmp->pbBits, 0, rcbmp->cbDst);
	value = 0;
	pos = 0;
	for (i = skip_newline(data, size, i); i < size; ++i)
		{
		byte c = data[i];
		if (c == ',' || c == '}')
			{
			if (pos < rcbmp->cbDst)
				{
				rcbmp->pbBits[pos++] = value;
				if (rcbmp->cx % 16 > 0 && rcbmp->cx % 16 < 9
					 && pos % rcbmp->cbRow == rcbmp->cbRow - 1)
					{
					++pos;
					}
				}
			value = 0;
			}
		else
			{
			if (c >= '0' && c <= '9')
				{
				value = (value >> 4) + (rev[c - '0'] << 4);
				}
			else if (c >= 'A' && c <= 'F')
				{
				value = (value >> 4) + (rev[c - 'A' + 10] << 4);
				}
			else if (c >= 'a' && c <= 'f')
				{
				value = (value >> 4) + (rev[c - 'a' + 10] << 4);
				}
			}
		}
}


/*  convert from a PBM bitmap to Pilot resource data */
void ConvertPBMBitmap(RCBITMAP *prcbmp, byte *pbData, int cb)
{
	char type;
	char c;
	int width, height;

	int x,y,r;
	byte *pb;
	byte *pbMac;

	pb = pbData;
	pbMac = pb+cb;
	if (*pb++ !='P')
		{ErrorLine("Not a PBM file.");}
	type=*pb++;
	if (type!='1' && type!='4')
		{ErrorLine("Not a monochrome bitmap.");}

	do
		{c=*pb++;} 
	while (isspace(c));  
	pb--;
	
	while (c=='#')
		{
		do
			{c=*pb++;} while (c!='\n');  
		c=*pb++; pb--;
		}

	/* fscanf(pFile, "%d %d", &width, &height); */
	sscanf(pb, "%d %d", &width, &height);
	do
		{c=*pb++;}
	while (c != '\n');
	pb--;

	do
		{c=*pb++;}
	while (isspace(c));
	pb--;

	while (c=='#')
		{
		do
			{c=*pb++;} while (c!='\n');  
		c=*pb++; pb--;
		}

	memset(prcbmp, 0, sizeof(RCBITMAP));

	prcbmp->cbRow = ((width + 15) & ~15) / 8;
	prcbmp->cbDst = prcbmp->cbRow * height;
	prcbmp->pbBits = malloc(prcbmp->cbDst);
	memset(prcbmp->pbBits, 0, prcbmp->cbDst);

	/*  read the bits */
	if (type=='1')
		{
		/* ASCII */
		for (y=0; y<height; y++)
			{
			for (x=0; x<width; x++)
				{
				do
					{c=*pb++;} 
				while (c!='0' && c!='1');
				if (c=='1')	 SetBit(width, height, prcbmp->pbBits, x, y, 16);
				}
			}
		}
	else
		{
		/* binary */
		for (y=0; y<height; y++)
			{
			for (r=0; r<width; r+=8)
				{
				c=*pb++;
				for (x=0; x+r<width & x<8; x++)
					{
					if (c&0x80)
						SetBit(width, height, 
								 prcbmp->pbBits, 
								 x+r, y, 16);
					c<<=1;
					}
				}
			}
		}
	prcbmp->cx = width;
	prcbmp->cy = height;
}


/* Compress bitmap data */
void CompressBitmap(RCBITMAP *rcbmp, int compress)
{
	unsigned char *bits;
	int size, i, j, k, flag;

	bits = malloc(2 + (rcbmp->cbRow + ((rcbmp->cbRow + 7) / 8)) * rcbmp->cy);
	size = 2;
	for (i = 0; i < rcbmp->cy; ++i)
		{
		flag = 0;
		for (j = 0; j < rcbmp->cbRow; ++j)
			{
			if (i == 0 || rcbmp->pbBits[i * rcbmp->cbRow + j] !=
				 rcbmp->pbBits[(i - 1) * rcbmp->cbRow + j])
				{
				flag |= (0x80 >> (j & 7));
				}
			if ((j & 7) == 7 || j == rcbmp->cbRow - 1)
				{
				bits[size++] = flag;
				for (k = (j & ~7); k <= j; ++k)
					{
					if (((flag <<= 1) & 0x100) != 0)
						{
						bits[size++] = rcbmp->pbBits[i * rcbmp->cbRow + k];
						}
					}
				flag = 0;
				}
			}
		}

	if (compress == rwForceCompress || size < rcbmp->cbDst)
		{
		free(rcbmp->pbBits);
		rcbmp->ff |= 0x8000;
		rcbmp->pbBits = bits;
		bits[0] = size >> 8;
		bits[1] = size;
		rcbmp->cbDst = size;
		}
	else
		{
		free(bits);
		}
}

/* Compress and dump Pilot resource data */
void CompressDumpBitmap(RCBITMAP *rcbmp, int isIcon, int compress)
{
	if (isIcon && (rcbmp->cx != 32 || rcbmp->cy != 32))
		{
		ErrorLine("Icon resource not 32x32");
		}

	if (compress == rwAutoCompress || compress == rwForceCompress)
		{
		CompressBitmap(rcbmp, compress);
		}

	CbEmitStruct(rcbmp, szRCBITMAP, NULL, fTrue);
	DumpBytes(rcbmp->pbBits, rcbmp->cbDst);
	free(rcbmp->pbBits);
}

void InvalidExtension(char *fileName)
{
	char pchLine[300];
	sprintf(pchLine, 
			  "Bitmap file extension not recognized for file %s\n"
			  "\tSupported extensions: .BMP (Windows bitmap)"
			  ", .pbitm (txt2bitm), .xbm (X11 bitmap)"
			  , fileName);
	ErrorLine(pchLine);
}

void DumpBitmap(char *fileName, int isIcon, int compress, BOOL fGray)
{
	byte *pBMPData;
	char *pchExt;
	FILE *pFile;
	long size;
	RCBITMAP rcbmp;

	pchExt = strrchr(fileName, '.');

	if ((strlen(fileName) < 5) || (pchExt == NULL))
		{
		InvalidExtension(fileName);
		}

	fileName = FindAndOpenFile(fileName, "rb", &pFile);

	fseek(pFile,0,SEEK_END);
	size = ftell(pFile);
	rewind(pFile);

	pBMPData = malloc(size);
	if (pBMPData == NULL)
		{
		char pchLine[200];
		sprintf(pchLine, "Resource %s too big to fit in memory", fileName);
		ErrorLine(pchLine);
		}

	fread(pBMPData, 1, size, pFile);
	fclose(pFile);

	pchExt = strrchr(fileName, '.');
	if (!pchExt)
		{
		InvalidExtension(fileName);
		}

	pchExt++;

	memset(&rcbmp, 0, sizeof(RCBITMAP));
	if (FSzEqI(pchExt, "bmp"))
		{
		ConvertWindowsBitmap(&rcbmp, pBMPData, size, fGray);
		CompressDumpBitmap(&rcbmp, isIcon, compress);
		}
	else if (FSzEqI(pchExt, "pbitm"))
		{
		ConvertTextBitmap(&rcbmp, pBMPData, size);
		CompressDumpBitmap(&rcbmp, isIcon, compress);
		}
	else if (FSzEqI(pchExt, "xbm"))
		{
		ConvertX11Bitmap(&rcbmp, pBMPData, size);
		CompressDumpBitmap(&rcbmp, isIcon, compress);
		}
	else if (FSzEqI(pchExt, "pbm"))
		{
		ConvertPBMBitmap(&rcbmp, pBMPData, size);
		CompressDumpBitmap(&rcbmp, isIcon, compress);
		}
	else
		{
		InvalidExtension(fileName);
		}
	free(pBMPData);
}
