"""
  PDA.Palm.DLP - Main part of PDA.Palm package
  $Id: DLP.py,v 1.9 1998/09/01 11:02:50 rob Exp $

  Copyright 1998 Rob Tillotson <rob@io.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License, version 2,
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.


  Warning!  Possibly incomplete code!  Use at your own risk and
  please send bug reports and suggestions to me at the address
  above.

  Please see the package documentation for complete details.
"""

__version__ = '$Id: DLP.py,v 1.9 1998/09/01 11:02:50 rob Exp $'

import _pdapalm

import re, string
import operator
import time
import struct

Types = None
PrefTypes = None

class DictAssoc:
    """An instance of this class maintains a list of dictionary->object
    associations.  When called with a dictionary parameter, it matches
    the parameter against each dictionary in the list in turn, returning
    the associated object when it finds a match.  The semantics of
    of the comparison are:
      - excess items in the parameter are ignored
      - string values in the association are treated as regular
        expressions to be matched against strings in the parameter
      - everything else is simply compared
      - all fields in the association must match in order for the
        association to be considered a match
    """
    def __init__(self, default):
	self.data = []
	self.default = default

    def register(self, cls, match):
	if (match, cls) not in self.data: self.data.append((match, cls))

    def unregister(self, cls, match):
	if (match, cls) in self.data: self.data.remove((match, cls))

    def __call__(self, target):
	for match, cls in self.data:
	    m = match.keys()
	    # Who says Python is always easy to read? :-)
	    if not filter(lambda x: not x,
		      map(lambda x,y: type(x) == type('') and \
			  re.match(x,y) or x == y,
			  map(operator.__getitem__, [match]*len(m), m),
			  map(lambda x,y: x.get(y,''), [target]*len(m), m))):
		return cls
	return self.default
    
class DLP:
    """A DLP connection to the PalmOS device."""
    def __init__(self, port=None):
	self.__socket = None
	self.__dlp = None
	self.__typemap = Types
	self.__prefmap = PrefTypes
	if port: self.openSocket(port)

    def openSocket(self, port):
	self.__socket = _pdapalm.openPort(port)
	self.__dlp = _pdapalm.accept(self.__socket)
	return self.__dlp.status()
    
    def open(self, name, mode=_pdapalm.OpenDBReadWrite, cardno=0):
	# find information about the database
	info = self.__dlp.findDBInfo(0, name, None, None, cardno)
	# do we have a mapper object?  if so, call it
	if self.__typemap:
	    cls = self.__typemap(info)
	else: cls = Database

	# open the database
	db = self.__dlp.open(name, mode, cardno)  # raw?

	# create the class
	return cls(db, info)

    def create(self, name, creator, type, flags=0, version=1, cardno=0):
	db = self.__dlp.create(name, creator, type, flags, version, cardno)
	info = self.__dlp.findDBInfo(0, name, None, None, cardno)
	if self.__typemap:
	    cls = self.__typemap(info)
	else: cls = Database

	return cls(db, info)

    def delete(self, name, cardno=0):
	return self.__dlp.delete(name, cardno)

    def status(self):
	return self.__dlp.status()

    def dirty(self):
	return self.__dlp.dirty()

    def getBattery(self):
	return self.__dlp.getBattery()   # new type?

    def reset(self):
	return self.__dlp.reset()

    def close(self, status=0):
	return self.__dlp.close(status)

    def abort(self):
	return self.__dlp.abort()

    def log(self, string):
	return self.__dlp.log(string)

    def getTime(self):
	return self.__dlp.getTime()

    def setTime(self, t):
	return self.__dlp.setTime(t)

    def cardInfo(self, cardno):
	return self.__dlp.cardInfo(cardno)  # new type?

    def sysInfo(self):
	return self.__dlp.sysInfo()

    def getUserInfo(self):
	return self.__dlp.getUserInfo()

    def setUserInfo(self, userinfo):
	return self.__dlp.setUserInfo(userinfo)

    def getPref(self, creator, id, version=0, backup=1):
	raw, creator, id, version, backup = self.__dlp.getPref(creator, id, backup)
	if self.__prefmap:
	    cls = self.__prefmap({'creator':creator, 'id':id, 'version':version})
	else: cls = PrefBlock

	return cls(creator, id, version, backup)

    def getPrefRaw(self, creator, id, backup=1):
	return self.__dlp.getPref(creator, id, backup)
    
    def setPref(self, pref):
	return self.__dlp.setPref(pref.creator, pref.id, pref.backup,
				  pref.version, pref.pack())

    def setPrefRaw(self, raw, creator, id, version=0, backup=0):
	return self.__dlp.setPref(creator, id, backup, version, raw)
    
    def getFeature(self, creator, number):
	return self.__dlp.getFeature(creator, number)

    def getDBInfo(self, start, ram=1, rom=0, cardno=0):
	return self.__dlp.getDBInfo(start, ram, rom, cardno)

    def findDBInfo(self, start, name, creator, type, cardno=0):
	return self.__dlp.findDBInfo(start, name, creator, type, cardno)

    def callApplication(self, creator, type, action, data, length, maxlength):
	return self.__dlp.callApp(creator, type, action, data, length, maxlength)

    def RPC():
	pass


#######################################################################
#  Records, Blocks, Appinfo, and friends
#######################################################################

# Field type specifiers (in Block.fields) are tuples of
# (type, default, [validation, ...])

FLD_UNKNOWN = None
FLD_ANY = '*'
FLD_BOOLEAN = 'b'
FLD_STRING = 's'  # validation: max length
FLD_INT = 'i'     # validation: min, max values
FLD_FLOAT = 'f'   # validation: min, max values
FLD_TIME = 't'    # validation: min, max date range
FLD_TIME_T = 'T'  # validation: min, max date range (as an integer/float time!)
FLD_LIST = '['    # validation: min, max length

# Blocks of all types are now expected to be accessed as dictionaries!

class Block:
    """Generic superclass for 'Things Which Can Be Packed And Unpacked',
    including but not limited to database records, appblocks, sortblocks,
    etc."""
    def __init__(self, raw=None):
	self.data = {}
	if hasattr(self, 'fields'):
	    for x in self.fields.keys():
		self.data[x] = self.fields[x][1]
	else:
	    self.fields = {}
	if raw: self.unpack(raw)
	else: self.raw = None

    def unpack(self, raw):
	self.raw = raw

    def pack(self):
	return self.raw

    # programming aids
    def packfields(self, fmt, names):
	r = apply(struct.pack,
		  (fmt,)+tuple(map(operator.getitem,
				   [self.data]*len(names),
				   names)))
	return r

    def unpackfields(self, fmt, names, r=''):
	map(operator.setitem, [self.data]*len(names), names,
	    struct.unpack(fmt, r))
	
    def __getitem__(self, k): return self.data[k]
    def __setitem__(self, k, v): self.data[k] = v
    def get(self, k, d): return self.data.get(k, d)
    def keys(self): return self.data.keys()
    def values(self): return self.data.values()
    def items(self): return self.data.items()
    def has_key(self, k): return self.data.has_key(k)
    

class Record(Block):
    """Generic database record type."""
    def __init__(self, raw='', index=0, id=0, attr=0, category=0):
	Block.__init__(self, raw)
	self.deleted = (attr & 0x80) != 0
	self.modified = (attr & 0x40) != 0
	self.busy = (attr & 0x20) != 0
	self.secret = (attr & 0x10) != 0
	self.archived = (attr & 0x08) != 0
	self.category = category
	self.index = index
	self.id = id

    def attributes(self):
	a = 0
	if self.deleted: a = a | 0x80
	if self.modified: a = a | 0x40
	if self.busy: a = a | 0x20
	if self.secret: a = a | 0x10
	if self.archived: a = a | 0x08
	return a
    
class AppBlock(Block):
    """Generic AppBlock type.  By default, an AppBlock is treated as
    just a hunk of raw data; this class is only provided for the sake
    of consistency."""
    pass

class SortBlock(Block):
    """Generic SortBlock type.  By default, a SortBlock is treated as
    just a hunk of raw data; this class is only provided for the sake
    of consistency."""
    pass

class Resource(Block):
    """Generic resource type."""
    def __init__(self, raw=None, type='    ', id=0):
	Block.__init__(self, raw)
	self.type = type
	self.id = id

class PrefBlock(Block):
    """Generic AppPref block."""
    def __init__(self, raw='', creator='    ', id=0, version=0, backup=0):
	Block.__init__(self, raw)
	self.creator = creator
	self.id = id
	self.version = version
	self.backup = backup

# Note: there is still a C->Python dependency in the various set*()
# methods; for now it is okay to leave these in, because the dependency
# is not major (attribute names and the pack() method).

# Note: The 'info' record is cached.  We need to keep this up to date
# with any changes... but there does not appear to be any way to
# simply set flags et. al. in a Hotsync-connected database. (?)  For
# now, suffice it to say that if the caller does not do a FindDBInfo
# every now and then, the cache may be out of date slightly... this is
# really no different than in the original module, where the caller
# might not have current info around at all.
#
class Database:
    """A PalmOS database.  This class is actually a wrapper around a
    back-end class, which may be anything that implements the same
    API; typical back-end objects include the file and database types
    from the C module.

    It is assumed that the back-end object deals only with raw data,
    and does not care about any higher-level formatting that may be
    imposed upon records in the database.  Instead, the wrapper
    handles the enclosure of raw data in appropriate Python objects,
    which should allow the use of a common set of Python objects
    to represent PalmOS data in any type of storage.
    """
    def __init__(self, db, info):
	self.__db = db
	self.info = info
	self.record_class = Record
	self.appblock_class = AppBlock
	self.sortblock_class = SortBlock
	self.__prefmap = PrefTypes

    # Issue: addRecord/addResource is implemented only for files in the
    # C module, while setRecord/setResource is implemented only for
    # DLP-connected databases.  Should there be a way to unify this
    # API, or do we just trust the application to know which sort of
    # thing it's working with?
    def setRecord(self, rec):
	id = self.__db.setRecord(rec.attributes(), rec.id, rec.category,
				 rec.pack())
	rec.id = id
	return id
    
    def setResource(self, rsc):
	return self.__db.setResource(rsc.type, rsc.id, rsc.pack())

    def addRecord(self, rec):
	try:
	    return self.__db.addRecord(rec.attributes(), rec.id, rec.category,
				       rec.pack())
	except AttributeError:
	    return self.setRecord(rec)

    def addResource(self, rsc):
	try:
	    return self.__db.addResource(rsc.type, rsc.id, rsc.pack())
	except AttributeError:
	    return self.setResource(rsc)
    
    def getNextRecord(self, cat, cls=None):
	if cls: return apply(cls, self.__db.getNextRecord(cat))
	else: return apply(self.record_class, self.__db.getNextRecord(cat))

    def getNextModRecord(self, cat=-1, cls=None):
	if cls: return apply(cls, self.__db.getNextModRecord(cat))
	else: return apply(self.record_class, self.__db.getNextModRecord(cat))

    def getResource(self, index, cls=None):
	if cls: return apply(cls, self.__db.getResource(index))
	else: return apply(self.resource_class, self.__db.getResource(index))

    def getResourceByID(self, id, cls=None):
	if cls: return apply(cls, self.__db.getResourceByID(index))
	else: return apply(self.resource_class, self.__db.getResourceByID(index))

    def getRecord(self, index, cls=None):
	if cls: return apply(cls, self.__db.getRecord(index))
	else: return apply(self.record_class, self.__db.getRecord(index))

    def getRecordByID(self, id, cls=None):
	if cls: return apply(cls, self.__db.getRecordByID(id))
	else: return apply(self.record_class, self.__db.getRecordByID(id))

    def deleteRecord(self, id):
	return self.__db.deleteRecord(id)

    def deleteRecords(self):
	return self.__db.deleteRecords()

    def deleteResource(self, type, id):
	return self.__db.deleteResource(type, id)

    def deleteResources(self):
	return self.__db.deleteResources()

    def close(self):
	return self.__db.close()

    def getRecords(self):
	return self.__db.getRecords()

    def getRecordIDs(self, sort=0):
	return self.__db.getRecordIDs(sort)

    def getAppBlock(self, cls=None):
	if cls: return cls(self.__db.getAppBlock())
	else: return self.appblock_class(self.__db.getAppBlock())

    def setAppBlock(self, blk):
	return self.__db.setAppBlock(blk.pack())

    def getSortBlock(self, cls=None):
	if cls: return cls(self.__db.getSortBlock())
	else: return self.sortblock_class(self.__db.getSortBlock())

    def setSortBlock(self, blk):
	return self.__db.setSortBlock(blk.pack())

    def moveCategory(self, frm, to):
	return self.__db.moveCategory(frm, to)

    def deleteCategory(self, cat):
	return self.__db.deleteCategory(cat)

    def purge(self):
	return self.__db.purge()

    def resetNext(self):
	return self.__db.resetNext()

    def resetFlags(self):
	return self.__db.resetFlags()

    def getPref(self, id=0, backup=1):
	raw, creator, id, version, backup = self.__db.getPref(self.info.creator,
							      id, backup)
	if self.__prefmap:
	    cls = self.__prefmap({'creator':self.info.creator,
				  'id': id, 'version': version})
	else: cls = PrefBlock

	return cls(raw, creator, id, version, backup)

    def setPref(self, pref):
	return self.__db.setPref(pref.creator, pref.id, pref.backup,
				 pref.version, pref.pack())

    def retrieve(self, dlp, cardno=0):
	return self.__db.retrieve(dlp._DLP__dlp, cardno)

    def install(self, dlp, cardno=0):
	return self.__db.install(dlp._DLP__dlp, cardno)
    
    # Python-isms
    def __len__(self): return self.getRecords()
    def __getitem__(self, i): return self.getRecord(i)

    def __setitem__(self, i, r):
	r.index = i
	self.setRecord(r)
	
    def __getslice__(self, low, high):
	return Slice(self, low, high)
    
    def Category(self, cat):
	return CategoryIterator(self, cat, self.getNextRecord)

    def ModifiedRecords(self, cat=-1):
	return CategoryIterator(self, cat, self.getNextModRecord)

class Slice:
    """A generic slice, which serves mainly to permit Python to not
    have to retrieve a bunch of records at once.  Doesn't support
    everything a real slice does, but should allow iteration and
    retrieval over a portion of a database."""
    def __init__(self, list, low, high):
	self.list = list
	self.low = low
	self.high = high
	self.len = high-low
    def __len__(self): return self.len
    def __getitem__(self, n):
	if n >= self.len: raise IndexError
	return self.list[n+self.low]
    
class CategoryIterator:
    """An iterator which returns each record in a category using
    getNextModRecord or getNextRecord."""
    def __init__(self, db, cat, func):
	self.db = db
	self.cat = cat
	self.func = func

    def __getitem__(self, n):
	return self.func(self.db, self.cat)

##################################

def protect_name(s):
    s = string.replace(s, '=', '=3D')
    s = string.replace(s, '/', '=2F')
    s = string.replace(s, '\n', '=0A')
    s = string.replace(s, '\r', '=0D')
    return s

def unprotect_name(s):
    s = string.replace(s, '=2F', '/')
    s = string.replace(s, '=0A', '\n')
    s = string.replace(s, '=0D', '\r')
    s = string.replace(s, '=3D', '=')
    return s


##################################
if not Types:
    Types = DictAssoc(Database)

if not PrefTypes:
    PrefTypes = DictAssoc(PrefBlock)
    

def openFile(name, typemapper=Types):
    """Open a local file in PalmOS database format."""
    f = _pdapalm.fileOpen(name)
    info = f.getDBInfo()

    if typemapper: cls = typemapper(info)
    else: cls = Database

    return cls(f, info)

def createFile(name, info=None, typemapper=Types):
    """Create a local file in PalmOS database format."""
    i = { 'name': name,
	  'type': 'DATA',
	  'creator': '    ',
	  'index': 0,
	  'more': 0,
	  'modnum': 0,
	  'version': 0,
	  'createDate': int(time.time()),
	  'modifyDate': int(time.time()),
	  'backupDate': 0,
	  'flagReset': 0,
	  'flagResource': 0,
	  'flagNewer': 0,
	  'flagExcludeFromSync': 0,
	  'flagAppInfoDirty': 0,
	  'flagReadOnly': 0,
	  'flagBackup': 0,
	  'flagOpen': 0
	  }
    if info: i.update(info)
    if len(i['name']) > 31: i['name'] = i['name'][:31]
    
    f = _pdapalm.fileCreate(name, i)
    info = f.getDBInfo()

    if typemapper: cls = typemapper(info)
    else: cls = Database

    return cls(f, info)

def openBackendDB(backend, *a, **kw):
    """Open any generic backend database, with type mapping.  Any parameters
    that are supplied beyond the first are passed along to the backend
    class."""
    typemapper = Types
    
    b = apply(backend, a, kw)
    info = b.getDBInfo()

    if typemapper: cls = typemapper(info)
    else: cls = Database

    return cls(f, info)
