/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: AudioData.cc,v $
 * Revision 1.6  1998/09/23 17:55:59  mueller
 * Improved error message about short audio file.
 *
 * Revision 1.5  1998/09/22 19:17:19  mueller
 * Added seeking to and reading of samples for GUI.
 *
 * Revision 1.4  1998/09/06 12:00:26  mueller
 * Used 'message' function for messages.
 *
 * Revision 1.3  1998/07/28 13:47:48  mueller
 * Automatic length determination of audio files is now done in 'AudioData'.
 *
 */

static char rcsid[] = "$Id: AudioData.cc,v 1.6 1998/09/23 17:55:59 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <errno.h>

#include "AudioData.h"
#include "Msf.h"
#include "util.h"

// creates an object representing a portion of a raw audio data file
AudioData::AudioData(const char *filename, unsigned long start,
		     unsigned long length)
{
  type_ = FILE;
  
  filename_ = strdupCC(filename);
  startPos_ = start;
  length_ = length;
  fileType_ = audioFileType(filename);

  open_ = 0;
  fd_ = 0;
  readPos_ = 0;
  autoLength_ = (length_ == 0 ? 1 : 0);
  autoLengthFailed_ = 0;
  headerLength_ = 0;
}

// creates an object that contains constant audio data
AudioData::AudioData(Type t, unsigned long length)
{
  assert(t != FILE);
  
  type_ = t;
  
  length_ = length;

  filename_ = NULL;
  fileType_ = 0;
  startPos_ = 0;
  open_ = 0;
  fd_ = 0;
  readPos_ = 0;
  autoLength_ = 0;
  autoLengthFailed_ = 0;
  headerLength_ = 0;
}

// copy constructor
AudioData::AudioData(const AudioData &obj)
{
  type_ = obj.type_;

  if (type_ == FILE) {
    filename_ = strdupCC(obj.filename_);
    startPos_ = obj.startPos_;
    fileType_ = obj.fileType_;
  }
  else {
    filename_ = NULL;
    startPos_ = 0;
    fileType_ = 0;
  }

  length_ = obj.length_;
  autoLength_ = obj.autoLength_;
  autoLengthFailed_ = obj.autoLengthFailed_;
  headerLength_ = 0;

  open_ = 0;
  fd_ = 0;
  readPos_ = 0;
}
    
AudioData::~AudioData()
{
  if (open_ != 0 && type_ == FILE) {
    close(fd_);
    delete[] filename_;
  }
}

// returns length of audio data, for audio files with auto length set the file
// will be visited to determine length if 'length_' is zero
unsigned long AudioData::length() const
{
  if (length_ != 0) {
    return length_;
  }

  if (type_ == FILE && autoLength_ && !autoLengthFailed_) {
    switch (audioDataLength(filename_, &(((AudioData*)this)->length_))) {
    case 0:
      if (startPos_ < length_) {
	((AudioData*)this)->length_ -= startPos_;
      }
      break;

    case 1:
      message(-2, "Cannot open audio file \"%s\": %s", filename_,
	      strerror(errno));
      ((AudioData*)this)->length_ = 0;
      ((AudioData*)this)->autoLengthFailed_ = 1;
      break;

    case 2:
      message(-2, "Cannot determine length of audio file \"%s\": %s",
	      filename_, strerror(errno));
      ((AudioData*)this)->length_ = 0;
      ((AudioData*)this)->autoLengthFailed_ = 1;
      break;

    case 3:
      message(-2, "Header of audio file \"%s\" is corrupted.",
	      filename_);
      ((AudioData*)this)->length_ = 0;
      ((AudioData*)this)->autoLengthFailed_ = 1;
      break;
    }
  }

  return length_;
}


// initiates reading audio data, an audio data file is opened
// return: 0: OK
//         1: file could not be opened
//         2: could not seek to start position
int AudioData::openData() const
{
  assert(open_ == 0);

  if (type_ == FILE) {
    long headerLength = 0;

    if ((((AudioData *)this)->fd_ = open(filename_, O_RDONLY)) < 0) {
      message(-2, "Cannot open audio file '%s': %s", filename_,
	      strerror(errno));
      return 1;
    }

    if (fileType_ == 2) {
      if ((headerLength = waveHeaderLength(filename_)) < 0) {
	message(-2, "%s: Unacceptable WAVE file.", filename_);
	return 1;
      }
    }

    if (lseek(fd_, headerLength + (startPos_ * sizeof(Sample)), SEEK_SET)
	< 0) {
      message(-2, "Cannot seek in audio file '%s': %s", filename_,
	      strerror(errno));
      return 2;
    }

    ((AudioData *)this)->headerLength_ = headerLength;
  }

  ((AudioData *)this)->readPos_ = 0;
  ((AudioData *)this)->open_ = 1;

  return 0;
}

// ends reading audio data, an audio data file is closed
void AudioData::closeData() const
{
  if (open_ != 0) {
    if (type_ == FILE) {
      close(fd_);
      ((AudioData *)this)->fd_ = 0;
    }

    ((AudioData *)this)->open_ = 0;
    ((AudioData *)this)->readPos_ = 0;
  }
}

// fills 'buffer' with 'len' samples
// return: number of samples written to buffer
long AudioData::readData(Sample *buffer, long len) const
{
  long readLen = 0;

  assert(open != 0);

  if (len == 0) {
    return 0;
  }

  if (readPos_ + len > length()) {
    if ((len = length() - readPos_) <= 0) {
      return 0;
    }
  }

  switch (type_) {
  case SILENCE:
    memset(buffer, 0, len * sizeof(Sample));
    readLen = len;
    break;

  case FILE:
    readLen = fullRead(fd_, buffer, len * sizeof(Sample));
    if (readLen < 0) {
      message(-2, "Read error while reading audio data from file '%s': %s",
	      filename_, strerror(errno));
    }
    else if (readLen != (long)(len * sizeof(Sample))) {
      message(-2, 
	      "Could not read expected amount of audio data from file '%s'.",
	      filename_);
      readLen = -1;
    }
    else {
      readLen = len;
    }
    break;
  }

  if (readLen > 0) {
    if (fileType_ == 2) {
      // wave file -> swap samples since wave files contain little endian data
      swapSamples(buffer, readLen);
    }
    ((AudioData *)this)->readPos_ += readLen;
  }

  return readLen;
}

// checks the consistency of object
// return: 0: OK
//         1: FILE: audio file cannot be opened or 'stat()' failed
//         2: FILE: audio file length  does not match requested portion
//         3: FILE: audio file length is not a multiple of 'sizeof(Sample)'
//         4: FILE: length is zero
int AudioData::check() const
{
  switch (type_) {
  case SILENCE:
    // always OK
    break;

  case FILE:
    {
      int fd;
      struct stat buf;
      long headerLength = 0;


      if ((fd = open(filename_, O_RDONLY)) < 0) {
	message(-2, "Cannot open audio file '%s': %s", filename_,
		strerror(errno));
	return 1;
      }

      if (fileType_ == 2) {
	if ((headerLength = waveHeaderLength(filename_)) < 0) {
	  message(-2, "%s: Unacceptable WAVE file.", filename_);
	  return 1;
	}
      }

      if (length() == 0) {
	message(-2, "Requested length for file '%s' is 0.",
		filename_);
	return 4;
      }

      if (fstat(fd, &buf) != 0) {
	message(-2, "Cannot fstat audio file '%s': %s", filename_,
		strerror(errno));
	close(fd);
	return 1;
      }

      if ((startPos_ + length()) * sizeof(Sample) + headerLength >
	  (unsigned long)buf.st_size) {
	// requested part exceeds file size
	message(-2,
		"Request length (%lu samples) exceeds length of file '%s'.",
		length(), filename_);
	close(fd);
	return 2;
      }

      if ((buf.st_size - headerLength) % sizeof(Sample) != 0) {
	// file size not a multiple of sizeof(Sample)
	message(-2, 
		"Length of file '%s' is not a multiple of sample size (4).",
		filename_);
	close(fd);
	return 3;
      }
    }
    break;
  }

  return 0;
}

// writes out contents of object in TOC file syntax
void AudioData::print(ostream &out) const
{
  switch (type()) {
  case FILE:
    out << "FILE \"" << filename_ << "\" ";

    if ((startPos() % SAMPLES_PER_BLOCK) == 0)
      out << Msf(startPos() / SAMPLES_PER_BLOCK).str();
    else
      out << startPos();
    
    out << " ";

    if ((length() % SAMPLES_PER_BLOCK) == 0)
      out << Msf(length() / SAMPLES_PER_BLOCK).str();
    else
      out << length();

    out << endl;
    break;

  case SILENCE:
    out << "SILENCE ";

    if ((length() % SAMPLES_PER_BLOCK) == 0)
      out << Msf(length() / SAMPLES_PER_BLOCK).str();
    else
      out << length();
    
    out << endl;
    break;
  }
}

// Seeks to specified sample.
// return: 0: OK
//        10: sample out of range
//        return code of 'AudioData::openData()'
int AudioData::seekSample(unsigned long sample) const
{
  assert(open_ != 0);

  if (sample >= length()) 
    return 10;

  if (type_ == FILE) {
    if (lseek(fd_, headerLength_ + (sample * sizeof(Sample)), SEEK_SET)	< 0) {
      message(-2, "Cannot seek in audio file '%s': %s", filename_,
	      strerror(errno));
      return 10;
    }
  }

  ((AudioData *)this)->readPos_ = sample;
  
  return 0;
}
