/*  VER 162   TAB P   $Id: putarticle.c,v 1.11 1998/09/11 16:37:44 src Exp $
 *
 *  POST articles via an NNTP server
 *
 *  copyright 1996, 1997 Egil Kvaleberg, egil@kvaleberg.no
 *  the GNU General Public License applies
 *
 *  $Log: putarticle.c,v $
 *  Revision 1.11  1998/09/11 16:37:44  src
 *  Lockfile for logfile of posted articles and for posted article folder.
 *
 *  Revision 1.10  1998/09/11 09:17:43  src
 *  Check path consistency (--no-path) and length (--max-path)
 *  GNU style option --help, --version, --dry-run, changed --noxx to --no-xx
 *  Check for putenv and setenv, added xstrcpy
 *
 *  Revision 1.9  1998/09/09 07:32:13  src
 *  Version 1.1
 *
 *  Revision 1.8  1998/09/02 06:50:31  src
 *  newsx version 1.0
 *
 *  Revision 1.7  1998/08/24 06:17:15  src
 *
 *  Revision 1.6  1998/07/22 10:56:31  src
 *  Implemented ad hoc support for "441 Duplicate" type response.
 */

#include "common.h"
#include "proto.h"
#include "options.h"
#include "statistics.h"
#include "news.h"
#include "nntp.h"

/*
 *  local stuff
 */
#define MAILFOLDER_TAG "From "
#define MAILFOLDER_TAG_LEN 5

/* for log etc. */
static char msgid[NNTP_STRLEN];
static char msgsender[NNTP_STRLEN];

static int article_lines = 0;

#ifndef HAVE_STRTOUL
/*
 *  for systems that doesn't have it, assuming base<=10
 */
long 
strtoul(char *str,char **endptr,int base)
{
    long u = 0L;
    char c;
    while (isspace(*str)) ++str;
    while (isdigit(c = *str++)) u = u*base + c-'0';
    *endptr = str;
    return u;
}
#endif
	
/*
 *  check if header tag
 */
int
is_tag(char *line, char *tag)
{
    char c,d;

    while ((c = *tag++)) {
	d = *line++;
	if (toupper(c) != toupper(d)) return 0;
    }
    return 1;
}
	
/*
 *  get current local time
 *  return static pointer
 */
static char *
current_time(void)
{
    time_t t;

    time(&t);
    return text_time(t);
}

/*
 *  check out header...
 *  update skip state
 */
static void 
check_header(char *line, int *skip)
{
    char *p;

    switch (toupper(line[0])) {
#if 0
    case 'C':
	if (is_tag(line,p="Content-Type: ")) {
	    if (!contenttype[0])
		strncpy(contenttype,line+strlen(p),sizeof(contenttype)-1);
	}
	break;
#endif
    case 'F':
	if (is_tag(line,p="From: ")) {
	    /* use From: if no Sender: */
	    if (!msgsender[0])
		strncpy(msgsender,line+strlen(p),sizeof(msgsender)-1);
	}
	break;
    case 'M':
	if (is_tag(line,p="Message-ID: ")) {
	    /* record local Message-ID */
	    strncpy(msgid,line+strlen(p),sizeof(msgid)-1);
	    if (nomsgid_opt && !ihave_opt) {
		/* but don't transmit it */
		*skip = 1;
		return;
	    }
	}
	break;
    case 'N':
	if (is_tag(line,"NNTP-Posting-Host:") && !ihave_opt) {
	    /*
	     * when POSTing, this header may cause
	     * messages to be rejected
	     * fix by: Riku Saikkonen <rjs@isil.lloke.dna.fi>               
	     *                 and Simon J. Mudd <sjmudd@bitmailer.net>
	     */
	    *skip = 1;
	    return;
	}
	break;
    case 'P':
	if (is_tag(line,"Path:")) {
	    int bangs = path_bangs(line+5);

	    if (bangs > max_path) {
		/* BUG: do something better */
		log_msg(L_ERR,"article path is %d steps long, max is %d",
					     bangs,max_path);
		/* BUG: don't send article */
		/* BUG: document */
	    }
	    if (!keep_path_opt && !ihave_opt) {
		/*
		 * skip local Path when POSTing:
		 * the main reason is that the local client
		 * may not have a registered host name
		 */
		*skip = 1;
		return;
	    }
	}
	break;
    case 'R':
	if (is_tag(line,p="Reply-to: ")) {
	    /* record Reply-to: */
	    strncpy(msgsender,line+strlen(p),sizeof(msgsender)-1);
	}
	break;
    case 'S':
	if (is_tag(line,p="Sender: ")) {
	    /* BUG: use Sender if no Reply-to */
	    strncpy(msgsender,line+strlen(p),sizeof(msgsender)-1);
	}
	break;
    case 'X':
	if (is_tag(line,"Xref:")) {
	    /* always omit local Xref */
	    *skip = 1;
	    return;
	} else if (is_tag(line,"X-Server-Date:")) {
	    if (!ihave_opt) {
		/* should not occur in postings */
		*skip = 1;
		return;
	    }
	    return;
	}
	break;
    case ' ':
    case '\t':
	/* continued header line, maintain skip state */
	return;
    }
    /* keep header line */
    *skip = 0;
}

/*
 *  sniff an article to find its message ID
 */
static char *
sniff_article(FILE *fp)
{
    char line[NNTP_STRLEN+2+1];
    int len;
    int dummy;
    int multi_1st = 0;
    int is_newline = 1;

    msgid[0] = '\0';
	
    /* read the article, header and body */
    progtitle("find message-ID");
    for (;;) {
	if (!fgets(line, NNTP_STRLEN, fp)) break;
	if ((len=strlen(line)) == 0) break;

	multi_1st = is_newline; /* full line? */
	if (line[len-1] == '\n') {
	    is_newline = 1;
	    line[len-1] = '\0';
	} else {
	    is_newline = 0;
	}
	if (multi_1st) {
	    if (len == 0) break; /* empty line: end of header */
	    check_header(line, &dummy);
	    if (msgid[0]) break; /* found header */
	}
    }
    /* rewind file */
    fseek(fp,0L,0);
    return msgid[0] ? msgid : 0;
}

/*
 *  transfer an article...
 *  set static variable article_lines
 */
static void
write_article(FILE *fp)
{
    char line[NNTP_STRLEN+2+1];
    int len;
    int hdr = 1;
    int skiphdr = 0;
    FILE *folder_file = 0;
    int lock_id = -1;
    int multi_1st = 0;
    int is_newline = 1;

    msgid[0] = '\0';
    msgsender[0] = '\0';
    article_lines = 0;

    /* transfer the article, header and body */
    progtitle("post: transfer article");
    for (;;) {
	line[NNTP_STRLEN] = '\0'; 
	if (!fgets(line, NNTP_STRLEN, fp)) break;

	/* must remove trailing linefeed */
	if ((len=strlen(line)) == 0) break;

	multi_1st = is_newline; /* if previous line ended here */
	if (line[len-1] == '\n') {
	    /* a complete line, or line termination */
	    is_newline = 1;
	    line[--len] = '\0';
	} else {
	    /*
	     * no newline detected - handle very long lines too
	     * problem pinpointed by Riku Saikkonen <rjs@isil.lloke.dna.fi>
	     */
	    is_newline = 0;
	}
	if (multi_1st) {
	    if (len == 0) {
		/* empty line: end of header */
		hdr = skiphdr = 0;
	    } else if (len == 1 && line[0] == '.') {
		/*
		 * posting contains EOF, so convert to something
		 * harmless according to RFC-977, section 3.10.1
		 * fix by Riku Saikkonen <rjs@isil.lloke.dna.fi>
		 */
		strcpy(line,"..");
		len = 2;
	    } 
	    if (hdr) {
		/* update skip state */
		check_header(line, &skiphdr);
	    }
	    ++article_lines;
	}
	if (!skiphdr) {
	    if (is_newline) {
		strcpy(line+len,newline);
		len += strlen(newline);
	    }
	    if (!put_server_msg(line)) exit_cleanup(9);
	
	    /* save to folder also */
	    if (folder) {
		if (!folder_file) {
		    char lockname[PATH_MAX];

		    /*
		     * lock other newsxes accessing same file
		     */
		    progtitle("locking folder");
		    build_filename(lockname,folder,LOCK_SUFFIX,NULL,NULL);
		    lock_id = lock("",lockname);

		    if (!(folder_file = fopen(folder,"a"))) {
			log_msg(L_ERRno,"can't open folder: %s",folder);
			folder = 0;
		    } else {
			/* folder item header, as for mail folders */
			fprintf(folder_file, "%s%s %s\n",
				MAILFOLDER_TAG, spoolname, current_time());
		    }
		}
		if (folder_file) {
		    /* save to folder too */
		    if (multi_1st 
		     && strncmp(line, MAILFOLDER_TAG, MAILFOLDER_TAG_LEN)==0) {
			/* mail folder hack */
			fputc('>', folder_file);
		    }
		    if (is_newline) {
			/* back to Unix convention */
			line[len-1] = '\0';
			line[len-2] = '\n';
		    }
		    fputs(line, folder_file);
		}
	    }
	    net_bytecount += len;
	}
    }
    if (!is_newline && !skiphdr) {
	/* no trailing newline, so we need to add one */
	if (!put_server_msg(newline)) exit_cleanup(9);
	if (folder_file) fputc('\n', folder_file);
    }

    if (folder_file) {
	/* BUG: what if posting failed... */
	/* BUG: or if noaction_opt.. */
	/* always a trailing blank line */
	fputc('\n',folder_file);
	fflush(folder_file);
	if (ferror(folder_file)) {
	    log_msg(L_ERRno,"error writing folder: %s",folder);
	}
	fclose(folder_file);
    }

    /* time to release lock */
    if (lock_id >= 0)
	unlock_one(lock_id);

    if (ferror(fp)) {
	/* panic */
	log_msg(L_ERRno,"error transferring article");
	exit_cleanup(1);
    }

    /* send termination */
    progtitle("post: send termination");
    sprintf(line, ".%s", newline);
    if (!put_server_msg(line)) exit_cleanup(9);

    /* article successfully read */
    log_msg(L_DEBUG,"(%d lines)", article_lines);
}

/*
 *  put_article following POST or IHAVE
 *  return string if all right
 *  return reason for failure
 */
static char *
put_article(FILE *fp, char **reason)
{
    static char status[NNTP_STRLEN+1];
    char *p;
    char *endptr;
    char *ok = 0;

    *reason = "?";

    /* read status line from server */
    if (!noaction_opt) {
	if (!get_server_nntp(status, sizeof(status))) {
	    exit_cleanup(9);
	}
    } else {
	sprintf(status,"%d",CONT_POST); /* fake OK */
    }
    switch (strtoul(status,&endptr,10)) {
    case CONT_POST:                 /* post: posting allowed, continue */
    case CONT_XFER:                 /* ihave: go ahead */
	write_article(fp);
	break;

    case ERR_POSTFAIL:              /* post: failed for some other reason */
	log_msg(L_ERR,"posting failed: got \"%s\"", status);
	++failed_articles;
	*reason = status;
	return 0; /* try again */

    case ERR_GOTIT:                 /* ihave: already got it */
	log_msg(L_DEBUG,"already got it");
	++duplicate_articles;
	return "Already got it";

    case ERR_XFERFAIL:              /* ihave: try again later */
	log_msg(L_ERR,"try again later");
	++failed_articles;
	*reason = status;
	return 0; /* try again */

    case ERR_NOPOST:                /* post: not allowed */
    case ERR_GOODBYE:               /* ihave: you do not have permission */
	log_msg(L_ERR,"sending prohibited: got \"%s\"", status);
	++failed_articles;
	*reason = status;
	return 0; /* don't give up */

    case ERR_XFERRJCT:              /* ihave: rejected - do not try again */
	log_msg(L_ERR,"rejected: got \"%s\"", status);
	++failed_articles;
	*reason = status;
	return 0; /* BUG: don't give up */

    /* otherwise must be a protocol error */
    default:
	log_msg(L_ERR,"NNTP send negotiation error: got \"%s\"", status);
	exit_cleanup(4);
    }

    /* get status of posting */
    progtitle("post: get status");
    if (!noaction_opt) {
	if (!get_server_nntp(status, sizeof(status))) {
	    exit_cleanup(9);
	} 
    } else {
	sprintf(status,"%d",OK_POSTED);
    }

    switch (strtoul(status,&endptr,10)) {
    case OK_XFERED:                 /* ihave: transferred OK */
    case OK_POSTED:                 /* post: article posted OK */
	++posted_articles;
	if (!noaction_opt)
	    ok = "OK";
	else
	    ok = "TEST";
	break;

    case ERR_NOPOST:                /* post: not allowed */
    case ERR_GOODBYE:               /* ihave: you do not have permission */
	log_msg(L_ERR,"sending not allowed: got \"%s\"", status);
	*reason = status;
	break; /* don't give up */

    case ERR_GOTIT:                 /* ihave: already got it */
	/* we'll just continue */
	log_msg(L_DEBUG,"already got it");
	++duplicate_articles;
	ok = "Already got it";
	break;

    case ERR_POSTFAIL:              /* post: failed */
    case ERR_XFERFAIL:              /* ihave: try again later */
    case ERR_XFERRJCT:              /* ihave: rejected - do not try again */
	/* see if secondary error message */
	if (strtoul(endptr,NULL,10) == ERR_GOTIT) {
	    /* already gotit: we'll just continue */
	    log_msg(L_DEBUG,"duplicate");
	    ++duplicate_articles;
	    ok = "Duplicate";

	} else if ((p = strchr(endptr,'D'))
		   && strncmp(p,"Duplicate",9)==0) {
	    /*
	     * this is really adhoc - the response is something like:
	     *  441 Posting Failed (Duplicate Message-ID)
	     */
	    log_msg(L_DEBUG,"assumed duplicate");
	    ++duplicate_articles;
	    ok = "Assumed duplicate";

	} else {
	    /* posting failed, some other reason */
	    log_msg(L_ERR,"article rejected: got \"%s\"", status);
	    ++failed_articles;
	    *reason = status;
	    break;
	}
	break;

    default:                        /* otherwise, protocol error */
	log_msg(L_ERR,"NNTP sending protocol error: got \"%s\"", status);
	exit_cleanup(4);
    }

    return ok;
}

/*
 *  submit an article with specified id to the server
 */
static void 
post_article(void)
{
    char buf[NNTP_STRLEN+1];
    progtitle("post: issuing POST");
    sprintf(buf, "POST%s",newline);
    if (!put_server(buf)) exit_cleanup(9);
}

/*
 *  as post_article, but use IHAVE mechanism 
 */
static void 
ihave_article(char *msgid)
{
    char buf[NNTP_STRLEN+1];
    progtitle("post: issuing IHAVE");
    sprintf(buf, "IHAVE %s%s", msgid,newline);
    if (!put_server(buf)) exit_cleanup(9);
}

/*
 *  log to file
 *  BUG: rather inefficient
 */
static void
log_to_file(char *articlename, char *ok)
{
    FILE *fp;
    int lock_id;
    char lockname[PATH_MAX];

    if (!logfile) return;

    /*
     *  lock other newsxes accessing same file
     */
    progtitle("locking log file");
    build_filename(lockname,logfile,LOCK_SUFFIX,NULL,NULL);
    lock_id = lock("",lockname);

    progtitle("post: logging");
    if (!(fp = fopen(logfile,"a"))) {
	log_msg(L_ERRno,"can't open logfile: %s",logfile);
    } else {
	fprintf(fp,"%s %s %s %s %s %s, %d lines\n",
		    current_time() + 4, /* skip "Day " */
		       spoolname,
			  msgid[0] ? msgid : "<?>",
			     articlename,
				msgsender[0] ? msgsender : "?",
				   ok,
				       article_lines);
	if (fclose(fp) == EOF) {
	    log_msg(L_ERRno,"can't write to logfile: %s",logfile);
	}
    }
    unlock_one(lock_id);
}

/*
 *  submit articles to the currently open NNTP server socket.
 *  return string if article should be removed from outgoing batch
 */
char *
submit_article(char *articlename)
{
    FILE *art;
    long age;
    char *ok;
    char *id;
    char *to;
    char *reason;
    char fullname[PATH_MAX];

    /* work with full and partial paths */
    build_alt_filename(fullname,spooldir,articlename,NULL);

    progtitle("post: reading article");
    log_msg(L_DEBUG,"reading %s", fullname);

    if ((art = fopen(fullname, "r")) == NULL) {
	/* article is missing - throw it away */
	log_msg(L_ERRno,"can't find article \"%s\"",fullname);
	++missing_articles;
	ok = "Missing";
    } else {
	if (ihave_opt) {
	    /* use ihave */
	    if (!(id = sniff_article(art))) {
		log_msg(L_ERR,"article \"%s\" lacks a message ID",fullname);
		++missing_articles;
		ok = "Missing-ID";
	    } else {
		ihave_article(id);
		ok = put_article(art,&reason);
	    }
	} else {
	    /* request post */
	    /* BUG: check if post_allowed? */
	    post_article();

	    /* and do it */
	    ok = put_article(art,&reason);
	}
    }

    if (ok) {
	log_to_file(articlename,ok);
    } else {
	if (failtime) {
	    /* timeout message... */
	    age = how_old(art,fullname);
	    log_msg(L_DEBUG,"article is %ds old, limit is %ds", age, failtime);
	    if (age > failtime) {
		log_msg(L_ERR,"failed article \"%s\" killed",fullname);
		ok = "Failed";
		if (!bounce || !bounce[0] || strcmp(bounce,"poster")==0) {
		    /* return to sender */
		    if (!msgsender[0]) {
			log_msg(L_ERR,"no From-specification in \"%s\"",
								    fullname);
			to = 0;
		    } else {
			to = msgsender;
		    }
		} else if (strcmp(bounce,"none")==0) {
		    to = 0;
		} else {
		    /* bounce address specified */
		    to = bounce;
		}
		if (to) {
		    ++bounced_articles;
		    log_msg(L_ERR,"bounce message to %s",to);
		    if (!bounce_msg(to,art,fullname,msgid,reason)) {
			/* BUG: what shall we do? */
			log_msg(L_ERR,"bounce failed");
		    }
		}
	    }
	}
    }
    if (art) fclose(art);

    return ok;
}


