
/***

hostmon.c - Host traffic monitor
Discovers LAN hosts and displays packet statistics for them
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#ifndef _I386_TYPES_H
#include <asm/types.h>
#endif

#include <curses.h>
#include <panel.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#include <linux/if_fddi.h>
#include <net/if_arp.h>
#include <stdlib.h>
#include <sys/time.h>
#include "dirs.h"
#include "deskman.h"
#include "packet.h"
#include "hostmon.h"
#include "stdwinset.h"
#include "attrs.h"
#include "log.h"
#include "timer.h"
#include "landesc.h"
#include "options.h"
#include "instances.h"
#include "error.h"

#define SCROLLUP 0
#define SCROLLDOWN 1

/* 
 * from log.c, applicable only to this module 
 */

extern void writeethlog(struct ethtabent *list, unsigned long nsecs, FILE * logfile);
extern char *ltrim(char *buf);

void ethlook(struct desclist *list, char *address, char *target)
{
    struct desclistent *ptmp = list->head;

    while (ptmp != NULL) {
	if (strcmp(address, ptmp->rec.address) == 0) {
	    strcpy(target, ptmp->rec.desc);
	    return;
	}
	ptmp = ptmp->next_entry;
    }
}

void initethtab(struct ethtab *table)
{
    table->head = table->tail = NULL;
    table->firstvisible = table->lastvisible = NULL;
    table->count = table->entcount = 0;

    table->borderwin = newwin(LINES - 2, 80, 1, 0);
    table->borderpanel = new_panel(table->borderwin);

    table->tabwin = newwin(LINES - 4, 78, 2, 1);
    table->tabpanel = new_panel(table->tabwin);

    wattrset(table->borderwin, BOXATTR);
    box(table->borderwin, ACS_VLINE, ACS_HLINE);
    wmove(table->borderwin, 0, 5);
    wprintw(table->borderwin, " PktsIn ");
    wmove(table->borderwin, 0, 16);
    wprintw(table->borderwin, " IP In ");
    wmove(table->borderwin, 0, 24);
    wprintw(table->borderwin, " BytesIn ");
    wmove(table->borderwin, 0, 34);
    wprintw(table->borderwin, " InRate ");

    wmove(table->borderwin, 0, 42);
    wprintw(table->borderwin, " PktsOut ");
    wmove(table->borderwin, 0, 53);
    wprintw(table->borderwin, " IP Out ");
    wmove(table->borderwin, 0, 61);
    wprintw(table->borderwin, " BytesOut ");
    wmove(table->borderwin, 0, 70);
    wprintw(table->borderwin, " OutRate ");

    wmove(table->borderwin, LINES - 3, 40);
    wprintw(table->borderwin, " InRate and OutRate are in kbits/sec ");

    wattrset(table->tabwin, STDATTR);
    colorwin(table->tabwin);
    stdwinset(table->tabwin);
    wtimeout(table->tabwin, -1);

    update_panels();
    doupdate();
}

struct ethtabent *addethnode(struct ethtab *table, int *nomem)
{
    struct ethtabent *ptemp;

    ptemp = malloc(sizeof(struct ethtabent));

    if (ptemp == NULL) {
	printnomem();
	*nomem = 1;
	return NULL;
    }
    if (table->head == NULL) {
	ptemp->prev_entry = NULL;
	table->head = ptemp;
	table->firstvisible = ptemp;
    } else {
	ptemp->prev_entry = table->tail;
	table->tail->next_entry = ptemp;
    }

    table->tail = ptemp;
    ptemp->next_entry = NULL;

    table->count++;
    ptemp->index = table->count;

    if (table->count <= LINES - 4)
	table->lastvisible = ptemp;

    return ptemp;
}

void convmacaddr(char *addr, char *result)
{
    unsigned int i;
    __u8 *ptmp = addr;
    char hexbyte[3];

    strcpy(result, "");
    for (i = 0; i <= 5; i++) {
	sprintf(hexbyte, "%02x", *ptmp);
	strcat(result, hexbyte);
	ptmp++;
    }
}

struct ethtabent *addethentry(struct ethtab *table, unsigned int linktype, 
			      char *ifname, char *addr, int *nomem,
			      struct desclist *list)
{
    struct ethtabent *ptemp;

    ptemp = addethnode(table, nomem);

    if (ptemp == NULL)
	return NULL;

    ptemp->type = 0;
    memcpy(&(ptemp->un.desc.eth_addr), addr, ETH_ALEN);
    strcpy(ptemp->un.desc.desc, "");
    convmacaddr(addr, ptemp->un.desc.ascaddr);
    ptemp->un.desc.linktype = linktype;
    ethlook(list, ptemp->un.desc.ascaddr, ptemp->un.desc.desc);
    strcpy(ptemp->un.desc.ifname, ifname);
    
    if (strcmp(ptemp->un.desc.desc, "") == 0)
	ptemp->un.desc.withdesc = 0;
    else
	ptemp->un.desc.withdesc = 1;

    ptemp->un.desc.printed = 0;

    ptemp = addethnode(table, nomem);

    if (ptemp == NULL)
	return NULL;

    ptemp->type = 1;
    ptemp->un.figs.inpcount = ptemp->un.figs.inpktact = 0;
    ptemp->un.figs.outpcount = ptemp->un.figs.outpktact = 0;
    ptemp->un.figs.inspanbr = ptemp->un.figs.outspanbr = 0;
    ptemp->un.figs.inippcount = ptemp->un.figs.outippcount = 0;
    ptemp->un.figs.inbcount = ptemp->un.figs.outbcount = 0;
    ptemp->un.figs.inrate = ptemp->un.figs.outrate = 0;
    ptemp->un.figs.past5 = 0;

    table->entcount++;

    wmove(table->borderwin, LINES - 3, 1);
    wprintw(table->borderwin, " %u entries ", table->entcount);

    return ptemp;
}

struct ethtabent *in_ethtable(struct ethtab *table, unsigned int linktype, char *addr)
{
    struct ethtabent *ptemp = table->head;

    while (ptemp != NULL) {
	if ((ptemp->type == 0) &&
	    (memcmp(addr, ptemp->un.desc.eth_addr, ETH_ALEN) == 0) &&
	    (ptemp->un.desc.linktype == linktype))
	    return ptemp->next_entry;

	ptemp = ptemp->next_entry;
    }

    return NULL;
}

void updateethent(struct ethtabent *entry, unsigned int pktsize,
		  int is_ip, int inout)
{
    if (inout == 0) {
	entry->un.figs.inpcount++;
	entry->un.figs.inbcount += pktsize;
	entry->un.figs.inspanbr += pktsize;
	if (is_ip)
	    entry->un.figs.inippcount++;
    } else {
	entry->un.figs.outpcount++;
	entry->un.figs.outbcount += pktsize;
	entry->un.figs.outspanbr += pktsize;
	if (is_ip)
	    entry->un.figs.outippcount++;
    }
}

void printethent(struct ethtab *table, struct ethtabent *entry,
		 unsigned int idx)
{
    unsigned int target_row;

    if ((entry->index < idx) || (entry->index > idx + LINES - 5))
	return;

    target_row = entry->index - idx;

    if (entry->type == 0) {
	wmove(table->tabwin, target_row, 1);
	wattrset(table->tabwin, STDATTR);
	
	if (entry->un.desc.linktype == ARPHRD_ETHER)
	    wprintw(table->tabwin, "Ethernet");
	else if (entry->un.desc.linktype == ARPHRD_FDDI)
	    wprintw(table->tabwin, "FDDI");
	
	wprintw(table->tabwin, " HW addr: %s", entry->un.desc.ascaddr);

	if (entry->un.desc.withdesc)
	    wprintw(table->tabwin, " (%s)", entry->un.desc.desc);

	wprintw(table->tabwin, " on %s", entry->un.desc.ifname);
	
	entry->un.desc.printed = 1;
    } else {
	wattrset(table->tabwin, PTRATTR);
	wmove(table->tabwin, target_row, 1);
	waddch(table->tabwin, ACS_LLCORNER);

	wattrset(table->tabwin, HIGHATTR);

	/* Inbound traffic counts */

	wmove(table->tabwin, target_row, 2);
	printlargenum(entry->un.figs.inpcount, table->tabwin);
	wmove(table->tabwin, target_row, 12);
	printlargenum(entry->un.figs.inippcount, table->tabwin);
	wmove(table->tabwin, target_row, 22);
	printlargenum(entry->un.figs.inbcount, table->tabwin);
	
	/* Outbound traffic counts */

	wmove(table->tabwin, target_row, 40);
	printlargenum(entry->un.figs.outpcount, table->tabwin);
	wmove(table->tabwin, target_row, 50);
	printlargenum(entry->un.figs.outippcount, table->tabwin);
	wmove(table->tabwin, target_row, 60);
	printlargenum(entry->un.figs.outbcount, table->tabwin);
    }
}

void destroyethtab(struct ethtab *table)
{
    struct ethtabent *ptemp = table->head;
    struct ethtabent *cnext = NULL;

    if (table->head != NULL)
	cnext = table->head->next_entry;

    while (ptemp != NULL) {
	free(ptemp);
	ptemp = cnext;

	if (cnext != NULL)
	    cnext = cnext->next_entry;
    }
}

void hostmonhelp()
{
    move(LINES - 1, 1);
    scrollkeyhelp();
    stdexitkeyhelp();
}

void printrates(struct ethtab *table, unsigned int target_row,
		struct ethtabent *ptmp)
{
    if (ptmp->un.figs.past5) {
	wmove(table->tabwin, target_row, 32);
	wprintw(table->tabwin, "%8.1f", ptmp->un.figs.inrate);
	wmove(table->tabwin, target_row, 69);
	wprintw(table->tabwin, "%8.1f", ptmp->un.figs.outrate);
    }
}

void updateethrates(struct ethtab *table, time_t starttime, time_t now,
		    unsigned int idx)
{
    struct ethtabent *ptmp = table->head;
    unsigned int target_row = 0;

    if (table->lastvisible == NULL)
	return;

    while (ptmp != NULL) {
	if (ptmp->type == 1) {
	    ptmp->un.figs.past5 = 1;
	    ptmp->un.figs.inrate =
		((float) (ptmp->un.figs.inspanbr * 8 / 1000)) / ((float) (now - starttime));
	    ptmp->un.figs.outrate =
		((float) (ptmp->un.figs.outspanbr * 8 / 1000)) / ((float) (now - starttime));
	    if ((ptmp->index >= idx) && (ptmp->index <= idx + LINES - 5)) {
		wattrset(table->tabwin, HIGHATTR);
		target_row = ptmp->index - idx;
		printrates(table, target_row, ptmp);
	    }
	    ptmp->un.figs.inspanbr = ptmp->un.figs.outspanbr = 0;
	}
	ptmp = ptmp->next_entry;
    }
}

void scrollethwin(struct ethtab *table, int direction, int *idx)
{
    wattrset(table->tabwin, STDATTR);
    if (direction == SCROLLUP) {
	if (table->lastvisible != table->tail) {
	    wscrl(table->tabwin, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;
	    (*idx)++;
	    wmove(table->tabwin, LINES - 5, 0);
	    scrollok(table->tabwin, 0);
	    wprintw(table->tabwin, "%78c", ' ');
	    scrollok(table->tabwin, 1);
	    printethent(table, table->lastvisible, *idx);
	    if (table->lastvisible->type == 1)
		printrates(table, LINES - 5, table->lastvisible);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->tabwin, -1);
	    table->lastvisible = table->lastvisible->prev_entry;
	    table->firstvisible = table->firstvisible->prev_entry;
	    (*idx)--;
	    wmove(table->tabwin, 0, 0);
	    wprintw(table->tabwin, "%78c", ' ');
	    printethent(table, table->firstvisible, *idx);
	    if (table->firstvisible->type == 1)
		printrates(table, 0, table->firstvisible);
	}
    }
}

void pageethwin(struct ethtab *table, int direction, int *idx)
{
    int i = 1;

    if (direction == SCROLLUP) {
	while ((i <= LINES - 7) && (table->lastvisible != table->tail)) {
	    i++;
	    scrollethwin(table, direction, idx);
	}
    } else {
	while ((i <= LINES - 7) && (table->firstvisible != table->head)) {
	    i++;
	    scrollethwin(table, direction, idx);
	}
    }
}

/*
 * The LAN station monitor
 */
 
void hostmon(const struct OPTIONS *options, int facilitytime)
{
    int logging = options->logging;
    int fd;
    struct ethtab table;
    struct ethtabent *entry;
    struct sockaddr_pkt fromaddr;

    int br;
    char buf[8192];
    char scratch_saddr[6];
    char scratch_daddr[6];
    unsigned int idx = 1;
    int is_ip;
    int ch;
    int endloop = 0;

    struct timeval tv;
    unsigned long starttime;
    unsigned long now = 0, unow = 0;
    unsigned long statbegin = 0, startlog = 0;
    unsigned long updtime = 0, updtime_usec = 0;

    struct desclist elist;             /* Ethernet description list */
    struct desclist flist;	       /* FDDI description list */
    struct desclist *list = NULL;

    FILE *logfile = NULL;

    int nomem = 0;

    if (!facility_active(LANMONIDFILE))
        mark_facility(LANMONIDFILE, "LAN monitor");
    else {
        errbox("LAN station monitor already active in another process",
               ANYKEY_MSG, &ch);
	return;
    }        

    hostmonhelp();

    initethtab(&table);
    loaddesclist(&elist, ARPHRD_ETHER);
    loaddesclist(&flist, ARPHRD_FDDI);

    if (logging) {
	opentlog(&logfile, LANLOG);

	if (logfile == NULL)
	    logging = 0;
    }
    writelog(logging, logfile, "******** LAN traffic monitor started");
    open_socket(&fd, "");

    if (fd < 0)
	return;
	
    leaveok(table.tabwin, TRUE);

    gettimeofday(&tv, NULL);
    starttime = statbegin = startlog = tv.tv_sec;

    do {
	getpacket(fd, buf, &fromaddr, &ch, &br, 5, table.tabwin);

	if (ch != ERR)
	    switch (ch) {
	    case KEY_UP:
		scrollethwin(&table, SCROLLDOWN, &idx);
		break;
	    case KEY_DOWN:
		scrollethwin(&table, SCROLLUP, &idx);
		break;
	    case KEY_PPAGE:
	    case '-':
		pageethwin(&table, SCROLLDOWN, &idx);
		break;
	    case KEY_NPAGE:
	    case ' ':
		pageethwin(&table, SCROLLUP, &idx);
		break;
	    case 'q':
	    case 'Q':
	    case 'x':
	    case 'X':
	    case 24:
		endloop = 1;
	    }
	if (br > 0) {
	    if ((fromaddr.spkt_family == ARPHRD_ETHER) ||
	        (fromaddr.spkt_family == ARPHRD_FDDI)) {
		if (fromaddr.spkt_protocol == htons(ETH_P_IP))
		    is_ip = 1;
		else
		    is_ip = 0;

		/*
		 * Check source address entry
		 */
		
		if (fromaddr.spkt_family == ARPHRD_ETHER) {
		    memcpy(scratch_saddr, ((struct ethhdr *) buf)->h_source, ETH_ALEN);
		    memcpy(scratch_daddr, ((struct ethhdr *) buf)->h_dest, ETH_ALEN);
		    list = &elist;
		} else if (fromaddr.spkt_family == ARPHRD_FDDI) {
		    memcpy(scratch_saddr, ((struct fddihdr *) buf)->saddr, FDDI_K_ALEN);
		    memcpy(scratch_daddr, ((struct fddihdr *) buf)->daddr, FDDI_K_ALEN);
		    list = &flist;
		}
		    
		entry = in_ethtable(&table, fromaddr.spkt_family, scratch_saddr);

		if ((entry == NULL) && (!nomem))
		    entry = addethentry(&table, fromaddr.spkt_family, fromaddr.spkt_device, scratch_saddr, &nomem, list);

		if (entry != NULL) {
		    updateethent(entry, br, is_ip, 1);
		    if (!entry->prev_entry->un.desc.printed)
			printethent(&table, entry->prev_entry, idx);

		    printethent(&table, entry, idx);
		}
		/*
		 * Check destination address entry
		 */

		entry = in_ethtable(&table, fromaddr.spkt_family, scratch_daddr);
		if ((entry == NULL) && (!nomem))
		    entry = addethentry(&table, fromaddr.spkt_family, fromaddr.spkt_device, scratch_daddr, &nomem, list);

		if (entry != NULL) {
		    updateethent(entry, br, is_ip, 0);
		    if (!entry->prev_entry->un.desc.printed)
			printethent(&table, entry->prev_entry, idx);

		    printethent(&table, entry, idx);
		}
	    }
	}
        gettimeofday(&tv, NULL);
	now = tv.tv_sec;
	unow = tv.tv_usec;
	
	if ((now - starttime) >= 5) {
	    printelapsedtime(statbegin, now, LINES - 3, 15, table.borderwin);
	    updateethrates(&table, starttime, now, idx);
	    starttime = now;
	}
	if (((now - startlog) >= options->logspan) && (logging)) {
	    writeethlog(table.head, now - statbegin, logfile);
	    startlog = now;
	}
	if (((options->updrate != 0) && (now - updtime >= options->updrate)) ||
	   ((options->updrate == 0) && (unow - updtime_usec >= HOSTMON_UPDATE_DELAY))) {
	    update_panels();
	    doupdate();
	    updtime = now;
	    updtime_usec = unow;
	}
	
	if ((facilitytime != 0) && (((now - statbegin) / 60) >= facilitytime))
	    endloop = 1;
    } while (!endloop);

    if (logging) {
	writeethlog(table.head, time((time_t *) NULL) - statbegin, logfile);
	writelog(logging, logfile, "******** LAN traffic monitor stopped");
	fclose(logfile);
    }
    del_panel(table.tabpanel);
    delwin(table.tabwin);
    del_panel(table.borderpanel);
    delwin(table.borderwin);
    update_panels();
    doupdate();
    destroyethtab(&table);
    destroydesclist(&elist);
    destroydesclist(&flist);
    unlink(LANMONIDFILE);
}
