
/*
 * FRADCFG	A configuration program for Frame Relay Access Devices in
 *              Linux.  Used along with 'fradcfg'.
 *
 * Version:	@(#)fradcfg.c	0.20	25 Jan 1998
 *
 * Author:	Mike McLagan <mike.mclagan@linux.org>
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#ifdef __GLIBC__
#define _LINUX_SOCKET_H
#endif

#include <linux/if.h>
#include <linux/if_frad.h>
#include <linux/sdla.h>

#include "cfgfile.h"
#include "tags.h"

char *banner = "FRAD configuration, v0.20(25 Jan 1998) by mike.mclagan@linux.org";

void usage(char *progname, char *invalid)
{
   char *temp;

   temp = strrchr(progname, '/');
   if (temp)
   {
      *temp = 0;
      progname = temp + 1;
   }

   if (invalid)
      printf("Invalid argument specified: %s\n", invalid);

   printf("Usage: %s config_file\n", progname);
   exit(-1); 
}

void read_dev_conf(void *cfg_info, char *dev, struct frad_conf *conf)
{
   char             *temp;
   int              len;
   char             flags[128];

   temp = config_value(cfg_info, dev, KTAG_ACCESS);
   if (temp)
   {
      len = strlen(temp);
      if (strncasecmp(VNODE_CPE, temp, len) == 0)
         conf->station = FRAD_STATION_CPE;
      else
         if (strncasecmp(VNODE_NODE, temp, len) == 0)
            conf->station = FRAD_STATION_NODE;
         else
         {
            fprintf(stderr, "Unknown access type %s for device %s\n", temp, dev);
            exit(-EINVAL);
         }
   }

   temp = config_value(cfg_info, dev, KTAG_CLOCK);
   if (temp)
   {
      len = strlen(temp);
      if (strncasecmp(VCLOCK_INTERNAL, temp, len) == 0)
         conf->clocking = FRAD_CLOCK_INT;
      else
         if (strncasecmp(VCLOCK_EXTERNAL, temp, len) == 0)
            conf->clocking = FRAD_CLOCK_EXT;
         else
         {
            fprintf(stderr, "Unknown clocking method %s for device %s\n", temp, dev);
            exit(-EINVAL);
         }
   }

   temp = config_value(cfg_info, dev, KTAG_MTU);
   if (temp)
      conf->mtu = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_BAUD);
   if (temp)
      conf->kbaud = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_T391);
   if (temp)
      conf->T391 = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_T392);
   if (temp)
      conf->T392 = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_N391);
   if (temp)
      conf->N391 = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_N392);
   if (temp)
      conf->N392 = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_N393);
   if (temp)
      conf->N393 = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_F_CIR);
   if (temp)
      conf->CIR_fwd = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_B_CIR);
   if (temp)
      conf->CIR_bwd = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_F_BC);
   if (temp)
      conf->Bc_fwd = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_B_BC);
   if (temp)
      conf->Bc_bwd = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_F_BE);
   if (temp)
      conf->Be_fwd = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_B_BE);
   if (temp)
      conf->Be_bwd = atoi(temp);

   temp = config_value(cfg_info, dev, KTAG_FLAGS);
   if (temp)
   {
      strcpy(flags, temp);
      for(temp = strtok(flags, ",");temp;temp = strtok(NULL, ","))
      {
         len = strlen(temp);
         if (strncasecmp(VFLAG_TXIGNORE, temp, len) == 0)
         {
            conf->flags |= FRAD_TX_IGNORE_CIR;
            continue;
         }

         if (strncasecmp(VFLAG_RXIGNORE, temp, len) == 0)
         {
            conf->flags |= FRAD_RX_ACCOUNT_CIR;
            continue;
         }

         if (strncasecmp(VFLAG_DROPABORT, temp, len) == 0)
         {
            conf->flags |= FRAD_DROP_ABORTED;
            continue;
         }

         if (strncasecmp(VFLAG_BUFFERIF, temp, len) == 0)
         {
            conf->flags |= FRAD_BUFFERIF;
            continue;
         }

         if (strncasecmp(VFLAG_STATS, temp, len) == 0)
         {
            conf->flags |= FRAD_STATS;
            continue;
         }

         if (strncasecmp(VFLAG_MCI, temp, len) == 0)
         {
            conf->flags |= FRAD_MCI;
            continue;
         }

         if (strncasecmp(VFLAG_AUTODLCI, temp, len) == 0)
         {
            conf->flags |= FRAD_AUTODLCI;
            continue;
         }

         fprintf(stderr, "Unknown or ambiguous flag %s for device %s\n", temp, dev);
         exit(-EINVAL);
      }
   }
}

void main(int argc, char *argv[])
{
   int              quiet;
   char             c;
   struct frad_conf def = {FRAD_STATION_CPE, 0, 64, FRAD_CLOCK_INT,
                           SDLA_MAX_MTU, 10, 15, 6, 3, 4, 64, 512, 0, 64, 512, 0};
   struct frad_conf conf;
   void             *cfg_info;
   char             *temp, *dev;
   int              count, i, len;
   char             buf[32];

   quiet = 0;

   while ((c=getopt(argc, argv, "Q")) > 0)
      switch (c)
      {
         case 'Q' : 
            quiet = 1;
            break;

         default:
            usage(argv[0], argv[optind - 1]);
            break;
      }

   if (!quiet)
      printf("%s.\n",  banner);
   
   if (argc <= optind)
      usage(argv[0], NULL);

   if (argc > optind + 1)
      usage(argv[0], argv[optind+1]);

   cfg_info = config_read(argv[optind]);
   if (!cfg_info)
   {
      fprintf(stderr, "Config file missing.\n");
      exit(-EINVAL);
   }

   temp = config_value(cfg_info, ATAG_MAIN, KTAG_DEVCOUNT);
   if (!temp)
   {
      fprintf(stderr, "Device count not found.\n");
      exit(-EINVAL);
   }

   read_dev_conf(cfg_info, ATAG_MAIN, &def);
   
   count = atoi(temp);

   for(i=1;i<=count;i++)
   {
      sprintf(buf, KTAG_DEV, i);
      dev = config_value(cfg_info, ATAG_MAIN, buf);
      if (!dev)
      {
         fprintf(stderr, "Device %i not found.\n", i);
         exit(-EINVAL);
      }

      memcpy(&conf, &def, sizeof(def));
      read_dev_conf(cfg_info, dev, &conf);

      temp = config_value(cfg_info, dev, KTAG_TYPE);
      if (!temp)
      {
         fprintf(stderr, "Type for device %s not found.\n", dev);
         exit(-EINVAL);
      }

      len = strlen(temp);
      if ((strncasecmp(VTYPE_SANGOMA, temp, len) == 0) && (len >= 1))
         sdla(cfg_info, dev, &conf, quiet);
      else
      {
         fprintf(stderr, "Unknown or ambiguous device type %s for device %s\n", temp, dev);
         exit(-EINVAL);
      }
   }
   config_clean(cfg_info);
}
