/*
   Pathetic Writer
   Copyright (C) 1997, 1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * matrix.c
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "../common/cmalloc.h"
#include "../common/fonts.h"

#include "pw.h"

int max_lines = BUFFER_ROWS;
int max_columns = BUFFER_COLS;


style styles[16] = {
	{"Default", TIMES | SIZE_12, 13, STY_DEFAULT},
	{"Header 1", HELVETICA | SIZE_24 | BOLD, 26, STY_DEFAULT},
	{"Header 2", HELVETICA | SIZE_18 | BOLD, 26, STY_DEFAULT},
	{"Header 3", HELVETICA | SIZE_14 | BOLD, 18, STY_DEFAULT},
	{"Header 4", HELVETICA | SIZE_12 | BOLD, 18, STY_DEFAULT},
	{"Header 5", HELVETICA | SIZE_10 | BOLD, 18, STY_DEFAULT},
	{"Header 6", HELVETICA | SIZE_8 | BOLD, 18, STY_DEFAULT},
	{"Address", TIMES | SIZE_12 | ITALIC, 13, STY_ADDRESS},
	{"Title", TIMES | SIZE_18 | BOLD, 26, STY_TITLE},
	{"Abstract", TIMES | SIZE_14 | BOLD, 13, STY_ABSTRACT},
	{"Preformatted", COURIER | SIZE_12, 13, STY_PREFORMAT},
	{"User 1", TIMES | SIZE_12, 13, STY_USER1},
	{"User 2", TIMES | SIZE_12, 13, STY_USER2},
	{"User 3", TIMES | SIZE_12, 13, STY_USER3},
	{"User 4", TIMES | SIZE_12, 13, STY_USER4},
	{"Embed", TIMES | SIZE_12, 13, STY_DEFAULT}
};


char *fmt_put(char *fmt, int mode)
{
	char *p;

	if (mode > STY_EMBED)	/* old file, adjust */
		mode >>= FMT_SHIFT;

	p = strtok(fmt, ",\n");
	if (!p) return NULL;
	styles[mode].name = cstrdup(p);
	p = strtok(NULL, ",\n");
	if (!p) return NULL;
	styles[mode].format = atol(p);
	p = strtok(NULL, ",\n");
	if (!p) return NULL;
	styles[mode].height = atoi(p);
	p = strtok(NULL, ",\n");
	if (!p) return NULL;
	styles[mode].follower = atoi(p);
	return fmt;
}

char *fmt_get(char *fmt, int mode)
{
	static char buf[80];

	if (fmt == NULL) {
		fmt = buf;
	}

	sprintf(fmt, "%s,%ld,%d,%d",
		styles[mode].name, styles[mode].format,
		styles[mode].height, styles[mode].follower);
	return fmt;
}

/*X
   void free_matrix(spread **matrix)
   Frees the memory used by the matrix.
   X */
void free_text(rich_text *c)
{
	cfree(c);
}

static rich_char empty_char = {
	'\0', HELVETICA | SIZE_12
};

#ifndef MAX
#define MAX(a,b) ((a)>(b)?(a):(b))
#endif


/* make sure the char is allocated */
/* note that this function changes the buffer */
/* If row is greater than any previously used line, new lines are allocated
   up to and including row. The new lines are initialized to be empty. */
void alloc_line(buffer *b, long row)
{
	long i = b->used_lines;
	if (row <= b->alloc_lines) {
		return;
	}
	b->alloc_lines = b->used_lines = MAX(b->used_lines, row);
	b->text = (rich_text *)crealloc(b->text,
			(b->alloc_lines+1)*sizeof(rich_text));
	i++;	/* don't overwrite the last used line */
	while (i <= b->alloc_lines) {
		b->text[i].height = styles[STY_DEFAULT].height;
		b->text[i].sty = STY_DEFAULT;
		b->text[i].adj = HADJ_LEFT;
		b->text[i++].p = NULL;
	}
	b->used_lines = b->alloc_lines;
}

/* kludge to keep things from breaking fatally */
/* any positions after row r are reset to (r,0) */
/* NB: this function does not take care of point */
void position_kludge(buffer *b, int r)
{
	window *w = w_list;
	position p = make_position(r, 0);

	do {
		if (w->buf == b) {
			if (w->top.row >= r) w->top = p;
			if (w->blku.row >= r) w->blku = p;
			if (w->blkl.row >= r) w->blkl = p;
		}
		w = w->next;
	} while (w != w_list);

	if (b->mark_pos.row >= r) b->mark_pos = p;
}

#ifndef MIN
#define MIN(x,y) ((x)<(y)?(x):(y))
#endif

/* this one only resets positions that don't exist */
/* it does take care of everything, including point */
void position_kludge2(void)
{
	window *w = w_list;
	buffer *b;
	unsigned long r;

	do {
		b = w->buf;
		r = line_last_used(b);
		if (r < 1) r = 1;
		w->top.row = MIN(w->top.row, r);
		w->top.col = MIN(w->top.col, col_last_used(b, w->top.row));
		w->blku.row = MIN(w->blku.row, r);
		w->blku.col = MIN(w->blku.col, col_last_used(b, w->blku.row));
		w->blkl.row = MIN(w->blkl.row, r);
		w->blkl.col = MIN(w->blkl.col, col_last_used(b, w->blkl.row));
		w->point_pos.row = MIN(w->point_pos.row, r);
		w->point_pos.col = MIN(w->point_pos.col,
				col_last_used(b, w->point_pos.row));
		w = w->next;
	} while (w != w_list);
	b = b_list;
	do {
		r = line_last_used(b);
		if (r < 1) r = 1;
		b->mark_pos.row = MIN(b->mark_pos.row, r);
		b->mark_pos.col = MIN(b->mark_pos.col,
				col_last_used(b, b->mark_pos.row));
		b = b->next;
	} while (b != b_list);
}

/* shift the next line down, make it the follower's style,
   copy the remainder of the line and truncate this line */
int split_line(buffer *buf, long row, long col)
{
	position_kludge(buf, row);

	alloc_line(buf, row);
	if (buf->text[row].sty == STY_EMBED) {
		if (col == 0) downshift_text(buf, row);
		else downshift_text(buf, row+1);
		return 1;
	}

	downshift_text(buf, row+1);	/* make room for the new line */
	buf->text[row+1].sty = styles[buf->text[row].sty].follower;
	buf->text[row+1].p = rc_strdup(buf->text[row].p+col);
	if (rc_strlen(buf->text[row].p) > col)
		buf->text[row].p[col] = empty_char;	/* truncate */
	buf->change = TRUE;
	return 1;
}

int join_lines(buffer *buf, long row)
{
	rich_char *p;
	position_kludge(buf, row);

	if (row >= buf->used_lines) return 0;	/* nothing to join */

	/* special case if either line is embedded object */
	if (buf->text[row].sty == STY_EMBED) {
		/* can only join with empty line */
		if (line_length(buf, row+1) == 0) {
			upshift_text(buf, row+1);
			return 1;
		} else return 0;	/* can't do that */
	}

	if (buf->text[row+1].sty == STY_EMBED) {
		if (line_length(buf, row) == 0) {
			upshift_text(buf, row);
			return 1;
		} else return 0;
	}

	p = buf->text[row].p;
	buf->text[row].p = rc_strins(p, buf->text[row+1].p, rc_strlen(p));
	cfree(p);
	upshift_text(buf, row+1);
	return 1;
}

int ins_text(buffer *buf, position pos, unsigned char *t, long fmt)
{
	rich_char *p;
	rich_char *q;
	if (ret_style(buf, pos.row) == STY_EMBED) return 0;	/* nope */

	alloc_line(buf, pos.row);
	p = buf->text[pos.row].p;
	q = rc_makerich(t, fmt);
	buf->text[pos.row].p = rc_strins(p, q, pos.col);
	cfree(p);
	cfree(q);
	return 1;
}

int ins_char(buffer *buf, long r, long c, int s, long fmt)
{
	unsigned char b[2];
	b[0] = s;
	b[1] = '\0';
	return ins_text(buf, make_position(r, c), b, fmt);
}

/* Make a plaintext copy of row r int the string p.
   The string must be freed by caller */
unsigned char *peek_line(buffer *b, long r)
{
	if (b == NULL) return NULL;
	if (r > b->used_lines) return NULL;
	if (b->text == NULL) return NULL;
	return rc_makeplain(b->text[r].p);
}

int peek_char(buffer *b, long r, long c)
{
	int result;
	unsigned char *p = peek_line(b, r);
	result = p[c];
	cfree(p);
	return result;
}

/* what the hell is this */
static int find_suitable_space(buffer *b, long r)
{
	long c = 0;
	int space = 0;	/* current "suitable" space */

	while (c < line_length(b, r) &&
			line_width(b, r, c) <=
				paper_width-left_margin-right_margin) {
		if (isspace(peek_char(b, r, c))) space = c;
		c++;
	}
	return space;
}

/* Checks if the line is too long for the paper width and breaks it if
   necessary. Continues with the next line. */
long rebreak_line(buffer *buf, long row)
{
	while (line_width(buf, row, line_length(buf, row)) >
			paper_width-left_margin-right_margin) {
		int col = find_suitable_space(buf, row);
		if (col <= 0) break;	/* none found */
		split_line(buf, row, col);
		row++;
		del_char(buf, row, 0);
		/* We now have a fragment from the last part of the newly
		   broken line. If the next line is of the same style and
		   also contains a non-empty fragment, we will want to
		   join the lines and potentially rebreak the result */
		if ((buf->text[row].sty == buf->text[row+1].sty) &&
				(line_length(buf, row+1) > 0)) {
			ins_char(buf, row+1, 0, ' ', ret_format(buf, row, 0));
			join_lines(buf, row);
		}
		pr_scr_flag = TRUE;
	}
	return row;
}

extern void embed_unload(char *);	/* FIXME */

/* returns number of deleted characters */
int del_char(buffer *buf, long row, long col)
{
	if (buf->text[row].sty == STY_EMBED) {
		if (col == 0) {		/* remove the object and the line */
			char *p = (char *)rc_makeplain(buf->text[row].p);
			embed_unload(p);	/* can't be NULL */
			cfree(p);
			upshift_text(buf, row);
			return 1;
		} else return 0;
	}

	rc_strcpy(buf->text[row].p+col, buf->text[row].p+col+1);
	buf->change = TRUE;
	return 1;
}

/* returns number of deleted characters */
long del_text(buffer *buf, position p, long length)
{
	long i;

	for (i = 0; i < length; i++)
		if (!del_char(buf, p.row, p.col)) break;
	return i;
}

long del_lines(buffer *buf, long row, long count)
{
	long i;

	/* don't delete lines that aren't there */
	if (row+count-1 > buf->alloc_lines) count = buf->alloc_lines-row+1;

	/* delete count lines, starting at row */
	for (i = 0; i < count; i++) {
		if (buf->text[row+i].sty == STY_EMBED) {
			char *p = (char *)rc_makeplain(buf->text[row].p);
			embed_unload(p);
			cfree(p);
		}
		cfree(buf->text[row+i].p);
	}

	/* shift the rest of the lines up */
	for (i = row; i+count <= buf->alloc_lines; i++) {
		buf->text[i] = buf->text[i+count];
	}

	/* clear the last count lines */
	for (; i <= buf->alloc_lines; i++) {
		buf->text[i].p = NULL;
	}
	buf->alloc_lines -= count;
	buf->used_lines -= count;
	if (count) buf->change = TRUE;
	return count;
}

int ins_format(buffer *buf, long row, long c1, long c2, long format)
{
	rich_char *line;
	alloc_line(buf, row);
	line = buf->text[row].p;
	while (c1 < c2 && c1 < rc_strlen(line)) {
		line[c1++].fmt = format;
	}
	return 1;
}

int ret_format(buffer *buf, long row, long col)
{
	rich_char *line;
	if (row > buf->used_lines) return styles[0].format;
	line = buf->text[row].p;
	if (!line || line[0].c == '\0') {
		/* empty line, try to get from previous line(s) */
		int pr = row-1;
		while (pr >= 1 && ret_style(buf, pr) == ret_style(buf, row)) {
			line = buf->text[pr].p;
			col = rc_strlen(line);
			if (col) return line[col-1].fmt;
			pr--;
		}
		/* give up and return the default format for the style */
		return styles[buf->text[row].sty].format;
	}
	if (col >= rc_strlen(line)) col = rc_strlen(line)-1;
	return line[col].fmt;
}

int ret_font(buffer *buf, long row, long col)
{
	return ret_format(buf, row, col) & 127;
}

void set_style(buffer *b, long row, int sty)
{
	alloc_line(b, row);
	b->text[row].sty = sty;
	b->text[row].height = styles[sty].height;
}

int ret_style(buffer *b, long row)
{
	if (row > b->used_lines) return STY_DEFAULT;
	return b->text[row].sty;
}

int ret_hadj(buffer *b, long row)
{
	if (row > b->used_lines) return HADJ_LEFT;
	return b->text[row].adj & HADJ_MASK;
}

long line_last_used(buffer *buf)
{
	return buf->used_lines;
}

/* this means strlen */
long col_last_used(buffer *buf, long row)
{
	return line_length(buf, row);	/* hmm */
}

void downshift_text(buffer *b, long row)
{
	long r;
	int i;
	b->used_lines++;
	alloc_line(b, b->used_lines);
	for (r = b->used_lines; r > row; r--) {
		b->text[r] = b->text[r-1];
	}
	b->text[r].sty = STY_DEFAULT;
	b->text[r].height = styles[STY_DEFAULT].height;
	b->text[r].adj = HADJ_LEFT;
	b->text[r].p = NULL;
	for (i = 0; i < b->nplugin; i++) {
		if (b->plugin[i].row >= row) b->plugin[i].row++;
	}
	b->change = pr_scr_flag = TRUE;
}

void upshift_text(buffer *b, long row)
{
	long r;
	int i, j;
	b->used_lines--;
#ifdef	WANNADIE
This causes problems with other stuff like join_lines that wants to keep
the old data from b->pwdoc[row]

	if (b->pwdoc[row])
		cfree_pwline(b->pwdoc[row]);
#endif
	for (r = row; r <= b->used_lines; r++) {
		b->text[r] = b->text[r+1];
	}
	/* this lines is still allocated. We clear it just in case */
	b->text[r].sty = STY_DEFAULT;
        b->text[r].height = styles[STY_DEFAULT].height;
        b->text[r].adj = HADJ_LEFT;
	b->text[r].p = NULL;
	for (i = 0; i < b->nplugin; i++) {
		if (b->plugin[i].row == row) {
			pw_plugin_stop(b->plugin[i].ph);
			cfree(b->plugin[i].name);
			b->nplugin--;
			for (j = i; j < b->nplugin; j++)
				b->plugin[j] = b->plugin[j+1];
		} else if (b->plugin[i].row > row) {
			b->plugin[i].row--;
		}
	}
	b->change = pr_scr_flag = TRUE;
}

