/*
   Pathetic Writer
   Copyright (C) 1997, 1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * fileio_rtf.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "pw.h"
#include "../common/fonts.h"
#include "../common/common.h"
#include "../common/cmalloc.h"

typedef struct s_rtf_font {
	int nr;
	char type[20];
	char name[40];
} rtf_font;

typedef struct s_rtf_style {
	int nr;		/* stylesheet */
	int li;		/* left indent */
	int sa;		/* skip after */
	int keepn;	/* keep with next */
	int bold;	/* guess */
	int font;	/* font number */
	int size;	/* foo */
	int basedon;	/* based on */
	int next;	/* next style */
	int brk;	/* paragraph break after every line */
	char name[40];	/* my name for this style */
} rtf_style;

static rtf_font fnt[] = {
	{0, "modern", "Courier"},
	{1, "swiss", "Helvetica"},
	{2, "roman", "New Century Schoolbook"},
	{3, "roman", "Times"}
};

#define RTF_NROFFONTS (sizeof fnt/sizeof *fnt)

static void fontdef(FILE *fpo, rtf_font f)
{
	fprintf(fpo, "{\\f%d\\f%s %s;}\n", f.nr, f.type, f.name);
}

static void rtf_fonts(FILE *fpo)
{
	int i;

	fprintf(fpo, "{\\fonttbl\n");
	for (i = 0; i < RTF_NROFFONTS; i++) {
		fontdef(fpo, fnt[i]);
	}
	fprintf(fpo, "}\n");
}

static void styledef(FILE *fpo, rtf_style s)
{
	fprintf(fpo, "{");
	/*if (s.nr)*/ fprintf(fpo, "\\s%d", s.nr);	/* stylesheet */
	if (s.li) fprintf(fpo, "\\li%d", s.li);	/* left indent */
	if (s.sa) fprintf(fpo, "\\sa%d", s.sa);	/* skip after */
	if (s.keepn) fprintf(fpo, "\\keepn");	/* keep with next */
	if (s.nr || s.li || s.sa || s.keepn) fprintf(fpo, " ");
	if (s.bold) fprintf(fpo, "\\b");	/* bold */
	if (s.font) fprintf(fpo, "\\f%d", s.font);	/* font number */
	/*if (s.size != 12)*/ fprintf(fpo, "\\fs%d", s.size);	/* font size */
	if (s.bold || s.font || s.size) fprintf(fpo, " ");
	if (s.nr) fprintf(fpo, "\\sbasedon%d", s.basedon);	/* based on */
	fprintf(fpo, "\\snext%d %s;}\n", s.next, s.name);	/* next style */
}

static int rtf_fontno(unsigned long fmt)
{
	return (fmt & FONT_MASK) >> FONT_SHIFT;
}

static int sizes[] = {8, 10, 12, 14, 18, 20, 24, 30};

static int rtf_fontsize(unsigned long fmt)
{
	return 2*sizes[fmt & SIZE_MASK];
}

static int rtf_isbold(unsigned long fmt)
{
	return (fmt & BOLD)?1:0;
}

static int rtf_isitalic(unsigned long fmt)
{
	return (fmt & ITALIC)?1:0;
}

static int rtf_isulined(unsigned long fmt)
{
	return (fmt & ULINE)?1:0;
}

static rtf_style *rtf_makestyle(rtf_style *s, int i)
{
	static rtf_style p;
	unsigned long fmt = styles[i].format;
	if (!s) s = &p;
	s->nr = i; 
	s->li = 0;
	s->sa = 0;
	s->keepn = FALSE;
	s->bold = rtf_isbold(fmt);
	s->font = rtf_fontno(fmt);
	s->size = rtf_fontsize(styles[i].format);
	s->basedon = STY_DEFAULT;
	s->next = styles[i].follower;
	s->brk = (i == STY_PREFORMAT ? 1 : 0);
	strcpy(s->name, styles[i].name);
	return s;
}

static void rtf_styles(FILE *fpo)
{
	int i;

	fprintf(fpo, "{\\stylesheet\n");
	for (i = STY_DEFAULT; i < STY_EMBED; i++) {
		rtf_style s;
		rtf_makestyle(&s, i);
		styledef(fpo, s);
	}
	fprintf(fpo, "}\n");
}

static rtf_style getstyle(int nr)
{
	int i;
	static rtf_style s;

	for (i = STY_DEFAULT; i < STY_EMBED; i++) {
		rtf_makestyle(&s, i);
		if (s.nr == nr) return s;
	}

	rtf_makestyle(&s, STY_DEFAULT);
	return s;
}

static void preblurb(FILE *fpo)
{
	/* default font Times; default language Swedish */
	fprintf(fpo, "{\\rtf1\\ansi \\deff3\\deflang1053\n");

	rtf_fonts(fpo);
	rtf_styles(fpo);

	/* paper size */
	fprintf(fpo, "\\paperw%d\\paperh%d\n",
			20*paper_width, 20*paper_height);

	/* marginals */
	fprintf(fpo, "\\margl%d\\margr%d\\margt%d\\margb%d\n",
			20*left_margin, 20*right_margin,
			20*top_margin, 20*bottom_margin);
	fprintf(fpo, "\\gutter0 \\deftab%d\\widowctrl\\ftnbj\\hyphhotz425 \n",
			20*36);
	fprintf(fpo, "\\sectd \n");
	fprintf(fpo, "\\binfsxn1\\binsxn1\\linex0\\endnhere \n");
}

static void postblurb(FILE *fpo)
{
	fprintf(fpo, "}\n");
}

static int last_sty;
static unsigned long last_fmt;

static void save_style(FILE *fp, int sty)
{
	rtf_style s = getstyle(sty);

	if (last_sty != -1) {
		fprintf(fp, "\\par\n");
	}

	/* paragraph definition */
	fprintf(fp, "\\pard\\plain \n");

	fprintf(fp, "\\s%d", s.nr);
	if (s.li) fprintf(fp, "\\li%d", s.li);
	if (s.sa) fprintf(fp, "\\sa%d", s.sa);
	if (s.keepn) fprintf(fp, "\\keepn");
	fprintf(fp, " ");

	/* save_fmt takes care of character formats */

	fprintf(fp, "\n");

	last_sty = sty;
	last_fmt = (unsigned long)-1;
}

static void save_fmt(FILE *fp, unsigned long fmt)
{
	fprintf(fp, "\\f%d\\fs%d", rtf_fontno(fmt), rtf_fontsize(fmt));
	if (rtf_isbold(fmt)) fprintf(fp, "\\b");
	else fprintf(fp, "\\b0");
	if (rtf_isitalic(fmt)) fprintf(fp, "\\i");
	else fprintf(fp, "\\i0");
	if (rtf_isulined(fmt)) fprintf(fp, "\\ul");
	else fprintf(fp, "\\ul0");
	fprintf(fp, " ");
	last_fmt = fmt;
}

static int save_char(FILE *fp, unsigned long fmt, int c)
{
	if (fmt != last_fmt)
		save_fmt(fp, fmt);

	switch (c) {
	case '\\':
		fprintf(fp, "\\\\");
		break;
	case '{':
		fprintf(fp, "\\{");
		break;
	case '}':
		fprintf(fp, "\\}");
		break;
	default:
		if (c > 127) fprintf(fp, "\\\'%02x", c);
		else putc(c, fp);
	}
	return 0;
}

static int save_line(FILE *fp, int sty, rich_char *p)
{
	int i;
	/* start new paragraph for each of these cases (heuristic alert):
	    - new style
	    - sty is preformat
	    - empty line
	    - line starts with white space (suggests indentation)
	*/
	if ((sty != last_sty) ||
			(sty == STY_PREFORMAT) ||
			(rc_strlen(p) == 0) ||
			(p && p[0].c && isspace(p[0].c))) {
		save_style(fp, sty);
	} else {
		save_char(fp, last_fmt, ' ');
	}
	/* This isn't really enough. We also need to handle adjustment,
	   line height and such. Get back to this... */
	for (i = 0; i < rc_strlen(p); i++) {
		if (save_char(fp, p[i].fmt, p[i].c))
			return 1;
	}
	save_char(fp, last_fmt, '\n');	/* separate lines */
	return 0;
}

static int save(char *fn, buffer *buf)
/* Returns: 0 if successful, otherwise 1 */
{
	FILE *fp;
	long i;

	if ((fp = fopen(fn, "w")) == NULL) return 1;

	/* do the header */
	preblurb(fp);

	last_sty = -1;	/* force new definitions */

	for (i = 1; i <= line_last_used(buf); i++) {
		if (save_line(fp, ret_style(buf, i), buf->text[i].p)) {
			fclose(fp);
			return 1;
		}
	}
	fprintf(fp, "\\par\n");		/* just in case */

	postblurb(fp);

	fclose(fp);
	return 0;
}

/* loader code */

static enum {START, BSLASH, TICK, HEX1, KEYWORD, END} state;
static int blocklevel;
static int errflag;
static char tagbuf[256];        /* for \keywords */
static char xchar[10];
static int tbi, pre;
static unsigned long fmt;
static unsigned long row;
static buffer *buf;
static rtf_style *rstyle;
static rtf_font *rfont;

static void emitchar(int c)
{
	if (blocklevel != 1) return;
	if (c == '\n') {
		split_line(buf, row, line_length(buf, row));
		row++;
	} else {
		ins_char(buf, row, line_length(buf, row), c, fmt);
		row = rebreak_line(buf, row);
	}
}

static void kw_par(int n)
{
	emitchar('\n');
	emitchar('\n');
}

static void kw_b(int n)
{
	if (blocklevel == 1) fmt |= BOLD;
}

static void kw_b0(int n)
{
	if (blocklevel == 1) fmt &= ~BOLD;
}

static void kw_i(int n)
{
	if (blocklevel == 1) fmt |= ITALIC;
}

static void kw_i0(int n)
{
	if (blocklevel == 1) fmt &= ~ITALIC;
}

static void kw_ul(int n)
{
	if (blocklevel == 1) fmt |= ULINE;
}

static void kw_ul0(int n)
{
	if (blocklevel == 1) fmt &= ~ULINE;
}

static void kw_fs(int n)
{
	if (blocklevel == 1) {
		int m = n/2;
		fmt &= ~SIZE_MASK;
		/* approximation of requested size */
		if (m <= 8) fmt |= SIZE_8;
		else if (m <= 10) fmt |= SIZE_10;
		else if (m <= 12) fmt |= SIZE_12;
		else if (m <= 14) fmt |= SIZE_14;
		else if (m <= 18) fmt |= SIZE_18;
		else if (m <= 20) fmt |= SIZE_20;
		else if (m <= 24) fmt |= SIZE_24;
		else fmt |= SIZE_30;
	}
}

static void kw_plain(int n)
{
	if (blocklevel == 1) {
		fmt = styles[STY_DEFAULT].format;
	}
}

/* this *only* works with my own stylesheet */
static void kw_f(int n)
{
	if (blocklevel == 1) {
		fmt &= ~FONT_MASK;
		fmt |= (n << FONT_SHIFT);
	}
}

/* these are all the keywords we use ourselves, but we ignore most of
   them while loading.
*/
static struct {
	char *name;
	void (*action)(int);
} keyword[] = {
	{"f", kw_f},			/* font number */
	{"fonttbl", NULL},		/* font table definition */
	{"s", NULL},			/* style number */
	{"li", NULL},			/* left indent */
	{"sa", NULL},			/* skip after */
	{"keepn", NULL},		/* keep with next */
	{"b", kw_b},			/* bold */
	{"b0", kw_b0},			/* not bold */
	{"i", kw_i},			/* italic */
	{"i0", kw_i0},			/* not italic */
	{"ul", kw_ul},			/* underlined */
	{"ul0", kw_ul0},		/* not underlined */
	{"fs", kw_fs},			/* font size */
	{"sbasedon", NULL},		/* based on */
	{"snext", NULL},		/* next style */
	{"stylesheet", NULL},		/* stylesheet */
	{"rtf", NULL},			/* rtf */
	{"ansi", NULL},			/* ansi */
	{"deff", NULL},			/* default font no */
	{"deflang", NULL},		/* default language */
	{"paperw", NULL},		/* paper width */
	{"paperh", NULL},		/* paper height */
	{"margl", NULL},		/* left margin */
	{"margr", NULL},		/* right margin */
	{"margt", NULL},		/* top margin */
	{"margb", NULL},		/* bottom margin */
	{"gutter", NULL},		/* don't ask */
	{"deftab", NULL},		/* default tab distance */
	{"widowctrl", NULL},		/* oomph there it is */
	{"ftnbj", NULL},		/* shaka shaka shaka */
	{"hyphhotz", NULL},		/* oobie doobie */
	{"sectd", NULL},		/* section definition */
	{"binfsxn", NULL},
	{"binsxn", NULL},
	{"linex", NULL},
	{"endnhere", NULL},
	{"par", kw_par},		/* end of paragraph */
	{"pard", NULL},			/* paragraph definition */
	{"plain", kw_plain},		/* reset format */
	{NULL, NULL}
};

static void emitkeyword(char *p)
{
	int i;
	int arg = 1;
	/* first try with arguments intact */
	for (i = 0; keyword[i].name; i++) {
		if (!strcmp(p, keyword[i].name)) {
			if (keyword[i].action)
				(*keyword[i].action)(arg);
			return;
		}
	}
	/* look for argument */
	for (i = 0; p[i]; i++) {
		if (isdigit(p[i])) {
			arg = strtol(p+i, NULL, 10);
			p[i] = '\0';
			for (i = 0; keyword[i].name; i++) {
				if (!strcmp(p, keyword[i].name)) {
					if (keyword[i].action)
						(*keyword[i].action)(arg);
					return;
				}
			}
			return;
		}
	}
}

static void rtf_char(int c)
{
	if (c == EOF) {
		state = END;
		return;
	}
	switch (state) {
	case START:
		if (isspace(c)) {
			if (c == ' ' || c == '\t')
				emitchar(c);
		} else if (c == '{') blocklevel++;
		else if (c == '}') blocklevel--;
		else if (c == '\\') state = BSLASH;
		else emitchar(c);
		break;
	case BSLASH:
		if (c == '{' || c == '}' || c == '\\') {
			emitchar(c);
			state = START;
		} else if (c == '\'') {
			state = TICK;
		} else if (islower(c)) {
			tbi = 0;
			tagbuf[tbi++] = c;
			state = KEYWORD;
		} else state = START;	/* unknown sequence */
		break;
	case TICK:
		if (isxdigit(c)) {
			xchar[0] = c;
			state = HEX1;
		} else state = START;
		break;
	case HEX1:
		if (isxdigit(c)) {
			int n;
			xchar[1] = c;
			xchar[2] = '\0';
			n = strtol(xchar, NULL, 16);
			emitchar(n);
		}
		state = START;
		break;
	case KEYWORD:
		if (c == '{') blocklevel++;
		else if (c == '}') blocklevel--;
		if (islower(c) || isdigit(c)) {
			tagbuf[tbi++] = c;
		} else {
			tagbuf[tbi] = '\0';
			emitkeyword(tagbuf);
			if (c == '\\') state = BSLASH;
			else state = START;
		}
		break;
	default:
		fprintf(stderr, "In rtf_char(): shouldn't be here!\n");
		errflag = 1;
		state = END;
		break;
	}
}

static int load(char *fn, buffer *b)
{
	int i;
        FILE *fp = fopen(fn, "r");
        if (!fp) return 1;

	rstyle = (rtf_style *)cmalloc(256*sizeof(rtf_style));
	rfont = (rtf_font *)cmalloc(256*sizeof(rtf_font));
	for (i = 0; i < 256; i++) {
		rstyle[i] = getstyle(i);
		rfont[i] = fnt[i%4];
	}
	state = START;
	blocklevel = 0;
        errflag = 0;
        tbi = 0;
        pre = 0;
        row = 1;
        buf = b;
        fmt = styles[STY_DEFAULT].format;
        while (state != END)
                rtf_char(getc(fp));
        emitchar('\n');         /* flush if anything left */
	cfree(rfont);
	cfree(rstyle);
        fclose(fp);

        return 0;
}


/* Format guessing:
   1. extension .rtf or .RTF
   2. contains the string "\rtf"
*/
static int myformat(char *fn)
{
        char *ext;
        FILE *fp;
        char b[256];

        ext = strrchr(fn, '.');
        if (!ext) return 0;     /* no extension */
        if (cstrcasecmp(ext, ".rtf"))
                return 0;       /* wrong extension */
        if ((fp = fopen(fn, "r")) == NULL) return 0;    /* can't open */
        while (fgets(b, sizeof b, fp)) {
                if (strstr(b, "\\rtf")) {
                        fclose(fp);
                        return 1;
                }
        }
        fclose(fp);
	return 0;
}

void fileio_rtf_init()
{
	register_format(load, save, myformat, "Rich Text Format (*.rtf)");
}

