/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

/* AIX requires this to be the first thing in the file.  */
#include <config.h>
#if __GNUC__
#define alloca __builtin_alloca
#else
#if HAVE_ALLOCA_H
#include <alloca.h>
#else
#ifdef _AIX
#pragma alloca
#else
#ifndef alloca			/* predefined by HP cc +Olibcalls */
char *alloca ();
#endif
#endif
#endif
#endif

#include <assert.h>
#include <stdlib.h>
#include "common.h"
#include "dfm.h"
#include "error.h"
#include "expr.h"
#include "file-handle.h"
#include "misc.h"
#include "som.h"
#include "tab.h"
#include "lexer.h"
#include "lexerP.h"
#include "var.h"

#undef DEBUGGING
/*#define DEBUGGING 1*/
#include "debug-print.h"

/* Describes what to do when an output field is encountered. */
enum
  {
    PRT_ERROR,			/* Invalid value. */
    PRT_NEWLINE,		/* Newline. */
    PRT_CONST,			/* Constant string. */
    PRT_VAR,			/* Variable. */
    PRT_SPACE			/* A single space. */
  };

/* Describes how to output one field. */
typedef struct prt_out_spec prt_out_spec;
struct prt_out_spec
  {
    prt_out_spec *next;
    int type;			/* PRT_* constant. */
    int fc;			/* 0-based first column. */
    union
      {
	char *c;		/* PRT_CONST: Associated string. */
	struct
	  {
	    variable *v;	/* PRT_VAR: Associated variable. */
	    fmt_spec f;		/* PRT_VAR: Output spec. */
	  }
	v;
      }
    u;
  }
/* prt_out_spec */ ;

/* Enums for use with print_trns's `options' field. */
enum
  {
    PRT_CMD_MASK = 1,		/* Command type mask. */
    PRT_PRINT = 0,		/* PRINT transformation identifier. */
    PRT_WRITE = 1,		/* WRITE transformation identifier. */
    PRT_EJECT = 002		/* Can be combined with CMD_PRINT only. */
  };

/* PRINT, PRINT EJECT, WRITE private data structure. */
typedef struct
  {
    trns_header h;
    struct file_handle *handle;	/* Output file, NULL=listing file. */
    int options;		/* PRT_* bitmapped field. */
    prt_out_spec *spec;		/* Output specifications. */
    int max_width;		/* Maximum line width including null. */
#if !PAGED_STACK
    char *line;			/* Buffer for sticking lines in. */
#endif
  }
print_trns;

/* PRT_PRINT or PRT_WRITE. */
int which_cmd;

/* Holds information on parsing the data file. */
static print_trns prt;

/* Last prt_out_spec in the chain.  Used for building the linked-list. */
static prt_out_spec *next;

/* Number of records. */
static int nrec;

static int internal_cmd_print (int flags);
static int print_trns_proc (any_trns *, ccase *);
static void print_trns_free (any_trns *);
static int parse_specs (void);
static void dump_table (void);
static void append_var_spec (prt_out_spec * spec);
static void alloc_line (void);

#if DEBUGGING
void debug_print (void);
#endif

/* Basic parsing. */

/* Parses PRINT command. */
int
cmd_print (void)
{
  int code;
  match_id (PRINT);
  code = internal_cmd_print (PRT_PRINT);
  if (!code)
    print_trns_free ((any_trns *) & prt);
  return code;
}

/* Parses PRINT EJECT command. */
int
cmd_print_eject (void)
{
  int code = internal_cmd_print (PRT_PRINT | PRT_EJECT);
  match_id (EJECT);
  if (!code)
    print_trns_free ((any_trns *) & prt);
  return code;
}

/* Parses WRITE command. */
int
cmd_write (void)
{
  int code = internal_cmd_print (PRT_WRITE);
  match_id (WRITE);
  if (!code)
    print_trns_free ((any_trns *) & prt);
  return code;
}

/* Parses the output commands.  F is PRT_PRINT, PRT_WRITE, or
   PRT_PRINT|PRT_EJECT. */
static int
internal_cmd_print (int f)
{
  /* 0=print no table, 1=print table.  (TABLE subcommand.)  */
  int table = 0;

  /* malloc()'d transformation. */
  print_trns *trns;

  /* Fill in prt to facilitate error-handling. */
  prt.h.proc = print_trns_proc;
  prt.h.free = print_trns_free;
  prt.handle = NULL;
  prt.options = f;
  prt.spec = NULL;
#if !PAGED_STACK
  prt.line = NULL;
#endif
  next = NULL;
  nrec = 0;

  which_cmd = f & PRT_CMD_MASK;

  /* Parse the command options. */
  while (!match_tok ('/'))
    {
      if (match_id (OUTFILE))
	{
	  match_tok ('=');
	  if (token == ID)
	    prt.handle = fh_get_handle_by_name (tokstr);
	  else if (token == STRING)
	    prt.handle = fh_get_handle_by_filename (tokstr);
	  else
	    return msg (SE, _("A file name or handle was expected in the "
			"OUTFILE subcommand."));
	  if (!prt.handle)
	    return 0;
	  get_token ();
	}
      else if (match_id (RECORDS))
	{
	  match_tok ('=');
	  match_tok ('(');
	  force_int ();
	  nrec = tokint;
	  get_token ();
	  match_tok (')');
	}
      else if (match_id (TABLE))
	table = 1;
      else if (match_id (NOTABLE))
	table = 0;
      else
	return syntax_error (_("while expecting a valid subcommand"));
    }

  /* Parse variables and strings. */
  if (!parse_specs ())
    return 0;

  /* Output the variable table if requested. */
  if (table)
    dump_table ();

  /* Count the maximum line width.  Allocate linebuffer if
     applicable. */
  alloc_line ();

  /* Put the transformation in the queue. */
  trns = xmalloc (sizeof (print_trns));
  memcpy (trns, &prt, sizeof (print_trns));
  add_transformation ((any_trns *) trns);

#if DEBUGGING
  debug_print ();
#endif

  return 1;
}

/* Appends the field output specification SPEC to the list maintained
   in prt. */
static void
append_var_spec (prt_out_spec * spec)
{
  if (next == 0)
    prt.spec = next = xmalloc (sizeof (prt_out_spec));
  else
    next = next->next = xmalloc (sizeof (prt_out_spec));

  memcpy (next, spec, sizeof (prt_out_spec));
  next->next = NULL;
}

/* Field parsing.  Mostly stolen from data-list.c. */

/* Used for chaining together fortran-like format specifiers. */
typedef struct fmt_list
{
  struct fmt_list *next;
  int count;
  fmt_spec f;
  struct fmt_list *down;
}
fmt_list;

/* Used as "local" variables among the fixed-format parsing funcs.  If
   it were guaranteed that PSPP were going to be compiled by gcc,
   I'd make all these functions a single set of nested functions. */
static struct
  {
    variable **v;		/* variable list */
    int nv;			/* number of variables in list */
    int cv;			/* number of variables from list used up so far
				   by the FORTRAN-like format specifiers */

    int recno;			/* current 1-based record number */
    int sc;			/* 1-based starting column for next variable */

    prt_out_spec spec;		/* next format spec to append to list */
    int fc, lc;			/* first, last 1-based column number of current
				   var */

    int level;			/* recursion level for FORTRAN-like format
				   specifiers */
  }
fx;

static int fixed_parse_compatible (void);
static fmt_list *fixed_parse_fortran (void);

static int parse_string_argument (void);
static int parse_variable_argument (void);

/* Parses all the variable and string specifications on a single
   PRINT, PRINT EJECT, or WRITE command into the prt structure.
   Returns success. */
static int
parse_specs (void)
{
  /* Return code from called function. */
  int code;

  fx.recno = 1;
  fx.sc = 1;

  while (token != '.')
    {
      while (match_tok ('/'))
	{
	  int prev_recno = fx.recno;

	  fx.recno++;
	  if (token == NUM)
	    {
	      force_int ();
	      if (tokint < fx.recno)
		return msg (SE, _("The record number specified, %ld, is "
			    "before the previous record, %d.  Data "
			    "fields must be listed in order of "
			    "increasing record number."),
			    tokint, fx.recno - 1);
	      fx.recno = tokint;
	      get_token ();
	    }

	  fx.spec.type = PRT_NEWLINE;
	  while (prev_recno++ < fx.recno)
	    append_var_spec (&fx.spec);

	  fx.sc = 1;
	}

      if (token == STRING)
	code = parse_string_argument ();
      else
	code = parse_variable_argument ();
      if (!code)
	return 0;
    }
  fx.spec.type = PRT_NEWLINE;
  append_var_spec (&fx.spec);

  if (!nrec)
    nrec = fx.recno;
  else if (fx.recno > nrec)
    return msg (SE, _("Variables are specified on records that "
		"should not exist according to RECORDS subcommand."));
  if (token != '.')
    return syntax_error (_("expecting end of command"));
  return 1;
}

/* Parses a string argument to the PRINT commands.  Returns success. */
static int
parse_string_argument (void)
{
  fx.spec.type = PRT_CONST;
  fx.spec.fc = fx.sc - 1;
  fx.spec.u.c = xstrdup (tokstr);
  get_token ();

  /* Parse the included column range. */
  if (token == NUM)
    {
      /* Width of column range in characters. */
      int c_len;

      /* Width of constant string in characters. */
      int s_len;

      /* 1-based index of last column in range. */
      int lc;

      if (tokint == NOT_LONG || tokint <= 0)
	{
	  msg (SE, _("%g is not a valid column location."), tokval);
	  goto fail;
	}
      fx.spec.fc = tokint - 1;

      get_token ();
      convert_negative_to_dash ();
      if (match_tok ('-'))
	{
	  if (token != NUM)
	    {
	      msg (SE, _("Column location expected following `%d-'."),
		   fx.spec.fc + 1);
	      goto fail;
	    }
	  if (token == NOT_LONG || tokint <= 0)
	    {
	      msg (SE, _("%g is not a valid column location."), tokval);
	      goto fail;
	    }
	  if (tokint < fx.spec.fc + 1)
	    {
	      msg (SE, _("%d-%ld is not a valid column range.  The second "
		   "column must be greater than or equal to the first."),
		   fx.spec.fc + 1, tokint);
	      goto fail;
	    }
	  lc = tokint - 1;

	  get_token ();
	}
      else
	/* If only a starting location is specified then the field is
	   the width of the provided string. */
	lc = fx.spec.fc + strlen (fx.spec.u.c) - 1;

      /* Apply the range. */
      c_len = lc - fx.spec.fc + 1;
      s_len = strlen (fx.spec.u.c);
      if (s_len > c_len)
	fx.spec.u.c[c_len] = 0;
      else if (s_len < c_len)
	{
	  fx.spec.u.c = xrealloc (fx.spec.u.c, c_len + 1);
	  memset (&fx.spec.u.c[s_len], ' ', c_len - s_len);
	  fx.spec.u.c[c_len] = 0;
	}

      fx.sc = lc + 1;
    }
  else
    /* If nothing is provided then the field is the width of the
       provided string. */
    fx.sc += strlen (fx.spec.u.c);

  append_var_spec (&fx.spec);
  return 1;

fail:
  free (fx.spec.u.c);
  return 0;
}

/* Parses a variable argument to the PRINT commands by passing it off
   to fixed_parse_compatible() or fixed_parse_fortran() as appropriate.
   Returns success. */
static int
parse_variable_argument (void)
{
  if (!parse_variables (NULL, &fx.v, &fx.nv, PV_DUPLICATE))
    return 0;

  if (token == NUM)
    {
      if (!fixed_parse_compatible ())
	goto fail;
    }
  else if (token == '(')
    {
      fx.level = 0;
      fx.cv = 0;
      if (!fixed_parse_fortran ())
	goto fail;
    }
  else
    {
      /* User wants dictionary format specifiers. */
      int i;

      match_tok ('*');
      for (i = 0; i < fx.nv; i++)
	{
	  /* Variable. */
	  fx.spec.type = PRT_VAR;
	  fx.spec.fc = fx.sc - 1;
	  fx.spec.u.v.v = fx.v[i];
	  fx.spec.u.v.f = fx.v[i]->print;
	  append_var_spec (&fx.spec);
	  fx.sc += fx.v[i]->print.w;

	  /* Space. */
	  fx.spec.type = PRT_SPACE;
	  fx.spec.fc = fx.sc - 1;
	  append_var_spec (&fx.spec);
	  fx.sc++;
	}
    }

  free (fx.v);
  return 1;

fail:
  free (fx.v);
  return 0;
}

/* Parses a column specification for parse_specs(). */
static int
fixed_parse_compatible (void)
{
  int dividend;
  int type;
  int i;

  type = fx.v[0]->type;
  for (i = 1; i < fx.nv; i++)
    if (type != fx.v[i]->type)
      return msg (SE, _("%s is not of the same type as %s.  To specify "
		  "variables of different types in the same variable "
		  "list, use a FORTRAN-like format specifier."),
		  fx.v[i]->name, fx.v[0]->name);

  force_int ();
  fx.fc = tokint - 1;
  if (fx.fc < 0)
    return msg (SE, _("Column positions for fields must "
		"be positive."));
  get_token ();

  convert_negative_to_dash ();
  if (match_tok ('-'))
    {
      force_int ();
      fx.lc = tokint - 1;
      if (fx.lc < 0)
	return msg (SE, _("Column positions for fields must "
		    "be positive."));
      else if (fx.lc < fx.fc)
	return msg (SE, _("The ending column for a field must be "
		    "greater than the starting column."));
      get_token ();
    }
  else
    fx.lc = fx.fc;

  fx.spec.u.v.f.w = fx.lc - fx.fc + 1;
  if (match_tok ('('))
    {
      fmt_desc *fdp;

      if (token == ID)
	{
	  const char *cp;

	  fx.spec.u.v.f.type = parse_format_specifier_name (&cp, 0);
	  if (fx.spec.u.v.f.type == -1)
	    return 0;
	  if (*cp)
	    return msg (SE, _("A format specifier on this line "
			"has extra characters on the end."));
	  get_token ();
	  match_tok (',');
	}
      else
	fx.spec.u.v.f.type = FMT_F;

      if (token == NUM)
	{
	  force_int ();
	  if (tokint < 1)
	    return msg (SE, _("The value for number of decimal places "
			"must be at least 1."));
	  fx.spec.u.v.f.d = tokint;
	  get_token ();
	}
      else
	fx.spec.u.v.f.d = 0;

      fdp = &formats[fx.spec.u.v.f.type];
      if (fdp->n_args < 2 && fx.spec.u.v.f.d)
	return msg (SE, _("Input format %s doesn't accept decimal "
		    "places."), fdp->name);
      if (fx.spec.u.v.f.d > 16)
	fx.spec.u.v.f.d = 16;

      force_match (')');
    }
  else
    {
      fx.spec.u.v.f.type = FMT_F;
      fx.spec.u.v.f.d = 0;
    }

  fx.sc = fx.lc + 1;

  if ((fx.lc - fx.fc + 1) % fx.nv)
    return msg (SE, _("The %d columns %d-%d can't be evenly divided into %d "
		"fields."), fx.lc - fx.fc + 1, fx.fc + 1, fx.lc + 1, fx.nv);

  dividend = (fx.lc - fx.fc + 1) / fx.nv;
  fx.spec.u.v.f.w = dividend;
  if (!check_output_specifier (&fx.spec.u.v.f))
    return 0;
  if ((type == ALPHA) ^ (formats[fx.spec.u.v.f.type].cat & FCAT_STRING))
    return msg (SE, _("%s variables cannot be displayed with format %s."),
		type == ALPHA ? _("String") : _("Numeric"),
		fmt_to_string (&fx.spec.u.v.f));

  /* Check that, for string variables, the user didn't specify a width
     longer than an actual string width. */
  if (type == ALPHA)
    {
      /* Minimum width of all the string variables specified. */
      int min_len = fx.v[0]->width;

      for (i = 1; i < fx.nv; i++)
	min_len = min (min_len, fx.v[i]->width);
      if (!check_string_specifier (&fx.spec.u.v.f, min_len))
	return 0;
    }

  fx.spec.type = PRT_VAR;
  for (i = 0; i < fx.nv; i++)
    {
      fx.spec.fc = fx.fc + dividend * i;
      fx.spec.u.v.v = fx.v[i];
      append_var_spec (&fx.spec);
    }
  return 1;
}

/* Destroy a format list and, optionally, all its sublists. */
static void
destroy_fmt_list (fmt_list * f, int recurse)
{
  fmt_list *next;

  for (; f; f = next)
    {
      next = f->next;
      if (recurse && f->f.type == FMT_DESCEND)
	destroy_fmt_list (f->down, 1);
      free (f);
    }
}

/* Recursively puts the format list F (which represents a set of
   FORTRAN-like format specifications, like 4(F10,2X)) into the
   structure prt. */
static int
dump_fmt_list (fmt_list * f)
{
  int i;

  for (; f; f = f->next)
    if (f->f.type == FMT_X)
      fx.sc += f->count;
    else if (f->f.type == FMT_T)
      fx.sc = f->f.w;
    else if (f->f.type == FMT_NEWREC)
      {
	fx.recno += f->count;
	fx.sc = 1;
	fx.spec.type = PRT_NEWLINE;
	for (i = 0; i < f->count; i++)
	  append_var_spec (&fx.spec);
      }
    else
      for (i = 0; i < f->count; i++)
	if (f->f.type == FMT_DESCEND)
	  {
	    if (!dump_fmt_list (f->down))
	      return 0;
	  }
	else
	  {
	    variable *v;

	    if (fx.cv >= fx.nv)
	      return msg (SE, _("The number of format "
			  "specifictions exceeds the number of variable "
			  "names given."));

	    v = fx.v[fx.cv++];
	    if ((v->type == ALPHA) ^ (formats[f->f.type].cat & FCAT_STRING))
	      return msg (SE, _("Display format %s may not be used with a "
			  "%s variable."), fmt_to_string (&f->f),
			  v->type == ALPHA ? _("string") : _("numeric"));
	    if (!check_string_specifier (&f->f, v->width))
	      return 0;

	    fx.spec.type = PRT_VAR;
	    fx.spec.u.v.v = v;
	    fx.spec.u.v.f = f->f;
	    fx.spec.fc = fx.sc - 1;
	    append_var_spec (&fx.spec);

	    fx.sc += f->f.w;
	  }
  return 1;
}

/* Recursively parses a list of FORTRAN-like format specifiers.  Calls
   itself to parse nested levels of parentheses.  Returns to its
   original caller NULL, to indicate error, non-NULL, but nothing
   useful, to indicate success (it returns a free()'d block). */
static fmt_list *
fixed_parse_fortran (void)
{
  fmt_list *head;
  fmt_list *fl = NULL;

  get_token ();			/* skip opening parenthesis */
  while (token != ')')
    {
      if (fl)
	fl = fl->next = xmalloc (sizeof (fmt_list));
      else
	head = fl = xmalloc (sizeof (fmt_list));

      if (token == NUM)
	{
	  if (tokint == NOT_LONG)
	    {
	      msg (SE, _("Integer expected for repeat count."));
	      goto fail;
	    }
	  fl->count = tokint;
	  get_token ();
	}
      else
	fl->count = 1;

      if (token == '(')
	{
	  fl->f.type = FMT_DESCEND;
	  fx.level++;
	  fl->down = fixed_parse_fortran ();
	  fx.level--;
	  if (!fl->down)
	    goto fail;
	}
      else if (match_tok ('/'))
	fl->f.type = FMT_NEWREC;
      else if (!parse_format_specifier (&fl->f, 1)
	       || !check_output_specifier (&fl->f))
	goto fail;

      match_tok (',');
    }
  fl->next = NULL;
  get_token ();

  if (fx.level)
    return head;

  fl->next = NULL;
  dump_fmt_list (head);
  destroy_fmt_list (head, 1);
  if (fx.cv < fx.nv)
    {
      msg (SE, _("There aren't enough format specifications "
	   "to match the number of variable names given."));
      goto fail;
    }
  return head;

fail:
  fl->next = NULL;
  destroy_fmt_list (head, 0);

  return NULL;
}

/* Prints the table produced by the TABLE subcommand to the listing
   file. */
static void
dump_table (void)
{
  prt_out_spec *spec;
  const char *filename;
  tab_table *t;
  int recno;
  int nspec;

  for (nspec = 0, spec = prt.spec; spec; spec = spec->next)
    if (spec->type == PRT_CONST || spec->type == PRT_VAR)
      nspec++;
  t = tab_create (4, nspec + 1, 0);
  tab_columns (t, SOM_COL_DOWN, 1);
  tab_box (t, TAL_1, TAL_1, TAL_0, TAL_1, 0, 0, 3, nspec);
  tab_hline (t, TAL_2, 0, 3, 1);
  tab_headers (t, 0, 0, 1, 0);
  tab_text (t, 0, 0, TAB_CENTER | TAT_TITLE, _("Variable"));
  tab_text (t, 1, 0, TAB_CENTER | TAT_TITLE, _("Record"));
  tab_text (t, 2, 0, TAB_CENTER | TAT_TITLE, _("Columns"));
  tab_text (t, 3, 0, TAB_CENTER | TAT_TITLE, _("Format"));
  tab_dim (t, tab_natural_dimensions);
  for (nspec = recno = 0, spec = prt.spec; spec; spec = spec->next)
    switch (spec->type)
      {
      case PRT_NEWLINE:
	recno++;
	break;
      case PRT_CONST:
	{
	  int len = strlen (spec->u.c);
	  nspec++;
	  tab_text (t, 0, nspec, TAB_LEFT | TAT_FIX | TAT_PRINTF,
			"\"%s\"", spec->u.c);
	  tab_text (t, 1, nspec, TAT_PRINTF, "%d", recno + 1);
	  tab_text (t, 2, nspec, TAT_PRINTF, "%3d-%3d",
			spec->fc + 1, spec->fc + len);
	  tab_text (t, 3, nspec, TAB_LEFT | TAT_FIX | TAT_PRINTF,
			"A%d", len);
	  break;
	}
      case PRT_VAR:
	{
	  nspec++;
	  tab_text (t, 0, nspec, TAB_LEFT, spec->u.v.v->name);
	  tab_text (t, 1, nspec, TAT_PRINTF, "%d", recno + 1);
	  tab_text (t, 2, nspec, TAT_PRINTF, "%3d-%3d",
			spec->fc + 1, spec->fc + spec->u.v.f.w);
	  tab_text (t, 3, nspec, TAB_LEFT | TAT_FIX,
			fmt_to_string (&spec->u.v.f));
	  break;
	}
      case PRT_SPACE:
	break;
      case PRT_ERROR:
	assert (0);
      }

  filename = fh_handle_name (prt.handle);
  tab_title (t, 1, (prt.handle != NULL
		    ? _("Writing %3d records to file %s.")
		    : _("Writing %3d records to the listing file.")),
	     recno, filename);
  tab_submit (t);
  fh_handle_name (NULL);
}

/* PORTME: The number of characters in a line terminator. */
#if __MSDOS__ 
#define LINE_END_WIDTH 2	/* \r\n */
#else
#define LINE_END_WIDTH 1	/* \n */
#endif

/* Calculates the maximum possible line width and allocates a buffer
   big enough to contain it, if necessary (otherwise sets max_width).
   (The action taken depends on compiler & OS as detected by pref.h.) */
static void
alloc_line (void)
{
  /* Cumulative maximum line width (excluding null terminator) so far. */
  int w = 0;

  /* Width required by current this prt_out_spec. */
  int pot_w;			/* Potential w. */

  /* Iterator. */
  prt_out_spec *i;

  for (i = prt.spec; i; i = i->next)
    {
      switch (i->type)
	{
	case PRT_NEWLINE:
	  pot_w = 0;
	  break;
	case PRT_CONST:
	  pot_w = i->fc + strlen (i->u.c);
	  break;
	case PRT_VAR:
	  pot_w = i->fc + i->u.v.f.w;
	  break;
	case PRT_SPACE:
	  pot_w = i->fc + 1;
	  break;
	case PRT_ERROR:
	  assert (0);
	  break;
	}
      if (pot_w > w)
	w = pot_w;
    }
  prt.max_width = w + LINE_END_WIDTH + 1;
#if !PAGED_STACK
  prt.line = xmalloc (prt.max_width);
#endif
}

/* Transformation. */

/* Performs the transformation inside print_trns T on case C. */
static int
print_trns_proc (any_trns * trns, ccase * c)
{
  /* Transformation. */
  print_trns *t = (print_trns *) trns;

  /* Iterator. */
  prt_out_spec *i;

  /* Line buffer. */
#if PAGED_STACK
#if __GNUC__ && !__STRICT_ANSI__
  char buf[t->max_width];
#else /* !__GNUC__ */
  char *buf = alloca (t->max_width);
#endif /* !__GNUC__ */
#else /* !PAGED_STACK */
  char *buf = t->line;
#endif /* !PAGED_STACK */

  /* Length of the line in buf. */
  int len = 0;
  memset (buf, ' ', t->max_width);

  if (t->options & PRT_EJECT)
    som_eject_page ();

  /* Note that a field written to a place where a field has already
     been written truncates the record.  `PRINT /A B (T10,F8,T1,F8).'
     only outputs B.  This is an example of bug-for-bug compatibility,
     in the author's opinion. */
  for (i = t->spec; i; i = i->next)
    switch (i->type)
      {
      case PRT_NEWLINE:
	if (t->handle == NULL)
	  {
	    buf[len] = 0;
	    tab_output_text (TAT_FIX | TAT_NOWRAP, buf);
	  }
	else
	  {
	    if ((t->options & PRT_CMD_MASK) == PRT_PRINT)
	      {
		/* PORTME: Line ends. */
#if __MSDOS__
		buf[len++] = '\r';
#endif
		buf[len++] = '\n';
	      }

	    dfm_put_record (t->handle, buf, len);
	  }

	memset (buf, ' ', t->max_width);
	len = 0;
	break;

      case PRT_CONST:
	/* FIXME: Should be revised to keep track of the string's
	   length outside the loop, probably in i->u.c[0]. */
	memcpy (&buf[i->fc], i->u.c, strlen (i->u.c));
	len = i->fc + strlen (i->u.c);
	break;

      case PRT_VAR:
	if (i->u.v.v->type == NUMERIC)
	  convert_format_to_string (&buf[i->fc], &i->u.v.f,
				    &c->data[i->u.v.v->fv]);
	else
	  {
	    value t;
	    t.c = c->data[i->u.v.v->fv].s;
	    convert_format_to_string (&buf[i->fc], &i->u.v.f, &t);
	  }
	len = i->fc + i->u.v.f.w;
	break;

      case PRT_SPACE:
	/* PRT_SPACE always immediately follows PRT_VAR. */
	buf[len++] = ' ';
	break;

      case PRT_ERROR:
	assert (0);
	break;
      }

  return -1;
}

/* Frees all the data inside print_trns T.  Does not free T. */
static void
print_trns_free (any_trns * t)
{
  prt_out_spec *i, *n;

  for (i = ((print_trns *) t)->spec; i; i = n)
    {
      switch (i->type)
	{
	case PRT_CONST:
	  free (i->u.c);
	  /* fall through */
	case PRT_NEWLINE:
	case PRT_VAR:
	case PRT_SPACE:
	  /* nothing to do */
	  break;
	case PRT_ERROR:
	  assert (0);
	  break;
	}
      n = i->next;
      free (i);
    }
#if !PAGED_STACK
  free (((print_trns *) t)->line);
#endif
}

/* PRINT SPACE. */

/* PRINT SPACE transformation. */
typedef struct
{
  trns_header h;

  file_handle *handle;		/* Output file, NULL=listing file. */
  struct expression *e;	/* Number of lines; NULL=1. */
}
print_space_trns;

static int print_space_trns_proc (any_trns *, ccase *);
static void print_space_trns_free (any_trns *);

int
cmd_print_space (void)
{
  print_space_trns *t;
  file_handle *handle;
  expression *e;

  match_id (SPACE);
  if (match_id (OUTFILE))
    {
      match_tok ('=');

      if (token == ID)
	handle = fh_get_handle_by_name (tokstr);
      else if (token == STRING)
	handle = fh_get_handle_by_filename (tokstr);
      else
	return msg (SE, _("A file name or handle was expected in the "
		    "OUTFILE subcommand."));
      if (!handle)
	return 0;
      get_token ();
    }
  else
    handle = NULL;

  if (token != '.')
    {
      e = parse_expression (PXP_NONE);
      if (e->type != EX_NUMERIC)
	{
	  free_expression (e);
	  return msg (SE, _("The argument must be a numeric expression."));
	}
      if (token != '.')
	{
	  free_expression (e);
	  return syntax_error (_("expecting end of command"));
	}
    }
  else
    e = NULL;

  t = xmalloc (sizeof (print_space_trns));
  t->h.proc = print_space_trns_proc;
  if (e)
    t->h.free = print_space_trns_free;
  else
    t->h.free = NULL;
  t->handle = handle;
  t->e = e;

  add_transformation ((any_trns *) t);
  return 1;
}

static int
print_space_trns_proc (any_trns * trns, ccase * c)
{
  print_space_trns *t = (print_space_trns *) trns;
  int n;

  if (t->e)
    {
      value v;

      evaluate_expression (t->e, c, &v);
      n = v.f;
      if (n < 0)
	{
	  msg (SW, _("The expression on PRINT SPACE evaluated to %d.  It's "
		     "not possible to PRINT SPACE a negative number of "
		     "lines."),
	       n);
	  n = 1;
	}
    }
  else
    n = 1;

  if (t->handle == NULL)
    while (n--)
      som_blank_line ();
  else
    {
      char buf[LINE_END_WIDTH];

      /* PORTME: Line ends. */
#if __MSDOS__
      buf[0] = '\r';
      buf[1] = '\n';
#else
      buf[0] = '\n';
#endif
      while (n--)
	dfm_put_record (t->handle, buf, LINE_END_WIDTH);
    }

  return -1;
}

static void
print_space_trns_free (any_trns * trns)
{
  free_expression (((print_space_trns *) trns)->e);
}

/* Debugging code. */

#if DEBUGGING
void
debug_print (void)
{
  prt_out_spec *p;

  if (prt.handle == NULL)
    {
      printf ("PRINT");
      if (prt.eject)
	printf (" EJECT");
    }
  else
    printf ("WRITE OUTFILE=%s", handle_name (prt.handle));
  printf (" MAX_WIDTH=%d", prt.max_width);
  printf (" /");
  for (p = prt.spec; p; p = p->next)
    switch (p->type)
      {
      case PRT_ERROR:
	printf (_("<ERROR>"));
	break;
      case PRT_NEWLINE:
	printf ("\n /");
	break;
      case PRT_CONST:
	printf (" \"%s\" %d-%d", p->u.c, p->fc + 1, p->fc + strlen (p->u.c));
	break;
      case PRT_VAR:
	printf (" %s %d %d-%d (%s)", p->u.v.v->name, p->u.v.v->fv, p->fc + 1,
		p->fc + p->u.v.v->print.w, fmt_to_string (&p->u.v.v->print));
	break;
      case PRT_SPACE:
	printf (" \" \" %d", p->fc + 1);
	break;
      }
  printf (".\n");
}
#endif /* DEBUGGING */
