;;; mew-mime.el --- MIME launcher for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Mar 23, 1997
;; Revised: Aug 24, 1998

;;; Code:

(defconst mew-mime-version "mew-mime.el version 0.11")

(require 'mew)

;;
;;
;;

(defmacro mew-attr-by-ct (ct)
  (` (mew-assoc-match2 (, ct) mew-mime-content-type 0)))

(defmacro mew-attr-by-file (ct)
  (` (mew-assoc-match2 (, ct) mew-mime-content-type 1)))

(defmacro mew-attr-get-ct (attr)
  (` (nth 0 (, attr))))

(defmacro mew-attr-get-cte (attr)
  (` (symbol-value (nth 2 (, attr)))))

(defun mew-attr-get-prog (attr)
  (let ((val (symbol-value (nth 3 attr))))
    (if (and (listp val) (equal 'if (car val)))
	(setq val (eval val)))
    (nth 0 val)))

(defun mew-attr-get-opt (attr)
  (let ((val (symbol-value (nth 3 attr))))
    (if (and (listp val) (equal 'if (car val)))
	(setq val (eval val)))
    (nth 1 val)))

(defun mew-attr-get-async (attr)
  (let ((val (symbol-value (nth 3 attr))))
    (if (and (listp val) (equal 'if (car val)))
	(setq val (eval val)))
    (nth 2 val)))

(defmacro mew-attr-get-icon (attr)
  (` (symbol-value (nth 4 (, attr)))))

(defvar mew-process-file-alist nil)

;;
;;
;;

(defun mew-mime-start-process (program options file)
  (let ((process-connection-type mew-connection-type1) pro)
    (message "Starting %s ..." program)
    (setq pro (apply (function start-process)
		     (format "*mew %s*" program)
		     mew-buffer-tmp
		     program
		     (append options (list file))))
    (set-process-sentinel pro 'mew-mime-start-process-sentinel)
    (message "Sending %s ... done" program)
    (setq mew-process-file-alist (cons (cons pro file) mew-process-file-alist))
    )
  t ;; to next part
  )

(defun mew-mime-start-process-sentinel (process event)
  (let* ((al (assoc process mew-process-file-alist))
	 (file (cdr al)))
    (if (and mew-delete-temp-file file) (delete-file file))
    (setq mew-process-file-alist (delete al mew-process-file-alist))))

(defun mew-mime-call-process (program options file)
  (message "Calling %s ..." program)
  (apply (function call-process) program file nil nil options)
  (message "Calling %s ... done" program)
  t ;; to next part
  )

;;
;;
;;

(defun mew-mime-part (fullpart num &optional execute)
  ;; called in message buffer
  ;; if num is nil, it means singlepart.
  (let* ((syntax  (mew-syntax-get-entry-strnum fullpart num))
	 (begin   (mew-syntax-get-begin syntax))
	 (end     (mew-syntax-get-end   syntax))
	 (ctl     (mew-syntax-get-ct    syntax))
	 (cte     (mew-syntax-get-cte   syntax))
	 (ct      (car ctl))
	 (cdpl    (mew-syntax-get-cdp syntax))
	 (fname   (and cdpl (mew-syntax-get-member cdpl "filename")))
	 (cd      (mew-syntax-get-cd syntax))
	 (params  (cdr ctl))
	 (attr    (mew-attr-by-ct ct))
	 (program (mew-attr-get-prog attr))
	 (options (mew-attr-get-opt attr))
	 (async   (mew-attr-get-async attr)))
    (if (symbolp program)
	(if (fboundp program)
	    (cond
	     ((eq program 'mew-mime-message/rfc822)
	      (funcall program syntax)) ;; for recursive MIME
	     ((eq program 'mew-mime-text/html)
	      (funcall program begin end params execute))
	     ((eq program 'mew-mime-external-body)
	      (funcall program begin end params execute))
	     ((eq program 'mew-mime-application/octet-stream)
	      (funcall program begin end params ct cte))
	     (t
	      (funcall program begin end params))))
      (insert " ######  ######  #######  #####  ######     #    #     #\n"
	      " #     # #     # #     # #     # #     #   # #   ##   ##\n"
	      " #     # #     # #     # #       #     #  #   #  # # # #\n"
	      " ######  ######  #     # #  #### ######  #     # #  #  #\n"
	      " #       #   #   #     # #     # #   #   ####### #     #\n"
	      " #       #    #  #     # #     # #    #  #     # #     #\n"
	      " #       #     # #######  #####  #     # #     # #     #\n"
	      "\n\n")
      (mew-insert "Content-Type:\t%s\n" ct)
      (mew-insert "Encoding: \t%s\n" cte)
      (mew-insert "Size:\t\t%d bytes\n" (- end begin))
      (mew-insert "Filename:\t%s\n" fname)
      (mew-insert "Description: \t%s\n" cd)
      (mew-insert "Program:\t%s\n" program)
      (if (not execute)
	  (insert "\nTo execute this external command, type "
		  (substitute-command-keys
		   "\\<mew-summary-mode-map>\\[mew-summary-execute-external].")
		  "\nTo save this part, type "
		  (substitute-command-keys
		   "\\<mew-summary-mode-map>\\[mew-summary-save].")
		  "\nTo display this part in Message mode, type "
		  (substitute-command-keys
		   "\\<mew-summary-mode-map>\\[mew-summary-insert]."))
	(if (mew-which program exec-path)
	    (let ((file (mew-make-temp-name fname)))
	      (save-excursion
		(set-buffer (mew-current-get 'cache))
		;; NEVER use call-process-region for privacy reasons
		(if (mew-member-case-equal ct mew-mime-content-type-text-list)
		    (mew-frwlet mew-cs-noconv mew-cs-outfile
		       (write-region begin end file nil 'no-msg))
		  (mew-flet
		    (write-region begin end file nil 'no-msg)))
		(if async
		    (mew-mime-start-process program options file)
		  (mew-mime-call-process program options file))))
	  (message "Program %s is not found" program))))))
;;
;;
;;

(defun mew-mime-image (begin end format)
  (let ((buffer-read-only nil))
    (message "Loading image...")
    (cond 
     ((eq format 'xbm) ;; use temporary file.
      (let ((temp-file-name (mew-make-temp-name))
	    glyph)
	(save-excursion
	  (set-buffer (mew-current-get 'cache))
	  (write-region begin end temp-file-name nil 'no-msg)
	  (set-buffer (mew-buffer-message))
	  (setq glyph (make-glyph (vector 
				   'xbm
				   :file
				   temp-file-name)))
	  (set-glyph-property glyph 'face 'x-face)
	  (set-extent-end-glyph (make-extent (point-min) (point-min)) glyph)
	  (if (file-exists-p temp-file-name)
	      (delete-file temp-file-name)))))
     (t
      (set-buffer (mew-buffer-message))
      (set-extent-end-glyph (make-extent (point-min) (point-min))
			    (make-glyph (vector 
					 format
					 :data
					 (buffer-substring 
					  begin end
					  (mew-current-get 'cache)))))))
    (message "Loading image...done")))

(defun mew-mime-image/jpeg (begin end &optional params)
  (mew-mime-image begin end 'jpeg))

(defun mew-mime-image/gif (begin end &optional params)
  (mew-mime-image begin end 'gif))

(defun mew-mime-image/xbm (begin end &optional params)
  (mew-mime-image begin end 'xbm))

(defun mew-mime-image/xpm (begin end &optional params)
  (mew-mime-image begin end 'xpm))

(defun mew-mime-image/png (begin end &optional params)
  (mew-mime-image begin end 'png))

(defun mew-mime-text/plain (begin end &optional params)
  (if (> end begin)
      (save-excursion
	(set-buffer (mew-buffer-message))
	(let ((buffer-read-only nil))
	  (insert-buffer-substring (mew-current-get 'cache) begin end)
	  ;; Highlight
	  (mew-highlight-url)
	  (mew-highlight-body))
	;; Page breaks
	(if mew-break-pages
	    (progn
	      (goto-char (point-min))
	      (mew-message-narrow-to-page))))))

(defun mew-mime-text/html (begin end &optional params execute)
  (let ((size (- end begin))
	(buffer-read-only nil))
    (insert " #     # ####### #     # #\n"
	    " #     #    #    ##   ## #\n"
	    " #     #    #    # # # # #\n"
	    " #######    #    #  #  # #\n"
	    " #     #    #    #     # #\n"
	    " #     #    #    #     # #\n"
	    " #     #    #    #     # #######\n"
	    "\n\n")
    (mew-insert "Size:\t\t%d bytes\n" size)
    (insert (format "Browser:\t%s\n"
		    (cond ((and (symbolp mew-prog-text/html)
				(fboundp mew-prog-text/html))
			   (symbol-name mew-prog-text/html))
			  ((stringp mew-prog-text/html) mew-prog-text/html)
			  (t "none")))
	    "\nTo save this part, type "
	    (substitute-command-keys
	     "\\<mew-summary-mode-map>\\[mew-summary-save].")
	    "\nTo display this part in Message mode, type "
	    (substitute-command-keys
	     "\\<mew-summary-mode-map>\\[mew-summary-insert]."))
    (if (null execute)
        (insert "\nTo display this text/html contents with browser, type "
                (substitute-command-keys
                 "\\<mew-summary-mode-map>\\[mew-summary-execute-external]."))
      (cond
       ((and (symbolp mew-prog-text/html) (fboundp mew-prog-text/html))
	(let (source)
	  (set-buffer (mew-current-get 'cache))
	  (setq source (buffer-substring begin end))
	  (set-buffer (mew-buffer-message))
	  (mew-erase-buffer)
	  (insert source)
	  (funcall mew-prog-text/html (point-min) (point-max))))
       ((stringp mew-prog-text/html)
	(if (> end begin)
	    (let ((file (format "%s.html" (mew-make-temp-name))))
	      (save-excursion
		(set-buffer (mew-current-get 'cache))
		(mew-frwlet
		 mew-cs-noconv mew-cs-outfile
		 (write-region begin end file nil 'no-msg)
		 (apply (function start-process)
			mew-prog-text/html mew-buffer-tmp mew-prog-text/html
			(append mew-prog-text/html-arg (list file))))))))))))

(defun mew-mime-message/rfc822 (part)
  (let* ((hbegin (mew-syntax-get-begin part))
	 (hend   (mew-syntax-get-end   part))
	 (cache  (mew-current-get 'cache))
	 (body   (mew-syntax-get-part part))
	 (buffer-read-only nil))
    (insert-buffer-substring cache hbegin hend)
    (mew-decode-syntax-insert-privacy)
    (insert "\n")
    (mew-header-arrange)
    (cond
     ;; Displaying the text/plain body or the first part of 
     ;; top level multipart if it is text/plain.
     ;; see also mew-syntax-singlepart
     ((mew-syntax-singlepart-p body)
      (mew-mime-part part nil)) ;; nil is single
     ((mew-syntax-multipart-p body)
      (let* ((first (mew-syntax-get-part body))
	     (ct (car (mew-syntax-get-ct first)))
	     (cdpl (mew-syntax-get-cdp first)))
	(if (and (mew-case-equal ct mew-ct-txt)
		 (or (null cdpl)
		     (null (mew-syntax-get-member cdpl "filename"))))
	    (let* ((syntax (mew-syntax-get-entry-strnum body "1"))
		   (begin   (mew-syntax-get-begin syntax))
		   (end     (mew-syntax-get-end   syntax)))
	      (mew-mime-text/plain begin end))))))))

(defun mew-mime-application/octet-stream (begin end &optional params ct cte)
  (let ((size (- end begin))
	(buffer-read-only nil))
    (insert " ######    ###   #     #    #    ######  #     #\n"
	    " #     #    #    ##    #   # #   #     #  #   #\n"
	    " #     #    #    # #   #  #   #  #     #   # #\n"
	    " ######     #    #  #  # #     # ######     #\n"
	    " #     #    #    #   # # ####### #   #      #\n"
	    " #     #    #    #    ## #     # #    #     #\n"
	    " ######    ###   #     # #     # #     #    #\n"
	    "\n\n")
    (mew-insert "Content-Type:\t%s\n" ct)
    (mew-insert "Encoding: \t%s\n" cte)
    (and params
	 (insert (format "Parameters:\t%s\n"
			 (mapconcat (function mew-header-sanity-check-string)
				    params "; "))))
    (mew-insert "Size:\t\t%d bytes\n" size)
    (insert "\nTo save this part, type "
	    (substitute-command-keys
	     "\\<mew-summary-mode-map>\\[mew-summary-save].")
	    "\nTo display this part in Message mode, type "
	    (substitute-command-keys
	     "\\<mew-summary-mode-map>\\[mew-summary-insert]."))))

(defun mew-summary-insert ()
  (interactive)
  (let* ((ofld-msg (mew-current-get 'message))
	 (msg (mew-summary-message-number))
	 (part (mew-syntax-number))
	 (buf (buffer-name)))
    (if (or msg (not part))
	(let ((mew-analysis nil))
	  (mew-summary-display))
      (unwind-protect
	  (progn
	    (mew-summary-toggle-disp-msg 'on)
	    (mew-window-configure buf 'message)
	    (set-buffer (mew-buffer-message))
	    (let* ((buffer-read-only nil)
		   (syntax (mew-cache-decode-syntax (mew-cache-hit ofld-msg)))
		   (stx (mew-syntax-get-entry-strnum syntax part))
		   (begin (mew-syntax-get-begin stx))
		   (end (mew-syntax-get-end stx)))
	      (erase-buffer)
	      (insert-buffer-substring (mew-current-get 'cache) begin end)
	      (run-hooks 'mew-message-hook)
	      (mew-message-set-end-of)
	      (goto-char (point-min))))
	(mew-pop-to-buffer buf)))))

(provide 'mew-mime)

;;; Copyright Notice:

;; Copyright (C) 1997, 1998 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-mime.el ends here
