/* DSTART                                                                    */
/*                                                                           */
/*           maildrop - mail delivery agent with filtering abilities         */
/*                                                                           */
/*  Copyright 1998, Double Precision Inc.                                    */
/*                                                                           */
/*  This program is distributed under the terms of the GNU General Public    */
/*  License. See COPYING for additional information.                         */
/* DEND                                                                      */
#include	<stdio.h>
#include	<ctype.h>
#include	<stdlib.h>
#include	<string.h>
#include	"rfc822.h"

static const char rcsid[]="$Id: rfc822.c 1.1 1998/04/16 23:53:22 mrsam Exp $";

static void tokenize(const char *p, struct rfc822token *tokp, int *toklen,
	void (*err_func)(const char *, int))
{
const char *addr=p;
int	i=0;
int	inbracket=0;

	*toklen=0;
	while (*p)
	{
		if (isspace(*p))
		{
			p++;
			i++;
			continue;
		}

		switch (*p)	{
		int	level;

		case '(':
			if (tokp)
			{
				tokp->token='(';
				tokp->ptr=p;
				tokp->len=0;
			}
			level=0;
			for (;;)
			{
				if (!*p)
				{
					if (err_func) (*err_func)(addr, i);
					if (tokp) tokp->token='"';
					++*toklen;
					return;
				}
				if (*p == '(')
					++level;
				if (*p == '\\' && p[1])
				{
					p++;
					i++;
					if (tokp)	tokp->len++;
				}

				i++;
				if (tokp)	tokp->len++;
				if (*p++ == ')' && --level == 0)
					break;
			}
			if (tokp)	++tokp;
			++*toklen;
			continue;

		case '"':
			p++;
			i++;

			if (tokp)
			{
				tokp->token='"';
				tokp->ptr=p;
			}
			while (*p != '"')
			{
				if (!*p)
				{
					if (err_func) (*err_func)(addr, i);
					++*toklen;
					return;
				}
				if (*p == '\\' && p[1])
				{
					if (tokp)	tokp->len++;
					p++;
					i++;
				}
				if (tokp)	tokp->len++;
				p++;
				i++;
			}
			++*toklen;
			if (tokp)	++tokp;
			p++;
			i++;
			continue;
		case '\\':
		case ')':
			if (err_func) (*err_func)(addr, i);
			++p;
			++i;
			continue;
		case '<':
		case '>':
		case '@':
		case ',':
		case ';':
		case ':':
		case '.':
		case '[':
		case ']':
		case '%':
		case '!':

			if ( (*p == '<' && inbracket) ||
				(*p == '>' && !inbracket))
			{
				if (err_func) (*err_func)(addr, i);
				++p;
				++i;
				continue;
			}

			if (*p == '<')
				inbracket=1;

			if (*p == '>')
				inbracket=0;

			if (tokp)
			{
				tokp->token= *p;
				tokp->ptr=p;
				tokp->len=1;
				++tokp;
			}
			++*toklen;
			++p;
			++i;
			continue;
		default:

			if (tokp)
			{
				tokp->token=0;
				tokp->ptr=p;
				tokp->len=0;
			}
			while (*p && !isspace(*p) && strchr(
				"<>@,;:.[]()%!\"\\", *p) == 0)
			{
				if (tokp)	++tokp->len;
				++p;
				++i;
			}
			if (i == 0)	/* Idiot check */
			{
				if (err_func) (*err_func)(addr, i);
				if (tokp)
				{
					tokp->token='"';
					tokp->ptr=p;
					tokp->len=1;
					++tokp;
				}
				++*toklen;
				++p;
				++i;
				continue;
			}
			if (tokp)	++tokp;
			++*toklen;
		}
	}
}

static void parseaddr(struct rfc822token *tokens, int ntokens,
		struct rfc822addr *addrs, int *naddrs)
{
int	flag, j, k;
struct	rfc822token	save_token;

	*naddrs=0;

	while (ntokens)
	{
	int	i;

		/* atoms (token=0) or quoted strings, followed by a : token
		is a list name. */

		for (i=0; i<ntokens; i++)
			if (tokens[i].token && tokens[i].token != '"')
				break;
		if (i < ntokens && tokens[i].token == ':')
		{
			++i;
			if (addrs)
			{
				addrs->tokens=0;
				addrs->ntokens=0;
				addrs->name=tokens;
				addrs->nname=i;
				addrs++;
			}
			++*naddrs;
			tokens += i;
			ntokens -= i;
			continue;  /* Group=phrase ":" */
		}

		/* Spurious commas are skipped, ;s are recorded */

		if (tokens->token == ',' || tokens->token == ';')
		{
			if (tokens->token == ';')
			{
				if (addrs)
				{
					addrs->tokens=0;
					addrs->ntokens=0;
					addrs->name=tokens;
					addrs->nname=1;
					addrs++;
				}
				++*naddrs;
			}
			++tokens;
			--ntokens;
			continue;
		}

		/* If we can find a '<' before the next comma or semicolon,
		we have new style RFC path address */

		for (i=0; i<ntokens && tokens[i].token != ';' &&
				tokens[i].token != ',' &&
					tokens[i].token != '<'; i++)
			;

		if (tokens[i].token == '<')
		{
		int	j;

			/* Ok -- what to do with the stuff before '>'???
			If it consists exclusively of atoms, leave them alone.
			Else, make them all a quoted string. */

			for (j=0; j<i && (tokens[j].token == 0 ||
					tokens[j].token == '('); j++)
				;

			if (j == i)
			{
				if (addrs)
				{
					addrs->name=tokens;
					addrs->nname=i;
				}
			}
			else	/* Intentionally corrupt the original toks */
			{
				if (addrs)
				{
					tokens->len= tokens[i-1].ptr
							+ tokens[i-1].len
							- tokens->ptr;
					/* We know that all the ptrs point
					to parts of the same string. */
					tokens->token='"';
						/* Quoted string. */
					addrs->name=tokens;
					addrs->nname=1;
				}
			}

		/* Any comments in the name part are changed to quotes */

			if (addrs)
			{
				for (j=0; j<addrs->nname; j++)
					if (addrs->name[j].token == '(')
						addrs->name[j].token='"';
			}

			/* Now that's done and over with, see what can
			be done with the <...> part. */

			++i;
			tokens += i;
			ntokens -= i;
			for (i=0; i<ntokens && tokens[i].token != '>'; i++)
				;
			if (addrs)
			{
				addrs->tokens=tokens;
				addrs->ntokens=i;
				++addrs;
			}
			++*naddrs;
			tokens += i;
			ntokens -= i;
			if (ntokens)	/* Skip the '>' token */
			{
				--ntokens;
				++tokens;
			}
			continue;
		}

		/* Ok - old style address.  Assume the worst */

		/* Try to figure out where the address ends.  It ends upon:
		a comma, semicolon, or two consecutive atoms. */

		flag=0;
		for (i=0; i<ntokens && tokens[i].token != ',' &&
			tokens[i].token != ';'; i++)
		{
			if (tokens[i].token == '(')	continue;
					/* Ignore comments */
			if (tokens[i].token == 0 || tokens[i].token == '"')
								/* Atom */
			{
				if (flag)	break;
				flag=1;
			}
			else	flag=0;
		}
		if (i == 0)	/* Must be spurious comma, or something */
		{
			++tokens;
			--ntokens;
			continue;
		}

		if (addrs)
		{
			addrs->name=0;
			addrs->nname=0;
		}

		/* Ok, now get rid of embedded comments in the address.
		Consider the last comment to be the real name */

		if (addrs)
		{

			save_token.ptr=0;
			save_token.len=0;

			for (j=k=0; j<i; j++)
			{
				if (tokens[j].token == '(')
				{
					save_token=tokens[j];
					continue;
				}
				tokens[k]=tokens[j];
				k++;
			}

			if (save_token.ptr)
			{
				tokens[i-1]=save_token;
				addrs->name=tokens+i-1;
				addrs->nname=1;
			}
			addrs->tokens=tokens;
			addrs->ntokens=k;
			++addrs;
		}
		++*naddrs;
		tokens += i;
		ntokens -= i;
	}
}

static void print_token(const struct rfc822token *token,
		void (*print_func)(char))
{
const char *p;
int	n;

	if (token->token == 0 || token->token == '(')
	{
		for (n=token->len, p=token->ptr; n; --n, ++p)
			(*print_func)(*p);
		return;
	}

	if (token->token != '"')
	{
		(*print_func)(token->token);
		return;
	}

	(*print_func)('"');
	n=token->len;
	p=token->ptr;
	while (n)
	{
		if (*p == '"' || (*p == '\\' && n == 1)) (*print_func)('\\');
		if (*p == '\\' && n > 1)
		{
			(*print_func)('\\');
			++p;
			--n;
		}
		(*print_func)(*p++);
		--n;
	}
	(*print_func)('"');
}

#define	is_atom(p)	( (p) == 0 || (p) == '"' || (p) == '(' )

static void print_tokens(const struct rfc822token *token, int ntokens,
		void (*print_func)(char))
{
	while (ntokens)
	{
		print_token(token, print_func);
		++token;
		--ntokens;
		if (ntokens && is_atom(token[-1].token) &&
				is_atom(token->token))
			(*print_func)(' ');
	}
}

void rfc822_print(struct rfc822 *rfcp, void (*print_func)(char),
	void (*print_separator)(const char *))
{
const struct rfc822addr *addrs=rfcp->addrs;
int naddrs=rfcp->naddrs;
int	i;

	while (naddrs)
	{
		if (addrs->ntokens == 0)
		{
			for (i=0; i<addrs->nname; i++)
			{
				print_token(addrs->name+i, print_func);
				if (i + 1 < addrs->nname &&
					is_atom( addrs->name[i].token ) &&
					is_atom( addrs->name[i+1].token ) )
					(*print_func)(' ');
			}
			++addrs;
			--naddrs;
			if (addrs[-1].nname && naddrs &&
				( addrs[-1].name[addrs[-1].nname-1].token==':'||
				  addrs[-1].name[addrs[-1].nname-1].token==';'))
				(*print_separator)(" ");
			continue;
		}
		else if (addrs->nname > 0 && addrs->name->token == '(')
		{	/* old style */

			print_tokens(addrs->tokens, addrs->ntokens, print_func);
			(*print_func)(' ');
			print_tokens(addrs->name, addrs->nname, print_func);
		}
		else
		{
		int	print_braces=0;

			if (addrs->nname)
			{
				print_tokens(addrs->name, addrs->nname,
						print_func);
				(*print_func)(' ');
				print_braces=1;
			}
			else
			{
				for (i=1; i<addrs->ntokens; i++)
					if (is_atom(addrs->tokens[i-1].token) &&
						is_atom(addrs->tokens[i].token))
					print_braces=1;
			}
			if (print_braces)
				(*print_func)('<');
			print_tokens(addrs->tokens, addrs->ntokens, print_func);
			if (print_braces)
			{
				(*print_func)('>');
			}
		}
		++addrs;
		--naddrs;
		if (naddrs)
			if (addrs->tokens || (addrs->nname &&
				is_atom(addrs->name->token)))
				(*print_separator)(", ");
	}
}

void rfc822_free(struct rfc822 *p)
{
	if (p->tokens)	free(p->tokens);
	if (p->addrs)	free(p->addrs);
	free(p);
}

void rfc822_deladdr(struct rfc822 *rfcp, int index)
{
	if (index < 0 || index >= rfcp->naddrs)	return;

int	i;

	for (i=index+1; i<rfcp->naddrs; i++)
		rfcp->addrs[i-1]=rfcp->addrs[i];
	if (--rfcp->naddrs == 0)
	{
		free(rfcp->addrs);
		rfcp->addrs=0;
	}
}

struct rfc822 *rfc822_alloc(const char *addr,
	void (*err_func)(const char *, int))
{
struct rfc822 *p=(struct rfc822 *)malloc(sizeof(struct rfc822));

	if (!p)	return (NULL);
	memset(p, 0, sizeof(*p));

	tokenize(addr, NULL, &p->ntokens, err_func);
	p->tokens=p->ntokens ? (struct rfc822token *)
			calloc(p->ntokens, sizeof(struct rfc822token)):0;
	if (p->ntokens && !p->tokens)
	{
		rfc822_free(p);
		return (NULL);
	}

	tokenize(addr, p->tokens, &p->ntokens, NULL);

	parseaddr(p->tokens, p->ntokens, NULL, &p->naddrs);
	p->addrs=p->naddrs ? (struct rfc822addr *)
			calloc(p->naddrs, sizeof(struct rfc822addr)):0;
	if (p->naddrs && !p->addrs)
	{
		rfc822_free(p);
		return (NULL);
	}
	parseaddr(p->tokens, p->ntokens, p->addrs, &p->naddrs);
	return (p);
}

void rfc822_praddr(const struct rfc822 *rfcp, int index, void (*print_func)(char))
{
	if (index < 0 || index >= rfcp->naddrs)	return;

const struct rfc822addr *addrs=rfcp->addrs+index;

	if (addrs->ntokens)
	{
		print_tokens(addrs->tokens, addrs->ntokens, print_func);
		(*print_func)('\n');
	}
}

void rfc822_addrlist(const struct rfc822 *rfcp, void (*print_func)(char))
{
int	i;

	for (i=0; i<rfcp->naddrs; i++)
		rfc822_praddr(rfcp, i, print_func);
}

void rfc822_prname(const struct rfc822 *rfcp, int index,
	void (*print_func)(char))
{
	if (index < 0 || index >= rfcp->naddrs)	return;

const struct rfc822addr *addrs=rfcp->addrs+index;

	if (!addrs->ntokens)	return;

	if (addrs->nname)
	{
	int i;

		for (i=0; i<addrs->nname; i++)
		{
			if (i && is_atom(addrs->name[i-1].token)
				&& is_atom(addrs->name[i].token))
				(*print_func)(' ');

			if (addrs->name[i].token != '(')
			{
				print_token(&addrs->name[i],
					print_func);
				continue;
			}

		int	n;
			for (n=2; n<addrs->name[i].len; n++)
				(*print_func)(
					addrs->name[i].ptr[n-1]
				);
		}
	} else
		print_tokens(addrs->tokens, addrs->ntokens,
				print_func);
	(*print_func)('\n');
}

void rfc822_namelist(const struct rfc822 *rfcp, void (*print_func)(char))
{
int	i;

	for (i=0; i<rfcp->naddrs; i++)
		rfc822_prname(rfcp, i, print_func);
}
