/*
 * GWT - General Windowing Toolkit
 *
 * Copyright (C) 1998 MenTaLguY - mentalg@geocities.com
 *                    Rodolphe Ortalo - ortalo@laas.fr
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Log: events.c,v $
 * Revision 1.3  1998/08/23 18:57:08  ortalo
 * Minor editing (mainly comments).
 *
 * Revision 1.2  1998/08/11 22:17:33  ortalo
 * Adapted to the gwt.h->ggi/gwt.h change. Incorporated the modifications
 * brought by Tristan Wibberley.
 *
 * Revision 1.1  1998/07/09 18:38:39  ortalo
 * Initial integration of libgwt in the GGI repository.
 * This is experimental code, for your eyes only !
 * (Don't execute... ;-)
 *
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>

#include "ggi/gwt.h"
#include "internal.h"

#ifdef USE_THREADS
#include <pthread.h>
static pthread_mutex_t event_record_store_mutex = PTHREAD_MUTEX_INITIALIZER;
#endif

#define MAX_EVENT_RECORD_STORE 32

static gwt_event_record *event_record_store=NULL;
static int event_record_store_count=0;

/************************************************************************
 *                      PRIVATE STATIC FUNCTIONS                        *
 ************************************************************************/

/*
 * Event record management
 */

/* MAllocate a new event record */
static gwt_event_record* alloc_event_record(void) {
	return (gwt_event_record *)malloc(sizeof(gwt_event_record));
}
/* Free a new event record */
static void free_event_record(gwt_event_record *rec) {
	free(rec);
}

/* Removes an event record from its queue */
static void extract_event_record(gwt_event_record *rec) {
	if ( rec->next != NULL ) {
		rec->next->prev = rec->prev;
	}
	if ( rec->prev != NULL ) {
		rec->prev->next = rec->next;
	}
	rec->next = rec->prev = NULL;
}
/* Finds a new or recycled event record */
static gwt_event_record* new_event_record(void) {
	gwt_event_record *rec;
#ifdef USE_THREADS
	pthread_mutex_lock(&event_record_store_mutex);
#endif
	if ( event_record_store != NULL ) {
		rec = event_record_store;
		event_record_store = rec->next;
		if ( rec->next != NULL ) {
			rec->next->prev = NULL;
		}
		rec->next = NULL;
		event_record_store_count--;
	} else {
		rec = alloc_event_record();
	}
#ifdef USE_THREADS
	pthread_mutex_unlock(&event_record_store_mutex);
#endif
	return rec;
}
/* Free an event record (and recycle it) */
static void discard_event_record(gwt_event_record *rec) {
	extract_event_record(rec);
#ifdef USE_THREADS
	pthread_mutex_lock(&event_record_store_mutex);
#endif
	if ( event_record_store_count >= MAX_EVENT_RECORD_STORE ) {
		free_event_record(rec);
	} else {
		rec->next = event_record_store;
		if ( rec->next != NULL ) {
			rec->next->prev = rec;
		}
		event_record_store = rec;
		event_record_store_count++;
	}
#ifdef USE_THREADS
	pthread_mutex_unlock(&event_record_store_mutex);
#endif
}

/*
 * Evqueue management
 */

int _gwt_evqueue_destroy(gwt_evqueue *queue) {
#ifdef USE_THREADS
	pthread_mutex_destroy(&queue->mutex);
#endif
	while (queue->head != NULL) {
		gwt_event_record *rec;
		rec = queue->head;
		queue->head = rec->next;
		discard_event_record(rec);
	}
	close(queue->fds[0]);
	close(queue->fds[1]);
	free(queue);
	return 0;
}

int _gwt_evqueue_increfcount(gwt_evqueue *queue) {
#ifdef USE_THREADS
	pthread_mutex_lock(&queue->mutex);
#endif
	queue->refcount++;
#ifdef USE_THREADS
	pthread_mutex_unlock(&queue->mutex);
#endif
	return 0;
}

int _gwt_evqueue_decrefcount(gwt_evqueue *queue) {
#ifdef USE_THREADS
	pthread_mutex_lock(&queue->mutex);
#endif
	queue->refcount--;
	if ( queue->refcount <= 0 ) {
#ifdef USE_THREADS
		pthread_mutex_unlock(&queue->mutex);
#endif
		return _gwt_evqueue_destroy(queue);
	} else {
#ifdef USE_THREADS
		pthread_mutex_unlock(&queue->mutex);
#endif
		return 0;
	}
}

int _gwt_event_rec_add(gwt_event_record *rec, gwt_evqueue_t queue) {
	char dummy;
	rec->prev = queue->head;
	if ( rec->prev == NULL ) queue->tail = rec;
	else rec->prev->next = rec;
	queue->head = rec;	
	queue->count++;
	write(queue->fds[1], &dummy, 1);	
	queue->seen |= 1 << rec->event.any.type;
	return 0;
}

int _gwt_event_rec_get(gwt_event_record **recp, gwt_evqueue_t queue) {
	gwt_event_record *rec;
	char dummy;

	rec = queue->tail;
	if ( rec->next == NULL ) queue->head = NULL;
	else rec->next->prev = NULL;
	queue->tail = rec->next;
	queue->count--;
	extract_event_record(rec);
	read(queue->fds[0], &dummy, 1);
	*recp = rec;
	queue->seen = 0;
	rec = queue->head;
	while ( rec != NULL ) {
		queue->seen |= 1 << rec->event.any.type;
		rec = rec->next;
	}
	return 0;
}

/************************************************************************
 *                  GWT PUBLIC EXPORTED FUNCTIONS                       *
 ************************************************************************/

/*
 * EvQueue management
 */
int gwtEvqueueCreate(gwt_evqueue_t *queue) {
	gwt_evqueue *qp;
	qp = (gwt_evqueue *)malloc(sizeof(gwt_evqueue));
	if ( qp == NULL ) return -errno;
	qp->refcount = 1;
	qp->head = qp->tail = NULL;
	qp->seen = 0;
	if ( pipe(qp->fds) != 0 ) {
		free(qp);
		return -errno;
	}
	qp->count = 0;
#ifdef USE_THREADS
	pthread_mutex_init(&qp->mutex);
#endif
	*queue = qp;
	return 0;
}
/*
 * Event queue destruction
 */
int gwtEvqueueDestroy(gwt_evqueue_t queue) {
	return _gwt_evqueue_decrefcount(queue);
}

/*
 * Returns the file descriptor of an Evqueue
 */
int gwtEventGetSelectFileDesc(gwt_evqueue_t queue) {
	return queue->fds[0];
}

/*
 * Post an event on an Evqueue
 */
int gwtEventPost(ggi_event *ev, gwt_evqueue_t queue) {
	gwt_event_record *rec;
	rec = new_event_record();

	memcpy(&rec->event, ev, sizeof(ggi_event));
	rec->recipient = NULL;
	rec->next = NULL;
#ifdef USE_THREADS
	pthread_mutex_lock(&queue->mutex);
#endif
	_gwt_event_rec_add(rec, queue);
#ifdef USE_THREADS
	pthread_mutex_unlock(&queue->mutex);
#endif
	return 0;
}

/*******************************************************************
 * Event functions used to "send" events to windows (manager side) *
 *            [NB: Personal interpretation, not sure :-( -- ortalo]*
 *******************************************************************/

/*
 * Deliver an event to a window
 */
int gwtEventDeliver(ggi_event *ev, gwt_window_t win) {
	int status=0;
	gwt_lock();
	if ( ( win->_ev_queue != NULL ) && ( win->accepts & ( 1 << ev->any.type ) ) ) {
		gwt_event_record *rec;
		rec = new_event_record();
		memcpy(&rec->event, ev, sizeof(ggi_event));
		rec->recipient = win;
		rec->next = NULL;
#ifdef USE_THREADS
		pthread_mutex_lock(&queue->mutex);
#endif
		_gwt_event_rec_add(rec, win->_ev_queue);
#ifdef USE_THREADS
		pthread_mutex_unlock(&queue->mutex);
#endif
	} else {
		status = -1;
	}
	gwt_unlock();
	return status;
}

/*
 * Process events of a window
 */
int gwtEventProcess(ggi_event *ev, gwt_window_t win) {
/* FIXME !!! this should descend into the window tree until it finds a window
   that will recieve the event (_ev_queue != NULL && the mask matches),
   then deliver the event to that window. */
  /* TODO: NB: This should not be very difficult, maybe also we should
     check for mouse position and window geometry ? --ortalo */
	return gwtEventDeliver(ev, win);
}

/*******************************************************************
 * Event functions used to "get" events from windows (user side)   *
 *            [NB: Personal interpretation, not sure :-( -- ortalo]*
 *******************************************************************/

/*
 * Get an event from a queue for a window (?)
 */
int gwtEventGet(ggi_event *event, gwt_window_t *recipient, gwt_evqueue_t queue, ggi_event_mask accept) {
	gwt_event_record *rec;
	int status=0;

#ifdef USE_THREADS
	pthread_mutex_lock(&queue->mutex);
#endif
	if ( queue->count <= 0 ) {
		status = -1;
	} else {
		rec = queue->tail;
		while ( ( rec != NULL ) &&
			!( accept & ( 1 << rec->event.any.type ) ) )
		  {
		  }
		if ( rec == NULL ) {
			status = -1;
		} else {
			char dummy;
			if ( queue->head == rec ) queue->head = rec->prev;
			if ( queue->tail == rec ) queue->tail = rec->prev;
			extract_event_record(rec);
			if ( event != NULL )
			  memcpy(event, &rec->event, sizeof(ggi_event));
			if ( recipient != NULL ) *recipient = rec->recipient;
			discard_event_record(rec);
			queue->count--;
			read(queue->fds[0], &dummy, 1);
			queue->seen = 0;
			rec = queue->head;
			while ( rec != NULL ) {
				queue->seen |= 1 << rec->event.any.type;
				rec = rec->next;
			}
		}
	}
#ifdef USE_THREADS
	pthread_mutex_unlock(&queue->mutex);
#endif
	return status;
}
