-- This file is  free  software, which  comes  along  with  SmallEiffel. This
-- software  is  distributed  in the hope that it will be useful, but WITHOUT 
-- ANY  WARRANTY;  without  even  the  implied warranty of MERCHANTABILITY or
-- FITNESS  FOR A PARTICULAR PURPOSE. You can modify it as you want, provided
-- this header is kept unaltered, and a notification of the changes is added.
-- You  are  allowed  to  redistribute  it and sell it, alone or as a part of 
-- another product.
--          Copyright (C) 1994-98 LORIA - UHP - CRIN - INRIA - FRANCE
--            Dominique COLNET and Suzanne COLLIN - colnet@loria.fr 
--                       http://www.loria.fr/SmallEiffel
--
expanded class DOUBLE
--
-- Note: An Eiffel DOUBLE is mapped as a C double or as a Java double.
--

inherit
   DOUBLE_REF
      redefine
	 infix "+", infix "-", infix "*", infix "/", infix "^", prefix "+",
	 prefix "-", divisible, infix "<", infix "<=", infix ">", infix ">=",
	 compare, one, zero, fill_tagged_out_memory, out_in_tagged_out_memory,
	 hash_code
      end;
   
feature {ANY}

   infix "+" (other: DOUBLE): DOUBLE is
	 -- Add `other' to Current.
      external "SmallEiffel"
      end;

   infix "-" (other: DOUBLE): DOUBLE is
	 -- Subtract `other' from Current.
      external "SmallEiffel"
      end;

   infix "*" (other: DOUBLE): DOUBLE is
	 -- Multiply `other' by Current.
      external "SmallEiffel"
      end;
   
   infix "/" (other: DOUBLE): DOUBLE is
	 -- Divide Current by `other'.
      external "SmallEiffel"
      end;

   infix "^" (e: INTEGER): DOUBLE is
	 -- Raise Current to `e'-th power.
	 -- Note: use ANSI C pow.
      external "SmallEiffel" 
      end;

   prefix "+" : DOUBLE is
      do
	 Result := Current
      end;

   prefix "-": DOUBLE is
      external "SmallEiffel"
      end;

   abs: DOUBLE is
      do
	 if Current < 0.0 then
	    Result := -Current;
	 else
	    Result := Current;
	 end;
      end;
         
   infix "<" (other: DOUBLE): BOOLEAN is
	 -- Is Current less than `other'?
      external "SmallEiffel"
      end;
   
   infix "<=" (other: DOUBLE): BOOLEAN is
	 -- Is Current smaller or equal to `other'?
      external "SmallEiffel"
      end;
   
   infix ">" (other: DOUBLE): BOOLEAN is
	 -- Is Current greater than `other'?
      external "SmallEiffel"
      end;
   
   infix ">=" (other: DOUBLE): BOOLEAN is
	 -- Is Current greater or equal to `other'?
      external "SmallEiffel"
      end;
   
   compare(other: DOUBLE): INTEGER is
	 -- Compare Current with `other'.
	 -- '<' <==> Result < 0
	 -- '>' <==> Result > 0
	 -- Otherwise Result = 0
      do
	 if Current < other then
	    Result := -1
	 else
	    if Current > other then
	       Result := 1
	    end;
	 end;
      end;

   divisible(other: DOUBLE): BOOLEAN is
      do
	 Result := (other /= 0.0)
      end;
   
   one: DOUBLE is 1.0;
   
   zero: DOUBLE is 0.0;

   double_floor: DOUBLE is
	 -- Greatest integral value no greater than Current.
      external "SmallEiffel"
      end;

   floor: INTEGER is
	 -- Greatest integral value no greater than Current.
      local
	 d: DOUBLE;
      do
	 d := double_floor;
	 Result := d.truncated_to_integer;
      ensure 
	 result_no_greater: Result.to_double <= Current;
	 close_enough: Current - Result < one;
      end;
      
   double_ceiling: DOUBLE is
	 -- Smallest integral value no smaller than Current.
      do
	 Result := ceil;
      end;

   ceiling: INTEGER is
	 -- Smallest integral value no smaller than Current.
      local
	 d: DOUBLE;
      do
	 d := double_ceiling;
	 Result := d.truncated_to_integer;
      ensure
	 result_no_smaller: Current <= Result;
	 close_enough: Current - Result < one;
      end;
   
   rounded: INTEGER is
      	 -- Rounded integral value.
      do
	 if double_floor + 0.5 < Current then
	    Result := double_ceiling.truncated_to_integer;
	 else
	    Result := double_floor.truncated_to_integer;
	 end;
      end;
   
   truncated_to_integer: INTEGER is
      	 -- Integer part (same sign, largest absolute value
       	 -- no greater than Current).
      external "SmallEiffel"
      ensure
	 Result.to_double <= Current
      end;
   
   to_real: REAL is
	 -- Note: C conversion from "double" to "float".
	 -- Thus, Result can be undefine (ANSI C).
      external "SmallEiffel"
      end;
   
   to_string: STRING is
	 -- Convert the DOUBLE into a new allocated STRING. 
	 -- As ANSI C, the default number of digit for the
	 -- fractionnal part is 6.
	 -- Note: see `append_in' to save memory.
      do
	 !!Result.make(8);
	 append_in(Result);
      end; 

   append_in(str: STRING) is
	 -- Append the equivalent of `to_string' at the end of 
	 -- `str'. Thus you can save memory because no other
	 -- STRING is allocate for the job.
      require
	 str /= Void
      do
	 append_in_format(str,6);
      end; 
   
   to_string_format(d: INTEGER): STRING is
	 -- Convert the DOUBLE into a new allocated STRING including 
	 -- only `d' digits in fractionnal part. 
	 -- Note: see `append_in_format' to save memory.
      do
	 !!Result.make(8);
	 append_in_format(Result,d);
      end; 

   append_in_format(str: STRING; f: INTEGER) is
	 -- Same as `append_in' but produce `f' digit of 
	 -- the fractionnal part.
      require
	 str /= Void;
	 f >= 0
      local
	 i: INTEGER;
      do
	 from
	    sprintf_double(tmp_native_array,f);
	    i := 0;
	 until
	    tmp_native_array.item(i) = '%U'
	 loop
	    str.extend(tmp_native_array.item(i));
	    i := i + 1;
	 end;
      end; 
   
feature -- Maths functions :

   sqrt: DOUBLE is
	 -- Square routine (ANSI C sqrt).
      require
	 Current >= 0;
      external "SmallEiffel"
      end;

   sin: DOUBLE is
	 -- Sinus (ANSI C sin).
      external "SmallEiffel"
      end;

   cos: DOUBLE is
	 -- Cosinus (ANSI C cos).
      external "SmallEiffel"
      end;

   tan: DOUBLE is
	 -- (ANSI C tan).
      external "SmallEiffel"
      end;

   asin: DOUBLE is
	 -- (ANSI C asin).
      external "SmallEiffel"
      end;

   acos: DOUBLE is
	 -- (ANSI C acos).
      external "SmallEiffel"
      end;

   atan: DOUBLE is
	 -- (ANSI C atan).
      external "SmallEiffel"
      end;

   sinh: DOUBLE is
	 -- (ANSI C sinh).
      external "SmallEiffel"
      end;

   cosh: DOUBLE is
	 -- (ANSI C cosh).
      external "SmallEiffel"
      end;

   tanh: DOUBLE is
	 -- (ANSI C tanh).
      external "SmallEiffel"
      end;

   exp: DOUBLE is
	 -- (ANSI C exp).
      external "SmallEiffel"
      end;

   log: DOUBLE is
	 -- (ANSI C log).
      external "SmallEiffel"
      end;

   log10: DOUBLE is
	 -- (ANSI C log10).
      external "SmallEiffel"
      end;

feature -- Object Printing :

   out_in_tagged_out_memory, fill_tagged_out_memory is
      do
	 Current.append_in(tagged_out_memory);
      end;

feature -- Hashing :

   hash_code: INTEGER is
      local
	 d: DOUBLE;
      do
	 d := double_floor;
	 if d < 0 then
	    d := - d;
	 end;
	 Result := d.truncated_to_integer;
      end;

feature {NONE}

   ceil: DOUBLE is 
      external "SmallEiffel"
      end;

   sprintf_double(native_array: NATIVE_ARRAY[CHARACTER]; f: INTEGER) is
	 -- Put in `native_array' a viewable version of Current with
	 -- `f' digit of the fractionnal part.
      external "SmallEiffel"
      end;
   
   tmp_native_array: NATIVE_ARRAY[CHARACTER] is
      once
	 Result := Result.calloc(128);
      end;

end -- DOUBLE

