/*
 * SXPGS.C - PGS extensions in SX
 *
 * Source Version: 3.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "sx.h"

static void
 SC_DECLARE(_SX_args_arr_2, 
            (object *argl, int *pn, REAL **px, REAL **py)),
 SC_DECLARE(_SX_args_arr_3,
         (object *argl, int *pn, REAL **px, REAL **py, REAL **pz));

static object
 *SC_DECLARE(SX_axis, (object *argl)),
 *SC_DECLARE(SX_center_label, (object *argl)),
 *SC_DECLARE(SX_clear_window, (object *argl)),
 *SC_DECLARE(SX_clear_viewport, (object *argl)),
 *SC_DECLARE(SX_clear_region, (object *argl)),
 *SC_DECLARE(SX_clr_mode, (object *argl)),
 *SC_DECLARE(SX_current_pal, (object *argl)),
 *SC_DECLARE(SX_def_mrk, (object *argl)),
 *SC_DECLARE(SX_ddp2, (object *argl)),
 *SC_DECLARE(SX_ddp3, (object *argl)),
 *SC_DECLARE(SX_draw_arc, (object *argl)),
 *SC_DECLARE(SX_draw_axis, (object *argl)),
 *SC_DECLARE(SX_draw_box, (object *argl)),
 *SC_DECLARE(SX_draw_line, (object *argl)),
 *SC_DECLARE(SX_draw_palette, (object *argl)),
 *SC_DECLARE(SX_draw_polyline, (object *argl)),
 *SC_DECLARE(SX_draw_rad, (object *argl)),
 *SC_DECLARE(SX_draw_text, (object *argl)),
 *SC_DECLARE(SX_fply, (object *argl)),
 *SC_DECLARE(SX_finish_plot, (object *argl)),
 *SC_DECLARE(SX_gcpw, (object *argl)),
 *SC_DECLARE(SX_gcss, (object *argl)),
 *SC_DECLARE(SX_gcuw, (object *argl)),
 *SC_DECLARE(SX_gclp, (object *argl)),
 *SC_DECLARE(SX_get_frame, (object *argl)),
 *SC_DECLARE(SX_get_viewport, (object *argl)),
 *SC_DECLARE(SX_get_window, (object *argl)),
 *SC_DECLARE(SX_gfin, (object *argl)),
 *SC_DECLARE(SX_glnc, (object *argl)),
 *SC_DECLARE(SX_glns, (object *argl)),
 *SC_DECLARE(SX_glnw, (object *argl)),
 *SC_DECLARE(SX_glop, (object *argl)),
 *SC_DECLARE(SX_gmxi, (object *argl)),
 *SC_DECLARE(SX_gtew, (object *argl)),
 *SC_DECLARE(SX_gtxc, (object *argl)),
 *SC_DECLARE(SX_gtxf, (object *argl)),
 *SC_DECLARE(SX_list_pal, (object *argl)),
 *SC_DECLARE(SX_mk_pal, (object *argl)),
 *SC_DECLARE(SX_mdvc, (object *argl)),
 *SC_DECLARE(SX_pal_list, (object *argl)),
 *SC_DECLARE(SX_pals, (object *argl)),
 *SC_DECLARE(SX_ptos, (object *argl)),
 *SC_DECLARE(SX_qdev, (object *argl)),
 *SC_DECLARE(SX_qwin, (object *argl)),
 *SC_DECLARE(SX_rd_pal, (object *argl)),
 *SC_DECLARE(SX_rdvc, (object *argl)),
 *SC_DECLARE(SX_sadm, (object *argl)),
 *SC_DECLARE(SX_sapl, (object *argl)),
 *SC_DECLARE(SX_sarn, (object *argl)),
 *SC_DECLARE(SX_sbwd, (object *argl)),
 *SC_DECLARE(SX_scpw, (object *argl)),
 *SC_DECLARE(SX_scuw, (object *argl)),
 *SC_DECLARE(SX_sclp, (object *argl)),
 *SC_DECLARE(SX_sdti, (object *argl)),
 *SC_DECLARE(SX_show_pal, (object *argl)),
 *SC_DECLARE(SX_sfic, (object *argl)),
 *SC_DECLARE(SX_sfin, (object *argl)),
 *SC_DECLARE(SX_sgrd, (object *argl)),
 *SC_DECLARE(SX_slnc, (object *argl)),
 *SC_DECLARE(SX_slns, (object *argl)),
 *SC_DECLARE(SX_slnw, (object *argl)),
 *SC_DECLARE(SX_slop, (object *argl)),
 *SC_DECLARE(SX_smxi, (object *argl)),
 *SC_DECLARE(SX_spal, (object *argl)),
 *SC_DECLARE(SX_sres_sf, (object *argl)),
 *SC_DECLARE(SX_ssct, (object *argl)),
 *SC_DECLARE(SX_stxc, (object *argl)),
 *SC_DECLARE(SX_stxf, (object *argl)),
 *SC_DECLARE(SX_swbk, (object *argl)),
 *SC_DECLARE(SX_set_clr_mode, (object *argl)),
 *SC_DECLARE(SX_set_frame, (object *argl)),
 *SC_DECLARE(SX_set_vect_attr, (object *argl)),
 *SC_DECLARE(SX_set_viewport, (object *argl)),
 *SC_DECLARE(SX_set_window, (object *argl)),
 *SC_DECLARE(SX_set_mrk_ornt, (object *argl)),
 *SC_DECLARE(SX_mrk_ornt, (object *argl)),
 *SC_DECLARE(SX_set_mrk_scal, (object *argl)),
 *SC_DECLARE(SX_mrk_scal, (object *argl)),
 *SC_DECLARE(SX_drw_mrk, (object *argl)),
 *SC_DECLARE(SX_show_mrk, (byte)),
 *SC_DECLARE(SX_stop, (object *argl)),
 *SC_DECLARE(SX_stow, (object *argl)),
 *SC_DECLARE(SX_update_vs, (object *argl)),
 *SC_DECLARE(SX_wr_pal, (object *argl)),
 *SC_DECLARE(SX_wtos, (object *argl));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_INSTALL_PGS_PRIMITIVES - install the PGS primitives */

void SX_install_pgs_primitives()
   {SS_install("pg-axis",
               "Draw a complete axis set on the given device",
               SS_nargs,
               SX_axis, SS_PR_PROC);

    SS_install("pg-character-path",
               "Return a pair specifying the character path direction on the given device",
               SS_nargs,
               SX_gcpw, SS_PR_PROC);

    SS_install("pg-character-size-ndc",
               "Return the character size in NDC for the given device",
               SS_nargs,
               SX_gcss, SS_PR_PROC);

    SS_install("pg-character-up",
               "Return a pair specifying the character up direction on the given device",
               SS_nargs,
               SX_gcuw, SS_PR_PROC);

    SS_install("pg-clear-mode",
               "Return the clear mode",
               SS_nargs,
               SX_clr_mode, SS_PR_PROC);

    SS_install("pg-clipping?",
               "Return #t iff clipping is on in the given device",
               SS_nargs,
               SX_gclp, SS_PR_PROC);

    SS_install("pg-clear-window",
               "Clear the screen and initialize a frame of the given device",
               SS_nargs,
               SX_clear_window, SS_PR_PROC);

    SS_install("pg-clear-viewport",
               "Clear the current viewport of the given device",
               SS_nargs,
               SX_clear_viewport, SS_PR_PROC);

    SS_install("pg-clear-region",
               "Clear the specified rectangular region of the given device",
               SS_nargs,
               SX_clear_region, SS_PR_PROC);

    SS_install("pg-center-label",
               "Print a label string centered on a line at an NDC height",
               SS_nargs,
               SX_center_label, SS_PR_PROC);

    SS_install("pg-current-palette",
               "Return the current palette for the given device",
               SS_nargs,
               SX_current_pal, SS_PR_PROC);

    SS_install("pg-define-marker",
               "Define a new marker character",
               SS_nargs,
               SX_def_mrk, SS_PR_PROC);

    SS_install("pg-draw-arc",
               "Draw a section of a circular arc on the given device",
               SS_nargs,
               SX_draw_arc, SS_PR_PROC);

    SS_install("pg-draw-axis",
               "Draw a single axis on the given device",
               SS_nargs,
               SX_draw_axis, SS_PR_PROC);

    SS_install("pg-draw-box",
               "Draw a box on the given device",
               SS_nargs,
               SX_draw_box, SS_PR_PROC);

    SS_install("pg-draw-disjoint-polyline-2d",
               "Draw a set of disjoint 2d line segments",
               SS_nargs,
               SX_ddp2, SS_PR_PROC);

    SS_install("pg-draw-disjoint-polyline-3d",
               "Draw a set of disjoint 3d line segments",
               SS_nargs,
               SX_ddp3, SS_PR_PROC);

    SS_install("pg-draw-line",
               "Draw a line on the given device",
               SS_nargs,
               SX_draw_line, SS_PR_PROC);

    SS_install("pg-draw-markers",
               "Draw the specified marker at the given points",
               SS_nargs,
               SX_drw_mrk, SS_PR_PROC);

    SS_install("pg-draw-palette",
               "Draw the current palette at the specified place on the given device",
               SS_nargs,
               SX_draw_palette, SS_PR_PROC);

    SS_install("pg-draw-polyline-2d",
               "Draw a set of connected 2d line segments",
               SS_nargs,
               SX_draw_polyline, SS_PR_PROC);

    SS_install("pg-draw-radius",
               "Draw a radial line segment on the given device",
               SS_nargs,
               SX_draw_rad, SS_PR_PROC);

    SS_install("pg-draw-text-abs",
               "Draw text at the given point on the given device",
               SS_nargs,
               SX_draw_text, SS_PR_PROC);

    SS_install("pg-fill-polygon",
               "Draw and fill a polygon on the specified device",
               SS_nargs,
               SX_fply, SS_PR_PROC);

    SS_install("pg-finish-plot",
               "Finish the plot frame on the given device",
               SS_nargs,
               SX_finish_plot, SS_PR_PROC);

    SS_install("pg-finish-state",
               "Get the finish flag of the given device",
               SS_nargs,
               SX_gfin, SS_PR_PROC);

    SS_install("pg-frame",
               "Return a list of numbers defining the extent of the frame in NDC",
               SS_nargs,
               SX_get_frame, SS_PR_PROC);

    SS_install("pg-line-color",
               "Get the line color on the given device",
               SS_nargs,
               SX_glnc, SS_PR_PROC);

    SS_install("pg-logical-op",
               "Get the logical operation on the given device",
               SS_nargs,
               SX_glop, SS_PR_PROC);

    SS_install("pg-line-style",
               "Get the line style on the given device",
               SS_nargs,
               SX_glns, SS_PR_PROC);

    SS_install("pg-line-width",
               "Get the line width on the given device",
               SS_nargs,
               SX_glnw, SS_PR_PROC);

    SS_install("list->pg-palette",
               "Convert a list of values into a palette and register it in the device",
               SS_nargs,
               SX_list_pal, SS_PR_PROC);

    SS_install("pg-make-device-current",
               "Make the given device the current device for drawing",
               SS_nargs,
               SX_mdvc, SS_PR_PROC);

    SS_install("pg-marker-orientation",
               "Get the orientation angle for marker characters",
               SS_nargs,
               SX_mrk_ornt, SS_PR_PROC);

    SS_install("pg-marker-scale",
               "Get the scale for marker characters",
               SS_nargs,
               SX_mrk_scal, SS_PR_PROC);

    SS_install("pg-maximum-intensity",
               "Return the maximum fractional intensities for colors (0.0 to 1.0)",
               SS_nargs,
               SX_gmxi, SS_PR_PROC);

    SS_install("pg-normalized->pixel",
               "Return a pair containing the PC version of the given NDC point",
               SS_nargs,
               SX_stop, SS_PR_PROC);

    SS_install("pg-normalized->world",
               "Return a pair containing the WC version of the given NDC point",
               SS_nargs,
               SX_stow, SS_PR_PROC);

    SS_install("pg-palette->list",
               "Return the named palette as a list of values ( ... r g b ... )",
               SS_nargs,
               SX_pal_list, SS_PR_PROC);

    SS_install("pg-palettes",
               "Return a list of palettes available for the given device",
               SS_nargs,
               SX_pals, SS_PR_PROC);

    SS_install("pg-pixel->normalized",
               "Return a pair containing the NDC version of the given PC point",
               SS_nargs,
               SX_ptos, SS_PR_PROC);

    SS_install("pg-query-device",
               "Return a list of device width, height, and number of colors",
               SS_nargs,
               SX_qdev, SS_PR_PROC);

    SS_install("pg-query-window",
               "Return a list of window width and height",
               SS_nargs,
               SX_qwin, SS_PR_PROC);

    SS_install("pg-release-current-device",
               "Release the given device as the current device",
               SS_nargs,
               SX_rdvc, SS_PR_PROC);

    SS_install("pg-set-autodomain!",
               "Set the autodomain flag of the given device",
               SS_nargs,
               SX_sadm, SS_PR_PROC);

    SS_install("pg-set-autoplot!",
               "Set the autoplot flag of the given device",
               SS_nargs,
               SX_sapl, SS_PR_PROC);

    SS_install("pg-set-autorange!",
               "Set the autorange flag of the given device",
               SS_nargs,
               SX_sarn, SS_PR_PROC);

    SS_install("pg-set-border-width!",
               "Set the device border width in pixels",
               SS_nargs,
               SX_sbwd, SS_PR_PROC);

    SS_install("pg-set-char-path!",
               "Set the character path direction for the given device",
               SS_nargs,
               SX_scpw, SS_PR_PROC);

    SS_install("pg-set-char-up!",
               "Set the character up direction for the given device",
               SS_nargs,
               SX_scuw, SS_PR_PROC);

    SS_install("pg-set-clear-mode!",
               "Set the clear mode",
               SS_nargs,
               SX_set_clr_mode, SS_PR_PROC);

    SS_install("pg-set-clipping!",
               "Set the clipping state of the given device",
               SS_nargs,
               SX_sclp, SS_PR_PROC);

    SS_install("pg-set-data-id-flag!",
               "Set the data-id flag of the given device",
               SS_nargs,
               SX_sdti, SS_PR_PROC);

    SS_install("pg-set-fill-color!",
               "Set the fill color on the given device",
               SS_nargs,
               SX_sfic, SS_PR_PROC);

    SS_install("pg-set-finish-state!",
               "Set the finish flag of the given device",
               SS_nargs,
               SX_sfin, SS_PR_PROC);

    SS_install("pg-set-grid-flag!",
               "Set the grid flag on the given device",
               SS_nargs,
               SX_sgrd, SS_PR_PROC);

    SS_install("pg-set-line-color!",
               "Set the line color on the given device",
               SS_nargs,
               SX_slnc, SS_PR_PROC);

    SS_install("pg-set-logical-op!",
               "Set the logical operation on the given device",
               SS_nargs,
               SX_slop, SS_PR_PROC);

    SS_install("pg-set-line-style!",
               "Set the line style on the given device",
               SS_nargs,
               SX_slns, SS_PR_PROC);

    SS_install("pg-set-line-width!",
               "Set the line width on the given device",
               SS_nargs,
               SX_slnw, SS_PR_PROC);

    SS_install("pg-set-marker-orientation!",
               "Set the orientation angle for marker characters",
               SS_nargs,
               SX_set_mrk_ornt, SS_PR_PROC);

    SS_install("pg-set-marker-scale!",
               "Set the scale for marker characters",
               SS_nargs,
               SX_set_mrk_scal, SS_PR_PROC);

    SS_install("pg-set-maximum-intensity!",
               "Set the maximum fractional intensity for colors (0.0 to 1.0) and optionally for RGB too",
               SS_nargs,
               SX_smxi, SS_PR_PROC);

    SS_install("pg-set-palette!",
               "Set the palette for the given device",
               SS_nargs,
               SX_spal, SS_PR_PROC);

    SS_install("pg-set-resolution-scale-factor!",
               "Set the resolution scale factor for the given device",
               SS_nargs,
               SX_sres_sf, SS_PR_PROC);

    SS_install("pg-set-scatter-flag!",
               "Set the scatter plot flag on the given device",
               SS_nargs,
               SX_ssct, SS_PR_PROC);

    SS_install("pg-set-text-color!",
               "Set the text color on the given device",
               SS_nargs,
               SX_stxc, SS_PR_PROC);

    SS_install("pg-set-text-font!",
               "Set the text type face, type style, and point size on the given device",
               SS_nargs,
               SX_stxf, SS_PR_PROC);

    SS_install("pg-set-white-background!",
               "Set the white background flag on the given device",
               SS_nargs,
               SX_swbk, SS_PR_PROC);

    SS_install("pg-set-frame!",
               "Set the frame for the given device",
               SS_nargs,
               SX_set_frame, SS_PR_PROC);

    SS_install("pg-set-vector-attributes!",
               "Set the drawing properties for vectors",
               SS_nargs,
               SX_set_vect_attr, SS_PR_PROC);

    SS_install("pg-set-viewport!",
               "Set the viewport for the given device",
               SS_nargs,
               SX_set_viewport, SS_PR_PROC);

    SS_install("pg-set-world-coordinate-system!",
               "Set the world coordinate system for the given device",
               SS_nargs,
               SX_set_window, SS_PR_PROC);

    SS_install("pg-show-markers",
               "Show the marker characters",
               SS_zargs,
               SX_show_mrk, SS_PR_PROC);

    SS_install("pg-make-palette",
               "Graphically make a palette and make it the current palette",
               SS_nargs,
               SX_mk_pal, SS_PR_PROC);

    SS_install("pg-show-palettes",
               "Show the available palettes and make selected palette the current one",
               SS_nargs,
               SX_show_pal, SS_PR_PROC);

    SS_install("pg-read-palette",
               "Read a palette from the given file",
               SS_nargs,
               SX_rd_pal, SS_PR_PROC);

    SS_install("pg-write-palette",
               "Write a palette to the given file",
               SS_nargs,
               SX_wr_pal, SS_PR_PROC);

    SS_install("pg-text-color",
               "Return the current text color on the given device",
               SS_nargs,
               SX_gtxc, SS_PR_PROC);

    SS_install("pg-text-extent",
               "Return a pair of numbers specifying the WC extent of the given string",
               SS_nargs,
               SX_gtew, SS_PR_PROC);

    SS_install("pg-text-font",
               "Return a list of current font specifications on the given device",
               SS_nargs,
               SX_gtxf, SS_PR_PROC);

    SS_install("pg-update-view-surface",
               "Update the view surface for the given device",
               SS_nargs,
               SX_update_vs, SS_PR_PROC);

    SS_install("pg-viewport",
               "Return a list of numbers defining the extent of the viewport in NDC",
               SS_nargs,
               SX_get_viewport, SS_PR_PROC);

    SS_install("pg-world-coordinate-system",
               "Return a list of numbers defining the WC system",
               SS_nargs,
               SX_get_window, SS_PR_PROC);

    SS_install("pg-world->normalized",
               "Return a pair containing the NDC version of the given WC point",
               SS_nargs,
               SX_wtos, SS_PR_PROC);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_ARGS_ARR_2 - map a list pairwise to numbers */

static void _SX_args_arr_2(argl, pn, px, py)
   object *argl;
   int *pn;
   REAL **px, **py;
   {int i, n;
    REAL *x, *y;
    object *obj;

    n = _SS_length(argl) >> 1;

    x = FMAKE_N(REAL, n, "_SX_ARGS_ARR_2:x");
    y = FMAKE_N(REAL, n, "_SX_ARGS_ARR_2:y");
    for (i = 0; !SS_nullobjp(argl); i++)
        {if (SS_consp(argl))
            SX_GET_FLOAT_FROM_LIST(x[i], argl,
                                   "CAN'T GET X VALUE - _SX_ARGS_ARR_2");
         if (SS_consp(argl))
            SX_GET_FLOAT_FROM_LIST(y[i], argl,
                                   "CAN'T GET Y VALUE - _SX_ARGS_ARR_2");};
    *px = x;
    *py = y;
    *pn = n;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_ARGS_ARR_3 - map a list triple-wise to numbers */

static void _SX_args_arr_3(argl, pn, px, py, pz)
   object *argl;
   int *pn;
   REAL **px, **py, **pz;
   {int i, n;
    REAL *x, *y, *z;
    object *obj;

    n = _SS_length(argl)/3;

    x = FMAKE_N(REAL, n, "_SX_ARGS_ARR_3:x");
    y = FMAKE_N(REAL, n, "_SX_ARGS_ARR_3:y");
    z = FMAKE_N(REAL, n, "_SX_ARGS_ARR_3:z");
    for (i = 0; !SS_nullobjp(argl); i++)
        {if (SS_consp(argl))
            SX_GET_FLOAT_FROM_LIST(x[i], argl,
                                   "CAN'T GET X VALUE - _SX_ARGS_ARR_3");
         if (SS_consp(argl))
            SX_GET_FLOAT_FROM_LIST(y[i], argl,
                                   "CAN'T GET Y VALUE - _SX_ARGS_ARR_3");
         if (SS_consp(argl))
            SX_GET_FLOAT_FROM_LIST(z[i], argl,
                                   "CAN'T GET Z VALUE - _SX_ARGS_ARR_3");};

    *px = x;
    *py = y;
    *pz = z;
    *pn = n;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_AXIS - draw a complete axis set */

static object *SX_axis(argl)
   object *argl;
   {PG_device *dev;
    int type;

    dev = NULL;
    type = CARTESIAN;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &type,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_AXIS", SS_null);

    PG_axis(dev, type);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_CLEAR_WINDOW - clear the screen */

static object *SX_clear_window(argl)
   object *argl;
   {PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_CLEAR_WINDOW", SS_null);

    PG_clear_window(dev);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_CLEAR_VIEWPORT - clear the viewport */

static object *SX_clear_viewport(argl)
   object *argl;
   {PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_CLEAR_VIEWPORT", SS_null);

    PG_clear_viewport(dev);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SET_CLR_MODE - set the clear mode */

static object *SX_set_clr_mode(argl)
   object *argl;
   {int mode;
    PG_device *dev;

    dev  = NULL;
    mode = CLEAR_SCREEN;
    SS_args(argl,
            G_DEVICE, &dev,
	    SC_INTEGER_I, &mode,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SET_CLR_MODE", SS_null);

    PG_set_clear_mode(mode);

    return(SS_mk_integer(mode));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_CLR_MODE - return the clear mode */

static object *SX_clr_mode(argl)
   object *argl;
   {int mode;
    PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SET_CLR_MODE", SS_null);

    PG_get_clear_mode(mode);

    return(SS_mk_integer(mode));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_CLEAR_REGION - clear the region */

static object *SX_clear_region(argl)
   object *argl;
   {PG_device *dev;
    REAL xmn, xmx, ymn, ymx;
    int pad;

    dev = NULL;
    xmn = 0.0;
    xmx = 1.0;
    ymn = 0.0;
    ymx = 1.0;
    pad = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_REAL_I, &xmn,
            SC_REAL_I, &xmx,
            SC_REAL_I, &ymn,
            SC_REAL_I, &ymx,
            SC_INTEGER_I, &pad,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_CLEAR_REGION", SS_null);

    PG_clear_region_NDC(dev, xmn, xmx, ymn, ymx, pad);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_CENTER_LABEL - print a label centered in a line */

static object *SX_center_label(argl)
   object *argl;
   {PG_device *dev;
    REAL sy;
    char *label;

    dev   = NULL;
    sy    = 0.0;
    label = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_REAL_I, &sy,
            SC_STRING_I, &label,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_CENTER_LABEL", SS_null);

    PG_center_label(dev, sy, label);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DEF_MRK - define a new marker character
 *            - a marker is defined by a set of line segments (4 numbers)
 *            - in the box -1 <= x <= 1 , -1 <= y <= 1
 *            - form (define-marker <x1a> <y1a> <x2a> <y2a> ...)
 */

static object *SX_def_mrk(argl)
   object *argl;
   {int i, ns, indx;
    REAL *x1, *y1, *x2, *y2;
    object *obj;

    ns = _SS_length(argl) >> 2;

    x1 = FMAKE_N(REAL, ns, "SX_DEF_MRK:x1");
    y1 = FMAKE_N(REAL, ns, "SX_DEF_MRK:y1");
    x2 = FMAKE_N(REAL, ns, "SX_DEF_MRK:x2");
    y2 = FMAKE_N(REAL, ns, "SX_DEF_MRK:y2");

    for (i = 0; i < ns; i++)
        {SX_GET_FLOAT_FROM_LIST(x1[i], argl,
	                        "CAN'T GET FIRST X VALUE - SX_DEF_MRK");
         SX_GET_FLOAT_FROM_LIST(y1[i], argl,
	                        "CAN'T GET FIRST Y VALUE - SX_DEF_MRK");
         SX_GET_FLOAT_FROM_LIST(x2[i], argl,
	                        "CAN'T GET SECOND X VALUE - SX_DEF_MRK");
         SX_GET_FLOAT_FROM_LIST(y2[i], argl,
	                        "CAN'T GET SECOND Y VALUE - SX_DEF_MRK");};

    indx = PG_def_marker(ns, x1, y1, x2, y2);

    SFREE(x1);
    SFREE(y1);
    SFREE(x2);
    SFREE(y2);

    return(SS_mk_integer(indx));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DRW_MRK - draw markers at the specified points
 *            - form (pg-draw-markers dev marker xlst ylst)
 */

static object *SX_drw_mrk(argl)
   object *argl;
   {int i, ns, mrk;
    REAL *x, *y;
    object *xa, *ya, *obj;
    PG_device *dev;

    dev = NULL;
    mrk = 0;
    xa  = SS_null;
    ya  = SS_null;
    SS_args(argl,
            G_DEVICE, &dev,
	    SC_INTEGER_I, &mrk,
	    SS_OBJECT_I, &xa,
	    SS_OBJECT_I, &ya,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DRW_MRK", SS_null);

    ns = _SS_length(xa);

    x = FMAKE_N(REAL, ns, "SX_DRW_MRK:x");
    y = FMAKE_N(REAL, ns, "SX_DRW_MRK:y");

    for (i = 0; i < ns; i++)
        {SX_GET_FLOAT_FROM_LIST(x[i], xa,
	                        "CAN'T GET X VALUE - SX_DRW_MRK");
         SX_GET_FLOAT_FROM_LIST(y[i], ya,
	                        "CAN'T GET Y VALUE - SX_DRW_MRK");};

    PG_draw_markers(dev, ns, x, y, mrk);

    SFREE(x);
    SFREE(y);

    return(SS_mk_integer(ns));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_MRK_ORNT - get the marker orientation */

static object *SX_mrk_ornt(argl)
   object *argl;
   {PG_device *dev;
    REAL theta;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_MRK_ORNT", SS_null);

    PG_get_marker_orientation(dev, theta);

    return(SS_mk_float(theta));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_MRK_SCAL - get the marker scale */

static object *SX_mrk_scal(argl)
   object *argl;
   {PG_device *dev;
    REAL scale;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_MRK_SCAL", SS_null);

    PG_get_marker_scale(dev, scale);

    return(SS_mk_float(scale));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SET_MRK_ORNT - set the marker orientation */

static object *SX_set_mrk_ornt(argl)
   object *argl;
   {PG_device *dev;
    REAL theta;

    dev   = NULL;
    theta = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_REAL_I, &theta,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SET_MRK_ORNT", SS_null);

    PG_set_marker_orientation(dev, theta);

    return(SS_mk_float(theta));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SET_MRK_SCAL - set the marker scale */

static object *SX_set_mrk_scal(argl)
   object *argl;
   {PG_device *dev;
    REAL scale;

    dev   = NULL;
    scale = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_REAL_I, &scale,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SET_MRK_SCAL", SS_null);

    PG_set_marker_scale(dev, scale);

    return(SS_mk_float(scale));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SHOW_MRK - show the current set of markers */

static object *SX_show_mrk()
   {PG_device *dev;
    int i;
    REAL b, dx, dy, x[1], y[1];

    dev = PG_make_device("WINDOW", "COLOR", "Markers");

    PG_white_background(dev, TRUE);
    PG_open_device(dev, 0.3, 0.3, 0.1, 0.2);

    PG_get_text_ext(dev, "0", &dx, &dy);
    dy *= 0.5;

    dev->marker_scale *= 3;
    b = 1.0/((double) (_PG_marker_index + 1));
    for (i = 0; i < _PG_marker_index; i++)
        {x[0] = 0.3;
         y[0] = b*(i + 1);
         PG_write_WC(dev, x[0], y[0], "%d", i+1);
         x[0] = 0.7;
         y[0] += dy;
         PG_draw_markers(dev, 1, x, y, i);};

    PRINT(stdout, "Press enter to continue\n");
    SC_pause();

    PG_close_device(dev);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DRAW_ARC - draw a portion of an arc */

static object *SX_draw_arc(argl)
   object *argl;
   {PG_device *dev;
    double a1, a2, r, x, y;
    int unit;

    dev = NULL;
    x = y = r = 0.0;
    a1 = a2 = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &r,
            SC_DOUBLE_I, &a1,
            SC_DOUBLE_I, &a2,
            SC_DOUBLE_I, &x,
            SC_DOUBLE_I, &y,
            SC_INTEGER_I, &unit,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DRAW_LINE", SS_null);

    if (unit == 1)
       unit = DEGREE;
    else
       unit = RADIAN;

    PG_set_line_color(dev, dev->line_color);
    PG_draw_arc(dev, r, a1, a2, x, y, unit);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DRAW_AXIS - draw a axis */

static object *SX_draw_axis(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1, x2, y2, t1, t2, v1, v2, sc;
    int tt, lt, td;
    char *fmt;

    dev = NULL;
    x1 = y1 = t1 = v1 = 0.0;
    x2 = y2 = t2 = v2 = 1.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &y1,
            SC_DOUBLE_I, &x2,
            SC_DOUBLE_I, &y2,
            SC_DOUBLE_I, &t1,
            SC_DOUBLE_I, &t2,
            SC_DOUBLE_I, &v1,
            SC_DOUBLE_I, &v2,
            SC_DOUBLE_I, &sc,
            SC_STRING_I, &fmt,
            SC_INTEGER_I, &tt,
            SC_INTEGER_I, &lt,
            SC_INTEGER_I, &td,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DRAW_AXIS", SS_null);

    switch (td)
       {case MAJOR             :
             PG_draw_axis(dev, x1, y1, x2, y2,
                          t1, t2, v1, v2,
                          sc, fmt, tt, lt, FALSE,
                          MAJOR, 0);
             break;

        case MINOR             :
             PG_draw_axis(dev, x1, y1, x2, y2,
                          t1, t2, v1, v2,
                          sc, fmt, tt, lt, FALSE,
                          MINOR, 0);
             break;

        case LABEL             :
             PG_draw_axis(dev, x1, y1, x2, y2,
                          t1, t2, v1, v2,
                          sc, fmt, tt, lt, FALSE,
                          LABEL, 0);
             break;

        case MAJOR_MINOR       :
             PG_draw_axis(dev, x1, y1, x2, y2,
                          t1, t2, v1, v2,
                          sc, fmt, tt, lt, FALSE,
                          MAJOR, MINOR, 0);
             break;

        case MAJOR_LABEL       :
             PG_draw_axis(dev, x1, y1, x2, y2,
                          t1, t2, v1, v2,
                          sc, fmt, tt, lt, FALSE,
                          MAJOR, LABEL, 0);
             break;

        case MINOR_LABEL       :
             PG_draw_axis(dev, x1, y1, x2, y2,
                          t1, t2, v1, v2,
                          sc, fmt, tt, lt, FALSE,
                          MINOR, LABEL, 0);
             break;

        case MAJOR_MINOR_LABEL :
             PG_draw_axis(dev, x1, y1, x2, y2,
                          t1, t2, v1, v2,
                          sc, fmt, tt, lt, FALSE,
                          MAJOR, MINOR, LABEL, 0);
             break;};

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DRAW_BOX - draw a box */

static object *SX_draw_box(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1, x2, y2;

    dev = NULL;
    x1 = y1 = 0.0;
    x2 = y2 = 1.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &x2,
            SC_DOUBLE_I, &y1,
            SC_DOUBLE_I, &y2,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DRAW_BOX", SS_null);

    PG_set_line_color(dev, dev->line_color);
    PG_draw_box(dev, x1, x2, y1, y2);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DDP2 - draw a set of disjoint 2d line segments */

static object *SX_ddp2(argl)
   object *argl;
   {PG_device *dev;
    REAL *x, *y;
    int n, flag, coord;

    dev   = NULL;
    flag  = TRUE;
    coord = TRUE;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &flag,
            SC_INTEGER_I, &coord,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DDP2", SS_car(argl));

    argl = SS_cddr(argl);
    _SX_args_arr_2(argl, &n, &x, &y);

    PG_set_line_color(dev, dev->line_color);
    PG_draw_disjoint_polyline_2(dev, x, y, (long) n/2, flag, coord);

    SFREE(x);
    SFREE(y);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DDP3 - draw a set of disjoint 3d line segments */

static object *SX_ddp3(argl)
   object *argl;
   {PG_device *dev;
    REAL *x, *y, *z;
    double ar, ad;
    int n, flag, norm;
    double theta, phi, chi;

    dev   = NULL;
    theta = phi = 0.0;
    flag  = TRUE;
    norm  = TRUE;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &theta,
            SC_DOUBLE_I, &phi,
            SC_DOUBLE_I, &chi,
            SC_INTEGER_I, &flag,
            SC_INTEGER_I, &norm,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DDP3", SS_car(argl));

    argl = SS_cdddr(SS_cdr(argl));
    _SX_args_arr_3(argl, &n, &x, &y, &z);

    ar = dev->autorange;
    ad = dev->autodomain;

    dev->autorange  = FALSE;
    dev->autodomain = FALSE;

    PG_set_line_color(dev, dev->line_color);
    PG_draw_disjoint_polyline_3(dev, x, y, z, theta, phi, chi,
                                (long) n/2, flag, norm);

    dev->autorange  = ar;
    dev->autodomain = ad;

    SFREE(x);
    SFREE(y);
    SFREE(z);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DRAW_LINE - draw a line segment */

static object *SX_draw_line(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1, x2, y2;

    dev = NULL;
    x1 = y1 = 0.0;
    x2 = y2 = 1.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &y1,
            SC_DOUBLE_I, &x2,
            SC_DOUBLE_I, &y2,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DRAW_LINE", SS_null);

    PG_set_line_color(dev, dev->line_color);
    PG_draw_line(dev, x1, y1, x2, y2);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DRAW_PALETTE - draw a palette */

static object *SX_draw_palette(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1, x2, y2, zn, zx, w;

    dev = NULL;
    x1 = y1 = 0.0;
    x2 = y2 = 1.0;
    zn = zx = w = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &y1,
            SC_DOUBLE_I, &x2,
            SC_DOUBLE_I, &y2,
            SC_DOUBLE_I, &zn,
            SC_DOUBLE_I, &zx,
            SC_DOUBLE_I, &w,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DRAW_PALETTE", SS_null);

    PG_draw_palette(dev, x1, y1, x2, y2, zn, zx, w);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DRAW_POLYLINE - draw a set of disjoint 2d line segments */

static object *SX_draw_polyline(argl)
   object *argl;
   {PG_device *dev;
    REAL *x, *y;
    int n, clip;

    dev  = NULL;
    clip = TRUE;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &clip,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DRAW_POLYLINE", SS_car(argl));

    argl = SS_cdr(argl);
    _SX_args_arr_2(argl, &n, &x, &y);

    PG_set_line_color(dev, dev->line_color);
    PG_draw_polyline(dev, x, y, n, clip);

    SFREE(x);
    SFREE(y);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DRAW_RAD - draw a radial line segment */

static object *SX_draw_rad(argl)
   object *argl;
   {PG_device *dev;
    double a, rn, rx, x, y;
    int unit;

    dev = NULL;
    x = y = 0.0;
    rn = rx = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &rn,
            SC_DOUBLE_I, &rx,
            SC_DOUBLE_I, &a,
            SC_DOUBLE_I, &x,
            SC_DOUBLE_I, &y,
            SC_INTEGER_I, &unit,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DRAW_LINE", SS_null);

    PG_set_line_color(dev, dev->line_color);
    PG_draw_rad(dev, rn, rx, a, x, y, unit);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_DRAW_TEXT - draw a text string */

static object *SX_draw_text(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1;
    char *txt;

    dev = NULL;
    x1 = y1 = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &y1,
            SC_STRING_I, &txt,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_DRAW_TEXT", SS_null);

    PG_set_text_color(dev, dev->text_color);
    PG_write_WC(dev, x1, y1, "%s", txt);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_FPLY - fill a polygon */

static object *SX_fply(argl)
   object *argl;
   {PG_device *dev;
    REAL *x, *y;
    int n, c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_FPLY", SS_car(argl));

    argl = SS_cddr(argl);
    _SX_args_arr_2(argl, &n, &x, &y);

    PG_set_color_fill(dev, c, TRUE);
    PG_shade_poly(dev, x, y, n);

    SFREE(x);
    SFREE(y);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_FINISH_PLOT - finish the frame */

static object *SX_finish_plot(argl)
   object *argl;
   {PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_FINISH_PLOT", SS_null);

    PG_finish_plot(dev);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GCPW - return the character path */

static object *SX_gcpw(argl)
   object *argl;
   {PG_device *dev;
    REAL x1, y1;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GCPW", SS_null);

    PG_get_char_path(dev, &x1, &y1);

    return(SS_make_list(SC_REAL_I, &x1,
                        SC_REAL_I, &y1,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GCSS - return the character size in NDC */

static object *SX_gcss(argl)
   object *argl;
   {PG_device *dev;
    REAL x1, y1;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GCSS", SS_null);

    PG_get_char_size_NDC(dev, &x1, &y1);

    return(SS_make_list(SC_REAL_I, &x1,
                        SC_REAL_I, &y1,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GCUW - return the character up direction */

static object *SX_gcuw(argl)
   object *argl;
   {PG_device *dev;
    REAL x1, y1;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GCUW", SS_null);

    PG_get_char_up(dev, &x1, &y1);

    return(SS_make_list(SC_REAL_I, &x1,
                        SC_REAL_I, &y1,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GCLP - return the clipping state */

static object *SX_gclp(argl)
   object *argl;
   {PG_device *dev;
    int clp;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GCLP", SS_null);

    PG_get_clipping(dev, &clp);

    return(clp ? SS_t : SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GFIN - report the state of the finished flag */

static object *SX_gfin(argl)
   object *argl;
   {int fl;
    PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GFIN", SS_null);

    PG_get_finish_state(dev, fl);

    return(SS_mk_integer(fl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GLNC - get the line color */

static object *SX_glnc(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GLNC", SS_null);

    PG_get_line_color(dev, &c);

    return(SS_mk_integer(c));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GLOP - get the logical operation */

static object *SX_glop(argl)
   object *argl;
   {PG_device *dev;
    int lop;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GLOP", SS_null);

    PG_get_logical_op(dev, &lop);

    return(SS_mk_integer(lop));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GLNS - get the line style */

static object *SX_glns(argl)
   object *argl;
   {PG_device *dev;
    int s;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GLNS", SS_null);

    PG_get_line_style(dev, &s);

    return(SS_mk_integer(s));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GLNW - get the line width */

static object *SX_glnw(argl)
   object *argl;
   {PG_device *dev;
    REAL w;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GLNW", SS_null);

    PG_get_line_width(dev, &w);

    return(SS_mk_float(w));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GMXI - get the maximum intensities */

static object *SX_gmxi(argl)
   object *argl;
   {PG_device *dev;
    double i, r, g, b;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GMXI", SS_null);

    PG_get_max_intensity(dev, i);
    PG_get_max_red_intensity(dev, r);
    PG_get_max_green_intensity(dev, g);
    PG_get_max_blue_intensity(dev, b);

    return(SS_make_list(SC_DOUBLE_I, &i,
                        SC_DOUBLE_I, &r,
                        SC_DOUBLE_I, &g,
                        SC_DOUBLE_I, &b,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GTEW - return the text extent */

static object *SX_gtew(argl)
   object *argl;
   {PG_device *dev;
    REAL x1, y1;
    char *s;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_STRING_I, &s,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GTEW", SS_null);

    PG_get_text_ext(dev, s, &x1, &y1);

    return(SS_make_list(SC_REAL_I, &x1,
                        SC_REAL_I, &y1,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GTXC - get the text color */

static object *SX_gtxc(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GTXC", SS_null);

    PG_get_text_color(dev, &c);

    return(SS_mk_integer(c));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GTXF - get the text font */

static object *SX_gtxf(argl)
   object *argl;
   {PG_device *dev;
    char *face, *style;
    int size;
    object *ret;

    dev  = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GTXF", SS_null);

    PG_get_font(dev, &face, &style, &size);

    ret = SS_make_list(SC_STRING_I, face,
                       SC_STRING_I, style,
                       SC_INTEGER_I, &size,
                       0);

    SFREE(face);
    SFREE(style);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GET_VIEWPORT - get the viewport */

static object *SX_get_viewport(argl)
   object *argl;
   {PG_device *dev;
    REAL x1, y1, x2, y2;
    object *ret;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GET_VIEWPORT", SS_null);

    PG_get_viewport(dev, &x1, &x2, &y1, &y2);

    ret = SS_make_list(SC_REAL_I, &x1,
                       SC_REAL_I, &x2,
                       SC_REAL_I, &y1,
                       SC_REAL_I, &y2,
                       0);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GET_FRAME - get the frame */

static object *SX_get_frame(argl)
   object *argl;
   {PG_device *dev;
    REAL x1, y1, x2, y2;
    object *ret;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GET_FRAME", SS_null);

    PG_get_frame(dev, &x1, &x2, &y1, &y2);

    ret = SS_make_list(SC_REAL_I, &x1,
                       SC_REAL_I, &x2,
                       SC_REAL_I, &y1,
                       SC_REAL_I, &y2,
                       0);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_GET_WINDOW - get the world coordinate system */

static object *SX_get_window(argl)
   object *argl;
   {PG_device *dev;
    REAL x1, y1, x2, y2;
    object *ret;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_GET_WINDOW", SS_null);

    PG_get_viewport_WC(dev, &x1, &x2, &y1, &y2);

    ret = SS_make_list(SC_REAL_I, &x1,
                       SC_REAL_I, &x2,
                       SC_REAL_I, &y1,
                       SC_REAL_I, &y2,
                       0);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_MDVC - make the device current */

static object *SX_mdvc(argl)
   object *argl;
   {PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_MDVC", SS_null);

    PG_make_device_current(dev);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_PTOS - convert from pixel to screen coordinates */

static object *SX_ptos(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1;
    long xi, yi;

    dev = NULL;
    x1 = y1 = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_LONG_I, &xi,
            SC_LONG_I, &yi,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_PTOS", SS_null);

    PtoS(dev, xi, yi, x1, y1);

    return(SS_make_list(SC_DOUBLE_I, &x1,
                        SC_DOUBLE_I, &y1,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_QDEV - query the device */

static object *SX_qdev(argl)
   object *argl;
   {PG_device *dev;
    int dx, dy, nc;
    object *ret;

    dev  = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_QDEV", SS_null);

    PG_query_device(dev, &dx, &dy, &nc);

    ret = SS_make_list(SC_INTEGER_I, &dx,
                       SC_INTEGER_I, &dy,
                       SC_INTEGER_I, &nc,
                       0);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_QWIN - query the window */

static object *SX_qwin(argl)
   object *argl;
   {PG_device *dev;
    int dx, dy, nc;
    object *ret;

    dev  = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_QDEV", SS_null);

    PG_query_device(dev, &dx, &dy, &nc);

    ret = SS_make_list(SC_INTEGER_I, &dx,
                       SC_INTEGER_I, &dy,
                       0);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_RDVC - release the device */

static object *SX_rdvc(argl)
   object *argl;
   {PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_RDVC", SS_null);

    PG_release_current_device(dev);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SADM - set the autodomain flag */

static object *SX_sadm(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SADM", SS_null);

    PG_turn_autodomain(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SAPL - set the autoplot flag */

static object *SX_sapl(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SAPL", SS_null);

    PG_turn_autoplot(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SARN - set the autorange flag */

static object *SX_sarn(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SARN", SS_null);

    PG_turn_autorange(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SBWD - set the border width in pixels */

static object *SX_sbwd(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SBWD", SS_null);

    PG_set_border_width(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SCPW - set the character path */

static object *SX_scpw(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1;

    dev = NULL;
    x1 = y1 = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &y1,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SCPW", SS_null);

    PG_set_char_path(dev, x1, y1);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SCUW - set the character up direction */

static object *SX_scuw(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1;

    dev = NULL;
    x1 = y1 = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &y1,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SCUW", SS_null);

    PG_set_char_path(dev, x1, y1);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SCLP - set the clipping state */

static object *SX_sclp(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SCLP", SS_null);

    PG_set_clipping(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SDTI - set the data id flag */

static object *SX_sdti(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SDTI", SS_null);

    PG_turn_data_id(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SFIC - set the fill color */

static object *SX_sfic(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SFIC", SS_null);

    PG_set_fill_color(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SFIN - set the finish state */

static object *SX_sfin(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SFIN", SS_null);

    PG_set_finish_state(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SGRD - set the grid flag */

static object *SX_sgrd(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SGRD", SS_null);

    PG_turn_grid(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SLNC - set the line color */

static object *SX_slnc(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SLNC", SS_null);

    PG_set_line_color(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SLOP - set the logical operation */

static object *SX_slop(argl)
   object *argl;
   {PG_device *dev;
    int lop;

    dev = NULL;
    lop = GS_COPY;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &lop,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SLOP", SS_null);

    PG_set_logical_op(dev, lop);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SLNS - set the line style */

static object *SX_slns(argl)
   object *argl;
   {PG_device *dev;
    int s;

    dev = NULL;
    s   = SOLID;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &s,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SLNS", SS_null);

    PG_set_line_style(dev, s);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SLNW - set the line width */

static object *SX_slnw(argl)
   object *argl;
   {PG_device *dev;
    double w;

    dev = NULL;
    w   = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &w,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SLNW", SS_null);

    PG_set_line_width(dev, w);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SMXI - set the maximum intensities */

static object *SX_smxi(argl)
   object *argl;
   {PG_device *dev;
    double i, r, g, b;

    i = r = g = b = 1.0;
    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &i,
            SC_DOUBLE_I, &r,
            SC_DOUBLE_I, &g,
            SC_DOUBLE_I, &b,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SMXI", SS_car(argl));

    i = max(0.0, i);
    i = min(1.0, i);

    r = max(0.0, r);
    r = min(1.0, r);

    g = max(0.0, g);
    g = min(1.0, g);

    b = max(0.0, b);
    b = min(1.0, b);

    PG_set_max_intensity(dev, i);
    PG_set_max_red_intensity(dev, r);
    PG_set_max_green_intensity(dev, g);
    PG_set_max_blue_intensity(dev, b);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SPAL - set the current palette */

static object *SX_spal(argl)
   object *argl;
   {PG_device *dev;
    char *txt;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_STRING_I, &txt,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SPAL", SS_null);

    if (!SX_OK_TO_DRAW(dev))
       return(SS_f);

    return((PG_set_palette(dev, txt) == NULL) ?
           SS_f : SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SRES_SF - set the resolution scale factor */

static object *SX_sres_sf(argl)
   object *argl;
   {PG_device *dev;
    int sf;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &sf,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SRES_SF", SS_null);

    PG_set_res_scale_factor(dev, sf);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SSCT - set the scatter flag */

static object *SX_ssct(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SSCT", SS_null);

    PG_turn_scatter(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SWBK - set the white background flag */

static object *SX_swbk(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SWBK", SS_null);

    PG_white_background(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_STXC - set the text color */

static object *SX_stxc(argl)
   object *argl;
   {PG_device *dev;
    int c;

    dev = NULL;
    c   = 1;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &c,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_STXC", SS_null);

    PG_set_text_color(dev, c);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_STXF - set the text font */

static object *SX_stxf(argl)
   object *argl;
   {PG_device *dev;
    char *face, *style;
    int size;

    dev  = NULL;
    size = 12;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_STRING_I, &face,
            SC_STRING_I, &style,
            SC_INTEGER_I, &size,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_STXF", SS_null);

    if (face == NULL)
       face = SC_strsavef("helvetica", "char*:SX_STXF:face");

    if (style == NULL)
       style = SC_strsavef("medium", "char*:SX_STXF:style");

    PG_set_font(dev, face, style, size);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SET_VIEWPORT - set the viewport */

static object *SX_set_viewport(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1, x2, y2;

    dev = NULL;
    x1 = y1 = 0.0;
    x2 = y2 = 1.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &x2,
            SC_DOUBLE_I, &y1,
            SC_DOUBLE_I, &y2,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SET_VIEWPORT", SS_null);

    PG_set_viewport(dev, x1, x2, y1, y2);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SET_FRAME - set the frame */

static object *SX_set_frame(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1, x2, y2;

    dev = NULL;
    x1 = y1 = 0.0;
    x2 = y2 = 1.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &x2,
            SC_DOUBLE_I, &y1,
            SC_DOUBLE_I, &y2,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SET_FRAME", SS_null);

    PG_set_frame(dev, x1, x2, y1, y2);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_SET_WINDOW - set the world coordinate system */

static object *SX_set_window(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1, x2, y2;

    dev = NULL;
    x1 = y1 = 0.0;
    x2 = y2 = 1.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &x2,
            SC_DOUBLE_I, &y1,
            SC_DOUBLE_I, &y2,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_SET_WINDOW", SS_null);

    PG_set_window(dev, x1, x2, y1, y2);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_UPDATE_VS - update the view surface with a flush */

static object *SX_update_vs(argl)
   object *argl;
   {PG_device *dev;

    dev = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_UPDATE_VS", SS_null);

    PG_update_vs(dev);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_STOP - convert from screen to pixel coordinates */

static object *SX_stop(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1;
    long xi, yi;

    dev = NULL;
    x1 = y1 = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &y1,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_STOP", SS_null);

    StoP(dev, x1, y1, xi, yi);

    return(SS_make_list(SC_INTEGER_I, &xi,
                        SC_INTEGER_I, &yi,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_STOW - convert from screen to world coordinates */

static object *SX_stow(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1;

    dev = NULL;
    x1 = y1 = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &y1,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_STOW", SS_null);

    StoW(dev, x1, y1);

    return(SS_make_list(SC_DOUBLE_I, &x1,
                        SC_DOUBLE_I, &y1,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* SX_WTOS - convert from world to screen coordinates */

static object *SX_wtos(argl)
   object *argl;
   {PG_device *dev;
    double x1, y1;

    dev = NULL;
    x1 = y1 = 0.0;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_DOUBLE_I, &x1,
            SC_DOUBLE_I, &y1,
            0);

    if (dev == NULL)
       SS_error("BAD DEVICE - SX_WTOS", SS_null);

    WtoS(dev, x1, y1);

    return(SS_make_list(SC_DOUBLE_I, &x1,
                        SC_DOUBLE_I, &y1,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/*

SX_plcn(FIXNUM *devid, REAL *px, REAL *py, REAL *pa
                                 REAL *pl, FIXNUM *pkx, FIXNUM *plx,
                                 FIXNUM *pnl, FIXNUM *pli),
SX_plln(FIXNUM *devid, REAL *px, REAL *py
                                 FIXNUM *pn, FIXNUM *pmod, FIXNUM *paxt,
                                 FIXNUM *pcol, REAL *pwid, FIXNUM *psty,
                                 FIXNUM *psca, FIXNUM *psta, FIXNUM *pl),
SX_plim(FIXNUM *devid, FIXNUM *pnc
                                 char *name, FIXNUM *pnct, char *type,
                                 REAL *px, REAL *py, REAL *pz,
                                 FIXNUM *pk, FIXNUM *pl,
                                 REAL *pxn, REAL *pxx, REAL *pyn,
                                 REAL *pyx, REAL *pzn, REAL *pzx),
SX_plsf(FIXNUM *devid, REAL *px, REAL *py, REAL *pz
                                 FIXNUM *pn, REAL *pxn, REAL *pxx, REAL *pyn,
                                 REAL *pyx, REAL *pzn, REAL *pzx,
                                 FIXNUM *pkx, FIXNUM *plx, REAL *pth, REAL *pph,
                                 FIXNUM *ptyp, FIXNUM *pcol, REAL *pwid,
                                 FIXNUM *psty, FIXNUM *pnc, char *label),
SX_plvc(FIXNUM *devid, REAL *px, REAL *py
                                 REAL *pu, REAL *pv, FIXNUM *pn),
SX_rvpa(FIXNUM *devid, FIXNUM *vwatid)
SX_svpa(FIXNUM *devid, FIXNUM *pn)
*/

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PAL_LIST - return a list of RGB values for the named palette
 *             - RGB components will be normalized to unity
 */

static object *SX_pal_list(argl)
   object *argl;
   {int i, nc;
    object *lst;
    char *name;
    PG_palette *pal;
    PG_device *dev;
    RGB_color_map *cm, cl;
    double imp, red, green, blue;

    name = NULL;
    dev  = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_STRING_I, &name,
            0);

    pal = dev->palettes;
    while (TRUE)
       {if (strcmp(name, pal->name) == 0)
           break;

        pal = pal->next;
        if (pal == dev->palettes)
           return(NULL);};

    if (pal == NULL)
       SS_error("NO SUCH PALETTE FOR DEVICE - SX_PAL_LIST", argl);

    cm = pal->true_colormap;
    nc = pal->n_pal_colors;

    imp = 1.0/((double) MAXPIX);
    lst = SS_null;
    for (i = 0; i < nc; i++)
        {cl = *cm++;
         red   = imp*cl.red;
         green = imp*cl.green;
         blue  = imp*cl.blue;
         lst   = _SS_append(SS_make_list(SC_DOUBLE_I, &red,
                                         SC_DOUBLE_I, &green,
                                         SC_DOUBLE_I, &blue,
                                         0),
                            lst);};

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_LIST_PAL - install a palette in the given device
 *             - built from the list of RGB values specified
 *             - RGB components must be normalized to unity
 */

static object *SX_list_pal(argl)
   object *argl;
   {int i, n_pal_colors, n_dev_colors;
    char *name;
    PG_palette *pal;
    PG_device *dev;
    RGB_color_map *cm, cl;
    object *colors;

    name = NULL;
    dev  = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_STRING_I, &name,
            0);

    colors       = SS_caddr(argl);
    n_pal_colors = _SS_length(colors)/3;
    n_dev_colors = dev->absolute_n_color;

    pal = FMAKE(PG_palette, "SX_LIST_PAL:pal");
    cm  = FMAKE_N(RGB_color_map, n_pal_colors,
                  "SX_LIST_PAL:cm");
    if (cm == NULL)
       return(SS_f);

/* the 256 max is to make it doable at all on a system with
 * 24 bits of color - nobody wants to wait a year for X to
 * allocate 16,000,000 colors
 */
    pal->true_colormap       = cm;
    pal->pseudo_colormap     = NULL;
    pal->max_red_intensity   = dev->max_red_intensity;
    pal->max_green_intensity = dev->max_green_intensity;
    pal->max_blue_intensity  = dev->max_blue_intensity;
    pal->n_pal_colors        = min(n_pal_colors, 256);
    pal->n_dev_colors        = min(n_dev_colors, 256);
    pal->name                = SC_strsavef(name, "char*:SX_LIST_PAL:name");

/* compute the color maps */
    for (i = 0; i < n_pal_colors; i++)
        {cl.red   = MAXPIX*SS_FLOAT_VALUE(SS_car(colors));
         colors   = SS_cdr(colors);
         cl.green = MAXPIX*SS_FLOAT_VALUE(SS_car(colors));
         colors   = SS_cdr(colors);
         cl.blue  = MAXPIX*SS_FLOAT_VALUE(SS_car(colors));
         colors   = SS_cdr(colors);

         *cm++    = cl;};

    _PG_register_palette(dev, pal, TRUE);

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_CURRENT_PAL - return the current palette for the given device */

static object *SX_current_pal(argl)
   object *argl;
   {object *obj;
    PG_device *dev;

    dev  = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    obj = SS_mk_string(dev->current_palette->name);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PALS - return a list of palettes available for the given device */

static object *SX_pals(argl)
   object *argl;
   {object *lst;
    PG_palette *pal;
    PG_device *dev;
    int first;

    dev  = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    lst   = SS_null;
    first = 0;
    for (pal = dev->palettes; TRUE; pal = pal->next)
        {first += (pal == dev->palettes);
         if (first > 1)
            break;
         lst = SS_mk_cons(SS_mk_string(pal->name), lst);};

    return(lst);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SHOW_PAL - show the available palettes
 *             - a background color flag is optional
 */

static object *SX_show_pal(argl)
   object *argl;
   {int wbck;
    PG_device *dev;

    dev  = NULL;
    wbck = TRUE;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_INTEGER_I, &wbck,
            0);

    PG_show_palettes(dev, "WINDOW", wbck);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MK_PAL - graphically construct a palette
 *           - make it the current palette
 */

static object *SX_mk_pal(argl)
   object *argl;
   {int ndims, wbck;
    int dims[5];
    char *name;
    PG_device *dev;

    dev  = NULL;
    name  = NULL;
    ndims = 1;
    dims[0] = 16;
    dims[1] = 1;
    dims[2] = 1;
    wbck = TRUE;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_STRING_I, &name,
            SC_INTEGER_I, &wbck,
            SC_INTEGER_I, &ndims,
            SC_INTEGER_I, &dims[0],
            SC_INTEGER_I, &dims[1],
            SC_INTEGER_I, &dims[2],
            0);

    dev->current_palette = PG_make_ndim_palette(dev, name, ndims, 
                                           dims, wbck);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_RD_PAL - read a palette from the specified file
 *           - make it the current palette
 */

static object *SX_rd_pal(argl)
   object *argl;
   {char *name;
    PG_device *dev;

    dev  = NULL;
    name = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_STRING_I, &name,
            0);

    dev->current_palette = PG_rd_palette(dev, name);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_WR_PAL - write a palette to the specified file */

static object *SX_wr_pal(argl)
   object *argl;
   {char *fname, *pname;
    PG_device *dev;
    PG_palette *pal;

    dev   = NULL;
    pname = NULL;
    fname = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            SC_STRING_I, &pname,
            SC_STRING_I, &fname,
            0);

    pal = PG_get_palette(dev, pname);

    PG_wr_palette(dev, pal, fname);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_VECT_ATTR - set vector drawing attributes */

static object *SX_set_vect_attr(argl)
   object *argl;
   {object *lst;
    PG_device *dev;
    int attr;
    double val;

    dev  = NULL;
    SS_args(argl,
            G_DEVICE, &dev,
            0);

    
    for (lst = SS_cdr(argl); lst != SS_null; lst = SS_cddr(lst))
        {SS_args(lst,
                 SC_INTEGER_I, &attr,
                 SC_DOUBLE_I, &val,
                 0);

         switch (attr)
            {case VEC_SCALE     :
             case VEC_ANGLE     :
             case VEC_HEADSIZE  :
             case VEC_FIXSIZE   :
             case VEC_MAXSIZE   :
             case VEC_LINETHICK :
                  PG_set_vec_attr(dev, attr, val, 0);
                  break;

             case VEC_LINESTYLE :
             case VEC_COLOR     :
             case VEC_FIXHEAD   :
                  PG_set_vec_attr(dev, attr, (int) val, 0);
                  break;

             default            : break;};};

    return(SS_cdr(argl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
