/*
 * GSRWI.C - read and write a portable user interface desription
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

char *ps;

static int
 SC_DECLARE(PG_write_interface_object,
         (FILE *fp, PG_interface_object *iob,
          char *indent, int nopf)),
 SC_DECLARE(_PG_get_xy, (REAL *px, REAL *py, double xmn, double ymn,
		      double dx, double dy));

static PG_interface_object
 *SC_DECLARE(PG_read_interface_object,
          (PG_device *dev, FILE *fp, char *bf,
           PG_interface_object *parent));

extern void
 SC_DECLARE(_PG_find_callback, (PG_interface_object *iob,
			     hashel **php, int *pityp));

/*--------------------------------------------------------------------------*/

/*                       INTERFACE READ/WRITE ROUTINES                      */

/*--------------------------------------------------------------------------*/

/* _PG_PARENT_LIMITS_NDC - find the limits of the parent entity */

void _PG_parent_limits_NDC(dev, parent, pxmn, pymn, pdx, pdy)
   PG_device *dev;
   PG_interface_object *parent;
   REAL *pxmn, *pymn, *pdx, *pdy;
   {double xmn, ymn, xmx, ymx, dx, dy, x0, y0;
    int i, n, *px, *py, ix, iy, ix0, iy0;
    PG_curve *crv;

    xmn  = 0.0;
    ymn  = 0.0;
    dx   = 1.0;
    dy   = 1.0;
    if (parent != NULL)
       {crv = parent->curve;
        n   = crv->n;

        px  = crv->x;
        py  = crv->y;
        ix0 = crv->x_origin;
        iy0 = crv->y_origin;
	xmn = 1.0;
	ymn = 1.0;
	xmx = 0.0;
	ymx = 0.0;
        for (i = 0; i < n; i++)
            {ix = px[i] + ix0;
             iy = py[i] + iy0;

	     PtoS(dev, ix, iy, x0, y0);

             xmn = min(xmn, x0);
             xmx = max(xmx, x0);
             ymn = min(ymn, y0);
             ymx = max(ymx, y0);};

        dx = xmx - xmn;
        dy = ymx - ymn;};

    *pxmn = xmn;
    *pymn = ymn;
    *pdx  = dx;
    *pdy  = dy;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RECTANGLE_P - return TRUE iff the curve is a rectangle */

static int _PG_rectangle_p(crv)
   PG_curve *crv;
   {int *x, *y, npt;

    npt = crv->n;
    x = crv->x;
    y = crv->y;

    if (npt == 5)
       return((x[0] == x[4]) &&
	      (x[0] == x[3]) &&
	      (x[1] == x[2]) &&
	      (y[0] == y[4]) &&
	      (y[0] == y[1]) &&
	      (y[2] == y[3]));
            
    else
       return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_WRITE_INTERFACE_OBJECT - write out an interface object to the
 *                           - given file
 *                           - return TRUE iff successful
 */

static int PG_write_interface_object(fp, iob, indent, nopf)
   FILE *fp;
   PG_interface_object *iob;
   char *indent;
   int nopf;
   {int i, nc, niobs, npt, ix, iy, xo, yo, fc, bc;
    int *x, *y, flags[N_IOB_FLAGS];
    REAL xs, ys, xso, yso;
    REAL dx, dy;
    char *type, s[MAXLINE];
    PG_device *dev;
    PG_interface_object **iobs;
    PG_curve *crv;

    PRINT(fp, "%s%s", indent, iob->type);
    nc = strlen(indent);
    if (nc > 0)
       indent[nc-1] = ' ';

    dev  = iob->device;
    type = iob->type;

    if (strcmp(type, PG_TEXT_OBJECT_S) == 0)
       {PG_text_box *b;

        SFREE(iob->name);
	b = (PG_text_box *) iob->obj;
	
	sprintf(s, "\"%s\"", b->text_buffer[0]);
	iob->name = SC_strsavef(s,
                    "char*:PG_WRITE_INTERFACE_OBJECT:name");};

    if (iob->name != NULL)
       PRINT(fp, " NAME(%s)", iob->name);

    fc = iob->foreground;
    if (fc == dev->BLACK)
       fc = 0;
    else if (fc == dev->WHITE)
       fc = 1;

    bc = iob->background;
    if (bc == dev->BLACK)
       bc = 0;
    else if (bc == dev->WHITE)
       bc = 1;

    PRINT(fp, " CLR(%d,%d)", fc, bc);

    flags[0] = iob->is_visible;
    flags[1] = iob->is_selectable;
    flags[2] = iob->is_active;
    if ((flags[0] != TRUE) || (flags[1] != TRUE) || (flags[2] != FALSE))
       {for (i = 0, nc = 0; i < N_IOB_FLAGS; i++)
            nc += flags[i];

        PRINT(fp, " FLG(");
        if (nc == 0)
	   PRINT(fp, " ");

        for (i = 0; i < N_IOB_FLAGS; i++)
            {if (flags[i])
                {switch (i)
                    {case 0 :
		          PRINT(fp, "IsVis");
			  break;
                     case 1 :
		          PRINT(fp, "IsSel");
			  break;
                     case 2 :
		          PRINT(fp, "IsAct");
			  break;};
		  if (nc-- > 1)
		     PRINT(fp, ",");};};
	PRINT(fp, ")");};

    if (iob->draw_name != NULL)
       PRINT(fp, " DRW(%s)", iob->draw_name);

    if (iob->action_name != NULL)
       {char *name;

	name = iob->action_name;
        if (strcmp(name, "toggle") == 0)
	   {char *type;
            PG_interface_object *trans;

            type = iob->type;
	    if (strcmp(type, PG_BUTTON_OBJECT_S) == 0)
	       {trans = (PG_interface_object *) iob->obj;
		PRINT(fp, " ACT(toggle,%s)", trans->name);}

	    else if (strcmp(type, PG_VARIABLE_OBJECT_S) == 0)
	       PRINT(fp, " ACT(toggle)");}

        else
	   PRINT(fp, " ACT(%s)", iob->action_name);};

    if (iob->select_name != NULL)
       PRINT(fp, " SEL(%s)", iob->select_name);

    crv = iob->curve;
    npt = crv->n;
    xo  = crv->x_origin;
    yo  = crv->y_origin;

    _PG_parent_limits_NDC(dev, iob->parent, &xso, &yso, &dx, &dy);

    x = crv->x;
    y = crv->y;
    if (_PG_rectangle_p(crv))
       {PRINT(fp, " BND(RECT)");
	ix = x[0] + xo;
	iy = y[0] + yo;
	PtoS(dev, ix, iy, xs, ys);
	xs = (xs - xso)/dx;
	ys = (ys - yso)/dy;
	PRINT(fp, " (%5.3f,%5.3f)", xs, ys);
	ix = x[2] + xo;
	iy = y[2] + yo;
	PtoS(dev, ix, iy, xs, ys);
	xs = (xs - xso)/dx;
	ys = (ys - yso)/dy;
	PRINT(fp, " (%5.3f,%5.3f)", xs, ys);}

    else
       {PRINT(fp, "\n%sBND(%d)", indent, npt);
	for (i = 0; i < npt; i++)
	    {ix = x[i] + xo;
	     iy = y[i] + yo;
	     PtoS(dev, ix, iy, xs, ys);
	     xs = (xs - xso)/dx;
	     ys = (ys - yso)/dy;
	     PRINT(fp, " (%5.3f,%5.3f)", xs, ys);};};

    iobs  = iob->children;
    niobs = iob->n_children;

    if (niobs > 0)
       {PRINT(fp, "\n");

        npt = strlen(indent);
        memset(indent, ' ', npt+1);
/*	indent[npt]   = ' '; */
	indent[npt+2] = '{';
	indent[npt+3] = '\0';

	for (i = 0; i < niobs; i++)
            {if (!PG_write_interface_object(fp, iobs[i], indent, FALSE))
	        return(FALSE);
	     if (i < niobs - 1)
                PRINT(fp, "\n");
	     indent[npt+2]   = ' ';};

	PRINT(fp, "}");
        indent[npt+3] = ' ';
	indent[npt]   = '\0';};

    if (nopf)
       PRINT(fp, "\n");

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_WRITE_INTERFACE - write out the interface descriptions to the
 *                    - given file
 *                    - return TRUE iff successful
 */

int PG_write_interface(dev, name)
   PG_device *dev;
   char *name;
   {int i, niobs, ret;
    char indent[MAXLINE];
    PG_interface_object **iobs;
    FILE *fp;

    fp = io_open(name, "w");
    if (fp == NULL)
       return(FALSE);

    memset(indent, ' ', MAXLINE);

    iobs  = dev->interface_objects;
    niobs = dev->n_interface_objects;

/* print documentation */
    PRINT(fp, "#\n");
    PRINT(fp, "# PGS User Interface Object Description\n");
    PRINT(fp, "#\n");
    PRINT(fp, "# <type> [<oper>(<parameters>)]* BND(<param>)");
    PRINT(fp, " <boundary points>*\n");
    PRINT(fp, "#\n");

    PRINT(fp, "# Operators:\n");
    PRINT(fp, "# NAME - object name\n");
    PRINT(fp, "# FLG  - state flags\n");
    PRINT(fp, "# CLR  - fore and background colors\n");
    PRINT(fp, "# DRW  - names function that draws object when visible\n");
    PRINT(fp, "# ACT  - names function that does object action when active\n");
    PRINT(fp, "# SEL  - names function that identifies object as selected");
    PRINT(fp, " when selectable\n");
    PRINT(fp, "# BND  - boundary specifier\n");
    PRINT(fp, "#\n");

    PRINT(fp, "# FLG parameters\n");
    PRINT(fp, "#   IsVis - object is visible\n");
    PRINT(fp, "#   IsAct - object is active\n");
    PRINT(fp, "#   IsSel - object is selectable\n");
    PRINT(fp, "#\n");

    PRINT(fp, "# BND parameters\n");
    PRINT(fp, "#   <n>  - n NDC points to follow (x1,y1) ...");
    PRINT(fp, " (xn, yn)\n");
    PRINT(fp, "#   RECT - two NDC points to follow (x1,y1) (x2,y2)\n");
    PRINT(fp, "#\n");
    PRINT(fp, "# BND takes n points defining a boundary relative to the");
    PRINT(fp, " parent object limits\n");

    PRINT(fp, "# The descriptions of children of an object are indented");
    PRINT(fp, " relative to their parent\n");
    PRINT(fp, "# with 3 spaces but are otherwise just like any");
    PRINT(fp, " interface object\n");
    PRINT(fp, "#\n\n");

    ret = TRUE;
    indent[0] = '\0';
    for (i = 0; i < niobs; i++)
        {ret = PG_write_interface_object(fp, iobs[i], indent, TRUE);
	 if (!ret)
	    break;
         PRINT(fp, "\n");};

    io_close(fp);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_GET_XY - pull an (x,y) value pair of the current token stream
 *            - return TRUE iff successful
 */

static int _PG_get_xy(px, py, xmn, ymn, dx, dy)
   REAL *px, *py;
   double xmn, ymn, dx, dy;
   {char *tok;

    tok = SC_firsttok(ps, "(,) ");
    if (tok == NULL)
       return(FALSE);

    if (*tok == '*')
       return(FALSE);

    *px = xmn + dx*SC_stof(tok);

    tok = SC_firsttok(ps, "(,) ");
    if (tok == NULL)
       return(FALSE);

    *py = ymn + dy*SC_stof(tok);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FIX_COLOR - parse the color spec and return the values */

static void _PG_fix_color(dev, pfc, pbc)
   PG_device *dev;
   int *pfc, *pbc;
   {int fc, bc;

    fc = SC_stoi(SC_firsttok(ps, ","));
    if (fc == 0)
       fc = dev->BLACK;
    else if (fc == 1)
       fc = dev->WHITE;

    bc = SC_stoi(SC_firsttok(ps, ")"));
    if (bc == 0)
       bc = dev->BLACK;
    else if (bc == 1)
       bc = dev->WHITE;

    *pfc = fc;
    *pbc = bc;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FIX_FLAGS - parse the flags */

static void _PG_fix_flags(dev, flags)
   PG_device *dev;
   int *flags;
   {char t[MAXLINE], *tok;

/* reset the flags if you come here */
    flags[0] = FALSE;
    flags[1] = FALSE;
    flags[2] = FALSE;

    strcpy(t, SC_firsttok(ps, ")"));
    while (TRUE)
       {tok = SC_firsttok(t, " ,");
        if (tok == NULL)
           break;

        if (strcmp(tok, "IsVis") == 0)
           flags[0] = TRUE;

        else if (strcmp(tok, "IsSel") == 0)
           flags[1] = TRUE;

        else if (strcmp(tok, "IsAct") == 0)
           flags[2] = TRUE;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_FIND_OBJECT - search the interface object tree in the device for
 *                - the named object
 */

PG_interface_object *PG_find_object(dev, s, parent)
   PG_device *dev;
   char *s;
   PG_interface_object *parent;
   {PG_interface_object *iob, **iobs;
    int i, niobs;

    if (parent != NULL)
       {if (parent->name != NULL)
	        {if (strcmp(parent->name, s) == 0)
                    return(parent);};

        iobs  = parent->children;
	niobs = parent->n_children;}

    else
       {iobs  = dev->interface_objects;
	niobs = dev->n_interface_objects;};

    iob = NULL;
    for (i = 0; i < niobs; i++)
        {iob = PG_find_object(dev, s, iobs[i]);
	 if (iob != NULL)
	    break;};

    return(iob);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FIX_ACTION - fix the action and object of the action */

static void _PG_fix_action(dev, pfct, pobj)
   PG_device *dev;
   char **pfct;
   byte **pobj;
   {char *fact, *item;
    byte *obj;

    obj  = NULL;
    fact = SC_firsttok(ps, ")");
    item = strchr(fact, ',');
    if (item != NULL)
       {*item++ = '\0';
        obj = (byte *) PG_find_object(dev, item, (PG_interface_object *) NULL);};

    *pfct = fact;
    *pobj = obj;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FIX_BOUNDARY - process the boundary specfication */

static void _PG_fix_boundary(dev, pnp, pxd, pyd, xmn, ymn, dx, dy)
   PG_device *dev;
   int *pnp;
   REAL **pxd, **pyd;
   double xmn, ymn, dx, dy;
   {REAL *xd, *yd;
    REAL xs, ys;
    int i, npt, ok;
    char *tok;

    tok = SC_firsttok(ps, ")");
    if (strcmp(tok, "RECT") == 0)
       {npt = 5;
	xd  = FMAKE_N(REAL, npt, "_PG_FIX_BOUNDARY:xd");
	yd  = FMAKE_N(REAL, npt, "_PG_FIX_BOUNDARY:yd");
        ok  = FALSE;
	if (_PG_get_xy(&xs, &ys, xmn, ymn, dx, dy))
           {xd[0] = xs;
            yd[0] = ys;
            yd[1] = ys;
            xd[3] = xs;
            xd[4] = xs;
            yd[4] = ys;
	    if (_PG_get_xy(&xs, &ys, xmn, ymn, dx, dy))
               {xd[1] = xs;
                xd[2] = xs;
                yd[2] = ys;
                yd[3] = ys;
                ok    = TRUE;};};
        if (!ok)
           {SFREE(xd);
            SFREE(yd);
            npt = -1;};}

    else
       {npt = SC_stoi(tok);
	xd  = FMAKE_N(REAL, npt, "_PG_FIX_BOUNDARY:xd");
	yd  = FMAKE_N(REAL, npt, "_PG_FIX_BOUNDARY:yd");
	for (i = 0; i < npt; i++)
	    {if (!_PG_get_xy(&xd[i], &yd[i], xmn, ymn, dx, dy))
                {SFREE(xd);
                 SFREE(yd);
                 npt = -1;
                 break;};};};

    *pnp = npt;
    *pxd = xd;
    *pyd = yd;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_READ_INTERFACE_OBJECT - read in an interface object from the
 *                          - given file
 *                          - return a PG_interface_object iff successful
 */

static PG_interface_object *PG_read_interface_object(dev, fp, bf, parent)
   PG_device *dev;
   FILE *fp;
   char *bf;
   PG_interface_object *parent;
   {int npt, fc, bc;
    int flags[N_IOB_FLAGS];
    byte *obj;
    REAL xmn, dx, ymn, dy;
    REAL *xd, *yd;
    char *tok, *type, *name;
    char *fsel, *fdrw, *fact;
    PG_interface_object *iob;
    PG_curve *crv;

    _PG_parent_limits_NDC(dev, parent, &xmn, &ymn, &dx, &dy);

    type = SC_firsttok(ps, " ");
    if (type == NULL)
       return(NULL);

    flags[0] = TRUE;
    flags[1] = TRUE;
    flags[2] = FALSE;

    fc   = 1;
    bc   = 0;
    npt  = -1;
    obj  = NULL;
    fsel = NULL;
    fdrw = NULL;
    fact = NULL;
    name = NULL;
    xd   = NULL;
    yd   = NULL;
    while (TRUE)
       {tok = SC_firsttok(ps, " ()\n");
        if (tok == NULL)
           {if (GETLN(bf, MAXLINE, fp) == NULL)
               return(NULL);
	    ps = bf;
            tok = SC_firsttok(ps, " ()\n");
            if (tok == NULL)
               return(NULL);};

        if (strcmp(tok, "FLG") == 0)
           _PG_fix_flags(dev, flags);

        else if (strcmp(tok, "CLR") == 0)
           _PG_fix_color(dev, &fc, &bc);

        else if (strcmp(tok, "NAME") == 0)
	   {if (ps[0] == '"')
	       name = SC_firsttok(ps, "\"");
	    else
	       name = SC_firsttok(ps, ")");}

        else if (strcmp(tok, "SEL") == 0)
           fsel = SC_firsttok(ps, ")");

        else if (strcmp(tok, "DRW") == 0)
           fdrw = SC_firsttok(ps, ")");

        else if (strcmp(tok, "ACT") == 0)
           _PG_fix_action(dev, &fact, &obj);

        else if (strcmp(tok, "BND") == 0)
           {_PG_fix_boundary(dev, &npt, &xd, &yd, xmn, ymn, dx, dy);
            break;}

        else
           break;};

    if (npt == -1)
       {PRINT(stdout, "Illegal boundary for %s (%s)\n", name, type);
        return(NULL);};

    crv = PG_make_curve(dev, NORMC, TRUE, npt, xmn, ymn, xd, yd);
    iob = PG_make_interface_object(dev, name, type, obj,
				   CENTER, 0.0, crv, flags, fc, bc,
                                   fsel, fdrw, fact, parent, NULL);

    return(iob);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_READ_INTERFACE - read in the interface descriptions from the
 *                   - given file
 *                   - return TRUE iff successful
 */

int PG_read_interface(dev, name)
   PG_device *dev;
   char *name;
   {int ns;
    PG_interface_object *iob, *piob, *parent;
    FILE *fp;
    char bf[MAXLINE];

    fp = io_open(name, "r");
    if (fp == NULL)
       return(FALSE);

    iob    = NULL;
    piob   = NULL;
    parent = NULL;
    while (TRUE)
       {if (GETLN(bf, MAXLINE, fp) == NULL)
           break;

        if (SC_blankp(bf, "#"))
           continue;

        for (ns = 0, ps = bf; *ps == ' '; ns++, ps++);
        
        switch (*ps)
           {case '{' :
                 parent = piob;
                 ps++;
                 break;
            case '}' :
	         for (parent = parent->parent;
		      *(++ps) == '}';
		      parent = parent->parent);
                 break;};

        if (*ps == '\n')
           continue;

        iob = PG_read_interface_object(dev, fp, bf, parent);
        if (parent == NULL)
           {SC_REMEMBER(PG_interface_object *, iob, dev->interface_objects,
                        dev->n_interface_objects, dev->max_interface_objects,
                        10);}
        else
           {PG_PUSH_CHILD_IOB(parent, iob);};

        piob = iob;

        ps = SC_firsttok(ps, " \t\n\r");
        if (ps != NULL)
	   {switch (*ps)
	       {case '{' :
		     parent = piob;
		     ps++;
		     break;
		case '}' :
	             for (parent = parent->parent;
			  *(++ps) == '}';
			  parent = parent->parent);
		     break;};};};

    io_close(fp);

    return(TRUE);}

/*--------------------------------------------------------------------------*/

/*                          DISPLAY TREE ROUTINES                           */

/*--------------------------------------------------------------------------*/

/* PG_DISPLAY_INTERFACE - display out the interface descriptions to the
 *                      - terminal
 */

DEBUG_FUNC int PG_display_interface(dev)
   PG_device *dev;
   {int i, niobs, ret;
    char indent[MAXLINE];
    PG_interface_object **iobs;

    memset(indent, ' ', MAXLINE);

    iobs  = dev->interface_objects;
    niobs = dev->n_interface_objects;

    ret = TRUE;
    indent[0] = '\0';
    for (i = 0; i < niobs; i++)
        {ret = PG_write_interface_object(stdout, iobs[i], indent, TRUE);
	 if (!ret)
	    break;
         PRINT(stdout, "\n");};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
