/*
 * PDRTST.C - test for the PDB I/O
 *
 * Source Version: 9.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pdb.h"

#define DATFILE "nat"

#define N_DOUBLE 3
#define N_INT    5
#define N_CHAR  10
#define N_FLOAT  4

#define FLOAT_EQUAL(d1, d2)  (PD_float_equal(d1, d2, float_tolerance) == 0)
#define DOUBLE_EQUAL(d1, d2) (PD_float_equal(d1, d2, double_tolerance) == 0)

struct s_l_frame
   {float x_min;
    float x_max;
    float y_min;
    float y_max;};

typedef struct s_l_frame l_frame;

struct s_plot
   {float x_axis[N_CHAR];
    float y_axis[N_CHAR];
    int npts;
    char *label;
    l_frame view;};

typedef struct s_plot plot;

struct s_lev2
   {char **s;
    int type;};

typedef struct s_lev2 lev2;

struct s_lev1
   {int *a;
    double *b;
    lev2 *c;};

typedef struct s_lev1 lev1;

struct s_st3
   {char a;
    short b;
    char c[2];
    int d;
    char e[3];
    float f;
    char g[4];
    double h;
    char i[5];
    char *j;
    char k[6];};

typedef struct s_st3 st3;

struct s_st4
   {short a;
    char  b;};

typedef struct s_st4 st4;

static st4
  *vr4_w,
  *vr4_r;

static st3
 vr1_w,
 vr1_r;

static lev1
 *tar_r,
 *tar_w;

static HASHTAB
 *tab4_r,
 *tab4_w;

static char
 *CHAR_S,
 *SHORT_S,
 *INT_S,
 *LONG_S,
 *FLOAT_S,
 *DOUBLE_S,
 *HASHEL_S,
 cs_w,
 cs_r,
 ca_w[N_CHAR],
 ca_r[N_CHAR],
 *cap_w[N_DOUBLE],
 *cap_r[N_DOUBLE];

static short
 ss_w,
 ss_r,
 sa_w[N_INT],
 sa_r[N_INT];

static int
 is_w,
 is_r,
 ia_w[N_INT],
 ia_r[N_INT],
 do_r,
 do_w,
 p_w[N_INT],
 p_r[N_INT],
 len;

static float
 fs_w,
 fs_r,
 fs_app_w,
 fs_app_r,
 fs_p1_r,
 fs_p2_r,
 fs_p3_r,
 fa1_w[N_FLOAT],
 fa1_r[N_FLOAT],
 fa2_w[N_FLOAT][N_DOUBLE],
 fa2_r[N_FLOAT][N_DOUBLE],
 fa2_app_w[N_FLOAT][N_DOUBLE],
 fa2_app_r[N_FLOAT][N_DOUBLE];

static double
 ds_w,
 ds_r,
 da_w[N_FLOAT],
 da_r[N_FLOAT];

static plot
 graph_w,
 graph_r;

static l_frame
 view_w,
 view_r;

/* variables for partial and member reads in test #2 */

static int
 *ap1,
 *ap2,
 aa[4];

static double
 *bp1,
 *bp2,
 ba[4];

static lev2
 *cp1,
 *cp2,
 ca[4];

static char
 **sp1,
 **sp2,
 **sp3,
 **sp4,
 *sa[5],
 *tp1,
 *tp2,
 *tp3,
 *tp4,
 *tp5,
 *tp6,
 *tp7,
 *tp8,
 ta[4];

int
 test_1(),
 compare_test_1_data(),
 test_2(),
 compare_test_2_data(),
 test_3(),
 compare_test_3_data(),
 test_4(),
 compare_test_4_data();

void
 run_test(),
 test_target(),
 dump_test_symbol_table(),

 prep_test_1_data(),
 print_test_1_data(),
 append_test_1_data(),
 read_test_1_data(),

 prep_test_2_data(),
 print_test_2_data(),
 append_test_2_data(),
 read_test_2_data(),

 prep_test_3_data(),
 print_test_3_data(),
 append_test_3_data(),
 read_test_3_data(),

 prep_test_4_data(),
 print_test_4_data(),
 append_test_4_data(),
 read_test_4_data();

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* MAIN - test the PDB Library system */

main()
   {SIGNAL(SIGINT, SIG_DFL);

    run_test(test_1, 1, DATFILE);
    run_test(test_2, 2, DATFILE);
    run_test(test_3, 3, DATFILE);
    run_test(test_4, 4, DATFILE);

    PRINT(STDOUT, "\n");

    exit(0);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* RUN_TEST - run a particular test through all targeting modes */

void run_test(test, n, host)
   PFInt test;
   int n;
   char *host;
   {char s[MAXLINE];
    double time;
    
    PRINT(STDOUT, "\nRunning test #%d\n", n);
    time = SC_wall_clock_time();

/* Cray target test */
    strcpy(s, "cray");
    if (!(*test)(host, s, n))
       PRINT(STDOUT, "Test #%d %s failed\n", n, s);

/* Sun4 target test */
    strcpy(s, "sun4");
    if (!(*test)(host, s, n))
       PRINT(STDOUT, "Test #%d %s failed\n", n, s);

/* Mips target test */
    strcpy(s, "mips");
    if (!(*test)(host, s, n))
       PRINT(STDOUT, "Test #%d %s failed\n", n, s);

/* Dos target test */
    strcpy(s, "dos");
    if (!(*test)(host, s, n))
       PRINT(STDOUT, "Test #%d %s failed\n", n, s);

/* Macintosh target test */
    strcpy(s, "mac");
    if (!(*test)(host, s, n))
       PRINT(STDOUT, "Test #%d %s failed\n", n, s);

/* Sun3 target test */
    strcpy(s, "sun3");
    if (!(*test)(host, s, n))
       PRINT(STDOUT, "Test #%d %s failed\n", n, s);

/* Vax target test */
    strcpy(s, "vax");
    if (!(*test)(host, s, n))
       PRINT(STDOUT, "Test #%d %s failed\n", n, s);

/* native test */
    if (!(*test)(host, NULL, n))
       PRINT(STDOUT, "Test #%d native failed\n", n);

    time = SC_wall_clock_time() - time;
    PRINT(STDOUT, "Completed in %.2g seconds\n", time);

    return;}

/*--------------------------------------------------------------------------*/

/*                            TEST #1 ROUTINES                              */

/*--------------------------------------------------------------------------*/

/* TEST_1 - test the fundamental PDBLib functions
 *        -
 *        - read scalars of all primitive types
 *        - read structures with no indirections
 *        - read structure members
 *        - read parts of arrays
 */

test_1(base, tgt, n)
   char *base, *tgt;
   int n;
   {PDBfile *strm;
    char datfile[MAXLINE], fname[MAXLINE];
    int err;
    FILE *fp;

/* build the file names */
    test_target(tgt, base, n, fname, datfile);

    fp = fopen(fname, "w");

    prep_test_1_data();

/* open the file */
    if ((strm = PD_open(datfile, "r")) == NULL)
       {fprintf(fp, "Test couldn't open file %s\r\n", datfile);
        return(-1);};
    fprintf(fp, "File %s opened\n", datfile);

/* dump the symbol table */
    dump_test_symbol_table(fp, strm->symtab, 1);

/* read the data from the file */
    read_test_1_data(strm);

/* compare the original data with that read in */
    err = compare_test_1_data(strm, fp);

/* close the file */
    if (!PD_close(strm))
       {fprintf(fp, "Test couldn't close file %s\r\n", datfile);
        exit(1);};
    fprintf(fp, "File %s closed\n", datfile);

/* print it out to STDOUT */
    print_test_1_data(fp);

    fclose(fp);
    if (err)
       remove(fname);

    return(err);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PREP_TEST_1_DATA - prepare the test data */

void prep_test_1_data()
   {int i, k;

/* set scalars */
    cs_r    = 0;
    ss_r    = 0;
    is_r    = 0;
    fs_r    = 0.0;
    fs_p1_r = 0.0;
    fs_p2_r = 0.0;
    fs_p3_r = 0.0;
    ds_r    = 0.0;

    cs_w     = 'Q';
    ss_w     = -514;
    is_w     =  10;
    fs_w     =  3.14159;
    fs_app_w = -3.14159;
    ds_w     =  exp(1.0);

/* set char array */
    for (i = 0; i < N_CHAR; i++)
        {ca_w[i] = '\0';
         ca_r[i] = '\0';};

/* set short array */
    for (i = 0; i < N_INT; i++)
        {sa_w[i] = 2 - i;
         sa_r[i] = 0;};

/* set int array */
    for (i = 0; i < N_INT; i++)
        {ia_w[i] = i - 2;
         ia_r[i] = 0;};

/* set float array */
    for (i = 0; i < N_FLOAT; i++)
        for (k = 0; k < N_DOUBLE; k++)
            {fa2_w[i][k]     = POW((double) (i+1), (double) (k+1));
             fa2_app_w[i][k] = POW((double) (k+1), (double) (i+1));
             fa2_r[i][k]     = 0.0;
             fa2_app_r[i][k] = 0.0;};

/* set double array */
    for (i = 0; i < N_FLOAT; i++)
        {da_w[i] = POW(ds_w, (double) (i+1));
         da_r[i] = 0.0;};

/* set strings */
    strcpy(ca_w, "Hi there!");
    len = strlen(ca_w) + 1;

    cap_w[0] = SC_strsavef("lev1",
               "char*:PREP_TEST_1_DATA:cap_w(0)");
    cap_w[1] = SC_strsavef("lev2",
               "char*:PREP_TEST_1_DATA:cap_w(1)");
    cap_w[2] = SC_strsavef("tar fu blat",
               "char*:PREP_TEST_1_DATA:cap_w(2)");
    cap_r[0] = NULL;
    cap_r[1] = NULL;
    cap_r[2] = NULL;

/* set structures */
    for (i = 0; i < N_CHAR; i++)
        {graph_w.x_axis[i] = ((float) i)/10.0;
         graph_w.y_axis[i] = 0.5 - graph_w.x_axis[i];
         graph_r.x_axis[i] = -1000.0;
         graph_r.y_axis[i] = -1000.0;};

    view_w.x_min = 0.1;
    view_w.x_max = 1.0;
    view_w.y_min = -0.5;
    view_w.y_max =  0.5;

    view_r.x_min = -1.e-10;
    view_r.x_max = -1.e-10;
    view_r.y_min = -1.e-10;
    view_r.y_max = -1.e-10;

    graph_w.npts  = N_CHAR;
    graph_w.label = SC_strsavef("test graph",
                    "char*:PREP_TEST_1_DATA:label");
    graph_w.view  = view_w;

    graph_r.npts  = 0;
    graph_r.label = NULL;

    graph_r.view.x_min = -1.e-10;
    graph_r.view.x_max = -1.e-10;
    graph_r.view.y_min = -1.e-10;
    graph_r.view.y_max = -1.e-10;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* READ_TEST_1_DATA - read the test data from the file */

void read_test_1_data(strm)
   PDBfile *strm;
   {long ind[6];

/* read the scalar data from the file */
    PD_read(strm, "cs", &cs_r);
    PD_read(strm, "ss", &ss_r);
    PD_read(strm, "is", &is_r);
    PD_read(strm, "fs", &fs_r);
    PD_read(strm, "ds", &ds_r);

/* read the primitive arrays from the file */
    PD_read(strm, "ca",  ca_r);
    PD_read(strm, "sa",  sa_r);
    PD_read(strm, "ia",  ia_r);
    PD_read(strm, "fa2", fa2_r);
    PD_read(strm, "da",  da_r);
    PD_read(strm, "cap", cap_r);

/* read the entire structures from the file */
    PD_read(strm, "view",  &view_r);
    PD_read(strm, "graph", &graph_r);

/* read the appended data from the file */
    PD_read(strm, "fs_app",  &fs_app_r);
    PD_read(strm, "fa2_app", fa2_app_r);

/* partial array read */
    ind[0] = 1;
    ind[1] = 2;
    ind[2] = 1;
    ind[3] = 0;
    ind[4] = 1;
    ind[5] = 1;
    PD_read_alt(strm, "fa2", fa1_r, ind);

/* struct member test */
    ind[0] = 8;
    ind[1] = 8;
    ind[2] = 1;
    PD_read_alt(strm, "graph.x_axis",     &fs_p1_r, ind);
    PD_read(    strm, "graph.view.x_max", &fs_p2_r);
    PD_read(    strm, "view.y_max",       &fs_p3_r);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PRINT_TEST_1_DATA - print it out to STDOUT */

void print_test_1_data(fp)
   FILE *fp;
   {int i, k;

/* print scalars */
    fprintf(fp, "short scalar:   ss = %d\n", ss_r);
    fprintf(fp, "integer scalar: is = %d\n", is_r);
    fprintf(fp, "float scalar:   fs = %14.6e\n", fs_r);
    fprintf(fp, "double scalar:  ds = %14.6e\n", ds_r);

    fprintf(fp, "\n");

/* print char array */
    fprintf(fp, "static char array:\n  ca = %s\n", ca_r);

/* print short array */
    fprintf(fp, "short array:\n");
    for (i = 0; i < N_INT; i++)
        fprintf(fp, "  sa[%d] = %d\n", i, sa_r[i]);

/* print int array */
    fprintf(fp, "integer array:\n");
    for (i = 0; i < N_INT; i++)
        fprintf(fp, "  ia[%d] = %d\n", i, ia_r[i]);

/* print float array */
    fprintf(fp, "float array:\n");
    for (i = 0; i < N_FLOAT; i++)
        for (k = 0; k < N_DOUBLE; k++)
            fprintf(fp, "  fa2[%d][%d] = %14.6e\n", i, k, fa2_r[i][k]);

/* print double array */
    fprintf(fp, "double array:\n");
    for (i = 0; i < N_FLOAT; i++)
        fprintf(fp, "  da[%d] = %14.6e\n", i, da_r[i]);

/* print character pointer array */
    fprintf(fp, "string array:\n");
    for (i = 0; i < N_DOUBLE; i++)
        fprintf(fp, "  cap[%d] = %s\n", i, cap_r[i]);

    fprintf(fp, "\n");

/* print appended scalars */
    fprintf(fp, "appended float scalar: fs_app = %14.6e\n", fs_app_r);

/* print float array */
    fprintf(fp, "appended float array:\n");
    for (i = 0; i < N_FLOAT; i++)
        for (k = 0; k < N_DOUBLE; k++)
            fprintf(fp, "  fa2_app[%d][%d] = %14.6e\n",
                    i, k, fa2_app_r[i][k]);

    fprintf(fp, "\n");

/* whole struct test */
    fprintf(fp, "struct view:\n");
    fprintf(fp, "  x-min = %14.6e\n", view_r.x_min);
    fprintf(fp, "  x-max = %14.6e\n", view_r.x_max);
    fprintf(fp, "  y-min = %14.6e\n", view_r.y_min);
    fprintf(fp, "  y-max = %14.6e\n", view_r.y_max);

    fprintf(fp, "\n");

    fprintf(fp, "struct graph:\n");
    fprintf(fp, "  #pts  = %d\n",     graph_r.npts);
    fprintf(fp, "  x-min = %14.6e\n", graph_r.view.x_min);
    fprintf(fp, "  x-max = %14.6e\n", graph_r.view.x_max);
    fprintf(fp, "  y-min = %14.6e\n", graph_r.view.y_min);
    fprintf(fp, "  y-max = %14.6e\n", graph_r.view.y_max);

    fprintf(fp, "      x values             y values\n");
    for (i = 0; i < N_CHAR; i++)
        fprintf(fp, "   %14.6e       %14.6e\n",
               graph_r.x_axis[i], graph_r.y_axis[i]);

    fprintf(fp, "\n");

/* partial read single elements */
    fprintf(fp, "\npartial read scalars:\n");
    fprintf(fp, "  graph.x_axis[8]  = %14.6e\n", fs_p1_r);
    fprintf(fp, "  graph.view.x_max = %14.6e\n", fs_p2_r);
    fprintf(fp, "  view.y_max       = %14.6e\n", fs_p3_r);

    fprintf(fp, "\n");

/* partial read arrays */
    fprintf(fp, "partial read float array:\n");
    for (i = 0; i < N_FLOAT; i++)
        fprintf(fp, "  fa2_p[%d] = %14.6e\n", i, fa1_r[i]);

    fprintf(fp, "\n");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* COMPARE_TEST_1_DATA - compare the test data */

compare_test_1_data(strm, fp)
   PDBfile *strm;
   FILE *fp;
   {int i, k, err, err_tot;
    int float_nm, double_nm;
    double float_tolerance, double_tolerance;

    float_nm  = min(strm->std->float_format[2],
                    strm->host_std->float_format[2]);
    double_nm = min(strm->std->double_format[2],
                    strm->host_std->double_format[2]);

/* pad the absolute tolerance */
    float_tolerance  = POW(2.0, -((double) float_nm));
    double_tolerance = POW(2.0, -((double) double_nm));
    err_tot = TRUE;

/* compare scalars */
    err  = TRUE;
    err &= (cs_r == cs_w);
    err &= (ss_r == ss_w);
    err &= (is_r == is_w);
    err &= FLOAT_EQUAL(fs_r, fs_w);
    err &= FLOAT_EQUAL(fs_app_r, fs_app_w);
    err &= DOUBLE_EQUAL(ds_r, ds_w);

    if (err)
       fprintf(fp, "Scalars compare\n");
    else
       fprintf(fp, "Scalars differ\n");
    err_tot &= err;

/* compare char array */
    err  = TRUE;
    for (i = 0; i < N_CHAR; i++)
        err &= (ca_r[i] == ca_w[i]);

/* compare short array */
    for (i = 0; i < N_INT; i++)
        err &= (sa_r[i] == sa_w[i]);

/* compare int array */
    for (i = 0; i < N_INT; i++)
        err &= (ia_r[i] == ia_w[i]);

/* compare float array */
    for (i = 0; i < N_FLOAT; i++)
        for (k = 0; k < N_DOUBLE; k++)
            {err &= FLOAT_EQUAL(fa2_r[i][k], fa2_w[i][k]);
             err &= FLOAT_EQUAL(fa2_app_r[i][k], fa2_app_w[i][k]);}

/* compare double array */
    for (i = 0; i < N_FLOAT; i++)
        err &= DOUBLE_EQUAL(da_r[i], da_w[i]);

    if (err)
       fprintf(fp, "Arrays compare\n");
    else
       fprintf(fp, "Arrays differ\n");
    err_tot &= err;

/* compare strings */
    err  = TRUE;

    for (i = 0; i < N_DOUBLE; i++)
        err &= (strcmp(cap_r[i], cap_w[i]) == 0);

    if (err)
       fprintf(fp, "Strings compare\n");
    else
       fprintf(fp, "Strings differ\n");
    err_tot &= err;

/* compare structures */
    err  = TRUE;
    for (i = 0; i < N_CHAR; i++)
        {err &= FLOAT_EQUAL(graph_r.x_axis[i], graph_w.x_axis[i]);
         err &= FLOAT_EQUAL(graph_r.y_axis[i], graph_w.y_axis[i]);};

    err &= FLOAT_EQUAL(view_r.x_min, view_w.x_min);
    err &= FLOAT_EQUAL(view_r.x_max, view_w.x_max);
    err &= FLOAT_EQUAL(view_r.y_min, view_w.y_min);
    err &= FLOAT_EQUAL(view_r.y_max, view_w.y_max);

    err &= (graph_r.npts == graph_w.npts);
    err &= (strcmp(graph_r.label, graph_w.label) == 0);

    err &= FLOAT_EQUAL(graph_r.view.x_min, graph_w.view.x_min);
    err &= FLOAT_EQUAL(graph_r.view.x_max, graph_w.view.x_max);
    err &= FLOAT_EQUAL(graph_r.view.y_min, graph_w.view.y_min);
    err &= FLOAT_EQUAL(graph_r.view.y_max, graph_w.view.y_max);

    if (err)
       fprintf(fp, "Structs compare\n");
    else
       fprintf(fp, "Structs differ\n");
    err_tot &= err;

/* compare partial read results */
    err  = TRUE;
    err &= FLOAT_EQUAL(fs_p1_r, graph_w.x_axis[8]);
    err &= FLOAT_EQUAL(fs_p2_r, graph_w.view.x_max);
    err &= FLOAT_EQUAL(fs_p3_r, view_w.y_max);

    if (err)
       fprintf(fp, "Partial reads compare\n");
    else
       fprintf(fp, "Partial reads differ\n");
    err_tot &= err;

    return(err_tot);}

/*--------------------------------------------------------------------------*/

/*                            TEST #2 ROUTINES                              */

/*--------------------------------------------------------------------------*/

/* TEST_2 - test the PDBLib functions handling indirections
 *        -
 *        - read and write structures with pointers
 *        - set default array base indexes
 *        - read structure members
 *        - read parts of arrays
 *        -
 *        - tests can be targeted
 */

test_2(base, tgt, n)
   char *base, *tgt;
   int n;
   {PDBfile *strm;
    char datfile[MAXLINE], fname[MAXLINE];
    int err;
    FILE *fp;

/* build the file names */
    test_target(tgt, base, n, fname, datfile);

    fp = fopen(fname, "w");

    prep_test_2_data();

/* open the file */
    if ((strm = PD_open(datfile, "r")) == NULL)
       {fprintf(fp, "Test couldn't open file %s\r\n", datfile);
        exit(2);};
    fprintf(fp, "File %s opened\n", datfile);

/* dump the symbol table */
    dump_test_symbol_table(fp, strm->symtab, 2);

/* read the structs */
    read_test_2_data(strm);

/* compare the original data with that read in */
    err = compare_test_2_data(strm, fp);

/* close the file */
    if (!PD_close(strm))
       {fprintf(fp, "Test couldn't close file %s\r\n", datfile);
        exit(2);};
    fprintf(fp, "File %s closed\n", datfile);

/* print out the results */
    print_test_2_data(fp);

    fclose(fp);
    if (err)
       remove(fname);

    return(err);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PREP_TEST_2_DATA - prepare the test data */

void prep_test_2_data()
   {int i, *p1, *p2;
    double *p3, *p4;

/*    do_w = -1; */

    for (i = 0; i < N_INT; i++)
        {p_w[i] = i;
         p_r[i] = 0;};

    tar_w = FMAKE_N(lev1, 2, "PREP_TEST_2_DATA:tar_w");

    p1 = tar_w[0].a = FMAKE_N(int, N_INT, "PREP_TEST_2_DATA:p1");
    p2 = tar_w[1].a = FMAKE_N(int, N_INT, "PREP_TEST_2_DATA:p2");
    for (i = 0; i < N_INT; i++)
        {p1[i] = i;
         p2[i] = i + 10;};

    p3 = tar_w[0].b = FMAKE_N(double, N_DOUBLE, "PREP_TEST_2_DATA:p3");
    p4 = tar_w[1].b = FMAKE_N(double, N_DOUBLE, "PREP_TEST_2_DATA:p4");
    for (i = 0; i < N_DOUBLE; i++)
        {p3[i] = exp((double) i);
         p4[i] = log(1.0 + (double) i);};

    tar_w[0].c = FMAKE_N(lev2, 2, "PREP_TEST_2_DATA:tar_w0");
    tar_w[1].c = FMAKE_N(lev2, 2, "PREP_TEST_2_DATA:tar_w1");

    tar_w[0].c[0].s    = FMAKE_N(char *, 2, "PREP_TEST_2_DATA:tar_w0.c0.s");
    tar_w[0].c[0].s[0] = SC_strsavef("Hello",
                         "char*:PREP_TEST_2_DATA:Hello");
    tar_w[0].c[0].s[1] = SC_strsavef(" ",
                         "char*:PREP_TEST_2_DATA:BLANK");
    tar_w[0].c[1].s    = FMAKE_N(char *, 2, "PREP_TEST_2_DATA:tar_w0.c1.s");
    tar_w[0].c[1].s[0] = SC_strsavef("world",
                         "char*:PREP_TEST_2_DATA:world");
    tar_w[0].c[1].s[1] = SC_strsavef("!",
                         "char*:PREP_TEST_2_DATA:!");

    tar_w[1].c[0].s    = FMAKE_N(char *, 2, "PREP_TEST_2_DATA:tar_w1.c0.s");
    tar_w[1].c[0].s[0] = SC_strsavef("Foo",
                         "char*:PREP_TEST_2_DATA:Foo");
    tar_w[1].c[0].s[1] = SC_strsavef(" ",
                         "char*:PREP_TEST_2_DATA:BLANK");
    tar_w[1].c[1].s    = FMAKE_N(char *, 2, "PREP_TEST_2_DATA:tar_w1.c1.s");
    tar_w[1].c[1].s[0] = SC_strsavef("Bar",
                         "char*:PREP_TEST_2_DATA:Bar");
    tar_w[1].c[1].s[1] = SC_strsavef("!!!",
                         "char*:PREP_TEST_2_DATA:!!!");

    tar_w[0].c[0].type = 1;
    tar_w[0].c[1].type = 2;
    tar_w[1].c[0].type = 3;
    tar_w[1].c[1].type = 4;

    tar_r = NULL;

    ap1 = NULL;
    ap2 = NULL;
    for (i = 0; i < 4; i++)
        aa[i] = 0;

    bp1 = NULL;
    bp2 = NULL;
    for (i = 0; i < 4; i++)
        ba[i] = 0.0;

    cp1 = NULL;
    cp2 = NULL;
    for (i = 0; i < 4; i++)
        {ca[i].s    = NULL;
         ca[i].type = 0;};

    sp1 = NULL;
    sp2 = NULL;
    sp3 = NULL;
    sp4 = NULL;
    for (i = 0; i < 5; i++)
        sa[i] = NULL;

    tp1 = NULL;
    tp2 = NULL;
    tp3 = NULL;
    tp4 = NULL;
    tp5 = NULL;
    tp6 = NULL;
    tp7 = NULL;
    tp8 = NULL;
    for (i = 0; i < 4; i++)
        ta[i] = 0;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* READ_TEST_2_DATA - read the test data from the file */

void read_test_2_data(strm)
   PDBfile *strm;
   {do_r = strm->default_offset;

    PD_read(strm, "tar", &tar_r);
    PD_read(strm, "p", p_r);

    PD_read(strm, "tar(0).a", &ap1);
    PD_read(strm, "tar(1).a", &ap2);

    PD_read(strm, "tar(0).a(0)", &aa[0]);
    PD_read(strm, "tar(0).a(1)", &aa[1]);
    PD_read(strm, "tar(1).a(0)", &aa[2]);
    PD_read(strm, "tar(1).a(1)", &aa[3]);

    PD_read(strm, "tar(0).b", &bp1);
    PD_read(strm, "tar(1).b", &bp2);

    PD_read(strm, "tar(0).b(0)", &ba[0]);
    PD_read(strm, "tar(0).b(1)", &ba[1]);
    PD_read(strm, "tar(1).b(0)", &ba[2]);
    PD_read(strm, "tar(1).b(1)", &ba[3]);

    PD_read(strm, "tar(0).c", &cp1);
    PD_read(strm, "tar(1).c", &cp2);

    PD_read(strm, "tar(0).c(0)", &ca[0]);
    PD_read(strm, "tar(0).c(1)", &ca[1]);
    PD_read(strm, "tar(1).c(0)", &ca[2]);
    PD_read(strm, "tar(1).c(1)", &ca[3]);

    PD_read(strm, "tar(0).c(0).s", &sp1);
    PD_read(strm, "tar(0).c(1).s", &sp2);
    PD_read(strm, "tar(1).c(0).s", &sp3);
    PD_read(strm, "tar(1).c(1).s", &sp4);

    PD_read(strm, "tar(0).c(0).s(0)", &tp1);
    PD_read(strm, "tar(0).c(0).s(1)", &tp2);
    PD_read(strm, "tar(0).c(1).s(0)", &tp3);
    PD_read(strm, "tar(0).c(1).s(1)", &tp4);

    PD_read(strm, "tar(0).c(0).s(0)(2)", &ta[0]);
    PD_read(strm, "tar(0).c(0).s(1)(1)", &ta[1]);
    PD_read(strm, "tar(0).c(1).s(0)(3)", &ta[2]);
    PD_read(strm, "tar(0).c(1).s(1)(2)", &ta[3]);

    PD_read(strm, "tar(1).c(0).s(0)", &tp5);
    PD_read(strm, "tar(1).c(0).s(1)", &tp6);
    PD_read(strm, "tar(1).c(1).s(0)", &tp7);
    PD_read(strm, "tar(1).c(1).s(1)", &tp8);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PRINT_TEST_2_DATA - print it out to the file */

void print_test_2_data(fp)
   FILE *fp;
   {int i;
    int *p1, *p2;
    double *p3, *p4;

    fprintf(fp, "\n");

    p1 = tar_r[0].a;
    p2 = tar_r[1].a;
    fprintf(fp, "TAR struct member A:\n");
    fprintf(fp, "    TAR[0].A    TAR[1].A\n");
    for (i = 0; i < N_INT; i++)
        fprintf(fp, "        %d          %d\n", p1[i], p2[i]);
    fprintf(fp, "\n");

    p3 = tar_r[0].b;
    p4 = tar_r[1].b;
    fprintf(fp, "TAR struct member B:\n");
    fprintf(fp, "    TAR[0].B    TAR[1].B\n");
    for (i = 0; i < N_DOUBLE; i++)
        fprintf(fp, "    %f    %f\n", p3[i], p4[i]);
    fprintf(fp, "\n");


    fprintf(fp, "TAR struct member C:\n");
    fprintf(fp, "    TAR[0].C[0]         TAR[0].C[1]\n");
    fprintf(fp, "   S[0]    S[1]        S[0]    S[1]\n");
    fprintf(fp, "  %s      %s        %s      %s\n",
                tar_r[0].c[0].s[0], tar_r[0].c[0].s[1],
                tar_r[0].c[1].s[0], tar_r[0].c[1].s[1]);
    fprintf(fp, "\n       TYPE\n");
    fprintf(fp, "     %d       %d\n",
                tar_r[0].c[0].type, tar_r[0].c[1].type);


    fprintf(fp, "\n    TAR[1].C[0]         TAR[1].C[1]\n");
    fprintf(fp, "   S[0]    S[1]        S[0]    S[1]\n");
    fprintf(fp, "   %s     %s           %s      %s\n",
                tar_r[1].c[0].s[0], tar_r[1].c[0].s[1],
                tar_r[1].c[1].s[0], tar_r[1].c[1].s[1]);

    fprintf(fp, "       TYPE\n");
    fprintf(fp, "     %d       %d\n",
                tar_r[1].c[0].type, tar_r[1].c[1].type);

    fprintf(fp, "\n");

/* print the offset */
    fprintf(fp, "file array index offset: %d\n\n", do_r);

    fprintf(fp, "\n");

/* read the an array which has offsets */
    fprintf(fp, "array p: \n");
    for (i = 0; i < N_INT; i++)
        fprintf(fp, "  %d\n", p_r[i]);

    fprintf(fp, "\n");

/* print the partial and member read stuff */
    if (ap1 != NULL)
       {fprintf(fp, "member read    TAR[0].A    TAR[1].A\n");
        for (i = 0; i < N_INT; i++)
            fprintf(fp, "                  %d          %d\n", ap1[i], ap2[i]);
        fprintf(fp, "\n");};
        
    if (bp1 != NULL)
       {fprintf(fp, "member read    TAR[0].B    TAR[1].B\n");
        for (i = 0; i < N_DOUBLE; i++)
            fprintf(fp, "               %f    %f\n", bp1[i], bp2[i]);
        fprintf(fp, "\n");};

    if (cp1 != NULL)
       {fprintf(fp, "member read TAR[0].C:\n");
        fprintf(fp, "    TAR[0].C[0]         TAR[0].C[1]\n");
        fprintf(fp, "   S[0]    S[1]        S[0]    S[1]\n");
        fprintf(fp, "  %s      %s        %s      %s\n",
                    cp1[0].s[0], cp1[0].s[1],
                    cp1[1].s[0], cp1[1].s[1]);
        fprintf(fp, "\n       TYPE\n");
        fprintf(fp, "     %d       %d\n",
                    cp1[0].type, cp1[1].type);

        fprintf(fp, "\nmember read TAR[1].C:\n");
        fprintf(fp, "    TAR[1].C[0]         TAR[1].C[1]\n");
        fprintf(fp, "   S[0]    S[1]        S[0]    S[1]\n");
        fprintf(fp, "   %s     %s           %s      %s\n",
                    cp2[0].s[0], cp2[0].s[1],
                    cp2[1].s[0], cp2[1].s[1]);
        fprintf(fp, "\n       TYPE\n");
        fprintf(fp, "     %d       %d\n",
                    cp2[0].type, cp2[1].type);};

/*
    fprintf(fp, "member read TAR[0].C[0]:\n");
    fprintf(fp, "   S[0]    S[1]\n");
    fprintf(fp, "  %s      %s\n", ca[0].s[0], ca[0].s[1]);

    fprintf(fp, "member read TAR[0].C[1]:\n");
    fprintf(fp, "   S[0]    S[1]\n");
    fprintf(fp, "  %s      %s\n", ca[1].s[0], ca[1].s[1]);

    fprintf(fp, "member read TAR[1].C[1]:\n");
    fprintf(fp, "   S[0]    S[1]\n");
    fprintf(fp, "  %s      %s\n", ca[2].s[0], ca[2].s[1]);

    fprintf(fp, "member read TAR[1].C[1]:\n");
    fprintf(fp, "   S[0]    S[1]\n");
    fprintf(fp, "  %s      %s\n", ca[3].s[0], ca[3].s[1]);
*/
/* TAR.C.S read */

    if (sp1 != NULL)
       {fprintf(fp, "\nmember read TAR[0].C[0].S:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", sp1[0], sp1[1]);

        fprintf(fp, "\nmember read TAR[0].C[1].S:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", sp2[0], sp2[1]);

        fprintf(fp, "\nmember read TAR[1].C[0].S:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", sp3[0], sp3[1]);

        fprintf(fp, "\nmember read TAR[1].C[1].S:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", sp4[0], sp4[1]);};

/* TAR.C.S[i] read */

    if (tp1 != NULL)
       {fprintf(fp, "\nmember read TAR[0].C[0].S[0]:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", tp1, tp1);

        fprintf(fp, "\nmember read TAR[0].C[1].S[1]:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", tp2, tp2);

        fprintf(fp, "\nmember read TAR[0].C[1].S[0]:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", tp3, tp3);

        fprintf(fp, "\nmember read TAR[0].C[1].S[1]:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", tp4, tp4);

        fprintf(fp, "\nmember read TAR[1].C[0].S[0]:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", tp5, tp5);

        fprintf(fp, "\nmember read TAR[1].C[0].S[1]:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", tp6, tp6);

        fprintf(fp, "\nmember read TAR[1].C[1].S[0]:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", tp7, tp7);

        fprintf(fp, "\nmember read TAR[1].C[1].S[1]:\n");
        fprintf(fp, "   S[0]    S[1]\n");
        fprintf(fp, "  %s      %s\n", tp8, tp8);

/* TAR.C.S[i][j] read */

        fprintf(fp, "\nmember read from TAR[0]:\n");
        fprintf(fp, "C[0].S[0][2]  C[0].S[1][1]  C[1].S[0][3]  C[1].S[1][2]\n");
        fprintf(fp, "     %c             %c           %c             %c\n",
                    ta[0], ta[1], ta[2], ta[3]);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* COMPARE_TEST_2_DATA - compare the test data */

compare_test_2_data(strm, fp)
   PDBfile *strm;
   FILE *fp;
   {int i, *p1w, *p2w, *p1r, *p2r;
    double *p3w, *p4w, *p3r, *p4r;
    int err, err_tot;
    int float_nm, double_nm;
    double float_tolerance, double_tolerance;

    float_nm  = min(strm->std->float_format[2],
                    strm->host_std->float_format[2]);
    double_nm = min(strm->std->double_format[2],
                    strm->host_std->double_format[2]);

/* pad the absolute tolerance */
    float_tolerance  = POW(2.0, -((double) float_nm));
    double_tolerance = POW(2.0, -((double) double_nm));
    err_tot = TRUE;

/* compare offset and array */
    err  = TRUE;
    err &= (do_r == do_w);

    for (i = 0; i < N_INT; i++)
        err &= (p_w[i] == p_r[i]);

    if (err)
       fprintf(fp, "Offset compare\n");
    else
       fprintf(fp, "Offset differ\n");
    err_tot &= err;

    err = TRUE;
    p1w = tar_w[0].a;
    p2w = tar_w[1].a;
    p1r = tar_r[0].a;
    p2r = tar_r[1].a;
    for (i = 0; i < N_INT; i++)
        {err &= (p1w[i] == p1r[i]);
         err &= (p2w[i] == p2r[i]);};

    p3w = tar_w[0].b;
    p4w = tar_w[1].b;
    p3r = tar_r[0].b;
    p4r = tar_r[1].b;
    for (i = 0; i < N_DOUBLE; i++)
        {err &= DOUBLE_EQUAL(p3r[i], p3r[i]);
         err &= DOUBLE_EQUAL(p4w[i], p4r[i]);};

    err &= (strcmp(tar_w[0].c[0].s[0], tar_r[0].c[0].s[0]) == 0);
    err &= (strcmp(tar_w[0].c[0].s[1], tar_r[0].c[0].s[1]) == 0);
    err &= (strcmp(tar_w[0].c[1].s[0], tar_r[0].c[1].s[0]) == 0);
    err &= (strcmp(tar_w[0].c[1].s[1], tar_r[0].c[1].s[1]) == 0);

    err &= (strcmp(tar_w[1].c[0].s[0], tar_r[1].c[0].s[0]) == 0);
    err &= (strcmp(tar_w[1].c[0].s[1], tar_r[1].c[0].s[1]) == 0);
    err &= (strcmp(tar_w[1].c[1].s[0], tar_r[1].c[1].s[0]) == 0);
    err &= (strcmp(tar_w[1].c[1].s[1], tar_r[1].c[1].s[1]) == 0);

    err &= (tar_w[0].c[0].type == tar_r[0].c[0].type);
    err &= (tar_w[0].c[1].type == tar_r[0].c[1].type);
    err &= (tar_w[1].c[0].type == tar_r[1].c[0].type);
    err &= (tar_w[1].c[1].type == tar_r[1].c[1].type);

    if (err)
       fprintf(fp, "Indirects compare\n");
    else
       fprintf(fp, "Indirects differ\n");
    err_tot &= err;

    fprintf(fp, "\n");

    if (err)
       fprintf(fp, "Offset compare\n");
    else
       fprintf(fp, "Offset differ\n");
    err_tot &= err;

    err = TRUE;

    if (ap1 != NULL)
       {p1w = tar_w[0].a;
        p2w = tar_w[1].a;
        for (i = 0; i < N_INT; i++)
            {err &= (p1w[i] == ap1[i]);
             err &= (p2w[i] == ap2[i]);};};

    if (bp1 != NULL)
       {p3w = tar_w[0].b;
        p4w = tar_w[1].b;
        for (i = 0; i < N_DOUBLE; i++)
            {err &= DOUBLE_EQUAL(p3w[i], bp1[i]);
             err &= DOUBLE_EQUAL(p4w[i], bp2[i]);};};

    if (cp1 != NULL)
       {err &= (strcmp(tar_w[0].c[0].s[0], cp1[0].s[0]) == 0);
        err &= (strcmp(tar_w[0].c[0].s[1], cp1[0].s[1]) == 0);
        err &= (strcmp(tar_w[0].c[1].s[0], cp1[1].s[0]) == 0);
        err &= (strcmp(tar_w[0].c[1].s[1], cp1[1].s[1]) == 0);

        err &= (strcmp(tar_w[1].c[0].s[0], cp2[0].s[0]) == 0);
        err &= (strcmp(tar_w[1].c[0].s[1], cp2[0].s[1]) == 0);
        err &= (strcmp(tar_w[1].c[1].s[0], cp2[1].s[0]) == 0);
        err &= (strcmp(tar_w[1].c[1].s[1], cp2[1].s[1]) == 0);

        err &= (tar_w[0].c[0].type == cp1[0].type);
        err &= (tar_w[0].c[1].type == cp1[1].type);
        err &= (tar_w[1].c[0].type == cp2[0].type);
        err &= (tar_w[1].c[1].type == cp2[1].type);};
/*
    err &= (strcmp(tar_w[0].c[0].s[0], ca[0].s[0]) == 0);
    err &= (strcmp(tar_w[0].c[0].s[1], ca[0].s[1]) == 0);
    err &= (strcmp(tar_w[0].c[1].s[0], ca[1].s[0]) == 0);
    err &= (strcmp(tar_w[0].c[1].s[1], ca[1].s[1]) == 0);

    err &= (strcmp(tar_w[1].c[0].s[0], ca[2].s[0]) == 0);
    err &= (strcmp(tar_w[1].c[0].s[1], ca[2].s[1]) == 0);
    err &= (strcmp(tar_w[1].c[1].s[0], ca[3].s[0]) == 0);
    err &= (strcmp(tar_w[1].c[1].s[1], ca[3].s[1]) == 0);
*/
    if (sp1 != NULL)
       {err &= (strcmp(tar_w[0].c[0].s[0], sp1[0]) == 0);
        err &= (strcmp(tar_w[0].c[0].s[1], sp1[1]) == 0);
        err &= (strcmp(tar_w[0].c[1].s[0], sp2[0]) == 0);
        err &= (strcmp(tar_w[0].c[1].s[1], sp2[1]) == 0);

        err &= (strcmp(tar_w[1].c[0].s[0], sp3[0]) == 0);
        err &= (strcmp(tar_w[1].c[0].s[1], sp3[1]) == 0);
        err &= (strcmp(tar_w[1].c[1].s[0], sp4[0]) == 0);
        err &= (strcmp(tar_w[1].c[1].s[1], sp4[1]) == 0);};

    if (tp1 != NULL)
       {err &= (strcmp(tar_w[0].c[0].s[0], tp1) == 0);
        err &= (strcmp(tar_w[0].c[0].s[1], tp2) == 0);
        err &= (strcmp(tar_w[0].c[1].s[0], tp3) == 0);
        err &= (strcmp(tar_w[0].c[1].s[1], tp4) == 0);

        err &= (strcmp(tar_w[1].c[0].s[0], tp5) == 0);
        err &= (strcmp(tar_w[1].c[0].s[1], tp6) == 0);
        err &= (strcmp(tar_w[1].c[1].s[0], tp7) == 0);
        err &= (strcmp(tar_w[1].c[1].s[1], tp8) == 0);

        err &= (tar_w[0].c[0].s[0][2] == ta[0]);
        err &= (tar_w[0].c[0].s[1][1] == ta[1]);
        err &= (tar_w[0].c[1].s[0][3] == ta[2]);
        err &= (tar_w[0].c[1].s[1][2] == ta[3]);};

    if (err)
       fprintf(fp, "Indirects compare\n");
    else
       fprintf(fp, "Indirects differ\n");
    err_tot &= err;

    fprintf(fp, "\n");

    return(err_tot);}

/*--------------------------------------------------------------------------*/

/*                            TEST #3 ROUTINES                              */

/*--------------------------------------------------------------------------*/

/* TEST_3 - test the PDBLib functions handling indirections
 *        -
 *        - read and write structures with alignment difficulties
 *        -
 *        - tests can be targeted
 */

test_3(base, tgt, n)
   char *base, *tgt;
   int n;
   {PDBfile *strm;
    char datfile[MAXLINE], fname[MAXLINE];
    int err;
    FILE *fp;

/* build the file names */
    test_target(tgt, base, n, fname, datfile);

    fp = fopen(fname, "w");

    prep_test_3_data();

/* open the file */
    if ((strm = PD_open(datfile, "r")) == NULL)
       {fprintf(fp, "Test couldn't open file %s\r\n", datfile);
        exit(2);};
    fprintf(fp, "File %s opened\n", datfile);

/* dump the symbol table */
    dump_test_symbol_table(fp, strm->symtab, 3);

/* read the structs */
    read_test_3_data(strm);

/* compare the original data with that read in */
    err = compare_test_3_data(strm, fp);

/* close the file */
    if (!PD_close(strm))
       {fprintf(fp, "Test couldn't close file %s\r\n", datfile);
        exit(2);};
    fprintf(fp, "File %s closed\n", datfile);

/* print out the results */
    print_test_3_data(fp);

    fclose(fp);
    if (err)
       remove(fname);

    return(err);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PREP_TEST_3_DATA - prepare the test data */

void prep_test_3_data()
   {vr1_w.a    = 'A';
    vr1_w.b    = -10;
    vr1_w.c[0] = 'B';
    vr1_w.c[1] = 'a';
    vr1_w.d    = 32589;
    vr1_w.e[0] = 'C';
    vr1_w.e[1] = 'a';
    vr1_w.e[2] = 'b';
    vr1_w.f    = 1.523e-19;
    vr1_w.g[0] = 'D';
    vr1_w.g[1] = 'a';
    vr1_w.g[2] = 'b';
    vr1_w.g[3] = 'c';
    vr1_w.h    = 4.2782918323832554e30;
    vr1_w.i[0] = 'E';
    vr1_w.i[1] = 'a';
    vr1_w.i[2] = 'b';
    vr1_w.i[3] = 'c';
    vr1_w.i[4] = 'd';
    vr1_w.j    = SC_strsavef("whoa there big fella!",
                 "char*:PREP_TEST_3_DATA:whoa");
    vr1_w.k[0] = 'F';
    vr1_w.k[1] = 'a';
    vr1_w.k[2] = 'b';
    vr1_w.k[3] = 'c';
    vr1_w.k[4] = 'd';
    vr1_w.k[5] = 'e';

    vr1_r.a    = '\0';
    vr1_r.b    = 0;
    vr1_r.c[0] = '\0';
    vr1_r.c[1] = '\0';
    vr1_r.d    = 0;
    vr1_r.e[0] = '\0';
    vr1_r.e[1] = '\0';
    vr1_r.e[2] = '\0';
    vr1_r.f    = 0.0;
    vr1_r.g[0] = '\0';
    vr1_r.g[1] = '\0';
    vr1_r.g[2] = '\0';
    vr1_r.g[3] = '\0';
    vr1_r.h    = 0.0;
    vr1_r.i[0] = '\0';
    vr1_r.i[1] = '\0';
    vr1_r.i[2] = '\0';
    vr1_r.i[3] = '\0';
    vr1_r.i[4] = '\0';
    vr1_r.j    = NULL;
    vr1_r.k[0] = '\0';
    vr1_r.k[1] = '\0';
    vr1_r.k[2] = '\0';
    vr1_r.k[3] = '\0';
    vr1_r.k[4] = '\0';
    vr1_r.k[5] = '\0';

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* READ_TEST_3_DATA - read the test data from the file */

void read_test_3_data(strm)
   PDBfile *strm;
   {PD_read(strm, "vr1", &vr1_r);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PRINT_TEST_3_DATA - print it out to the file */

void print_test_3_data(fp)
   FILE *fp;
   {fprintf(fp, "struct vr1:\n");
    fprintf(fp, "   vr1.a    = %c\n", vr1_r.a);
    fprintf(fp, "   vr1.b    = %d\n", vr1_r.b);
    fprintf(fp, "   vr1.c[0] = %c\n", vr1_r.c[0]);
    fprintf(fp, "   vr1.c[1] = %c\n", vr1_r.c[1]);
    fprintf(fp, "   vr1.d    = %d\n", vr1_r.d);
    fprintf(fp, "   vr1.e[0] = %c\n", vr1_r.e[0]);
    fprintf(fp, "   vr1.e[1] = %c\n", vr1_r.e[1]);
    fprintf(fp, "   vr1.e[2] = %c\n", vr1_r.e[2]);
    fprintf(fp, "   vr1.f    = %14.7e\n", vr1_r.f);
    fprintf(fp, "   vr1.g[0] = %c\n", vr1_r.g[0]);
    fprintf(fp, "   vr1.g[1] = %c\n", vr1_r.g[1]);
    fprintf(fp, "   vr1.g[2] = %c\n", vr1_r.g[2]);
    fprintf(fp, "   vr1.g[3] = %c\n", vr1_r.g[3]);
    fprintf(fp, "   vr1.h    = %20.13e\n", vr1_r.h);
    fprintf(fp, "   vr1.i[0] = %c\n", vr1_r.i[0]);
    fprintf(fp, "   vr1.i[1] = %c\n", vr1_r.i[1]);
    fprintf(fp, "   vr1.i[2] = %c\n", vr1_r.i[2]);
    fprintf(fp, "   vr1.i[3] = %c\n", vr1_r.i[3]);
    fprintf(fp, "   vr1.i[4] = %c\n", vr1_r.i[4]);
    fprintf(fp, "   vr1.j    = %s\n", vr1_r.j);
    fprintf(fp, "   vr1.k[0] = %c\n", vr1_r.k[0]);
    fprintf(fp, "   vr1.k[1] = %c\n", vr1_r.k[1]);
    fprintf(fp, "   vr1.k[2] = %c\n", vr1_r.k[2]);
    fprintf(fp, "   vr1.k[3] = %c\n", vr1_r.k[3]);
    fprintf(fp, "   vr1.k[4] = %c\n", vr1_r.k[4]);
    fprintf(fp, "   vr1.k[5] = %c\n", vr1_r.k[5]);

    fprintf(fp, "\n");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* COMPARE_TEST_3_DATA - compare the test data */

compare_test_3_data(strm, fp)
   PDBfile *strm;
   FILE *fp;
   {int err, err_tot;
    int float_nm, double_nm;
    double float_tolerance, double_tolerance;

    float_nm  = min(strm->std->float_format[2],
                    strm->host_std->float_format[2]);
    double_nm = min(strm->std->double_format[2],
                    strm->host_std->double_format[2]);

/* pad the absolute tolerance */
    float_tolerance  = POW(2.0, -((double) float_nm));
    double_tolerance = POW(2.0, -((double) double_nm));
    err_tot = TRUE;

/* compare offset and array */
    err  = TRUE;
    err &= (vr1_w.a == vr1_r.a);
    err &= (vr1_w.b == vr1_r.b);
    err &= (vr1_w.c[0] == vr1_r.c[0]);
    err &= (vr1_w.c[1] == vr1_r.c[1]);
    err &= (vr1_w.d == vr1_r.d);
    err &= (vr1_w.e[0] == vr1_r.e[0]);
    err &= (vr1_w.e[1] == vr1_r.e[1]);
    err &= (vr1_w.e[2] == vr1_r.e[2]);
    err &= FLOAT_EQUAL(vr1_w.f, vr1_r.f);
    err &= (vr1_w.g[0] == vr1_r.g[0]);
    err &= (vr1_w.g[1] == vr1_r.g[1]);
    err &= (vr1_w.g[2] == vr1_r.g[2]);
    err &= (vr1_w.g[3] == vr1_r.g[3]);
    err &= DOUBLE_EQUAL(vr1_w.h, vr1_r.h);
    err &= (vr1_w.i[0] == vr1_r.i[0]);
    err &= (vr1_w.i[1] == vr1_r.i[1]);
    err &= (vr1_w.i[2] == vr1_r.i[2]);
    err &= (vr1_w.i[3] == vr1_r.i[3]);
    err &= (vr1_w.i[4] == vr1_r.i[4]);
    err &= (strcmp(vr1_w.j, vr1_r.j) == 0);
    err &= (vr1_w.k[0] == vr1_r.k[0]);
    err &= (vr1_w.k[1] == vr1_r.k[1]);
    err &= (vr1_w.k[2] == vr1_r.k[2]);
    err &= (vr1_w.k[3] == vr1_r.k[3]);
    err &= (vr1_w.k[4] == vr1_r.k[4]);
    err &= (vr1_w.k[5] == vr1_r.k[5]);

    if (err)
       fprintf(fp, "Alignments compare\n");
    else
       fprintf(fp, "Alignments differ\n");
    err_tot &= err;

    fprintf(fp, "\n");

    return(err_tot);}

/*--------------------------------------------------------------------------*/

/*                            TEST #4 ROUTINES                              */

/*--------------------------------------------------------------------------*/

/* TEST_4 - test the PDBLib functions handling indirections
 *        -
 *        - read and write structures with alignment difficulties
 *        -
 *        - tests can be targeted
 */

test_4(base, tgt, n)
   char *base, *tgt;
   int n;
   {PDBfile *strm;
    char datfile[MAXLINE], fname[MAXLINE];
    int err;
    FILE *fp;

/* build the file names */
    test_target(tgt, base, n, fname, datfile);

    fp = fopen(fname, "w");

    prep_test_4_data();

/* open the file */
    if ((strm = PD_open(datfile, "r")) == NULL)
       {fprintf(fp, "Test couldn't open file %s\r\n", datfile);
        exit(2);};
    fprintf(fp, "File %s opened\n", datfile);

/* dump the symbol table */
    dump_test_symbol_table(fp, strm->symtab, 4);

/* read the structs */
    read_test_4_data(strm);

/* compare the original data with that read in */
    err = compare_test_4_data(strm, fp);

/* close the file */
    if (!PD_close(strm))
       {fprintf(fp, "Test couldn't close file %s\r\n", datfile);
        exit(2);};
    fprintf(fp, "File %s closed\n", datfile);

/* print out the results */
    print_test_4_data(fp);

    fclose(fp);
    if (err)
       remove(fname);

    return(err);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PREP_TEST_4_DATA - prepare the test data */

void prep_test_4_data()
   {char *pc;
    short *ps;
    int *pi;
    long *pl;
    float *pf;
    double *pd;
    hashel *hp;

    tab4_w = SC_make_hash_table(3, NODOC);

    CHAR_S   = SC_strsavef("char *", "char*:PREP_TEST_4_DATA:char");
    SHORT_S  = SC_strsavef("short *", "char*:PREP_TEST_4_DATA:short");
    INT_S    = SC_strsavef("integer *", "char*:PREP_TEST_4_DATA:int");
    LONG_S   = SC_strsavef("long *", "char*:PREP_TEST_4_DATA:long");
    FLOAT_S  = SC_strsavef("float *", "char*:PREP_TEST_4_DATA:float");
    DOUBLE_S = SC_strsavef("double *", "char*:PREP_TEST_4_DATA:double");
    HASHEL_S = SC_strsavef("hashel *", "char*:PREP_TEST_4_DATA:hashel");

    pc  = FMAKE(char, "PREP_TEST_4_DATA:pc");
    *pc = 'A';   
    SC_install("pc", pc, CHAR_S, tab4_w);

    ps  = FMAKE(short, "PREP_TEST_4_DATA:ps");
    *ps = -1024;
    SC_install("ps", ps, SHORT_S, tab4_w);

    pi  = FMAKE(int, "PREP_TEST_4_DATA:pi");
    *pi = 16384;
    SC_install("pi", pi, INT_S, tab4_w);

    pl  = FMAKE(long, "PREP_TEST_4_DATA:pl");
    *pl = -1048576;
    SC_install("pl", pl, LONG_S, tab4_w);

    pf  = FMAKE(float, "PREP_TEST_4_DATA:pf");
    *pf = 3.141596;
    SC_install("pf", pf, FLOAT_S, tab4_w);

    pd  = FMAKE(double, "PREP_TEST_4_DATA:pd");
    *pd = -1.0e-30;
    hp = SC_install("pd", pd, DOUBLE_S, tab4_w);

    SC_install("ph", hp, HASHEL_S, tab4_w);

    tab4_r = NULL;

    vr4_w = FMAKE_N(st4, 3, "PREP_TEST_4_DATA:vr4_w");

    vr4_w[0].a =  2048;
    vr4_w[0].b =  'G';
    vr4_w[1].a = -2048;
    vr4_w[1].b =  'H';
    vr4_w[2].a =  4096;
    vr4_w[2].b =  'I';

    vr4_r = NULL;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* READ_TEST_4_DATA - read the test data from the file */

void read_test_4_data(strm)
   PDBfile *strm;
   {PD_read(strm, "tab4", &tab4_r);
    PD_read(strm, "vr4", &vr4_r);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PRINT_TEST_4_DATA - print it out to the file */

void print_test_4_data(fp)
   FILE *fp;
   {char *pc, **names;
    short *ps;
    int *pi, i;
    long *pl;
    float *pf;
    double *pd;
    hashel *ph;

    names = SC_hash_dump(tab4_r, NULL);
    for (i = 0; i < tab4_r->nelements; i++)
        fprintf(fp, "%s\n", names[i]);

    fprintf(fp, "\n");

    pc = (char *) SC_def_lookup("pc", tab4_r);
    ps = (short *) SC_def_lookup("ps", tab4_r);
    pi = (int *) SC_def_lookup("pi", tab4_r);
    pl = (long *) SC_def_lookup("pl", tab4_r);
    pf = (float *) SC_def_lookup("pf", tab4_r);
    pd = (double *) SC_def_lookup("pd", tab4_r);
    ph = (hashel *) SC_def_lookup("ph", tab4_r);

    fprintf(fp, "Table values:\n");
    fprintf(fp, "   pc = %c %s\n", *pc, CHAR_S);
    fprintf(fp, "   ps = %d %s\n", *ps, SHORT_S);
    fprintf(fp, "   pi = %d %s\n", *pi, INT_S);
    fprintf(fp, "   pl = %ld %s\n", *pl, LONG_S);
    fprintf(fp, "   pf = %f %s\n", *pf, FLOAT_S);
    fprintf(fp, "   pd = %e %s\n", *pd, DOUBLE_S);

    fprintf(fp, "\n   ph : %s %s\n\n", ph->name, ph->type);

    fprintf(fp, "VR4[0]:\n");
    fprintf(fp, "VR4[0].A = %d\n", vr4_r[0].a);
    fprintf(fp, "VR4[0].B = %c\n", vr4_r[0].b);
    fprintf(fp, "VR4[1].A = %d\n", vr4_r[1].a);
    fprintf(fp, "VR4[1].B = %c\n", vr4_r[1].b);
    fprintf(fp, "VR4[2].A = %d\n", vr4_r[2].a);
    fprintf(fp, "VR4[2].B = %c\n", vr4_r[2].b);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* COMPARE_TEST_4_DATA - compare the test data */

compare_test_4_data(strm, fp)
   PDBfile *strm;
   FILE *fp;
   {int err, err_tot;
    int float_nm, double_nm;
    double float_tolerance, double_tolerance;
    char *pc_w, *pc_r;
    short *ps_w, *ps_r;
    int *pi_w, *pi_r;
    long *pl_w, *pl_r;
    float *pf_w, *pf_r;
    double *pd_w, *pd_r;
    hashel *ph_w, *ph_r;

    float_nm  = min(strm->std->float_format[2],
                    strm->host_std->float_format[2]);
    double_nm = min(strm->std->double_format[2],
                    strm->host_std->double_format[2]);

/* pad the absolute tolerance */
    float_tolerance  = POW(2.0, -((double) float_nm));
    double_tolerance = POW(2.0, -((double) double_nm));
    err_tot = TRUE;

    pc_r = (char *) SC_def_lookup("pc", tab4_r);
    ps_r = (short *) SC_def_lookup("ps", tab4_r);
    pi_r = (int *) SC_def_lookup("pi", tab4_r);
    pl_r = (long *) SC_def_lookup("pl", tab4_r);
    pf_r = (float *) SC_def_lookup("pf", tab4_r);
    pd_r = (double *) SC_def_lookup("pd", tab4_r);
    ph_r = (hashel *) SC_def_lookup("ph", tab4_r);

    pc_w = (char *) SC_def_lookup("pc", tab4_w);
    ps_w = (short *) SC_def_lookup("ps", tab4_w);
    pi_w = (int *) SC_def_lookup("pi", tab4_w);
    pl_w = (long *) SC_def_lookup("pl", tab4_w);
    pf_w = (float *) SC_def_lookup("pf", tab4_w);
    pd_w = (double *) SC_def_lookup("pd", tab4_w);
    ph_w = (hashel *) SC_def_lookup("ph", tab4_w);

/* compare offset and array */
    err  = TRUE;
    err &= (*pc_r == *pc_w);
    err &= (*ps_r == *ps_w);
    err &= (*pi_r == *pi_w);
    err &= (*pl_r == *pl_w);
    err &= FLOAT_EQUAL(*pf_r, *pf_w);
    err &= DOUBLE_EQUAL(*pd_r, *pd_w);

    if (err)
       fprintf(fp, "Hash tables compare\n");
    else
       fprintf(fp, "Hash tables differ\n");
    err_tot &= err;

/* check the new alignments */
    err  = TRUE;
    err &= (vr4_w[0].a == vr4_r[0].a);
    err &= (vr4_w[0].b == vr4_r[0].b);
    err &= (vr4_w[1].a == vr4_r[1].a);
    err &= (vr4_w[1].b == vr4_r[1].b);
    err &= (vr4_w[2].a == vr4_r[2].a);
    err &= (vr4_w[2].b == vr4_r[2].b);

    if (err)
       fprintf(fp, "Alignments compare\n");
    else
       fprintf(fp, "Alignments differ\n");
    err_tot &= err;

    fprintf(fp, "\n");

    return(err_tot);}

/*--------------------------------------------------------------------------*/

/*                         GENERAL PURPOSE ROUTINES                         */

/*--------------------------------------------------------------------------*/

/* TEST_TARGET - set up the target for the data file */

void test_target(tgt, base, n, fname, datfile)
   char *tgt, *base;
   int n;
   char *fname, *datfile;
   {if (tgt != NULL)
       {sprintf(fname, "%s-%s.rs%d", base, tgt, n);
        sprintf(datfile, "%s-%s.db%d", base, tgt, n);}
    else
       {sprintf(fname, "%s-nat.rs%d", base, n);
        sprintf(datfile, "%s-nat.db%d", base, n);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* DUMP_TEST_SYMBOL_TABLE - dump the symbol table */

void dump_test_symbol_table(fp, tab, n)
   FILE *fp;
   HASHTAB *tab;
   int n;
   {int i;
    char **names;

    fprintf(fp, "\nTest %d Symbol table:\n", n);

    names = SC_hash_dump(tab, NULL);
    for (i = 0; i < tab->nelements; i++)
        fprintf(fp, "%s\n", names[i]);

    fprintf(fp, "\n");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PD_FLOAT_EQUAL - compare two floating point numbers
 *                - which have different precision
 */

PD_float_equal(d1, d2, tol)
   double d1, d2, tol;
   {double del;

    if (d1 == d2)
       return(0);

    del = (d1 - d2)/(ABS(d1) + ABS(d2) + SMALL);
    if (del < -tol)
       return(-1);
    else if (tol < del)
       return(1);
    else
       return(0);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
