/*	$Id: PosixFileDescr.c,v 1.12 1998/06/02 19:15:23 acken Exp $	*/
/*  Generalized access to POSIX-style file descriptors.
    Copyright (C) 1997, 1998  Michael van Acken

    This file is part of OOC.

    OOC is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.  

    OOC is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details. 

    You should have received a copy of the GNU General Public License
    along with OOC. If not, write to the Free Software Foundation, 59
    Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <stddef.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <limits.h>
#include <string.h>

#include "__oo2c.h"
#include "__mini_gc.h"
#include "__StdTypes.h"
#include "__config.h"

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#if HAVE_SOCKET_H
#include <socket.h>
#endif

#ifndef SSIZE_MAX
/* for every POSIX.1 system the macro SSIZE_MAX is the limit on the number of
   bytes that can be read or written in a single operation; assume 2^15 if the
   macro isn't part of limits.h */
#define SSIZE_MAX 32768
#endif

/* --- begin #include "PosixFileDescr.d" */
#include "PosixFileDescr.h"
#include "C.h"
#include "CharClass.h"

/* local definitions */
typedef int PosixFileDescr_FileDescriptor;

/* function prototypes */

/* module and type descriptors */
static const struct {
  int length;
  void* pad;
  const char name[15];
} _n0 = {15, NULL, {"PosixFileDescr"}};
static struct _MD PosixFileDescr__md = {
  NULL, 
  &Kernel_ModuleDesc__td.td, 
  {
    NULL, 
    (const unsigned char*)_n0.name, 
    -1, 
    NULL
  }
};

static const struct {
  int length;
  void* pad;
  const char name[12];
} _n1 = {12, NULL, {"ChannelDesc"}};
static const struct {
  int length;
  void* pad;
  _Type btypes[2];
} PosixFileDescr_ChannelDesc__tdb = {
  2, 
  NULL, 
  {
    &Channel_ChannelDesc__td.td, 
    &PosixFileDescr_ChannelDesc__td.td
  }
};
static const struct {
  int length;
  void* pad;
  const void* tbprocs[8];
} _tb0 = {8, NULL, {
  (void*)PosixFileDescr_ChannelDesc_Length, 
  (void*)PosixFileDescr_ChannelDesc_GetModTime, 
  (void*)PosixFileDescr_ChannelDesc_NewReader, 
  (void*)PosixFileDescr_ChannelDesc_NewWriter, 
  (void*)PosixFileDescr_ChannelDesc_Flush, 
  (void*)PosixFileDescr_ChannelDesc_Close, 
  (void*)Channel_ChannelDesc_ErrorDescr, 
  (void*)Channel_ChannelDesc_ClearError
}};
struct _TD PosixFileDescr_ChannelDesc__td = {
  NULL, 
  &Types_TypeDesc__td.td, 
  {
    PosixFileDescr_ChannelDesc__tdb.btypes, 
    _tb0.tbprocs, 
    (const unsigned char*)_n1.name, 
    &PosixFileDescr__md.md, 
    1, 
    '0', '1', 
    sizeof(PosixFileDescr_ChannelDesc), 
    NULL
  }
};

static const struct {
  int length;
  void* pad;
  const char name[11];
} _n2 = {11, NULL, {"ReaderDesc"}};
static const struct {
  int length;
  void* pad;
  _Type btypes[2];
} PosixFileDescr_ReaderDesc__tdb = {
  2, 
  NULL, 
  {
    &Channel_ReaderDesc__td.td, 
    &PosixFileDescr_ReaderDesc__td.td
  }
};
static const struct {
  int length;
  void* pad;
  const void* tbprocs[7];
} _tb1 = {7, NULL, {
  (void*)PosixFileDescr_ReaderDesc_Pos, 
  (void*)PosixFileDescr_ReaderDesc_Available, 
  (void*)PosixFileDescr_ReaderDesc_SetPos, 
  (void*)PosixFileDescr_ReaderDesc_ReadByte, 
  (void*)PosixFileDescr_ReaderDesc_ReadBytes, 
  (void*)PosixFileDescr_ReaderDesc_ErrorDescr, 
  (void*)Channel_ReaderDesc_ClearError
}};
struct _TD PosixFileDescr_ReaderDesc__td = {
  NULL, 
  &Types_TypeDesc__td.td, 
  {
    PosixFileDescr_ReaderDesc__tdb.btypes, 
    _tb1.tbprocs, 
    (const unsigned char*)_n2.name, 
    &PosixFileDescr__md.md, 
    1, 
    '0', '1', 
    sizeof(PosixFileDescr_ReaderDesc), 
    &PosixFileDescr_ChannelDesc__td.td
  }
};

static const struct {
  int length;
  void* pad;
  const char name[11];
} _n3 = {11, NULL, {"WriterDesc"}};
static const struct {
  int length;
  void* pad;
  _Type btypes[2];
} PosixFileDescr_WriterDesc__tdb = {
  2, 
  NULL, 
  {
    &Channel_WriterDesc__td.td, 
    &PosixFileDescr_WriterDesc__td.td
  }
};
static const struct {
  int length;
  void* pad;
  const void* tbprocs[6];
} _tb2 = {6, NULL, {
  (void*)PosixFileDescr_WriterDesc_Pos, 
  (void*)PosixFileDescr_WriterDesc_SetPos, 
  (void*)PosixFileDescr_WriterDesc_WriteByte, 
  (void*)PosixFileDescr_WriterDesc_WriteBytes, 
  (void*)PosixFileDescr_WriterDesc_ErrorDescr, 
  (void*)Channel_WriterDesc_ClearError
}};
struct _TD PosixFileDescr_WriterDesc__td = {
  NULL, 
  &Types_TypeDesc__td.td, 
  {
    PosixFileDescr_WriterDesc__tdb.btypes, 
    _tb2.tbprocs, 
    (const unsigned char*)_n3.name, 
    &PosixFileDescr__md.md, 
    1, 
    '0', '1', 
    sizeof(PosixFileDescr_WriterDesc), 
    &PosixFileDescr_ReaderDesc__td.td
  }
};

/* local strings */

/* --- end #include "PosixFileDescr.d" */


/* this are the ids of type-bound procedures called in this file; they
   equal the respective index in the type descriptor: */
#define Channel_Flush 4
#define Channel_ClearError 7

/* keep track whether the file descriptors 0-2 refer to the standard IO
   descriptors passed from the shell, or should be treated like any other
   descriptor */
static int standard_io[3] = {1, 1, 1};

static _ModId _mid;


/* function definitions */

static void adjust_pos(PosixFileDescr_Channel ch, int pos) {
  if (ch->positionable && (ch->pos != pos)) {
    (void)lseek(ch->fd, pos, SEEK_SET);
    ch->pos = pos;
  }
}

static INTEGER write_error(void) {
  if (errno == EBADF) {
    return Channel_invalidChannel;
  } else if (errno == ENOSPC) {
    return Channel_noRoom;
#ifdef EDQUOT
  } else if (errno == EDQUOT) {
    return Channel_noRoom;
#endif
  } else {
    return Channel_writeError;
  }
}

static INTEGER read_error(void) {
  if (errno == EBADF) {
    return Channel_invalidChannel;
  } else {
    return Channel_readError;
  }
}

static INTEGER read_bytes(PosixFileDescr_Channel ch, LONGINT pos, LONGINT n,
                          BYTE *x, LONGINT *bytes_read) {
/* Reads only a single line for line buffered input from terminal; in this case
   `n should be large enough to hold a single line.  */
  LONGINT size;
  size_t acc, res;
  
  adjust_pos(ch, pos);
  /* read bytes from stream; repeat until all have been read successfully */
  acc = 0;
  do {
    /* make sure that no read request larger than the system limit is 
       issued */
    size = n-acc;
    if (size > SSIZE_MAX) size = SSIZE_MAX;
    res = read(((PosixFileDescr_Channel)ch)->fd, x+acc, size);
    /* note: for a terminal in canonical input mode the above read will at most
       return a single line, independent of the requested size; I hope that
       SSIZE_MAX is larger than MAX_CANON... */
    if (res >= 0) acc += res;
  } while (((res == -1) && (errno == EINTR)) ||
	   ((res > 0) && (acc < n) && 
	    (ch->buffering != PosixFileDescr_lineBuffer)));
  *bytes_read = acc;
  ch->pos += acc;
  
  /* check error condition */
  if (res == -1) {
    return read_error();
  } else if ((res == 0) && (size != 0)) {
    /* we tried to read behind the end of file */
    return Channel_readAfterEnd;
  } else {
    return Channel_done;
  }
}

static INTEGER write_bytes(PosixFileDescr_Channel ch, LONGINT pos, LONGINT n,
                           BYTE *x, LONGINT *bytes_written) {
  LONGINT size;
  size_t acc, res;
  
  adjust_pos(ch, pos);
  /* write bytes to stream; repeat until all have been written successfully */
  acc = 0;
  do {
    /* make sure that no write request larger than the system limit is 
       issued */
    size = n-acc;
    if (size > SSIZE_MAX) size = SSIZE_MAX;
    res = write(((PosixFileDescr_Channel)ch)->fd, x+acc, size);
    if (res >= 0) acc += res;
  } while (((res == -1) && (errno == EINTR)) ||
           ((res > 0) && (acc < n)));
  *bytes_written = acc;
  ch->pos += acc;
  
  /* check error condition */
  if (res == -1) {
    return write_error();
  } else {
    return Channel_done;
  }
}

static INTEGER flush_buffer(PosixFileDescr_Channel ch) {
  LONGINT bytesWritten;
  
  if ((ch->buffering != PosixFileDescr_noBuffer) && ch->dirty) {
    ch->dirty = 0;
    return write_bytes(ch, ch->bufStart, ch->bufEnd - ch->bufStart, 
                       ch->buf, &bytesWritten);
  } else {
    return Channel_done;
  }
}

void PosixFileDescr_ErrorDescr(short int res, unsigned char* descr, int descr_0d) {
  Channel_ErrorDescr(res, descr, descr_0d);
}

void PosixFileDescr_ReaderDesc_ErrorDescr(PosixFileDescr_Reader r, unsigned char* descr, int descr_0d) {
  PosixFileDescr_ErrorDescr(r->res, descr, descr_0d);
}

void PosixFileDescr_InitReader(PosixFileDescr_Reader r, PosixFileDescr_Channel ch) {
  r->base = (Channel_Channel)ch;
  r->res = Channel_done;
  r->bytesRead = -1;
  r->positionable = ch->positionable;
  r->pos = 0;
  TB_CALL(ch, Channel_ClearError, (void(*)(PosixFileDescr_Channel)), (ch));
  if (!ch->positionable) {
    ch->reader = r;
  }
}

void PosixFileDescr_InitWriter(PosixFileDescr_Writer w, PosixFileDescr_Channel ch) {
  w->base = (Channel_Channel)ch;
  w->res = Channel_done;
  w->bytesWritten = -1;
  w->positionable = ch->positionable && !ch->append;
  w->pos = 0;
  TB_CALL(ch, Channel_ClearError, (void(*)(PosixFileDescr_Channel)), (ch));
  if (!ch->positionable) {
    ch->writer = w;
  }
}



LONGINT PosixFileDescr_ReaderDesc_Pos(PosixFileDescr_Reader r) {
  if (r->positionable) {
    return r->pos;
  } else {
    return Channel_noPosition;
  }
}

int PosixFileDescr_ReaderDesc_Available(PosixFileDescr_Reader r) {
  struct stat stat_buf;
  int res;
  LONGINT len;
  PosixFileDescr_Channel ch = (PosixFileDescr_Channel)r->base;

  res = fstat(ch->fd, &stat_buf);
  
  if ((!r->base->open) || (res == -1)) {
    /* error; assume that channel has been closed */
    return -1;
  } else if (S_ISREG(stat_buf.st_mode)) {
    /* regular file; check position and size */
    len = stat_buf.st_size;
    if ((ch->buffering != PosixFileDescr_noBuffer) &&
	ch->dirty && (ch->bufEnd > len)) {
      return len = ch->bufEnd;
    }

    res = len - r->pos;
    if (res < 0) {
      /* a previous SetPos might have moved the reading position past the end
         of the file; this is no error, put obviously no bytes can be read 
         there -- although this might change if the file is expanded */
      return 0;
    } else {
      return res;
    }
  } else {
    /* something else, like terminal or socket */
    fd_set set;
    struct timeval timeout;
    
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    FD_ZERO(&set);
    FD_SET(ch->fd, &set);
    
    do {
      res = select(FD_SETSIZE, (void*)&set, NULL, NULL, &timeout);
    } while ((res == -1) && (errno == EINTR));
    
    /* res==-1: error; assume that channel has been closed
       res== 0: no input availbale
       res== 1: input available from channel */
    return res;
  }
}

void PosixFileDescr_ReaderDesc_SetPos(PosixFileDescr_Reader r, LONGINT newPos) {
  if (r->res == Channel_done) {
    if (!r->base->open) {
      r->res = Channel_channelClosed;
    } else if ((r->positionable) && (newPos >= 0)) {
      r->pos = newPos;
    } else {
      r->res = Channel_outOfRange;
    }
  }
}

void PosixFileDescr_ReaderDesc_ReadByte(PosixFileDescr_Reader r, BYTE *x) {
  INTEGER res;
  LONGINT bytesRead;
  PosixFileDescr_Channel ch = (PosixFileDescr_Channel)r->base;
      
  if (r->res == Channel_done) {
    if (!r->base->open) {
      r->res = Channel_channelClosed;
      r->bytesRead = 0;

    } else if (ch->buffering == PosixFileDescr_noBuffer) {
      if (ch->dirty) {
	res = flush_buffer(ch);
	if (res != Channel_done) {
	  r->res = res;
	  r->bytesRead = 0;
	  return;
	}
      }
      res = read_bytes(ch, r->pos, 1, x, &r->bytesRead);
      if (res != Channel_done) r->res = res;
      r->pos += r->bytesRead;
      
    } else {
      /* line or block buffering is enabled; check if the byte is currently in
         the buffer; refill if it isn't */
      if ((r->pos < ch->bufStart) || (ch->bufEnd <= r->pos)) {
        /* requested byte isn't in buffer: flush and refill */
        res = flush_buffer(ch);
        if (res == Channel_done) {
          res = read_bytes(ch, r->pos, ch->sizeBuffer, ch->buf, &bytesRead);
          ch->bufStart = r->pos;
          ch->bufEnd = r->pos + bytesRead;
        }
        if ((res != Channel_done) && 
	    ((res != Channel_readAfterEnd) || (bytesRead == 0))) {
          /* reaching the end of the file is acceptable, all other errors have
             to be reported back */
          r->res = res;
          r->bytesRead = 0;
	} else {
	  *x = ch->buf[0];
	  r->pos++;
	  r->bytesRead = 1;
        }
      } else {
	*x = ch->buf[r->pos - ch->bufStart];
	r->pos++;
	r->bytesRead = 1;
      }
    }
  } else {
    r->bytesRead = 0;
  }
}

void PosixFileDescr_ReaderDesc_ReadBytes(PosixFileDescr_Reader r, BYTE* x, int x_0d, LONGINT start, LONGINT n) {
  INTEGER res;
  LONGINT bytesRead, size;
  PosixFileDescr_Channel ch = (PosixFileDescr_Channel)r->base;

  if (r->res == Channel_done) {
    if (!r->base->open) {
      r->res = Channel_channelClosed;
      r->bytesRead = 0;

    } else if (n == 0) {
      /* ignore request for zero bytes, but only after we have made
	 sure that the channel is still open */

    } else if (ch->buffering == PosixFileDescr_noBuffer) {
      if (ch->dirty) {
	res = flush_buffer(ch);
	if (res != Channel_done) {
	  r->res = res;
	  r->bytesRead = 0;
	  return;
	}
      }
      res = read_bytes(ch, r->pos, n, x+start, &r->bytesRead);
      if (res != Channel_done) r->res = res;
      r->pos += r->bytesRead;

    } else {
      /* the following cases deal with block and line buffering */
      if (ch->dirty && (ch->bufStart > r->pos)) {
	/* make sure that any holes we might notice are actually created for
	   the file descriptor */
	res = flush_buffer(ch);
	if (res != Channel_done) {
	  r->res = res;
	  r->bytesRead = 0;
	  return;
	}
      }
      
      if ((ch->bufStart == ch->bufEnd) ||
	  (r->pos+n <= ch->bufStart) ||
	  (r->pos >= ch->bufEnd)) {
	/* the buffer is empty, or the whole requested interval is located in 
	   front or behind the buffer */

	if (ch->buffering == PosixFileDescr_lineBuffer) {
 	  /* we are reading from a terminal in canonical input mode; 
	     successively grab lines until the whole request is fulfilled */
	  r->bytesRead = 0;
	  do {  /* here holds: n>0, buffer is empty */
	    /* get a single line of input into the buffer */
	    res = read_bytes(ch, r->pos, ch->sizeBuffer, ch->buf, &bytesRead);
	    if (res != Channel_done) r->res = res;
	    ch->bufStart = r->pos;
	    ch->bufEnd = r->pos + bytesRead;
	  
	    /* copy available bytes from buffer to x+start */
	    size = n;
	    if (size > bytesRead) size = bytesRead;
	    (void)memcpy(x+start, ch->buf, size);
	    start += size;
	    n -= size;
	    r->bytesRead += size;
	    r->pos += size;
	  } while ((res == Channel_done) && (n > 0));
	    
	} else if (n >= ch->sizeBuffer) {
	  /* the requested block is larger than the buffer, so don't bother
	     filling it and transfer bytes directly to x+start*/
	  res = read_bytes(ch, r->pos, n, x+start, &r->bytesRead);
	  if (res != Channel_done) r->res = res;
	  r->pos += r->bytesRead;
	  
	} else {
	  /* fill buffer with file contents starting at r->pos */
	  res = flush_buffer(ch);
	  if (res == Channel_done) {
	    res = read_bytes(ch, r->pos, ch->sizeBuffer, ch->buf, &bytesRead);
	    ch->bufStart = r->pos;
	    ch->bufEnd = r->pos + bytesRead;
	    
	    /* copy available bytes from buffer to x+start */
	    size = n;
	    if (size > bytesRead) size = bytesRead;
	    (void)memcpy(x+start, ch->buf, size);
	    r->bytesRead = size;
	    r->pos += size;
	  } else {
	    r->bytesRead = 0;
	  }
          
	  if ((res != Channel_done) && 
	      ((res != Channel_readAfterEnd) || (r->bytesRead < n))) {
	    r->res = res;
	  }
	}
	
      } else {
	/* the intersection of the requested and the buffered file interval 
	   isn't empty; first read the bytes in front of the buffer, then copy
	   the bytes from the buffer, and finally fill in the rest of the 
	   request */
	if (r->pos < ch->bufStart) {
	  res = read_bytes(ch, r->pos, ch->bufStart - r->pos, x+start, 
			   &r->bytesRead);
	  r->pos += r->bytesRead;
	  if (res != Channel_done) {
	    r->res = res;
	    return;
	  } else {
	    n -= r->bytesRead;
	  }
	} else {
	  r->bytesRead = 0;
	}

	size = ch->bufEnd - r->pos;
	if (size > n) size = n;
	(void)memcpy(x + start + r->bytesRead, 
		     ch->buf + r->pos - ch->bufStart, size);
	r->bytesRead += size;
	r->pos += size;
	n -= size;
	
	if (n > 0) {
	  size = r->bytesRead;
	  PosixFileDescr_ReaderDesc_ReadBytes(r, x, x_0d, start+size, n);
	  r->bytesRead += size;
	}
      }
    }
  } else {
    r->bytesRead = 0;
  }
}



void PosixFileDescr_WriterDesc_ErrorDescr(PosixFileDescr_Writer w, unsigned char* descr, int descr_0d) {
  PosixFileDescr_ErrorDescr(w->res, descr, descr_0d);
}

int PosixFileDescr_WriterDesc_Pos(PosixFileDescr_Writer w) {
  if (w->positionable) {
    return w->pos;
  } else {
    return Channel_noPosition;
  }
}

void PosixFileDescr_WriterDesc_SetPos(PosixFileDescr_Writer w, LONGINT newPos) {
  if (w->res == Channel_done) {
    if (!w->base->open) {
      w->res = Channel_channelClosed;
    } else if ((w->positionable) && (newPos >= 0)) {
      w->pos = newPos;
    } else {
      w->res = Channel_outOfRange;
    }
  }
}

void PosixFileDescr_WriterDesc_WriteByte(PosixFileDescr_Writer w, BYTE x) {
  INTEGER res;
  PosixFileDescr_Channel ch = (PosixFileDescr_Channel)w->base;
  
  if (w->res == Channel_done) {
    if (!w->base->open) {
      w->res = Channel_channelClosed;
      w->bytesWritten = 0;
    
    } else if (ch->buffering == PosixFileDescr_noBuffer) {
      res = write_bytes(ch, w->pos, 1, &x, &w->bytesWritten);
      if (res != Channel_done) w->res = res;
      w->pos += w->bytesWritten;
      
    } else {                    /* buffering is enabled */
      if (ch->dirty && 
          ((w->pos < ch->bufStart) || 
           (w->pos > ch->bufEnd) ||
           ((w->pos == ch->bufEnd) && 
            (ch->bufEnd - ch->bufStart == ch->sizeBuffer)))) {
        /* current buffer is dirty, and the new byte can't be added to it;
           flush it to make room for next output */
        res = flush_buffer(ch);
        if (res != Channel_done) {
          w->res = res;
          w->bytesWritten = 0;
          return;
        }
      }
        
      if (!ch->dirty) {  /* convert buffer from read to write state */
        ch->bufStart = w->pos;
        ch->bufEnd = w->pos+1;
	ch->dirty = 1;
      } else if (w->pos == ch->bufEnd) { /* add to end of buffer */
        ch->bufEnd++;
      }
      ch->buf[w->pos - ch->bufStart] = x;
      w->pos++;
      w->bytesWritten = 1;

      if ((ch->buffering == PosixFileDescr_lineBuffer) &&
          ((CHAR)x == CharClass_eol)) {
        res = flush_buffer(ch);
        if (res != Channel_done) {
          w->res = res;
          w->bytesWritten = 0;
        }
      }
    }
  } else {
    w->bytesWritten = 0;
  }
}

static INTEGER flush_lines (PosixFileDescr_Channel ch, LONGINT start, 
                            LONGINT end) {
  LONGINT i;

  i = start;
  while ((i < end) && (ch->buf[i] != CharClass_eol)) {
    i++;
  }
  if (i != end) {
    return flush_buffer(ch);
  } else {
    return Channel_done;
  }
}

void PosixFileDescr_WriterDesc_WriteBytes(PosixFileDescr_Writer w, BYTE* x, int x_0d, LONGINT start, LONGINT n) {
  INTEGER res;
  LONGINT size, s, e;
  PosixFileDescr_Channel ch = (PosixFileDescr_Channel)w->base;

  if (w->res == Channel_done) {
    if (!w->base->open) {
      w->res = Channel_channelClosed;
      w->bytesWritten = 0;

    } else if (ch->buffering == PosixFileDescr_noBuffer) {
      res = write_bytes(ch, w->pos, n, x+start, &w->bytesWritten);
      if (res != Channel_done) w->res = res;
      w->pos += w->bytesWritten;
      
    } else if (!ch->dirty ||
               (w->pos+n <= ch->bufStart) ||
               (w->pos > ch->bufEnd) ||
               ((w->pos == ch->bufEnd) && 
		(ch->bufEnd - ch->bufStart == ch->sizeBuffer))) {
      /* the buffer contains no written data, or the whole requested interval 
         is located in front or behind the buffer, or the buffer is full */
      res = flush_buffer(ch);
      if (res != Channel_done) {
        w->res = res;
        w->bytesWritten = 0;
        return;
      }
      
      if (n >= ch->sizeBuffer) {
        /* the written block is larger than the buffer, so don't bother
           filling it and transfer bytes directly from x+start*/
        res = write_bytes(ch, w->pos, n, x+start, &w->bytesWritten);
        if (res != Channel_done) w->res = res;
	/* determine the intersection between buffer and write request */
	s = ch->bufStart;
	if (w->pos > s) s = w->pos;
	e = ch->bufEnd;
	if (w->pos + w->bytesWritten < e) e = w->pos + w->bytesWritten;
	if (s < e) {
	  /* someone was reading in the area we have just overwritten; update 
	     buffer contents in intersection of buffer and write request 
	     instead of invalidating the buffer */
	  (void)memcpy(ch->buf + (s - ch->bufStart),
		       x + start + (s - w->pos), e - s);
	}
        w->pos += w->bytesWritten;

      } else {
        /* copy bytes into buffer */
        (void)memcpy(ch->buf, x+start, n);
        ch->bufStart = w->pos;
        ch->bufEnd = w->pos + n;
        ch->dirty = 1;
        w->bytesWritten = n;
        w->pos += n;
	if (ch->buffering == PosixFileDescr_lineBuffer) {
	  res = flush_lines(ch, 0, n);
	  if (res != Channel_done) {
	    w->res = res;
	    w->bytesWritten = 0;
	  }
	}
      }
    } else {
      /* the intersection of the written interval and the buffered file 
         interval isn't empty, or the new data extends the current buffer;
         first write the bytes in front of the buffer, then extend the buffer,
         and finally write the bytes after the buffer */
      if (w->pos < ch->bufStart) {
        res = write_bytes(ch, w->pos, ch->bufStart - w->pos, x+start, 
                          &w->bytesWritten);
        w->pos += w->bytesWritten;
        if (res != Channel_done) {
          w->res = res;
          return;
        } else {
          n -= w->bytesWritten;
        }
      } else {
        w->bytesWritten = 0;
      }
      
      s = w->pos - ch->bufStart;
      size = ch->sizeBuffer - s;
      if (size > n) size = n;
      (void)memcpy(ch->buf + s, x + start + w->bytesWritten, size);
      if (w->pos + size > ch->bufEnd) {
	ch->bufEnd = w->pos + size;
      }
      w->bytesWritten += size;
      w->pos += size;
      n -= size;

      if (n > 0) {
        size = w->bytesWritten;
        PosixFileDescr_WriterDesc_WriteBytes(w, x, x_0d, start+size, n);
	if (w->res == Channel_done) w->bytesWritten += size;
      } else if (ch->buffering == PosixFileDescr_lineBuffer) {
	res = flush_lines(ch, s, size);
	if (res != Channel_done) {
	  w->res = res;
	  w->bytesWritten = 0;
	}
      }
    }
  } else {
    w->bytesWritten = 0;
  }
}



LONGINT PosixFileDescr_ChannelDesc_Length(PosixFileDescr_Channel ch) {
  int res;
  struct stat stat_buf;
  LONGINT len;

  res = fstat(ch->fd, &stat_buf);
  if (res == -1) {
    return Channel_noLength;
  } else {
    len = stat_buf.st_size;
    if ((ch->buffering != PosixFileDescr_noBuffer) &&
	ch->dirty && (ch->bufEnd > len)) {
      return ch->bufEnd;
    } else {
      return len;
    }
  }
}

/* define the day count of the Unix epoch (Jan 1 1970 00:00:00 GMT) for the
   Time.TimeStamp format */
#define days_to_epoch 40587
#define secs_per_day 86400

void PosixFileDescr_ChannelDesc_GetModTime(PosixFileDescr_Channel ch, Time_TimeStamp *mtime, _Type mtime__tag) {
  int res;
  struct stat stat_buf;

  res = fstat(ch->fd, &stat_buf);
  if (res == -1) {
    ch->res = Channel_noModTime;
  } else {
    mtime->days = days_to_epoch + stat_buf.st_mtime / secs_per_day;
    mtime->msecs = (stat_buf.st_mtime % secs_per_day) * 1000;
#if HAVE_ST_MTIME_USEC
    mtime->msecs += (stat_buf.st_mtime_usec / 1000);
#endif
    ch->res = Channel_done;
  }
}

PosixFileDescr_Reader PosixFileDescr_ChannelDesc_NewReader(PosixFileDescr_Channel ch) {
  PosixFileDescr_Reader r = NULL;

  if (!ch->open) {
    ch->res = Channel_channelClosed;
  } else if (ch->readable) {
    if (ch->positionable || (ch->reader == NULL)) {
      NEW_REC(r, PosixFileDescr_ReaderDesc);
      PosixFileDescr_InitReader (r, ch);
    } else {
      /* channel doesn't support multiple readers, so just return the 
         one previously created */
      r = ch->reader;
    }
  } else {
    ch->res = Channel_noReadAccess;
  }

  return r;
}

PosixFileDescr_Writer PosixFileDescr_ChannelDesc_NewWriter(PosixFileDescr_Channel ch) {
  PosixFileDescr_Writer w = NULL;

  if (!ch->open) {
    ch->res = Channel_channelClosed;
  } else if (ch->writable) {
    if (ch->positionable || (ch->writer == NULL)) {
      NEW_REC(w, PosixFileDescr_WriterDesc);
      PosixFileDescr_InitWriter (w, ch);
    } else {
      /* channel doesn't support multiple writers, so just return the 
         one previously created */
      w = ch->writer;
    }
  } else {
    ch->res = Channel_noWriteAccess;
  }

  return w;
}

void PosixFileDescr_ChannelDesc_Flush(PosixFileDescr_Channel ch) {
  ch->res = flush_buffer(ch);
  if (ch->buffering == PosixFileDescr_blockBuffer) {
    /* invalidate whole buffer to force next read operation to get the
       data from the OS; of course this wont work for unbuffered input or
       when reading lines from a terminal in canonical input mode */
    ch->bufEnd = ch->bufStart;
  }
}

void PosixFileDescr_ChannelDesc_Close(PosixFileDescr_Channel ch) {
  int res;

  /* flush the channel; this may be an upcall */
  TB_CALL(ch, Channel_Flush, (void(*)(PosixFileDescr_Channel)), (ch));

  /* close the file descriptor; try again if the primitive is
     interrupted by signal */
  do {
    res = close(ch->fd);
  } while ((res == -1) && (errno == EINTR));
  if (ch->fd <= PosixFileDescr_stderrFileno) { 
    /* this fd isn't used for standard IO anymore */
    standard_io[ch->fd] = 0;
  }
  ch->fd = -1;

  /* only put a close error into ch->res if the flush succeeded; 
     otherwise keep the old error indication */
  if ((res == -1) && (ch->res == Channel_done)) {
    if (errno == EBADF) {
      ch->res = Channel_invalidChannel;
    } else if (errno == ENOSPC) {
      ch->res = Channel_noRoom;
#ifdef EDQUOT
    } else if (errno == EDQUOT) {
      ch->res = Channel_noRoom;
#endif
    } else {
      ch->res = Channel_writeError;
    }
  }

  /* free buffer */
  if (ch->buf) {
    GC_free(ch->buf);
    ch->buf = NULL;
  }

  /* mark channel as closed */
  ch->open = 0;
}

void PosixFileDescr_Init(PosixFileDescr_Channel ch, int fd, SHORTINT mode) {
/* Initializes channel `ch' to use file descriptor `fd' and access rights 
   `mode'.  If `fd' is a file, block buffering is enabled; if it's a terminal,
   output is line buffered; otherwise no buffering is applied.
   The standard file descriptors that were passed from the shell are handled
   specially: positioning is disabled, and stderr is never buffered.  */
  struct stat stat_buf;
  int size;

  ch->fd = fd;
  ch->pos = lseek(fd, 0, SEEK_CUR);
  ch->positionable = (ch->pos != -1);
  ch->append = 0;
  ch->dirty = 0;
  ch->bufStart = 0;
  ch->bufEnd = 0;
  ch->reader = NULL;
  ch->writer = NULL;

  /* assume that this call never fails; otherwise someone handed us a bad file
     descriptor, which is forbidden :-) */
  (void)fstat(fd, &stat_buf);
  
  /* decide which buffering to use, get memory for buffer */
  if (isatty(fd)) {
    /* do line buffering for anything that is connected to a terminal; 
       canonical input mode is assumed when reading from a terminal; likewise
       it is assumed that positioning is not possible */
    ch->buffering = PosixFileDescr_lineBuffer;
  } else if (S_ISREG(stat_buf.st_mode)) {
    /* files are buffered on a per block basis; this can only work for files,
       since other input types would block when filling the buffer */
    ch->buffering = PosixFileDescr_blockBuffer;
  } else {  /* the conservative approach: don't buffer unknown fd */
    ch->buffering = PosixFileDescr_noBuffer;
  }
  
  /* handle standard file descriptors: no positioning and don't buffer 
     stderr */
  if (((fd == PosixFileDescr_stdoutFileno) || 
       (fd == PosixFileDescr_stdinFileno)) && standard_io[fd]) {
    ch->positionable = 0;
  } else if ((fd == PosixFileDescr_stderrFileno) && standard_io[fd]) {
    ch->positionable = 0;
    ch->buffering = PosixFileDescr_noBuffer;
  }
  
  if (ch->buffering != PosixFileDescr_noBuffer) {
#if HAVE_ST_BLKSIZE
    size = stat_buf.st_blksize;
    if (size < 1024) {  /* impose a lower and upper limit on block size */
      size = 1024;
    } else if (size > 8192) {
      size = 8192;
    }
#else
    size = 2048;
#endif
    /* in any case `size should be sufficiently large compared with 
       MAX_CANON */
    ch->buf = GC_malloc_atomic(size);
    ch->sizeBuffer = size;
  } else {
    ch->buf = NULL;
    ch->sizeBuffer = 0;
  }

  TB_CALL(ch, Channel_ClearError, (void(*)(PosixFileDescr_Channel)), (ch));
  ch->readable = (mode == PosixFileDescr_readOnly) || (mode == PosixFileDescr_readWrite);
  ch->writable = (mode == PosixFileDescr_writeOnly) || (mode == PosixFileDescr_readWrite);
  ch->open = 1;
}

void PosixFileDescr_Truncate(PosixFileDescr_Writer w, int newLength) {
  int res;

  if (w->res == Channel_done) {
    if (!w->base->open) {
      w->res = Channel_channelClosed;
    } else {
      PosixFileDescr_Channel ch = (PosixFileDescr_Channel)w->base;
      do {
	/* mh, ftruncate is neither ANSI nor POSIX; if there is a system out
           there that doesn't support it we need to extend configure to check
	   for its presence and provide alternative code... --mva */
        res = ftruncate(ch->fd, newLength);
      } while ((res == -1) && (errno == EINTR));
      
      if (res == -1) {
        w->res = write_error();
      } else if (ch->bufEnd > newLength) {
        /* truncate buffer */
        if (ch->bufStart >= newLength) {
          ch->bufEnd = ch->bufStart; /* empty interval means empty buffer */
        } else {
          ch->bufEnd = newLength;
        }
      }
    }
  }
}

void PosixFileDescr__init(void) {
  _mid = _register_module(&PosixFileDescr__md.md, &PosixFileDescr_WriterDesc__td.td);
}
