/*
 *  $Id: dmpi.c,v 1.37 1996/11/24 20:19:11 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      See COPYRIGHT in top-level directory.
 */

/*  dmpi.c - routines in mpir that are called by the device */

#include "dmpi.h"
#include "mpiimpl.h"
#ifndef MPI_ADI2
#include "mpisys.h"
#endif

#define MPIR_MIN(a,b) (a) < (b) ? (a) : (b)
#define DEVICE_PREFERS_MEMCPY 1

#ifndef MPI_ADI2

/* called by device when a message arrives.  Returns 1 if there is posted
 * receive, 0 otherwise.
 *
 * This puts the responsibility for searching the unexpected queue and
 * posted-receive queues on the device.  If it is operating asynchronously
 * with the user code, the device must provide the necessary locking mechanism.
 */
void DMPI_msg_arrived( src, tag, context_id, dmpi_recv_handle, foundflag )
int               src, tag, *foundflag;
MPIR_CONTEXT      context_id;
MPIR_RHANDLE      **dmpi_recv_handle;
{
    int          found;
    MPIR_RHANDLE *handleptr;

    MPID_THREAD_DS_LOCK(&MPIR_posted_recvs)
    MPIR_search_posted_queue( src, tag, context_id, &found, 1, 
			      dmpi_recv_handle);
    MPID_THREAD_DS_UNLOCK(&MPIR_posted_recvs);
    if ( found )
    {
	*foundflag = 1;	
	/* note this overwrites any wild-card values in the posted handle */
	handleptr         	= *dmpi_recv_handle;
	handleptr->source 	= src;
	handleptr->tag  	= tag;
	/* count is set in the put and get routines */
    }
    else
    {
	/* allocate handle and put in unexpected queue */
	handleptr = ( MPIR_RHANDLE * ) MPIR_SBalloc ( MPIR_rhandles );
	if (!handleptr) {
	    MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, 
		        "Could not dynamically allocate internal handle" );
	    MPI_Abort( MPI_COMM_WORLD, 1 );
	}
	*dmpi_recv_handle       = handleptr;
	handleptr->handle_type  = MPIR_RECV;
	handleptr->source 	= src;
	handleptr->tag  	= tag;
	handleptr->contextid    = context_id;
	handleptr->datatype     = MPI_BYTE;
	DMPI_Clr_recv_completed( handleptr );
	MPID_Alloc_recv_handle( handleptr->comm->ADIctx,
			        &(handleptr->dev_rhandle) );
	
	MPID_THREAD_DS_LOCK(&MPIR_unexpected_recvs);	
	MPIR_enqueue( &MPIR_unexpected_recvs, (void * ) *dmpi_recv_handle,
                      MPIR_QRHANDLE );
	MPID_THREAD_DS_UNLOCK(&MPIR_unexpected_recvs);
	*foundflag = 0;
    }
}


#ifdef MPID_HAS_HETERO
/* Returns MPIR_MSGFORM_XDR if the data needs XDR conversion,
           MPIR_MSGFORM_SWAP if the data needs byteswapping,
	   MPIR_MSGFORM_OK if none of the above... 
   This is also used for receives, where "dest" is source, and may 
   be MPI_ANY_SOURCE, in which case the "worst-case" for the communicator
   is chosen (XDR if there is any heterogeneity in the communicator).

   dest is the RELATIVE source (i.e., the normal part of the
   (comm,dest) pair)
*/
int MPIR_Dest_needs_conversion(comm,dest)
MPI_Comm comm;
int      dest;
{
  /* Use the form that matches the communicator (this allows homogenous
     communicators in an inhomogenous MPI_COMM_WORLD).
     In this case (sending to an unknown destination, including possibly
     ourselves), we must use a "sender-side" or "cannonical-form" 
     representations.  Since we don't yet have "sender-side", choose the
     cannonical form, which is XDR.

     This is for ADI-1 only.
   */
  if (dest < 0) {
      if (comm->msgrep != 0) 
	  return MPIR_MSGFORM_XDR;
      return MPIR_MSGFORM_OK;
      }

  /* We must convert dest to the GLOBAL id */
  dest = comm->group->lrank_to_grank[dest];
  /* Now, test the byte ordering */
  if ( (MPID_Dest_byte_order(MPIR_tid) == MPID_H_XDR) ||
       (MPID_Dest_byte_order(dest) == MPID_H_XDR))
      return MPIR_MSGFORM_XDR;
  else if (MPID_Dest_byte_order(MPIR_tid) != MPID_Dest_byte_order(dest))
      /* Sender will swap in this case */
      return MPIR_MSGFORM_SWAP;
  else
      return MPIR_MSGFORM_OK;
}


int 
MPIR_Comm_needs_conversion(comm)
MPI_Comm comm;
{
  if (comm->msgrep) return MPIR_MSGFORM_XDR;
#ifdef FOO
  for (i = 0; i < comm->local_group->np; i++) {
    if (MPIR_Dest_needs_conversion(comm,comm->local_group->lrank_to_grank[i]))
      return 2;
  }
#endif
  return 0;
}
#endif



/*
   Let the device tell the API that a handle can be freed (this handle
   was generated by an unexpected receive and inserted by DMPI_msg_arrived.
 */
void DMPI_free_unexpected( dmpi_recv_handle  )
MPIR_RHANDLE      *dmpi_recv_handle;
{
MPID_Free_recv_handle( dmpi_recv_handle->comm->ADIctx, 
		       &dmpi_recv_handle->dev_rhandle );
MPIR_SBfree( MPIR_rhandles, dmpi_recv_handle );
}

/*
   This sets up the buffers for a send.  The logic is:
   if can use in place, setup buffer for that, 
   else use MPI_PACK.

   Note that in the homogeneous case with contiguous data, we can inline
   this routine and the receive counterpart.  It can always handle 
   contiguous messages, but may be bypassed by the macro below.

   A macro of the form (passed the actual shandle structure) could be used:
   MPIR_Send_setup_contig( request, shandle ) \
   if (shandle.datatype->is_contig) {\
   shandle.active       = 1;\
   shandle.dev_shandle.bytes_as_contig =\
   shandle.count * shandle.datatype->size;\
   if (shandle.dev_shandle.bytes_as_contig > 0 && shandle.bufadd == 0)\
       mpi_errno = MPI_ERR_BUFFER;\
   shandle.dev_shandle.start = shandle.bufadd;\
   shandle.bufpos		 = 0;}\
   else{mpi_errno = MPIR_Send_setup( request );} 

 */
int MPIR_Send_setup(request)
MPI_Request *request;
{
  register MPIR_SHANDLE *shandle;
  register int mpi_errno = MPI_SUCCESS;
  int dest_type = 0;
  struct MPIR_DATATYPE *dtype_ptr;

  shandle = &(*request)->shandle;
  if (shandle->dest == MPI_PROC_NULL) return mpi_errno;

  shandle->active       = 1;

/* 
  Even though there is some overlap in the code, it is clearer to have 
  separate heterogeneous and uniform sections.
 */

/* First, packed data.  This is a spacial case, because the data format is
   that set in comm->msgrep */

dtype_ptr   = MPIR_GET_DTYPE_PTR(shandle->datatype);
if (dtype_ptr->dte_type == MPIR_PACKED) {
    shandle->msgrep = shandle->comm->msgrep;
    shandle->dev_shandle.bytes_as_contig =
	shandle->count * dtype_ptr->size;
    if (shandle->dev_shandle.bytes_as_contig > 0 && shandle->bufadd == 0)
	mpi_errno = MPI_ERR_BUFFER;
    shandle->dev_shandle.start = shandle->bufadd;
    shandle->bufpos		 = 0;
    return mpi_errno;
    }

/*
 * Handle all of the other datatypes here.
 */
#ifdef MPID_HAS_HETERO
  /* Get the dest type if we might be heterogeneous */
  if (MPID_IS_HETERO == 1)
      dest_type = MPIR_Dest_needs_conversion(shandle->comm,shandle->dest);
  
  if (dest_type || !dtype_ptr->is_contig) {
      /* We need to do some conversion.  In both the contiguous and
	 non-contiguous cases, we need to allocate space and call
	 the packing routines.  
       */
      if ((mpi_errno = 
	  MPIR_PackMessage(shandle->bufadd, shandle->count, 
			   shandle->datatype, shandle->dest, dest_type, 
			   *request ))) {
	  MPIR_ERROR( MPI_COMM_WORLD, mpi_errno, 
		     "Could not pack message in MPIR_Send_setup" );
	  }
      }
  else {
      /* Contiguous and homogeneous */
      shandle->msgrep = MPIR_MSGREP_RECEIVER; 
      shandle->dev_shandle.bytes_as_contig =
	  shandle->count * dtype_ptr->size;
      if (shandle->dev_shandle.bytes_as_contig > 0 && shandle->bufadd == 0)
	  mpi_errno = MPI_ERR_BUFFER;
      shandle->dev_shandle.start = shandle->bufadd;
      shandle->bufpos		 = 0;
      }
#else
  /* NOT heterogeneous */
  if (dtype_ptr->is_contig) {
      /* Contiguous, no conversion form */
      /* This SHOULD have been handled in the macro that expands into
	 a call to this routine */
      shandle->dev_shandle.bytes_as_contig =
	shandle->count * dtype_ptr->size;
      if (shandle->dev_shandle.bytes_as_contig > 0 && shandle->bufadd == 0)
	  mpi_errno = MPI_ERR_BUFFER;
      shandle->dev_shandle.start = shandle->bufadd;
      shandle->bufpos		 = 0;
    }
  else {
      /* non-contiguous message */
      /* This used to be conditional on MPID_PACK_IN_ADVANCE */
      if (mpi_errno = 
	  MPIR_PackMessage(shandle->bufadd, shandle->count, 
			   shandle->datatype, shandle->dest, dest_type, 
			   *request )) {
	  MPIR_ERROR( MPI_COMM_WORLD, mpi_errno, 
		     "Could not pack message in MPIR_Send_setup" );
	  }
      }

#endif /* MPID_HAS_HETERO */
  return mpi_errno;
}


int 
MPIR_SendBufferFree( request )
MPI_Request request;
{
  FREE( request->chandle.bufpos );
  request->chandle.bufpos = 0;
  return MPI_SUCCESS;
}


/* 
   This handles setting up a receive request; primarily handling the
   allocation of a receive buffer for aggressive receives.

   NOTE: If the rendevous protocol is used, then in the heterogeneous 
   case, we don't need to do this aggressive (in space) buffer allocation.
 */
int 
MPIR_Receive_setup(request)
MPI_Request *request;
{
  struct MPIR_DATATYPE *dtype_ptr;
  MPIR_RHANDLE *rhandle;
  int mpi_errno = MPI_SUCCESS;
  int dest_type = 0;

  rhandle = &(*request)->rhandle;
  if (rhandle->source == MPI_PROC_NULL) return mpi_errno;
  rhandle->active       = 1;

  /* Just as for the send setup, we separate the heterogeneous and 
     homogeneous cases for simplicity.  
   */

  /* Even for contiguous data, if heterogeneous, we may need to 
     allocate a larger buffer ... */

  dtype_ptr   = MPIR_GET_DTYPE_PTR(rhandle->datatype);

#if defined(MPID_HAS_HETERO)
  if (MPID_IS_HETERO == 1)
      dest_type = MPIR_Dest_needs_conversion(rhandle->comm,rhandle->source);
  if (dest_type || !dtype_ptr->is_contig) {
      /* This is OK but not optimal, as this will use MPI_Pack_size
	 to allocate the buffer; pack_size will use the most
	 pessimistic value */
      if ((mpi_errno = 
	  MPIR_SetupUnPackMessage( rhandle->bufadd, rhandle->count, 
				   rhandle->datatype, rhandle->source, 
				   *request ))) {
	  MPIR_ERROR( MPI_COMM_WORLD, mpi_errno, 
	     "Could not setup unpack area for message in MPIR_Receive_setup" );
	  }
      }
  else {
      /* Contiguous and homogeneous */
      rhandle->dev_rhandle.start = rhandle->bufadd;
      rhandle->dev_rhandle.bytes_as_contig =
	  rhandle->count * dtype_ptr->extent;
      if (rhandle->dev_rhandle.bytes_as_contig > 0 && 
	  rhandle->bufadd == 0) 
	  mpi_errno = MPI_ERR_BUFFER;
      rhandle->bufpos                      = 0;
      }
#else
  if (dtype_ptr->is_contig) {
      rhandle->dev_rhandle.start = rhandle->bufadd;
      rhandle->dev_rhandle.bytes_as_contig =
	  rhandle->count * dtype_ptr->extent;
      if (rhandle->dev_rhandle.bytes_as_contig > 0 && 
	  rhandle->bufadd == 0) 
	  mpi_errno = MPI_ERR_BUFFER;
      rhandle->bufpos                      = 0;
      }
  else {
      /* This used to test for MPID_RETURN_PACKED */
      if (mpi_errno = 
	  MPIR_SetupUnPackMessage( rhandle->bufadd, rhandle->count, 
				   rhandle->datatype, rhandle->source, 
				   *request )) {
	  MPIR_ERROR( MPI_COMM_WORLD, mpi_errno, 
             "Could not setup unpack area for message in MPIR_Receive_setup" );
	  }
      }
#endif

#ifdef FOO
  if (dtype_ptr->is_contig) {
#if defined(MPID_HAS_HETERO)
    if (MPID_IS_HETERO == 1 && 
     (dest_type = MPIR_Dest_needs_conversion(rhandle->comm,rhandle->source))) {
      /* This is a heterogeneous case - we MAY need a longer buffer, 
	 depending on whether XDR may be used in this communicator 
	 and whether the source is known... */
	if (rhandle->count > 0 && rhandle->bufadd == 0) 
	    return MPI_ERR_BUFFER;
	rhandle->dev_rhandle.bytes_as_contig =
	  MPIR_Mem_convert_len( dest_type, rhandle->datatype, rhandle->count );
	/* This is needed only if a non-rendevous, non-packetized approach
	   is used */
	if (rhandle->dev_rhandle.bytes_as_contig) {
	    rhandle->bufpos = 
		(char *)MALLOC(rhandle->dev_rhandle.bytes_as_contig);
	    if (!rhandle->bufpos) 
		return MPI_ERR_EXHAUSTED;
	    }
	else
	    rhandle->bufpos = 0;
	rhandle->dev_rhandle.start = rhandle->bufpos;
	/* PRINTF( "Receive buffer size is %d for count of %d\n", 
	         rhandle->dev_rhandle.bytes_as_contig, rhandle->count ); */
    } else 
#endif
#endif

  return mpi_errno;
}
#endif
