/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	GDB
 *
 *	$Id: tkill.c,v 6.1 96/11/23 19:37:42 nevin Rel $
 *
 *	Function:	- kills OTB environment
 *			- reads process IDs from the kill file, aka
 *			  reset file, aka lock file for historical reasons
 *			- makes a very good attempt at ensuring the
 *			  death of each pid in the file
 */

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <args.h>
#include <debug.h>
#include <terror.h>

#define TIMEOUT		10000000	/* timeout on process death */

/*
 * external functions
 */
extern void		microsleep();
extern char		*killname();
extern char		*sockname();
extern char		*iosockname();
extern void		_rpi_mop_up();

/*
 * local functions
 */
static void		help();


int
main(argc, argv)

int			argc;
char			*argv[];

{
	FILE		*fp_kill;	/* kill file ptr */
	int		fd_kill;	/* kill file desc. */
	int		fl_debug;	/* debugging option */
	int		fl_pretend;	/* pretend option */
	int		fl_verbose;	/* verbose option */
	int		n;		/* favourite counter */
	int		pid;		/* lots of space to hold a PID */
	unsigned int	usec;		/* microseconds to sleep */
	char		*f_kill;	/* kill file */
	char		*f_sock;	/* socket file */
	char		*f_iosock;	/* io daemon socket file */
	char		syscmd[256];	/* system command */
/*
 * Initialize option parser.
 */
	validopts("dhvN");
	followed("f");

	if ((do_args(&argc, argv)) || (errno = (argc == 1) ? 0 : EUSAGE)) {
	    fprintf(stderr, "tkill [-hdvN] [-f <killfile>]\n");
	    exit(errno);
	}

	if (opt_taken('h')) {
	    help();
	    exit(0);
	}

	fl_debug = opt_taken('d');
	fl_pretend = opt_taken('N');
	fl_verbose = opt_taken('v') || fl_debug;
/*
 * Remove the socket file.
 */
	DBUG("tkill: removing socket file ...\n");
	f_sock = sockname();

	if (f_sock == 0) {
	    perror("tkill (sockname)");
	    exit(errno);
	}

	if (!fl_pretend) {
	    if (unlink(f_sock)) {
		if (errno != ENOENT) {
		    fprintf(stderr, "tkill: cannot unlink \"%s\": ", f_sock);
		    perror((char *) 0);
		}
	    }
	}
/*
 * Remove the IO daemon socket file.
 */
	DBUG("tkill: removing IO daemon socket file ...\n");
	f_iosock = iosockname();

	if (f_iosock == 0) {
	    perror("tkill (iosockname)");
	    exit(errno);
	}

	if (!fl_pretend) {
	    if (unlink(f_iosock)) {
		if (errno != ENOENT) {
		    fprintf(stderr, "tkill: cannot unlink \"%s\": ", f_iosock);
		    perror((char *) 0);
		}
	    }
	}
/*
 * Get the kill filename.
 */
	if (opt_taken('f')) {
	    f_kill = getparam('f');
	} else {
	    f_kill = killname();

	    if (f_kill == 0) {
		perror("tkill (killname)");
		exit(errno);
	    }
	}
/*
 * Dump debugging information.
 */
	DBUG("tkill: f_kill = \"%s\"\n", f_kill);
/*
 * Try to open the file.
 */
	if ((fd_kill = open(f_kill, O_RDWR, 0)) < 0) {

	    if (errno == ENOENT) {
		VERBOSE("tkill: nothing to kill: \"%s\"\n", f_kill);
		exit(0);
	    } else {
		fprintf(stderr, "tkill: can't access kill file \"%s\": ",
			f_kill);
		perror("");
		exit(errno);
	    }
	}

	fp_kill = fdopen(fd_kill, "r");

	if (! fp_kill) {
	    perror("tkill (fdopen)");
	    exit(errno);
	}

	VERBOSE("kill ...\n");

	n = fscanf(fp_kill, "%d", &pid);

	while (n > 0) {
	    DBUG("tkill: %d ...", pid);

	    if (fl_debug) fflush(stdout);
/*
 * Send SIGHUP to the process.
 */
	    if (fl_pretend) {
		DBUG("\n");
		n = fscanf(fp_kill, "%d", &pid);
		continue;
	    }

	    if (kill(pid, SIGHUP)) {

		if (errno == ESRCH) {
		    DBUG(" already dead");
		} else {
		    DBUG("\n");
		    perror("tkill (kill)");
		    exit(errno);
		}
	    } else {
		usec = 2;

		while ((kill(pid, SIGHUP) == 0) && (usec <= TIMEOUT)) {
		    microsleep(usec);
		    usec *= 2;
		}
	
		if (usec > TIMEOUT) {
		    DBUG(" trying -9 ...");
		    kill(pid, SIGKILL);
		    microsleep(500000);

		    if (kill(pid, SIGHUP)) {
			DBUG(" killed");
		    } else {
			DBUG(" cannot kill");

			if (!fl_debug) {
			    fprintf(stderr,
	"tkill: cannot kill pid %d, use ps(1) and kill(1)\n",
				   pid);
			}
		    }
		} else {
		    DBUG(" killed");
		}
	    }
		
	    DBUG("\n");
	    n = fscanf(fp_kill, "%d", &pid);
	}

	if (fl_pretend) {
	    exit(0);
	}
/*
 * Remove the kill file.
 */
	DBUG("tkill: removing kill file ...\n");

	if (unlink(f_kill)) {
	    fprintf(stderr, "tkill: cannot unlink \"%s\": ", f_kill);
	    perror((char *) 0);
	    exit(errno);
	}
/*
 * Mop up any persistent MPI RPI state.
 */
	_rpi_mop_up();
/*
 * Delete any stray temporary files.
 */
	sprintf(syscmd, "%s %s*", RM, f_kill);
	system(syscmd);

	free(f_kill);
	free(f_sock);
	free(f_iosock);
	return(0);
}

/*
 *	help
 *
 *	Function:	- prints helpful information on the tkill command
 */
static void
help()

{
    printf("\nSynopsis:	tkill [options]\n");
    printf("\nDescription:	Terminate LAM on one node.\n");
    printf("\nOptions:	-h		Print this message.\n");
    printf("\t\t-d		Print debug information.\n");
    printf("\t\t-v		Be verbose.\n");
    printf("\t\t-N		Pretend to tkill.\n");
    printf("\t\t-f <kill>	Use <kill> as the kill file.\n\n");
}
