/*
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB
 *
 *	$Id: ndi_parse.c,v 6.1.1.1 97/01/27 17:46:39 nevin Exp $
 *
 *	Function:	- node identifier parsing package
 *			- id_range() based in part on Trollius 2.0,
 *			  Copyright 1990 The Ohio State University and Cornell
 *			  Research Foundation
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "all_list.h"
#include "args.h"
#include "ndi.h"
#include "lam.h"

#define NEW		1
#define ONE		2
#define MANY		3

/*
 * global functions
 */
LIST *			ndi_parse();
int4			ndi_parse1();
void			ndi_fmt();
void			ndi_usage();

/*
 * local functions
 */
static int		id_range();	/* parse a range of identifiers */

/*
 *	ndi_parse
 *
 *	Functions:	- parses node identifier mnemonics
 *	Accepts:	- command line count
 *			- command line vector
 *			- leftover vector (returned)
 *	Returns:	- node list or null
 */
LIST *
ndi_parse(argc, argv, othersv)

int			argc;
char **			argv;
char ***		othersv;

{
	int		index = 0;		/* current index */
	int		othersc;		/* leftover count */
	struct ndi	new;			/* new list element */	
	LIST *		nodelist;		/* node ID list */
/*
 * Initialize the list.
 */
	nodelist = al_init(sizeof(struct ndi), 0);
	if (nodelist == 0) return(0);
/*
 * Pass argument 0 through.
 */
	if (othersv) {
		othersc = 0;
		*othersv = 0;

		if (argvadd(&othersc, othersv, argv[0])) {
			al_free(nodelist);
			return(0);
		}
	}

	argc--;
	argv++;
/*
 * Loop through all the arguments.
 */
	while (argc) {
/*
 * Is it a generic node identifier?
 */
		if ((**argv == 'n') && isdigit(*(*argv + 1))) {

			if (id_range(*argv + 1, 0, &index, nodelist)) {
				al_free(nodelist);
				if (othersv) argvfree(*othersv);
				return(0);
			}
		}
/*
 * Is it the local host identifier (h)?
 */
		else if ((**argv == 'h') && (*(*argv + 1) == '\0')) {
			new.ndi_node = LOCAL;
			new.ndi_flags = NT_ME;
			new.ndi_count = 1;
			new.ndi_index = index++;

			if (al_append(nodelist, &new) == 0) {
				al_free(nodelist);
				if (othersv) argvfree(*othersv);
				return(0);
			}
		}
/*
 * Is it a multicast destination?
 */
		else if ((**argv == 'O') && (*(*argv + 1) == '\0')) {
			new.ndi_node = HOST2OTB;
			new.ndi_flags = NT_CAST;
			new.ndi_count = -1;
			new.ndi_index = index++;

			if (al_append(nodelist, &new) == 0) {
				al_free(nodelist);
				if (othersv) argvfree(*othersv);
				return(0);
			}
		}

		else if ((**argv == 'C') && (*(*argv + 1) == '\0')) {
			new.ndi_node = HOST2COMP;
			new.ndi_flags = NT_CAST;
			new.ndi_count = -1;
			new.ndi_index = index++;

			if (al_append(nodelist, &new) == 0) {
				al_free(nodelist);
				if (othersv) argvfree(*othersv);
				return(0);
			}
		}

		else if ((**argv == 'N') && (*(*argv + 1) == '\0')) {
			new.ndi_node = HOST2ALL;
			new.ndi_flags = NT_CAST;
			new.ndi_count = -1;
			new.ndi_index = index++;

			if (al_append(nodelist, &new) == 0) {
				al_free(nodelist);
				if (othersv) argvfree(*othersv);
				return(0);
			}
		}
/*
 * Bump the non-nodeid pointer and counter.
 */
		else if (othersv) {

			if (argvadd(&othersc, othersv, argv[0])) {
				al_free(nodelist);
				argvfree(*othersv);
				return(0);
			}
		}

		argc--;
		argv++;
	}

	return(nodelist);
}

/*
 *	ndi_fmt
 *
 *	Function:	- translates a node identifier into a mnemonic string
 *	Accepts:	- node identifier
 *			- string buffer
 */
void
ndi_fmt(node, fmt_str)

int4			node;
char			*fmt_str;

{
	if (node == NOTNODEID) {
		strcpy(fmt_str, "?");
	} else if (node == LOCAL) {
		strcpy(fmt_str, "local");
	} else if (node == HOST2OTB) {
		strcpy(fmt_str, "O");
	} else if (node == HOST2ALL) {
		strcpy(fmt_str, "N");
	} else if (node == HOST2COMP) {
		strcpy(fmt_str, "C");
	} else {
		sprintf(fmt_str, "n%d", node);
	}
}

/*
 *	ndi_parse1
 *
 *	Function:	- parses one node identifier
 *	Accepts:	- mnemonic string
 *	Returns:	- node identifier
 */
int4
ndi_parse1(s)

char			*s;

{
	int4		node;

	if (strcmp(s, "h") == 0) {
		node = LOCAL;
	} else if (strcmp(s, "local") == 0) {
		node = LOCAL;
	} else if (strcmp(s, "O") == 0) {
		node = HOST2OTB;
	} else if (strcmp(s, "N") == 0) {
		node = HOST2ALL;
	} else if (strcmp(s, "C") == 0) {
		node = HOST2COMP;
	}

	else if (sscanf(s, " n%d", &node) == 1) {
	}

	else {
		node = NOTNODEID;
	}

	return(node);
}

/*
 *	ndi_usage
 *
 *	Function:	- prints a nodespec usage message
 */
void
ndi_usage()

{
	printf("\nNodes:		n<list>, eg., n0-3,5,0xa,12-15\n");
}

/*
 *	id_range
 *
 *	Function:	- parses the ID range format
 *			- example: 1,3-5,7
 *	Accepts:	- range string
 *			- flags to use with every value found
 *			- current index ptr
 *			- list descriptor ptr
 *	Returns:	- 0 or ERROR
 */
static int
id_range(s, flags, ixp, lp)

char			*s;
int4			flags;
int			*ixp;
LIST			*lp;

{
	int		c;			/* current character */
	int4		id;			/* current identifier */
	int4		oldid;			/* previous identifier */
	int		mode;			/* parsing state */
	struct ndi	new;			/* new list entry */
	char		*t;			/* bumped up s */

	mode = NEW;

	while ((c = *s) != '\0') {

		if (c == ',') {

			if (mode == NEW) {
				errno = EBADIDSPEC;
				return(LAMERROR);
			}

			++s;
			mode = NEW;
		}

		else if (c == '-') {

			if (mode != ONE) {
				errno = EBADIDSPEC;
				return(LAMERROR);
			}

			++s;
			mode = MANY;
		}

		else {
			id = strtol(s, &t, 0);

			if ((id == 0) && (s == t)) {
				errno = EBADIDSPEC;
				return(LAMERROR);
			}

			s = t;

			if (mode == NEW) {
				new.ndi_node = id;
				new.ndi_flags = flags;
				new.ndi_count = 1;
				new.ndi_index = (*ixp)++;

				if (al_append(lp, &new) == 0) return(LAMERROR);

				oldid = id;
				mode = ONE;
			}

			else if (mode == MANY) {

				if (id < oldid) {
					errno = EBADIDSPEC;
					return(LAMERROR);
				}

				while (++oldid <= id) {
					new.ndi_node = oldid;
					new.ndi_flags = flags;
					new.ndi_count = 1;
					new.ndi_index = (*ixp)++;

					if (al_append(lp, &new) == 0)
							return(LAMERROR);
				}
			}

			else {
				errno = EIMPOSSIBLE;
				return(LAMERROR);
			}
		}
	} /* while() */

	return (0);
}
