/*
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB
 *
 *	$Id: pqdetach.c,v 6.1.1.1 97/01/27 17:48:17 nevin Exp $
 * 
 *	Function:	- OTB kenyad
 *			- attaches and detaches kenya processes
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/wait.h>
#include <t_types.h>
#include <unistd.h>

#include <args.h>
#include <events.h>
#include <kio.h>
#include <net.h>
#include <preq.h>
#include <terror.h>
#include <typical.h>

/*
 * external variables
 */
extern struct pprocess	ptable[];	/* kenya process table */
extern struct preq	pq;		/* kenya request */
extern struct preply	pr;		/* kenya reply */

/*
 * external functions
 */
extern void		psendr();
extern int		pfind();

/*
 * global functions
 */
void			childdetach();
void			childtrap();
void			pdetachindex();
void			pqattach();
void			pqdetach();

void
pqdetach()

{
	int		idetach;	/* descriptor index of target */
/*
 * Find the process descriptor for the given identifier.
 */
	idetach = pfind(pq.pq_pid);
	pr.pr_reply = (idetach < 0) ? ltot(ENOTPROCESS) : 0;
/*
 * Send reply of detach operation.
 */
	psendr((char *) &pr, sizeof(pr), 0);

	if (pr.pr_reply == 0) {
		pdetachindex(idetach);
	}
}

/*
 *	pdetachindex
 *
 *	Function:	- detaches a process identified by table index
 *	Accepts:	- table index
 */
void
pdetachindex(idetach)

int			idetach;

{
	int		i;		/* favourite index */
	int		found = FALSE;	/* found load module flag */
	int4		reqnode;	/* node of requester */
	int4		reqevent;	/* event of requester */
	struct pprocess	*pdetach;	/* descriptor ptr of target */

	pdetach = ptable + idetach;
/*
 * Flush IO.
 */
	if (pdetach->p_rtf & RTF_IO) {
		fflush(stderr);
		fflush(stdout);
	}

	if (pdetach->p_rtf & RTF_WAIT) {
/*
 * Reply to the parent process. This may not be the process which made the
 * kenyad request so we must save and restore the requester's synchronization.
 */
		reqnode = pq.pq_src_node;
		reqevent = pq.pq_src_event;

		pq.pq_src_node = pdetach->p_nodeid;
		pq.pq_src_event = pdetach->p_event & 0xBFFFFFFF;
		pr.pr_reply = ltot(pq.pq_status);
		pr.pr_nodeid = ltot(getnodeid());
		pr.pr_pid = ltot(pdetach->p_pid);

		psendr((char *) &pr, sizeof(pr), 0);

		pq.pq_src_node = reqnode;
		pq.pq_src_event = reqevent;
	}
/*
 * Detach target process.
 */
	pdetach->p_pid = NOTPROCESS;
	pdetach->p_status = NOTSTATUS;

	if (pdetach->p_argc) {
		free((char *) pdetach->p_argv);
	}
/*
 * Check to see if anybody else is using the load module.
 */
	if (pdetach->p_rtf & RTF_FLAT) {

		for (i = 0; (i < PMAX) && !found; ++i) {

			if ((ptable[i].p_pid != NOTPROCESS) &&
					((ptable + i) != pdetach) &&
					(! strcmp(ptable[i].p_loadpt,
					pdetach->p_loadpt))) {

				found = TRUE;
			}
		}

		if (!found) {
/*
 * Remove the load module file and free the filename buffer.
 */
			unlink(pdetach->p_loadpt);
		}
	}

	free(pdetach->p_loadpt);
}

/*
 *	pqattach
 *
 *	Function:	- attaches a process to kenyad
 */
void
pqattach()

{
	struct pprocess	*pattach;	/* new process ptr */
	int		iattach;	/* index of new process */
/*
 * Is the process already attached?
 */
	iattach = pfind(pq.pq_pid);

	if (iattach >= 0) {
		pr.pr_reply = 0;
		pattach = ptable + iattach;
		pattach->p_rtf = pq.pq_rtf & ~RTF_KENYA;
/*
 * If the request doesn't specify a new name, ignore the name.
 */
		if (pq.pq_name[0] != '\0') {
			free(pattach->p_argv[0]);
			pattach->p_argv[0] = malloc((unsigned)
					strlen(pq.pq_name) + 1);
			if (pattach->p_argv[0] == 0)
					lampanic("kenyad (malloc)");
			strcpy(pattach->p_argv[0], pq.pq_name);
		}
	}
/*
 * Find a slot in the process table.
 */
	else if ((iattach = pfind((int4) NOTPROCESS)) < 0) {
		pr.pr_reply = ltot(ENOPDESCRIPTORS);
	}
/*
 * Fill in new kenya descriptor.
 */
	else {
		pattach = ptable + iattach;

		pattach->p_loadpt = malloc((unsigned) strlen(pq.pq_name) + 1);
		if (pattach->p_loadpt == 0) lampanic("kenyad (malloc)");

		strcpy(pattach->p_loadpt, pq.pq_name);

		if (pq.pq_name[0] != '\0') {
			pattach->p_argc = 1;
			pattach->p_argv = argvbreak(pq.pq_name, 0xa);
			if (pattach->p_argv == 0)
					lampanic("kenyad (pack_argv)");
		}
		else {
			pattach->p_argc = 0;
		}

		pattach->p_pid = pq.pq_pid;
		pattach->p_rtf = pq.pq_rtf;
		pattach->p_nodeid = NOTNODEID;
		pattach->p_event = NOTEVENT;
		pr.pr_reply = 0;
	}
/*
 * Reply to client.
 *
 *	psendr((char *) &pr, sizeof(pr), 0);
 */
}

/*
 *	childtrap
 *
 *	Function:	- traps child process termination
 *			- stores exit status in process table
 */
void
childtrap()

{
	int		i;		/* favourite index */
	int		pid;		/* child process ID */
	int		status;		/* child exit status */
#if 0
	struct sigaction
			chld;		/* SIGCHLD action */
#endif
	pid = wait(&status);

	if (pid < 0) {
		return;
	}

	i = pfind((int4) pid);

	if (i >= 0) {
		ptable[i].p_status = (int4) (WIFEXITED(status) ?
				WEXITSTATUS(status) : 1);
	}
/*
 * Don't reset the SIGCHLD handler.  POSIX.1 says it is not reset to the
 * default when the handler is entered.  We are using sigaction() which
 * is POSIX.1 so it should be safe to assume POSIX.1 semantics.  
 */
#if 0	
/*
 * Reinstall signal handler.
 */
	chld.sa_handler = childtrap;
	chld.sa_flags = SA_NOCLDSTOP;
	sigemptyset(&chld.sa_mask);
	sigaction(SIGCHLD, &chld, (struct sigaction *) 0);
#endif	
}

/*
 *	childdetach
 *
 *	Function:	- detaches child process indicated via SIGCHLD
 */
void
childdetach()

{
	int		i;		/* favourite index */

	for (i = 0; i < PMAX; ++i) {

		if (ptable[i].p_status != NOTSTATUS) {
			pq.pq_status = ptable[i].p_status;
			pdetachindex(i);
		}
	}
}
