/*
   PXKMenuWindow.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@net-community.com>
   Date: May 1997
   
   This file is part of the GNUstep GUI X/DPS Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <config.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <Foundation/NSDictionary.h>
#include <Foundation/NSRunLoop.h>
#include <Foundation/NSDate.h>
#include <Foundation/NSTimer.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSValue.h>

#include <AppKit/NSCursor.h>
#include <AppKit/NSApplication.h>
#include <AppKit/NSColor.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSButton.h>

#include <gnustep/xdps/PXKColor.h>
#include <gnustep/xdps/PXKScreen.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKMenu.h>
#include <gnustep/xdps/PXKCell.h>
#include <gnustep/xdps/PXKMenuWindow.h>

#include "drawingfuncs.h"

#if LIB_FOUNDATION_LIBRARY
# include <Foundation/NSPosixFileDescriptor.h>
#elif defined(NeXT_PDO)
# include <Foundation/NSFileHandle.h>
# include <Foundation/NSNotification.h>
#endif

#define ASSIGN(variable, value) \
  [value retain]; \
  [variable release]; \
  variable = value;

#define XWINDOW (((PXKWindow_struct *)be_wind_reserved)->xWindow)
#define XGC (((PXKWindow_struct *)be_wind_reserved)->context)
#define XEXPOSE (((PXKWindow_struct *)be_wind_reserved)->is_exposed)
#define exposedRectangles \
  (((PXKWindow_struct *)be_wind_reserved)->exposedRectangles)
#define xPixmap \
  (((PXKWindow_struct *)be_wind_reserved)->xPixmap)
#define X (((PXKWindow_struct *)be_wind_reserved)->x)
#define Y (((PXKWindow_struct *)be_wind_reserved)->y)
#define WIDTH (((PXKWindow_struct *)be_wind_reserved)->width)
#define HEIGHT (((PXKWindow_struct *)be_wind_reserved)->height)
#define DEPTH (((PXKWindow_struct *)be_wind_reserved)->depth)
#define region (((PXKWindow_struct *)be_wind_reserved)->region)


@implementation PXKMenuWindow

/* This is the mode used by run loop to handle mouse tracking on menus */
static NSString* NSMenuTrackingRunLoopMode = @"NSMenuTrackingRunLoopMode";

+ (void)initialize
{
	if (self == [PXKMenuWindow class])
		[self setVersion:1];								// Initial version
}

- (void)setTitleHeight:(int)_height		{ titleHeight = _height; }
- (void)setMenu:(PXKMenu*)aMenu			{ menu = aMenu; }
- (PXKMenu*)menu				{ return menu; }

- initWithContentRect:(NSRect)contentRect
  styleMask:(unsigned int)aStyle
  backing:(NSBackingStoreType)bufferingType
  defer:(BOOL)flag
  screen:aScreen
{
  NSApplication* theApp = [NSApplication sharedApplication];

  [super initWithContentRect:contentRect
	  styleMask:aStyle
	  backing:bufferingType
	  defer:flag
	  screen:aScreen];
  [theApp removeWindowsItem:self];
  return self;
}

- (void)createXWindow:(NSRect)rect
{
  Display *xDisplay = [(PXKDPSContext *)[NSDPSContext currentContext]
					xDisplay];
  XSetWindowAttributes winattrs;
  unsigned long valuemask;

  [super createXWindow:rect];

  valuemask = (CWSaveUnder|CWOverrideRedirect);
  winattrs.save_under = True;
  winattrs.override_redirect = True;
  XChangeWindowAttributes (xDisplay, XWINDOW, valuemask, &winattrs);
}

- (void)setContentSize:(NSSize)aSize
{
  [super setContentSize:NSMakeSize (aSize.width, aSize.height + titleHeight)];
}

- (void)setFrame:(NSRect)frameRect display:(BOOL)flag
{
  NSRect parentFrame = frameRect;

  parentFrame.size.height += titleHeight;
  [super setFrame:parentFrame display:flag];
}

- (void)_displayWithoutFlushing
{
  PXKDPSContext *context = [NSDPSContext currentContext];
  PXKWindow* oldXDrawable = [context drawableWindow];

  [context setXDrawable:self];

  /* Temporary disable displaying */
  [self disableFlushWindow];

  [[menu titleView] display];
  [[menu menuCells] display];

  /* Reenable displaying and flush the window */
  [self enableFlushWindow];

  [context setXDrawable:oldXDrawable];
}

/* Avoid performing the default action on ConfigureNotify requests */
- (void)setFrameFromXFrame:(NSRect)rect
{
}

/* The following variables are used to communicate with -handleEvent method.
   Don't worry about thread-safe here because menu moving on the screen should
   really performed in the main thread by the user. */
static Display *xDisplay;
static int xScreen;
static XWindowAttributes winAttrs;
static int previousX, previousY;
static int currentX, currentY;
static int winX, winY;
static int originalX, originalY;
static BOOL done;

- (void)mouseDown:(NSEvent*)theEvent
{
  PXKDPSContext* context = (PXKDPSContext*)[NSDPSContext currentContext];
  NSDate* limitDate = [NSDate distantFuture];
  NSRunLoop* currentRunLoop;
  Window rootWin, childWin;
  int xEventQueueFd;
  NSTimer* timer;
  unsigned mask;
#if defined(LIB_FOUNDATION_LIBRARY) || defined(NeXT_PDO)
  id fileDescriptor;
#endif

  xDisplay = [context xDisplay];
  xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
  xEventQueueFd = XConnectionNumber (xDisplay);

  XQueryPointer (xDisplay, RootWindow(xDisplay, xScreen),
		 &rootWin, &childWin,
		 &currentX, &currentY,
		 &winX, &winY, /* These doesn't matter */
		 &mask);
  previousX = currentX;
  previousY = currentY;

  if (!XGetWindowAttributes (xDisplay, XWINDOW, &winAttrs)) {
    NSLog (@"cannot get window attributes");
    return;
  }
  originalX = winX = winAttrs.x;
  originalY = winY = winAttrs.y;

  /* Create a local run loop to handle mouse moves. Create a new mode
     NSMenuTrackingRunLoopMode used for this purpose. Invoke limitDateForMode:
     first to setup the current mode of the run loop (the doc says that this
     method and acceptInputForMode:beforeDate: are the only ones that setup
     the current mode).
  */
  currentRunLoop = [NSRunLoop currentRunLoop];
  [currentRunLoop limitDateForMode:NSMenuTrackingRunLoopMode];

#if LIB_FOUNDATION_LIBRARY
  fileDescriptor = [[[NSPosixFileDescriptor alloc]
			      initWithFileDescriptor:xEventQueueFd]
			      autorelease];

  [fileDescriptor setDelegate:self];
  [fileDescriptor monitorFileActivity:NSPosixReadableActivity];
#elif NeXT_PDO
  fileDescriptor = [[[NSFileHandle alloc]
			      initWithFileDescriptor:xEventQueueFd]
			      autorelease];
  [[NSNotificationCenter defaultCenter]
	  addObserver:self
	  selector:@selector(activityOnFileHandle:)
	  name:NSFileHandleDataAvailableNotification
	  object:fileDescriptor];
  [fileDescriptor waitForDataInBackgroundAndNotifyForModes:
		      [NSArray arrayWithObject:NSMenuTrackingRunLoopMode]];
#else
  /* Assume gnustep-base */
  [(id)currentRunLoop addReadDescriptor:xEventQueueFd
			  object:self
			 forMode:NSMenuTrackingRunLoopMode];
#endif

  timer = [NSTimer timerWithTimeInterval:0.05
		    target:self
		    selector:@selector(timerFired:)
		    userInfo:nil
		    repeats:YES];
  [currentRunLoop addTimer:timer forMode:NSMenuTrackingRunLoopMode];

  done = NO;
  while(!done)
    [currentRunLoop runMode:NSMenuTrackingRunLoopMode beforeDate:limitDate];

  [timer invalidate];

#if LIB_FOUNDATION_LIBRARY
  [fileDescriptor ceaseMonitoringFileActivity];
#elif !defined(NeXT_PDO)
  [(id)currentRunLoop removeReadDescriptor:xEventQueueFd
		  forMode:NSMenuTrackingRunLoopMode];
#endif
}

- (void)handleEvent
{
  NSApplication* theApp = [NSApplication sharedApplication];
  XEvent event;

  while (XPending(xDisplay)) {
    XNextEvent (xDisplay, &event);

    switch (event.type) {
      case MotionNotify:
	currentX = event.xmotion.x_root;
	currentY = event.xmotion.y_root;
	break;

      case ButtonRelease: {
	PXKDPSContext* context = (PXKDPSContext*)[NSDPSContext currentContext];
	NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];
	NSMutableDictionary* menuLocations
	    = [[[defaults objectForKey:NSMenuLocationsKey]
			  mutableCopy]
			  autorelease];
	PXKMenuWindow* menuWindow;
	PXKMenu* attachedMenu;
	NSPoint origin;
	NSArray* array;
	NSString* key;

	if (currentX != previousX || currentY != previousY) {
	  winX += currentX - previousX;
	  winY += currentY - previousY;
	  XMoveWindow (xDisplay, XWINDOW, winX, winY);
	}
	[context setXDrawable:self];
	[self _updateWindowGeometry];
	attachedMenu = (PXKMenu*)[menu attachedMenu];
	while (attachedMenu) {
	  menuWindow = [attachedMenu menuWindow];
	  [menuWindow _updateWindowGeometry];
	  attachedMenu = (PXKMenu*)[attachedMenu attachedMenu];
	}
	DPSFlushContext ([context xDPSContext]);
	done = YES;

	/* Don't save the menu position and don't create a torn off window
	   if the menu position is the same as the original one. */
	if (winX == originalX && winY == originalY)
	  break;

	/* Save the window position */
	if (!menuLocations)
	  menuLocations = [NSMutableDictionary dictionaryWithCapacity:2];
	origin = [self frame].origin;
	array = [NSArray arrayWithObjects:
		    [[NSNumber numberWithInt:origin.x] stringValue],
		    [[NSNumber numberWithInt:origin.y] stringValue],
		    nil];
	if (menu == (PXKMenu*)[theApp mainMenu])
	  key = @"Main menu";
	else
	  key = [menu title];
	[menuLocations setObject:array forKey:key];
	[defaults setObject:menuLocations forKey:NSMenuLocationsKey];
	[defaults synchronize];

	NSDebugLog (@"move ready, save under name '%@'!", key);

	/* Show the window close button if the menu is not the main menu */
	if (menu != (PXKMenu*)[theApp mainMenu] && ![menu isTornOff])
	  [menu windowBecomeTornOff];

	break;
      }

      case Expose: {
	XRectangle rectangle;
	PXKWindow *window;

	window = [PXKWindow windowForXWindow:event.xexpose.window];
	[window setXExposed: YES];

	rectangle.x = event.xexpose.x;
	rectangle.y = event.xexpose.y;
	rectangle.width = event.xexpose.width;
	rectangle.height = event.xexpose.height;

	[window addExposedRectangle:rectangle];

	if (event.xexpose.count == 0)
	  [window processExposedRectangles];
	break;
      }

      default:
	break;
    }
  }
}

- (void)timerFired:(NSTimer*)timer
{
  if (currentX != previousX || currentY != previousY) {
    PXKMenu* attachedMenu;
    PXKMenu* currentMenu = menu;
    PXKMenuWindow* menuWindow;
    int x, y;

    winX += currentX - previousX;
    winY += currentY - previousY;
    XMoveWindow (xDisplay, XWINDOW, winX, winY);
    previousX = currentX;
    previousY = currentY;

    /* Move the attached menu windows */
    attachedMenu = (PXKMenu*)[currentMenu attachedMenu];
    x = winX;
    y = winY;
    while (attachedMenu) {
      menuWindow = [attachedMenu menuWindow];
      x += ((PXKWindow_struct*)[currentMenu menuWindow]->be_wind_reserved)->width + 1;
      XMoveWindow (xDisplay, [menuWindow xWindow], x, y);
      currentMenu = attachedMenu;
      attachedMenu = (PXKMenu*)[currentMenu attachedMenu];
    }
  }
}

#if LIB_FOUNDATION_LIBRARY
- (void)activity:(NSPosixFileActivities)activity
  posixFileDescriptor:(NSPosixFileDescriptor*)fileDescriptor
{
  [self handleEvent];
}
#elif defined(NeXT_PDO)
- (void)activityOnFileHandle:(NSNotification*)notification
{
  id fileDescriptor = [notification object];
  id runLoopMode = [[NSRunLoop currentRunLoop] currentMode];

  [fileDescriptor waitForDataInBackgroundAndNotifyForModes:
		    [NSArray arrayWithObject:runLoopMode]];
  [self handleEvent];
}
#else
/* Assume gnustep-base */
- (void)readyForReadingOnFileDescriptor:(int)fd
{
  [self handleEvent];
}
#endif

@end /* PXKMenuWindow */


@implementation PXKMenuWindowTitleView

static PXKCell* titleCell = nil;

+ (void)initialize
{
  titleCell = [[PXKCell alloc] initTextCell:@""];
  [titleCell setBordered:NO];
  [titleCell setEnabled:NO];
  [titleCell setAlignment:NSLeftTextAlignment];
}

- (void)drawRect:(NSRect)rect
{
  PSgsave();
  PSWDrawMenuBorder (rect.origin.x, rect.origin.y,
		    rect.size.width, rect.size.height);
  [[NSColor whiteColor] set];

  /* Ugly, we use a private method to draw; but this way we avoid creating
     another cell class. */
  [titleCell _displayTitle:[[(PXKMenuWindow*)[self window] menu] title]
	      inFrame:NSMakeRect(rect.origin.x + 3, rect.origin.y + 3,
				rect.size.width - 4, rect.size.height - 4)
	      titleGray:NSWhite];
  PSgrestore();
}

- (void)mouseDown:(NSEvent*)theEvent
{
  [self lockFocus];
  [[self window] mouseDown:theEvent];
  [self unlockFocus];
}

- (void)windowBecomeTornOff
{
  if ([[(PXKMenuWindow*)[self window] menu] isTornOff])
    return;
  else {
    NSImage* closeImage = [NSImage imageNamed:@"common_Close"];
    NSImage* closeHImage = [NSImage imageNamed:@"common_CloseH"];
    NSSize imageSize = [closeImage size];
    NSRect rect = {
	{ frame.size.width - imageSize.width - 4,
	  (frame.size.height - imageSize.height) / 2
	},
	{ imageSize.height, imageSize.width }
    };
    NSButton* button = [[[NSButton alloc] initWithFrame:rect] autorelease];
    int mask = NSViewMinXMargin | NSViewMinYMargin | NSViewMaxYMargin;

    [button setButtonType:NSToggleButton];
    [button setImagePosition:NSImageOnly];
    [button setImage:closeImage];
    [button setAlternateImage:closeHImage];
    [button setBordered:NO];
    [button setTarget:[(PXKMenuWindow*)[self window] menu]];
    [button setAction:@selector(_performMenuClose:)];

    [self addSubview:button];
    [self setAutoresizingMask:mask];
#if 1
    [self setNeedsDisplay:YES];
#else
    [self display];
    [[self window] flushWindow];
#endif
  }
}

@end /* PXKMenuWindowTitleView */
