/* 
   NSException.h

   Copyright (C) 1995, 1996, 1997 Ovidiu Predescu and Mircea Oancea.
   All rights reserved.

   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>

   This file is part of the Foundation Extensions Library.

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies and that both that
   copyright notice and this permission notice appear in supporting
   documentation.

   We disclaim all warranties with regard to this software, including all
   implied warranties of merchantability and fitness, in no event shall
   we be liable for any special, indirect or consequential damages or any
   damages whatsoever resulting from loss of use, data or profits, whether in
   an action of contract, negligence or other tortious action, arising out of
   or in connection with the use or performance of this software.
*/

#ifndef __NSException_h__
#define __NSException_h__

#include <Foundation/NSException.h>
#include <Foundation/NSArray.h>

@interface NSException (Extensions)
- (BOOL)exceptionIsKindOfClass:(Class)class;
				/* return [self isKindOfClass:class] */
- (BOOL)exceptionIsIn:(NSArray*)exceptions;
- (NSString*)errorString;
- initWithFormat:(NSString*)format, ...;
- setName:(NSString*)name;
- setReason:(NSString*)reason;
- setUserInfo:(NSDictionary*)userInfo;
@end /* NSException (Extension) */


/*
 * The new macros for handling exceptions.
 */

#define TRY \
({ \
    __label__ _quit; \
    NSHandler exceptionHandler; \
    int __setjmp_ret = setjmp(exceptionHandler.jumpState); \
    if(!__setjmp_ret) { \
	_NSAddHandler(&exceptionHandler);

#define END_TRY \
	_NSRemoveHandler(&exceptionHandler); \
	goto _quit; /* to remove compiler warning about unused label*/ \
    } \
_quit: \
    { \
	void handler(id localException) \
	{ \
	    BOOL _caught = NO; \
	    if(!localException) {

#define CATCH(exception_class) \
	    } else if([localException isKindOfClass:[exception_class class]]) { \
		_caught = YES;

#ifndef PRECOMP
# define MULTICATCH(exception_classes...) \
	    } else if([localException exceptionIsIn: \
		    [NSArray arrayWithObjects:##exception_classes, nil]]) { \
		_caught = YES;
#endif /* PRECOMP */

#define OTHERWISE \
	    } else { \
		_caught = YES;

#define CLEANUP \
	    } \
	    if(localException && !_caught) {

#define FINALLY \
	    } \
	    if(1) {

#define END_CATCH \
	    } \
	    if(!localException) return; \
	    if(!_caught) \
		[localException raise]; \
	} \
	handler(__setjmp_ret == 1 ? exceptionHandler.exception : nil); \
    } \
});

    /*  Use BREAK inside a TRY block to get out of it */
#define BREAK	({_NSRemoveHandler(&exceptionHandler); goto _quit;})

#ifndef PRECOMP
    /*  If you want to generate an exception issue a THROW with the exception
	an object derived from the NSException class. */
# define THROW(exception...)	[##exception raise]
#else
# define THROW(exception)		[exception raise]
#endif /* PRECOMP */

    /*  If you want to reraise an exception inside an exception handler
	just say RERAISE. */
#define RERAISE                 THROW(localException)


/*
 * Assertions.
 */

#ifndef __FoundationException_definition__
#define __FoundationException_definition__

@interface FoundationException : NSException
@end

#endif /* __FoundationException_definition__ */

@interface AssertException : FoundationException
@end


#ifndef PRECOMP

#define Assert(condition) \
    ({if(!(condition)) {\
	NSLog([@"Assertion failed: " stringByAppendingString:(@#condition)]); \
	THROW([AssertException new]); \
    } \
    0;})

#endif /* PRECOMP */


#endif /* __NSException_h__ */
