/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "gladeconfig.h"

#include "gbwidget.h"
#include "editor.h"
#include "tree.h"

/* FIXME: only use tree with gtk1.1, 1.0 gtktree is unstable :(
 */

/* This key is used on the data hash of the tree items to hold a pointer to
   the corresponding widget. */
#define GB_TREE_CHILD_WIDGET	"TREE_CHILD_WIDGET"

static GtkWidget *win_tree = NULL;
static GtkWidget *tree_notebook = NULL, *tree_widget_label = NULL;
static GtkWidget *tree_file_label = NULL;
static GtkWidget *nyi = NULL;
static GtkWidget *widget_tree = NULL;
static gboolean shown = FALSE;

/* private functions */
static GtkWidget *tree_search_parent_tree (GtkWidget * widget);
static GtkWidget *tree_find_child_label (GtkWidget * widget,
					 GtkWidget * tree);
static GtkWidget *tree_find_child (GtkWidget * widget,
				   GtkWidget * tree);
static void tree_on_select_child (GtkTree * tree,
				  GtkWidget * tree_item,
				  gpointer data);
static void tree_delete (gpointer widget,
			 gpointer dummy);
static void tree_on_delete (GtkWidget *,
			    gpointer);
static gint tree_on_key_press (GtkWidget * widget,
			       GdkEventKey * event,
			       gpointer data);
static void tree_on_tree_item_destroy (GtkWidget * tree_item,
				       gpointer data);

static gint tree_on_button_press (GtkWidget * event_widget,
				  GdkEventButton * event,
				  gpointer data);

/* Hack to evade problems with refcounting in 1.0.x gtk_tree
   courtesy of Elrond on the #gimp channel
 */
static void item_destroy (GtkTreeItem * treeitem,
			  gpointer dummy);

/* This creates the widget tree window, with just the root tree. */
void
tree_init ()
{
  GtkWidget *scroll_win;

  win_tree = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_uposition (win_tree, 450, 0);

  gtk_signal_connect (GTK_OBJECT (win_tree), "delete_event",
		      GTK_SIGNAL_FUNC (tree_hide), NULL);
  gtk_window_set_title (GTK_WINDOW (win_tree), _("Widget Tree"));
  gtk_container_border_width (GTK_CONTAINER (win_tree), 0);

  tree_notebook = gtk_notebook_new ();
  gtk_container_add (GTK_CONTAINER (win_tree), tree_notebook);
  gtk_widget_show (tree_notebook);

  scroll_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_win),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  tree_widget_label = gtk_label_new ("Widgets");
  gtk_notebook_append_page (GTK_NOTEBOOK (tree_notebook),
			    scroll_win, tree_widget_label);
  gtk_widget_show (scroll_win);
  tree_file_label = gtk_label_new ("Files");
  nyi = gtk_label_new ("Not yet implemented");
  gtk_notebook_append_page (GTK_NOTEBOOK (tree_notebook),
			    nyi, tree_file_label);
  gtk_widget_show (nyi);
  gtk_container_border_width (GTK_CONTAINER (scroll_win), 2);
  gtk_widget_set_usize (scroll_win, 250, 350);

  widget_tree = gtk_tree_new ();
  gtk_tree_set_selection_mode (GTK_TREE (widget_tree), GTK_SELECTION_MULTIPLE);
  gtk_container_add (GTK_CONTAINER (scroll_win), widget_tree);
  gtk_widget_show (widget_tree);
  gtk_signal_connect (GTK_OBJECT (widget_tree), "select_child",
		      GTK_SIGNAL_FUNC (tree_on_select_child), NULL);
  gtk_signal_connect (GTK_OBJECT (widget_tree), "unselect_child",
		      GTK_SIGNAL_FUNC (tree_on_select_child), NULL);
  gtk_signal_connect (GTK_OBJECT (widget_tree), "key_press_event",
		      GTK_SIGNAL_FUNC (tree_on_key_press), NULL);

}

void
tree_do_select (gpointer widget, gpointer dummy)
{
  tree_select_widget (GTK_WIDGET (widget), TRUE);
}

/* This shows the widget tree window. */
void
tree_show (GtkWidget * widget, gpointer data)
{
  if (!win_tree)
    tree_init ();
  gtk_widget_show (win_tree);
  gdk_window_show (GTK_WIDGET (win_tree)->window);
  gdk_window_raise (GTK_WIDGET (win_tree)->window);
  shown = TRUE;
  g_list_foreach (editor_get_selection (), tree_do_select, NULL);
}

/* This hides the widget tree window. */
gint
tree_hide (GtkWidget * widget, gpointer data)
{
  gtk_widget_hide (win_tree);
  shown = FALSE;
  return TRUE;
}


/* This adds a widget to the widget tree. Toplevel widgets (windows/dialogs)
   are added to the root tree, other widgets are added to the tree
   corresponding to their parent. */
void
tree_add_widget (GtkWidget * widget)
{
  GtkWidget *tree_item, *subtree;

  if (!GB_IS_GB_WIDGET (widget))
    return;
  MSG ("In tree_add");
  if (!widget->parent)
    subtree = widget_tree;
  else
    subtree = tree_search_parent_tree (widget->parent);
  if (subtree)
    {
      tree_item = gtk_tree_item_new_with_label (gtk_widget_get_name (widget));
      gtk_tree_append (GTK_TREE (subtree), tree_item);
      gtk_widget_show (tree_item);
      gtk_object_set_data (GTK_OBJECT (tree_item), GB_TREE_CHILD_WIDGET,
			   widget);
      gtk_widget_ref (widget);
      gtk_signal_connect (GTK_OBJECT (tree_item), "destroy",
			  GTK_SIGNAL_FUNC (tree_on_tree_item_destroy), NULL);
      gtk_signal_connect (GTK_OBJECT (tree_item), "button_press_event",
			  GTK_SIGNAL_FUNC (tree_on_button_press), NULL);
    }
  MSG ("Out tree_add");
}

void
tree_insert_widget_parent (GtkWidget * parent, GtkWidget * child)
{
  GtkWidget *tree, *subtree, *tree_child;

  /* find the tree where child is. parent is inserted, so child->parent
     is now parent->parent */
  tree = tree_search_parent_tree (parent->parent);
  tree_child = tree_find_child (child, tree);
  gtk_widget_ref (tree_child);
  subtree = GTK_TREE_ITEM (tree_child)->subtree;
  gtk_object_ref (GTK_OBJECT (subtree));
  gtk_container_remove (GTK_CONTAINER (tree), tree_child);
  tree_add_widget (parent);
  tree = tree_search_parent_tree (child->parent);
  gtk_tree_append (GTK_TREE (tree), tree_child);
  gtk_tree_item_set_subtree (GTK_TREE_ITEM (tree_child), subtree);
  gtk_signal_connect (GTK_OBJECT (tree_child), "destroy",
		      (GtkSignalFunc) item_destroy, NULL);
  gtk_widget_unref (tree_child);
}

/* This sets the name of a widget in the tree, i.e. after a widget has been
   renamed. */
void
tree_rename_widget (GtkWidget * widget, gchar * name)
{
  GtkWidget *subtree, *label;

  MSG ("In tree_set_name");
  if (!widget->parent)
    subtree = widget_tree;
  else
    subtree = tree_search_parent_tree (widget->parent);

  if (subtree)
    {
      label = tree_find_child_label (widget, subtree);
      if (label == NULL)
	{
	  MSG ("#### Couldn't find child label");
	  return;
	}
      gtk_label_set (GTK_LABEL (label), name);
    }
  MSG ("Out tree_set_name");
}


void
tree_delete_widget (GtkWidget * widget)
{
  GList *children;
  GtkWidget *child, *subtree;

  MSG ("In tree_delete_child");

  if (!widget->parent)
    subtree = widget_tree;
  else
    subtree = tree_search_parent_tree (widget->parent);

  if (!subtree)
    return;

  children = g_list_first (GTK_TREE (subtree)->children);
  while (children)
    {
      child = children->data;
      children = children->next;

      if (gtk_object_get_data (GTK_OBJECT (child), GB_TREE_CHILD_WIDGET)
	  == widget)
	{
	  gtk_widget_destroy (child);
	  return;
	}
    }
  MSG ("Out tree_delete_child");
}


/* This returns the tree corresponding to the parent of the given widget.
   It recursively calls itself to find the widget starting from the root
   tree down. If any parent widgets are not in the tree they are added. */
static GtkWidget *
tree_search_parent_tree (GtkWidget * widget)
{
  GtkWidget *insert_at = NULL, *child;
  GList *children;

  MSG ("In tree_search_parent_tree");
  if (widget->parent)
    insert_at = tree_search_parent_tree (widget->parent);
  else
    insert_at = widget_tree;

  if (insert_at == NULL)
    {
      MSG ("#### Couldn't find parent in tree");
      return NULL;
    }

  MSG ("...searching children");
  children = g_list_first (GTK_TREE (insert_at)->children);
  while (children)
    {
      child = children->data;
      if (gtk_object_get_data (GTK_OBJECT (child), GB_TREE_CHILD_WIDGET)
	  == widget)
	{
	  insert_at = GTK_TREE_ITEM_SUBTREE (child);
	  if (!insert_at)
	    {
	      /* TODO: should only execute when adding a widget */
	      insert_at = gtk_tree_new ();
	      gtk_tree_item_set_subtree (GTK_TREE_ITEM (child), insert_at);
	      gtk_tree_item_expand (GTK_TREE_ITEM (child));
	      gtk_tree_set_selection_mode (GTK_TREE (insert_at),
					   GTK_SELECTION_MULTIPLE);
	      gtk_signal_connect (GTK_OBJECT (insert_at), "select_child",
				  GTK_SIGNAL_FUNC (tree_on_select_child),
				  NULL);
	      gtk_signal_connect (GTK_OBJECT (insert_at), "unselect_child",
				  GTK_SIGNAL_FUNC (tree_on_select_child),
				  NULL);
	    }
	  return (insert_at);
	}
      children = children->next;
    }
  return (insert_at);
}

/* This returns the label in the tree item corresponding to the given widget */
static GtkWidget *
tree_find_child_label (GtkWidget * widget, GtkWidget * tree)
{
  GList *children;
  GtkWidget *child;

  children = g_list_first (GTK_TREE (tree)->children);
  while (children)
    {
      child = children->data;
      if (gtk_object_get_data (GTK_OBJECT (child), GB_TREE_CHILD_WIDGET)
	  == widget)
	return (GTK_BIN (child)->child);
      children = children->next;
    }
  return NULL;
}

static GtkWidget *
tree_find_child (GtkWidget * widget, GtkWidget * tree)
{
  GList *children;
  GtkWidget *child;

  children = g_list_first (GTK_TREE (tree)->children);
  while (children)
    {
      child = children->data;
      if (gtk_object_get_data (GTK_OBJECT (child), GB_TREE_CHILD_WIDGET)
	  == widget)
	return (child);
      children = children->next;
    }
  return NULL;
}

static void
tree_on_select_child (GtkTree * atree, GtkWidget * tree_item, gpointer data)
{
  GtkWidget *widget;

  widget = gtk_object_get_data (GTK_OBJECT (tree_item), GB_TREE_CHILD_WIDGET);
  g_return_if_fail (widget != NULL);
  editor_select_widget_control (widget);
}

static void
tree_delete (gpointer data, gpointer userdata)
{
  GtkWidget *widget;

  if (!GTK_IS_TREE_ITEM (data))
    return;

  widget = gtk_object_get_data (GTK_OBJECT (data), GB_TREE_CHILD_WIDGET);
  g_return_if_fail (widget != NULL);
  tree_select_widget (widget, FALSE);
}

static void
tree_on_delete (GtkWidget * atree, gpointer data)
{
  tree_delete (g_list_first (GTK_TREE (atree)->selection)->data, NULL);
}


static gint
tree_on_key_press (GtkWidget * widget, GdkEventKey * event, gpointer data)
{
  guint key = event->keyval;

  MSG ("In tree_on_key_press");
  switch (key)
    {
    case GDK_Delete:
      MSG ("Delete pressed");
      break;
    case GDK_Escape:

      break;
    }
  return FALSE;
}

void
tree_select_widget (GtkWidget * widget, gboolean select)
{
  GtkWidget *child, *tree;

  if (!shown)
    return;
  if (GB_IS_PLACEHOLDER (widget))
    return;
  if (widget->parent)
    {
      tree = tree_search_parent_tree (widget->parent);
    }
  else
    {
      tree = widget_tree;
}

  if (!tree)
    return;
  child = tree_find_child (widget, tree);
  gtk_signal_handler_block_by_func (GTK_OBJECT (tree),
			      GTK_SIGNAL_FUNC (tree_on_select_child), NULL);
  if (select)
    {
      gtk_tree_select_child (GTK_TREE (tree), child);
    }
  else
    {
      gtk_tree_unselect_child (GTK_TREE (tree), child);
    }
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (tree),
			      GTK_SIGNAL_FUNC (tree_on_select_child), NULL);
}

void
tree_clear_selection ()
{

}

/* When a tree item is destroyed we unref the corresponding widget. */
static void
tree_on_tree_item_destroy (GtkWidget * tree_item, gpointer data)
{
  GtkWidget *widget;

  MSG ("In on_tree_item_destroy");
  widget = gtk_object_get_data (GTK_OBJECT (tree_item), GB_TREE_CHILD_WIDGET);
  g_return_if_fail (widget != NULL);
  gtk_widget_unref (widget);
}

void
tree_remove_widget_parent (GtkWidget * parent, GtkWidget * child)
{
  GtkWidget *tree, *subtree, *tree_child;

  tree = tree_search_parent_tree (child->parent);
  tree_child = tree_find_child (child, tree);
  gtk_widget_ref (tree_child);
  subtree = GTK_TREE_ITEM (tree_child)->subtree;
  gtk_object_ref (GTK_OBJECT (subtree));
  gtk_container_remove (GTK_CONTAINER (tree), tree_child);
  tree = tree_search_parent_tree (parent->parent);
  gtk_tree_append (GTK_TREE (tree), tree_child);
  tree_delete_widget (parent);
  gtk_tree_item_set_subtree (GTK_TREE_ITEM (tree_child), subtree);
  gtk_signal_connect (GTK_OBJECT (tree_child), "destroy",
		      (GtkSignalFunc) item_destroy, NULL);
  gtk_widget_unref (tree_child);
}

//gtk_signal_connect(GTK_OBJECT(treeitem), "destroy",
//                   (GtkSignalFunc) item_destroy, NULL);

static void
item_destroy (GtkTreeItem * treeitem, gpointer unused)
{
  if (treeitem->subtree)
    {
      gtk_widget_unparent (treeitem->subtree);
      gtk_widget_destroy (treeitem->subtree);
      gtk_object_unref (GTK_OBJECT (treeitem->subtree));
      treeitem->subtree = NULL;
    }
}

static gint
tree_on_button_press (GtkWidget * event_widget,
		      GdkEventButton * event,
		      gpointer data)
{
  GtkWidget *widget;

  if (event->button == 3)
    {
      widget = gtk_object_get_data (GTK_OBJECT (event_widget), GB_TREE_CHILD_WIDGET);
      gb_widget_show_popup_menu (widget, event);
      return (TRUE);
    }
  return (FALSE);

}
