/*
 *	cook - file construction tool
 *	Copyright (C) 1994, 1995, 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate the persistent fingerprint cache
 */

%{
#include <errno.h>
#include <ac/stdio.h>
#include <ac/stdlib.h>
#include <ac/unistd.h>

#include <archive.h>
#include <error_intl.h>
#include <fngrprnt.h>
#include <fngrprnt_lex.h>
#include <fp/combined.h>
#include <mem.h>
#include <option.h>
#include <progname.h>
#include <quit.h>
#include <str_list.h>
#include <symtab.h>
#include <cook.h>
#include <trace.h>

/*
 * forward declaration of the generated parser
 */
int yyparse _((void));

%}

%token	STRING
%token	JUNK
%token	NUMBER
%token	EQ
%token	LB
%token	RB

%union
{
	string_ty	*lv_string;
	long		lv_number;
}

%type	<lv_string>	STRING
%type	<lv_number>	NUMBER

%{

static string_ty *fp_path;
static symtab_ty *symtab;
static int	dirty;
static int	inited;


static void reap _((void *));

static void
reap(p)
	void	*p;
{
	fp_ty	*fp;

	trace(("reap(p = %08lX)\n{\n"/*}*/, (long)p));
	fp = p;
	str_free(fp->fingerprint);
	mem_free(fp);
	trace((/*{*/"}\n"));
}


static char *fp_filename _((void));

static char *
fp_filename()
{
	static string_ty *s;

	if (!s)
		s = str_format(".%.10s.fp", progname_get());
	return s->str_text;
}


static void walk _((symtab_ty *, string_ty *, void *, void *));

static void
walk(sp, key, p, arg)
	symtab_ty	*sp;
	string_ty	*key;
	void		*p;
	void		*arg;
{
	fp_ty		*data;
	FILE		*fp;

	trace(("walk(sp = %08lX, key = \"%s\", p = %08lX, arg = %08lX)\n{\n"/*}*/, (long)sp, key->str_text, (long)p, (long)arg));
	data = p;
	fp = arg;
	if (data->oldest >= data->newest)
		fprintf
		(
			fp,
			"\"%s\" = { %ld\n\"%s\" }\n",
			key->str_text,
			(long)data->newest,
			data->fingerprint->str_text
		);
	else
		fprintf
		(
			fp,
			"\"%s\" = { %ld %ld\n\"%s\" }\n",
			key->str_text,
			(long)data->oldest,
			(long)data->newest,
			data->fingerprint->str_text
		);
	trace((/*{*/"}\n"));
}


static void fp_close _((void));

static void
fp_close()
{
	char	*fn;
	FILE	*fp;

	if (!dirty)
		return;
	if (!inited)
		return;
	trace(("fp_close()\n{\n"/*}*/));
	fn = fp_filename();
	if (unlink(fn) && errno != ENOENT)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set(scp, "File_Name", "%s", fn);
		error_intl(scp, i18n("warning: unlink $filename: $errno"));
		sub_context_delete(scp);
	}
	trace(("open\n"));
	fp = fopen(fn, "w");
	if (!fp)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set(scp, "File_Name", "%s", fn);
		fatal_intl(scp, i18n("open $filename: $errno"));
		/* NOTREACHED */
		sub_context_delete(scp);
	}
	trace(("walk\n"));
	trace(("symtab = %08lX;\n", (long)symtab));
	symtab_walk(symtab, walk, fp);
	trace(("close\n"));
	fflush(fp);
	if (ferror(fp))
	{
		int		err;
		sub_context_ty	*scp;

		err = errno;
		unlink(fn); /* ignore error */
		scp = sub_context_new();
		sub_errno_setx(scp, err);
		sub_var_set(scp, "File_Name", "%s", fn);
		fatal_intl(scp, i18n("write $filename: $errno"));
	}
	fclose(fp);
	dirty = 0;
	trace((/*{*/"}\n"));
}


static void init _((void));

static void
init()
{
	string_ty	*file_name;
	string_list_ty	sl;
	int		j;

	if (inited)
		return;
	trace(("init()\n{\n"/*}*/));
	inited = 1;
	symtab = symtab_alloc(100);
	symtab->reap = reap;

	/*
	 * Read all of the fingerprint caches, deepest to shallowest
	 * (cook_search_list will make "." the first entry.)
	 */
	file_name = str_from_c(fp_filename());
	cook_search_list(&sl);
	for (j = sl.nstrings - 1; j >= 0; --j)
	{
		char		*dir;
		string_ty	*full;

		dir = sl.string[j]->str_text;
		while (dir[0] == '.' && dir[1] == '/')
		{
			dir += 2;
			while (*dir == '/')
				++dir;
		}
		if (dir[0] == 0 || (dir[0] == '.' && dir[1] == 0))
		{
			fp_path = 0;
			full = str_copy(file_name);
		}
		else
		{
			fp_path = str_format("%s/", dir);
			full = str_catenate(fp_path, file_name);
		}
		trace(("read %s\n", full->str_text));
		yylex_open(full->str_text);
		yyparse();
		yylex_close();
		str_free(full);
		if (fp_path)
			str_free(fp_path);
		fp_path = 0;
	}
	string_list_destructor(&sl);
	str_free(file_name);

	dirty = 0;
	quit_handler(fp_close);
	trace((/*{*/"}\n"));
}


fp_ty *
fp_search(path)
	string_ty	*path;
{
	fp_ty		*fp;

	trace(("fp_search(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (!inited)
		init();
	fp = symtab_query(symtab, path);
	trace(("return %08lX;\n", (long)fp));
	trace((/*{*/"}\n"));
	return fp;
}


void
fp_assign(path, fp)
	string_ty	*path;
	fp_ty		*fp;
{
	fp_ty		*data;

	trace(("fp_assign(path = \"%s\", fp = %08lX)\n{\n"/*}*/,
		path->str_text, (long)fp));
	if (!inited)
		init();
	data = mem_alloc(sizeof(fp_ty));
	if (fp->oldest > fp->newest)
	{
		/* should not happen */
		data->oldest = fp->newest;
		data->newest = fp->newest;
	}
	else
	{
		data->oldest = fp->oldest;
		data->newest = fp->newest;
	}
	data->fingerprint = str_copy(fp->fingerprint);
	symtab_assign(symtab, path, data);
	if (option_test(OPTION_FINGERPRINT_WRITE))
		dirty = 1;
	trace((/*{*/"}\n"));
}


void
fp_delete(path)
	string_ty	*path;
{
	if (!inited)
		return;
	trace(("fp_delete(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (symtab_query(symtab, path))
	{
		symtab_delete(symtab, path);
		dirty = 1;
	}
	trace((/*{*/"}\n"));
}


string_ty *
fp_fingerprint(path)
	string_ty	*path;
{
	fingerprint_ty	*fp;
	string_ty	*result;
	int		err;
	char		buffer[1000];
	sub_context_ty	*scp;

	trace(("fp_fingerprint(path = \"%s\")\n{\n"/*}*/, path->str_text));
	fp = fingerprint_new(&fp_combined);
	err = fingerprint_file_sum(fp, path->str_text, buffer);
	if (err && errno == ENOENT)
		err = archive_fingerprint(fp, path, buffer);
	if (err)
	{
		switch (errno)
		{
		case ENOTDIR:
		case EISDIR:
		case ENOENT:
			break;

		default:
			scp = sub_context_new();
			sub_errno_set(scp);
			sub_var_set(scp, "File_Name", "%S", path);
			fatal_intl
			(
				scp,
				i18n("fingerprint \"$filename\": $errno")
			);
			/* NOTREACHED */
			sub_context_delete(scp);
			break;
		}
		result = 0;
	}
	else
		result = str_from_c(buffer);
	fingerprint_delete(fp);
	trace(("return \"%s\";\n", result ? result->str_text : ""));
	trace((/*{*/"}\n"));
	return result;
}

%}

%%

cache
	: /* empty */
	| cache entry
	;

entry
	: STRING EQ LB NUMBER NUMBER STRING RB
		{
			fp_ty	data;

			if ($4 > $5)
			{
				/* should not happen */
				data.oldest = $5;
				data.newest = $5;
			}
			else
			{
				data.oldest = $4;
				data.newest = $5;
			}
			data.fingerprint = $6;
			if (fp_path && $1->str_text[0] != '/')
			{
				string_ty	*abspath;

				abspath = str_catenate(fp_path, $1);
				fp_assign(abspath, &data);
				str_free(abspath);
			}
			else
				fp_assign($1, &data);
			str_free($1);
			str_free($6);
		}
	| STRING EQ LB NUMBER STRING RB
		{
			fp_ty	data;

			data.oldest = $4;
			data.newest = $4;
			data.fingerprint = $5;
			if (fp_path && $1->str_text[0] != '/')
			{
				string_ty	*abspath;

				abspath = str_catenate(fp_path, $1);
				fp_assign(abspath, &data);
				str_free(abspath);
			}
			else
				fp_assign($1, &data);
			str_free($1);
			str_free($5);
		}
	;
