/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994, 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to implement the builtin functions
 *
 * The builtin function all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 *
 * Only a limited set of this are candidates for builtin functions,
 * these are 
 *	- string manipulation [dirname, stringset, ect ]
 *	- environment manipulation [getenv(3), etc]
 *	- stat(3) related functions [exists, mtime, pathname, etc]
 *	- launching OS commands [execute, collect]
 * The above list is though to be exhaustive.
 *
 * This explicitly and forever excluded from being a builtin function
 * is anything which known or understands the format of some secific 
 * class of files.
 *
 * Access to stdio(3) has been thought of, and explicitly avoided.
 * Mostly because a specialist program used through [collect]
 * will almost always be far faster.
 */

#include <ac/stdlib.h>

#include <builtin/cook.h>
#include <cook.h>
#include <error_intl.h>
#include <expr/position.h>
#include <str_list.h>
#include <trace.h>


/*
 * NAME
 *	builtin_mtime - file last modified time
 *
 * SYNOPSIS
 *	int builtin_mtime(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	Defined is a built-in function of cook, described as follows:
 *	This function requires a single argument,
 *	the name of a file of which to get the last modified time.
 *
 * RETURNS
 *	It returns a string containing the last modified time
 *	(suitable for comparing with others) of the named file,
 *	and "" (false) if the files does not exist
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a string_list_destructor() call.
 */

int
builtin_mtime(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	int		j;

	trace(("mtime\n"));
	assert(result);
	assert(args);
	assert(args->nstrings);
	if (args->nstrings < 2)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_var_set(scp, "Name", "%S", args->string[0]);
		error_with_position
		(
			pp,
			scp,
			i18n("$name: requires one or more arguments")
		);
		sub_context_delete(scp);
		return -1;
	}
	for (j = 1; j < args->nstrings; j++)
	{
		time_t		mtime;
		long		depth;

		mtime = cook_mtime_newest(args->string[j], &depth);
		if (mtime < 0)
			return -1;
		if (mtime == 0)
			string_list_append(result, str_false);
		else
		{
			struct tm	*tm;
			string_ty	*s;

			tm = localtime(&mtime);
			s =
				str_format
				(
					"%02d%02d%02d%02d%02d%02d",
					tm->tm_year,
					tm->tm_mon+1,
					tm->tm_mday,
					tm->tm_hour,
					tm->tm_min,
					tm->tm_sec
				);
			string_list_append(result, s);
			str_free(s);
		}
	}
	return 0;
}


/*
 * NAME
 *	builtin_resolve - resolve search path
 *
 * SYNOPSIS
 *	int builtin_resolve(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	The resolve function is a built-in of cook, described as follows:
 *	This builtin function is used to resolve file names when using
 *	the search_list variable to locate files.  This builtin
 *	function produces resolved file names as output.  This is
 *	useful when taking partial copies of a source to perform
 *	controlled updates.  The targets of recipes are always cooked
 *	into the current directory.
 *
 * RETURNS
 *	A word list containing the resolved names.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a string_list_destructor() call.
 */

int
builtin_resolve(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	trace(("resolve\n"));
	assert(result);
	assert(args);
	assert(args->nstrings);
	return cook_mtime_resolve(result, args, 1);
}


/*
 * NAME
 *	builtin_sort_newest - sort the arguments
 *
 * SYNOPSIS
 *	int builtin_sort_newest(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	The sort_newest function is a built-in of cook, described as
 *	follows: sorts the arguments my their last-modified file times,
 *	youngest to oldest.
 *	This function requires zero or more arguments.
 *
 * RETURNS
 *	A sorted word list.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a string_list_destructor() call.
 */

static int cmp _((const void *, const void *));

static int
cmp(va, vb)
	const void	*va;
	const void	*vb;
{
	string_ty	*a;
	string_ty	*b;
	long		tmp;
	long		da;
	long		db;

	a = *(string_ty **)va;
	b = *(string_ty **)vb;
	tmp = cook_mtime_newest(b, &db) - cook_mtime_newest(a, &da);
	if (tmp == 0)
		return 0;
	return (tmp < 0 ? -1 : 1);
}


int
builtin_sort_newest(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	int		j;
	int		start;

	trace(("sort_newest\n"));
	assert(result);
	assert(args);
	switch (args->nstrings)
	{
	case 0:
		assert(0);

	case 1:
		return 0;

	case 2:
		string_list_append(result, args->string[1]);
		return 0;
	}
	start = result->nstrings;
	for (j = 1; j < args->nstrings; ++j)
		string_list_append(result, args->string[j]);
	qsort
	(
		&result->string[start],
		args->nstrings - 1,
		sizeof(result->string[0]),
		cmp
	);
	return 0;
}


/*
 * NAME
 *	builtin_cando - test if know how to cook given files
 *
 * SYNOPSIS
 *	int builtin_cando(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	The cando function is a built-in of cook, described as follows:
 *	returns true if all the arguments are known how to be cooked.
 *	This function requires one or more arguments.
 *
 * RETURNS
 *	A word list containing true ("1") if can do all arguments,
 *	or false ("") if one or more could not.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a string_list_destructor() call.
 */

int
builtin_cando(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	int		j;
	int		ret;

	trace(("cando\n"));
	assert(result);
	assert(args);
	assert(args->nstrings);
	for (j = 1; j < args->nstrings; ++j)
	{
		ret = cando(args->string[j]);
		if (!ret)
		{
			string_list_append(result, str_false);
			return 0;
		}
	}
	string_list_append(result, str_true);
	return 0;
}


/*
 * NAME
 *	builtin_uptodate - test if files are up to date
 *
 * SYNOPSIS
 *	int builtin_uptodate(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	The uptodate function is a built-in of cook, described as follows:
 *	returns true if all the arguments are up-to-date.
 *	Does not build them if they are not.
 *	This function requires one or more arguments.
 *
 * RETURNS
 *	A word list containing true ("1") if all arguments are up-to-date,
 *	or false ("") if one or more could not.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a string_list_destructor() call.
 */

int
builtin_uptodate(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	int		j;
	int		ret;

	trace(("uptodate\n"));
	assert(result);
	assert(args);
	assert(args->nstrings);
	for (j = 1; j < args->nstrings; ++j)
	{
		ret = isit_uptodate(args->string[j]);
		if (ret < 0)
			return -1;
		if (!ret)
		{
			string_list_append(result, str_false);
			return 0;
		}
	}
	string_list_append(result, str_true);
	return 0;
}
