/*
 *	VME Linux/m68k Loader
 *
 *  Dump the contents of a boot loader file (for debugging)
 *
 *  (C) Copyright 1996 by Geert Uytterhoeven
 *		       (Geert.Uytterhoeven@cs.kuleuven.ac.be)
 *
 *  --------------------------------------------------------------------------
 *
 *  Usage:
 *
 *      dumpmapfile [-v|--verbose] [<loaderfile>]
 *
 *  With:
 *
 *      <loaderfile>   : path of the map file (default: `/boot/boot.loader')
 *
 *  --------------------------------------------------------------------------
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License.  See the file COPYING for more details.
 */


#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/fcntl.h>

#include "../loader.h"
#include "../vmelilo.h"


static const char *ProgramName;


void __attribute__ ((noreturn, format (printf, 1, 2))) Die(const char *fmt, ...)
{
	va_list ap;

	fflush(stdout);
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	exit(1);
}


void __attribute__ ((noreturn)) Usage(void)
{
	Die("Usage: %s [-v|--verbose] [loaderfile]\n", ProgramName);
}


int main(int argc, char *argv[])
{
	int fh, size = 0;
	u_char *data;
	const TAGRECORD *tr;
	u_long offset = 0;
	const char *loaderfile = NULL;
	int verbose = 0;

	ProgramName = argv[0];
	argc--;
	argv++;

	while (argc) {
		if (!strcmp(argv[0], "-h") || !strcmp(argv[0], "--help"))
			Usage();
		else if (!strcmp(argv[0], "-v") || !strcmp(argv[0], "--verbose"))
			verbose = 1;
		else if (!loaderfile)
			loaderfile = argv[0];
		else
			Usage();
		argc--;
		argv++;
	}
	if (!loaderfile)
		loaderfile = LILO_LOADERFILE;

	if ((fh = open(loaderfile, O_RDONLY)) == -1)
		Die("open %s: %s\n", loaderfile, strerror(errno));
	if ((size = lseek(fh, 0, SEEK_END)) == -1)
		Die("lseek %s: %s\n", loaderfile, strerror(errno));
	if (!(data = malloc(size)))
		Die("No memory\n");
	if (lseek(fh, 0, SEEK_SET) == -1)
		Die("lseek %s: %s\n", loaderfile, strerror(errno));
	if (read(fh, data, size) != size)
		Die("read %s: %s\n", loaderfile, strerror(errno));
	close(fh);

	{
		const u_long	pattern[]		= { LILO_ID, LILO_MAPDATAID };
		int				i;
	
		for (i = 0; i < (size - sizeof(pattern) + 1); i++)
		{
			if (memcmp(&data[i], pattern, sizeof(pattern)) == 0)
			{
				/* get offset immediately following pattern */
				offset = i + sizeof(pattern) + 4;
				goto found_it;
			}
		}

		Die("Couldn't find magic key in loader\n");

	found_it:
	}

	while (1) {
		tr = (TAGRECORD *)&data[offset];
		if (verbose)
			printf("0x%08lx: tag = 0x%08lx, size = 0x%08lx\n\t", offset,
				   tr->tagid, tr->size);
		switch (tr->tagid) {
			case TAG_LILO:
				printf("  TAG_LILO: File Identification\n");
				break;
			case TAG_EOF:
				printf("  TAG_EOF: End of File\n");
				break;
			case TAG_HEADER:
				printf("    TAG_HEADER: Start of Boot Options\n");
				break;
			case TAG_HEADER_END:
				printf("    TAG_HEADER_END: End of Boot Options\n");
				break;
			case TAG_DEFAULT:
				printf("      TAG_DEFAULT: Default Boot Record = `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_MESSAGE:
				printf("      TAG_MESSAGE: Boot Message File = `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_PROMPT:
				printf("      TAG_PROMPT: Prompt = %ld\n", tr->data[0]);
				break;
			case TAG_RESTRICTED:
				printf("      TAG_RESTRICTED: Restricted = %ld\n", tr->data[0]);
				break;
			case TAG_CALLMONITOR:
				printf("      TAG_CALLMONITOR: Call monitor before boot = %ld\n", tr->data[0]);
				break;
			case TAG_DELAY:
				printf("      TAG_DELAY: Delay = %ld seconds\n",
					   tr->data[0]);
				break;
			case TAG_TIMEOUT:
				printf("      TAG_TIMEOUT: Timeout = %ld seconds\n",
					   tr->data[0]);
				break;
			case TAG_DEBUG:
				printf("      TAG_DEBUG: Debug = %ld\n", tr->data[0]);
				break;
			case TAG_BOOT_RECORD:
				printf("    TAG_BOOT_RECORD: Start of Boot Record `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_BOOT_RECORD_END:
				printf("    TAG_BOOT_RECORD_END: End of Boot Record\n");
				break;
			case TAG_ALIAS:
				printf("      TAG_ALIAS: Alias Label = `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_KERNEL:
				printf("      TAG_KERNEL: Kernel Image = `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_SYMTAB:
				printf("      TAG_SYMTAB: Symbol File = `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_ARGUMENTS:
				printf("      TAG_ARGUMENTS: Boot Arguments = `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_PASSWORD:
				printf("      TAG_PASSWORD: Boot Record Password = `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_RAMDISK:
				printf("      TAG_RAMDISK: Ramdisk Image = `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_MEMSIZE:
				printf("      TAG_MEMSIZE: DRAM size = `%ld'\n",
					   tr->data[0]);
				break;
			case TAG_FILE_DEF:
				printf("    TAG_FILE_DEF: Start of File Definition `%s'\n",
					   (char *)tr->data);
				break;
			case TAG_FILE_DEF_END:
				printf("    TAG_FILE_DEF_END: End of File Definition\n");
				break;
			case TAG_FILEMAP:
				printf("      TAG_FILEMAP: File map containing %ld bytes in %ld fragments\n",
					   tr->data[0], tr->data[1]);
				break;
			default:
				printf("    <UNKNOWN>\n");
				break;
		}
		offset += sizeof(TAGRECORD) + tr->size;
		if (offset > size) {
			puts("Warning: Beyond end of file");
			break;
		}
		if (tr->tagid == TAG_EOF)
			break;
	}
	if (offset < size)
		printf("Warning: Left %ld bytes\n", size-offset);
	return(0);
}
