#include "config.h"
#if HAVE_MSDOS_PARTITION

/*
 * glibc 2.1 strxxx macros conflicts with linux/string.h included from
 * asm/unaligned.h if the following name is not defined
 */
#define __NO_STRING_INLINES

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <asm/types.h>
#include "byteorder.h"

#include "fdisk.h"

/*
 * Many architectures don't like unaligned accesses, which is
 * frequently the case with the nr_sects and start_sect partition
 * table entries.
 */
#include <asm/unaligned.h>

#define SYS_IND(p)	(get_unaligned(&p->sys_ind))
#define NR_SECTS(p)	({ __typeof__(p->nr_sects) __a =	\
				get_unaligned(&p->nr_sects);	\
				__le32_to_cpu(__a); \
			})

#define START_SECT(p)	({ __typeof__(p->start_sect) __a =	\
				get_unaligned(&p->start_sect);	\
				__le32_to_cpu(__a); \
			})

/* These three have identical behaviour; use the second one if DOS fdisk gets
   confused about extended/logical partitions starting past cylinder 1023. */
#define DOS_EXTENDED_PARTITION 5
#define LINUX_EXTENDED_PARTITION 0x85
#define WIN98_EXTENDED_PARTITION 0x0f
#define LINUX_SWAP_PARTITION	0x82
#define SOLARIS_X86_PARTITION	LINUX_SWAP_PARTITION
    
	
struct partition {
	__u8 boot_ind;		/* 0x80 - active */
	__u8 head;		/* starting head */
	__u8 sector;		/* starting sector */
	__u8 cyl;		/* starting cylinder */
	__u8 sys_ind;		/* What partition type */
	__u8 end_head;		/* end head */
	__u8 end_sector;	/* end sector */
	__u8 end_cyl;		/* end cylinder */
	__u32 start_sect;	/* starting sector counting from 0 */
	__u32 nr_sects;		/* nr of sectors in partition */
};


#if HAVE_BSD_DISKLABEL
/*
 * BSD disklabel support by Yossi Gottlieb <yogo@math.tau.ac.il>
 */

#define BSD_PARTITION		0xa5	/* Partition ID */

#define BSD_DISKMAGIC	(0x82564557UL)	/* The disk magic number */
#define BSD_MAXPARTITIONS	8
#define BSD_FS_UNUSED		0	/* disklabel unused partition entry ID */
struct bsd_disklabel {
	__u32	d_magic;		/* the magic number */
	__s16	d_type;			/* drive type */
	__s16	d_subtype;		/* controller/d_type specific */
	char	d_typename[16];		/* type name, e.g. "eagle" */
	char	d_packname[16];		/* pack identifier */ 
	__u32	d_secsize;		/* # of bytes per sector */
	__u32	d_nsectors;		/* # of data sectors per track */
	__u32	d_ntracks;		/* # of tracks per cylinder */
	__u32	d_ncylinders;		/* # of data cylinders per unit */
	__u32	d_secpercyl;		/* # of data sectors per cylinder */
	__u32	d_secperunit;		/* # of data sectors per unit */
	__u16	d_sparespertrack;	/* # of spare sectors per track */
	__u16	d_sparespercyl;		/* # of spare sectors per cylinder */
	__u32	d_acylinders;		/* # of alt. cylinders per unit */
	__u16	d_rpm;			/* rotational speed */
	__u16	d_interleave;		/* hardware sector interleave */
	__u16	d_trackskew;		/* sector 0 skew, per track */
	__u16	d_cylskew;		/* sector 0 skew, per cylinder */
	__u32	d_headswitch;		/* head switch time, usec */
	__u32	d_trkseek;		/* track-to-track seek, usec */
	__u32	d_flags;		/* generic flags */
#define NDDATA 5
	__u32	d_drivedata[NDDATA];	/* drive-type specific information */
#define NSPARE 5
	__u32	d_spare[NSPARE];	/* reserved for future use */
	__u32	d_magic2;		/* the magic number (again) */
	__u16	d_checksum;		/* xor of data incl. partitions */

			/* filesystem and partition information: */
	__u16	d_npartitions;		/* number of partitions in following */
	__u32	d_bbsize;		/* size of boot area at sn0, bytes */
	__u32	d_sbsize;		/* max size of fs superblock, bytes */
	struct	bsd_partition {		/* the partition table */
		__u32	p_size;		/* number of sectors in partition */
		__u32	p_offset;	/* starting sector */
		__u32	p_fsize;	/* filesystem basic fragment size */
		__u8	p_fstype;	/* filesystem type, see below */
		__u8	p_frag;		/* filesystem fragments per block */
		__u16	p_cpg;		/* filesystem cylinders per group */
	} d_partitions[BSD_MAXPARTITIONS];	/* actually may be more */
};
#endif	/* HAVE_BSD_DISKLABEL */

#if HAVE_SOLARIS_X86_PARTITION
#define SOLARIS_X86_NUMSLICE	8
#define SOLARIS_X86_VTOC_SANE	(0x600DDEEEUL)

struct solaris_x86_slice {
	__u16	s_tag;			/* ID tag of partition */
	__u16	s_flag;			/* permision flags */
	__u32	s_start;		/* start sector no of partition */
	__s32	s_size;			/* # of blocks in partition */
};

struct solaris_x86_vtoc {
	__u32 v_bootinfo[3];	/* info needed by mboot (unsupported) */
	__u32 v_sanity;		/* to verify vtoc sanity */
	__u32 v_version;	/* layout version */
	char	v_volume[8];		/* volume name */
	__u16	v_sectorsz;		/* sector size in bytes */
	__u16	v_nparts;		/* number of partitions */
	__u32 v_reserved[10];	/* free space */
	struct solaris_x86_slice
		v_slice[SOLARIS_X86_NUMSLICE]; /* slice headers */
	time_t	timestamp[SOLARIS_X86_NUMSLICE]; /* timestamp (unsupported) */
	char	v_asciilabel[128];	/* for compatibility */
};
#endif /* CONFIG_SOLARIS_X86_PARTITION */

static int current_minor;

static inline int is_extended_partition(struct partition *p)
{
	return (SYS_IND(p) == DOS_EXTENDED_PARTITION ||
		SYS_IND(p) == WIN98_EXTENDED_PARTITION ||
		SYS_IND(p) == LINUX_EXTENDED_PARTITION);
}

/*
 * Create devices for each logical partition in an extended partition.
 * The logical partitions form a linked list, with each entry being
 * a partition table with two entries.  The first entry
 * is the real data partition (with a start relative to the partition
 * table start).  The second is a pointer to the next logical partition
 * (with a start relative to the entire extended partition).
 * We do not create a Linux partition for the partition tables, but
 * only for the actual data partitions.
 */

#define MSDOS_LABEL_MAGIC		0xAA55

static void extended_partition(char *device, int fd, unsigned long startsec,
			       unsigned long size)
{
    struct partition *p;
    unsigned int first_sector, first_size, this_sector, this_size;
    int i;
    unsigned char data[512];

    first_sector = startsec;
    first_size = size;
    this_sector = first_sector;
    this_size = size;
    
    while (1) {
	if (!sread( fd, this_sector, data ))
	    return;
	if ((*(unsigned short *) (data+510)) != __cpu_to_le16(MSDOS_LABEL_MAGIC))
	    return;
	
	p = (struct partition *) (0x1BE + data);
	
	/*
	 * Usually, the first entry is the real data partition,
	 * the 2nd entry is the next extended partition, or empty,
	 * and the 3rd and 4th entries are unused.
	 * However, DRDOS sometimes has the extended partition as
	 * the first entry (when the data partition is empty),
	 * and OS/2 seems to use all four entries.
	 */
	
	/* 
	 * First process the data partition(s)
	 */
	for (i=0; i<4; i++, p++) {
	    if (!NR_SECTS(p) || is_extended_partition(p))
		continue;
	    
	    /* Check the 3rd and 4th entries -
	       these sometimes contain random garbage */
	    if (i >= 2
		&& START_SECT(p) + NR_SECTS(p) > this_size
		&& (this_sector + START_SECT(p) < first_sector ||
		    this_sector + START_SECT(p) + NR_SECTS(p) >
		    first_sector + first_size))
		continue;
	    
	    fdisk_add_partition(device, current_minor, SYS_IND(p),
				NR_SECTS(p)/2);
	    current_minor++;
	}
	/*
	 * Next, process the (first) extended partition, if present.
	 * (So far, there seems to be no reason to make
	 *  extended_partition()  recursive and allow a tree
	 *  of extended partitions.)
	 * It should be a link to the next logical partition.
	 * Create a minor for this just long enough to get the next
	 * partition table.  The minor will be reused for the next
	 * data partition.
	 */
	p -= 4;
	for (i=0; i<4; i++, p++)
	    if(NR_SECTS(p) && is_extended_partition(p))
		break;
	if (i == 4)
	    return;	 /* nothing left to do */
	
	this_sector = first_sector + START_SECT(p);
	this_size = NR_SECTS(p);
    }
}

#if HAVE_BSD_DISKLABEL
/* 
 * Create devices for BSD partitions listed in a disklabel, under a
 * dos-like partition. See extended_partition() for more information.
 */
static void
bsd_disklabel_partition(char *device, int fd, unsigned long startsec)
{
    struct bsd_disklabel *l;
    struct bsd_partition *p;
    unsigned char data[512];

    if (!sread( fd, startsec, data ))
	return;
    l = (struct bsd_disklabel *) (data+512);
    if (l->d_magic != __cpu_to_le32(BSD_DISKMAGIC))
	return;

    p = &l->d_partitions[0];
    while (p - &l->d_partitions[0] <= BSD_MAXPARTITIONS) {
	if (p->p_fstype != BSD_FS_UNUSED) {
	    fdisk_add_partition(device, current_minor,
				PTYPE_PREFIX_BSD | p->p_fstype,
				__le32_to_cpu(p->p_size)/2);
	    current_minor++;
	}
	p++;
    }
}
#endif /* HAVE_BSD_DISKLABEL */

#if HAVE_SOLARIS_X86_PARTITION
static void
solaris_x86_partition(char *device, int fd, long offset)
{
    struct solaris_x86_vtoc *v;
    struct solaris_x86_slice *s;
    unsigned char data[512];
    int i;

    if (!sread( fd, offset, data ))
	return;
    v = (struct solaris_x86_vtoc *)(data + 512);
    if(v->v_sanity != __cpu_to_le32(SOLARIS_X86_VTOC_SANE))
	return;
    if(v->v_version != __cpu_to_le32(1))
	return;
    for(i=0; i<SOLARIS_X86_NUMSLICE; i++) {
	s = &v->v_slice[i];
	
	if (s->s_tag == __cpu_to_le16(0))
	    continue;
	/* solaris partitions are relative to current MS-DOS
	 * one but add_partition starts relative to sector
	 * zero of the disk.  Therefore, must add the offset
	 * of the current partition */
	fdisk_add_partition(device, current_minor,
			    /* tag currently ignored, don't have infos about
			     * it yet :-( List all partitions in here as
			     * "Solaris/x86" */
			    PTYPE_PREFIX_SOLARIS_X86 /*| __le16_to_cpu(s->s_tag)*/,
			    __le32_to_cpu(s->s_size)/2);
	current_minor++;
    }
}
#endif

int parse_msdos_partition(char *device, int fd)
{
    int i, minor = 1;
    struct partition *p;
    unsigned char data[512];

    if (!sread( fd, 0, data ))
	return -1;
    if (*(unsigned short *) (0x1fe + data) != __cpu_to_le16(MSDOS_LABEL_MAGIC))
	return 0;
    p = (struct partition *) (0x1be + data);

    current_minor = minor+4; /* first "extra" minor (for extended partitions) */
    for (i=1 ; i<=4 ; minor++,i++,p++) {
	if (!NR_SECTS(p) || !SYS_IND(p))
	    continue;
	fdisk_add_partition(device, minor, SYS_IND(p), NR_SECTS(p)/2);
	if (is_extended_partition(p)) {
	    extended_partition(device, fd, START_SECT(p), NR_SECTS(p));
	}
#if HAVE_BSD_DISKLABEL
	if (SYS_IND(p) == BSD_PARTITION) {
	    bsd_disklabel_partition(device, fd, START_SECT(p));
	}
#endif
#if HAVE_SOLARIS_X86_PARTITION
	/* james@bpgc.com: Solaris has a nasty indicator: 0x82
	 * which also means linux swap.  For that reason, all
	 * of the prints are done inside the
	 * solaris_x86_partition routine */
	if (SYS_IND(p) == SOLARIS_X86_PARTITION) {
	    solaris_x86_partition(device, fd, START_SECT(p));
	}
#endif
    }
    /*
     *  Check for old-style Disk Manager partition table
     */
    if (*(unsigned short *) (data+0xfc) == __cpu_to_le16(MSDOS_LABEL_MAGIC)) {
	p = (struct partition *) (0x1be + data);
	for (i = 4 ; i < 16 ; i++, current_minor++) {
	    p--;
	    if (!(START_SECT(p) && NR_SECTS(p)))
		continue;
	    fdisk_add_partition(device, current_minor, SYS_IND(p),
				NR_SECTS(p)/2);
	}
    }
    return 1;
}

#endif /* HAVE_MSDOS_PARTITION */
