// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include <qmsgbox.h>
#include "CatalogEntry.h"
#include "Options.h"
#include "Request.h"
#include "SgmlCatalog.h"
#include "StyleParser.h"
#include "StyleSheetManager.h"

StyleSheetManager::StyleSheetManager()
{
}

StyleSheetManager::~StyleSheetManager()
{
    while ( _styleSheets.first() ) {
	delete _styleSheets.first();
	_styleSheets.remove();
    }
    while ( _receivers.first() ) {
	delete _receivers.first();
	_receivers.remove();
    }
    while ( _requests.first() ) {
	delete _requests.first();
	_requests.remove();
    }
}

StyleSheet* StyleSheetManager::findStyleSheet( const QString& soi )
{
    QString tmp = "file:";
    tmp += options->catalogFile();
    Url url( tmp );
    Url styleSheetUrl( &url, soi );
    QListIterator<StyleSheet> i( _styleSheets );
    for ( i.toFirst(); i.current(); ++i ) {
	if ( i.current()->soi() == styleSheetUrl.url() ) {
	    return i.current();
	}
    }

    return 0;
}

void StyleSheetManager::request( StyleSheetReceiver* receiver, QString doctype, QString id )
{
    // Look for the public ID in the catalog, to get the SOI.
    CatalogEntry* e = sgmlCatalog->find( CatalogEntry::Style, id );
    if ( !e ) {
        // Try doctype.
        if ( !( e = sgmlCatalog->find( CatalogEntry::Style, doctype ) ) ) {
            QString error;
            error.sprintf( "Cannot resolve Style Sheet reference '%s'.\nPlease check the entry in the catalog.", id.data() );
            QMessageBox::message( "QWeb: Error", error );
            return;
        }
    }

    requestBySOI( receiver, e->soi() );
}

void StyleSheetManager::requestBySOI( StyleSheetReceiver* receiver, QString soi )
{
    StyleSheet* styleSheet;

    if ( ( styleSheet = findStyleSheet( soi ) ) ) {
        receiver->styleSheet( styleSheet );
        return;
    }
	
    // Create a new request.
    Receiver* par = new Receiver( receiver, soi );
	
    // Has the same Style Sheet already been requested?
    for ( StyleSheetRequest* r = _requests.first(); r; r = _requests.next() ) {
        if ( r->soi == soi ) {
            _receivers.append( par );
            return;
        }
    }
	
    // Search the catalog for the dtd.
    styleSheet = new StyleSheet;
    par->styleSheet = styleSheet;
    StyleParser* styleParser = new StyleParser( styleSheet );
    connect( styleParser, SIGNAL( done( StyleSheet* ) ), this, SLOT( done( StyleSheet* ) ) );

    // Resolve Style Sheet URLs relative to the location of the catalog.
    QString tmp = "file:";
    tmp += options->catalogFile();
    Url url( tmp );
    Url styleUrl( &url, soi );
    styleSheet->soi( styleUrl.url() );

    // Issue requests for the DTD and Style Sheet.
    Request* req = new Request( styleUrl );
    connect( req, SIGNAL( data( const char*, int ) ), styleParser, SLOT( data( const char*, int ) ) );
    connect( req, SIGNAL( endOfData() )             , styleParser, SLOT( endOfData() ) );

    if ( req->open() ) {
        _requests.append( new StyleSheetRequest( req, styleSheet, soi ) );
        _receivers.append( par );
    } else {
        // Cannot open connection.
        delete req;
        delete styleParser;
        delete styleSheet;
        delete par;
            
        QString error;
        error.sprintf( "Cannot open Style Sheet '%s'.\nPlease check the entry in the catalog.", styleUrl.url().data() );
        QMessageBox::message( "QWeb: Error", error );
        
        receiver->styleSheet( 0 );
    }
}

void StyleSheetManager::clear()
{
    while ( _styleSheets.first() ) {
	delete _styleSheets.first();
	_styleSheets.remove();
    }
}

void StyleSheetManager::done( StyleSheet* styleSheet )
{
    for ( StyleSheetRequest* r = _requests.first(); r; r = _requests.next() ) {
        if ( r->styleSheet == styleSheet ) {
            _styleSheets.append( styleSheet );
            _requests.remove( r );
            delete r;
        }
    }
	
    for ( Receiver* p = _receivers.first(); p; ) {
        if ( p->styleSheet == styleSheet ) {
            p->receiver->styleSheet( styleSheet );
            _receivers.remove();
            delete p;
            p = _receivers.current();
        } else {
            p = _receivers.next();
        }
    }
}

StyleSheet* StyleSheetManager::find( QString styleSheetName )
{
    // Look for the public ID in the catalog, to get the SOI.
    CatalogEntry* e = sgmlCatalog->find( CatalogEntry::Style, styleSheetName );
    if ( !e ) {
        QString error;
        error.sprintf( "Cannot resolve Style Sheet reference '%s'.\nPlease check the entry in the catalog.", styleSheetName.data() );
        QMessageBox::message( "QWeb: Error", error );
        return 0;
    }

    return findStyleSheet( e->soi() );
}
