/* Copyright (C) 1994 
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

 This library is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <X11/Xlib.h>
#include <stdio.h>
#include <string.h>
#include "McAlloc.h"
#include "McApp.h"
#include "McGadget.h"
#include "McSlider.h"

static void GadgetUpdate(McGadget *gadget, int busy, int all);
static void GadgetBorderRedraw(McGadget *gadget);

McGadget *McCreateGadget(McWindow *mcw, unsigned int flags, unsigned int type,
			 int x, int y, int w, int h) {
  McGadget *gadget = (McGadget *) McAlloc(sizeof(McGadget));
  McApp *app = mcw->app;

  gadget->type = type;
  if (app->flags & MCAPP_CLIP) {
    gadget->flags = flags | GAD_CLIP;
  } else {
    gadget->flags = flags;
  }
  gadget->normalBitmap = gadget->selectBitmap = NULL;
  gadget->normalLabel = gadget->selectLabel = NULL;
  gadget->callbackDown = gadget->callbackUp = NULL;
  gadget->topLeftGravity = NorthWestGravity;
  gadget->bottomRightGravity = ForgetGravity;
  gadget->x = x;
  gadget->y = y;
  gadget->width = w;
  gadget->height = h;
  gadget->id = 0;
  gadget->mcw = mcw;
  gadget->customData = gadget->specialInfo = NULL;
  McAddGadgetToList(mcw, gadget);
  return gadget;
}

void McMoveGadget(McGadget *gadget, int x, int y) {
  if (x>=0) gadget->x=x;
  if (y>=0) gadget->y=y;
}

void McResizeGadget(McGadget *gadget, int w, int h) {
  if (w>=0) gadget->width=w;
  if (h>=0) gadget->height=h;
}

void McInitGadgets(McWindow *mcw) {
  McGadget *next;
  XSetWindowAttributes attr;

  /* We can use NorthWestGravity only if no gadget tends to resize itself.  */
  /* But if we can use it, we will since it gives a small performance boost */
  /* (Or a big boost if the user uses an opaque-resizing window manager.)   */
  attr.bit_gravity = NorthWestGravity;
  next=mcw->firstGadget;
  while(next) {
    if ((next->topLeftGravity != NorthWestGravity) ||
	(next->bottomRightGravity != ForgetGravity)) {
      attr.bit_gravity = ForgetGravity;
      break;
    }
    next=next->next;
  }

  XChangeWindowAttributes(mcw->app->display, mcw->window, CWBitGravity, &attr);
}

static void GadgetBorderRedraw(McGadget *gadget) {
  int i;
  McWindow *mcw=gadget->mcw;
  McApp *app=mcw->app;

  if (gadget->flags & GAD_INVIS) return;
  if (gadget->flags & GAD_STATE) {  /* Highlighted */
    if (gadget->flags & GAD_3D) {
      i = _3D_IN;
    } else {
      switch (gadget->flags & GAD_HMASK) {
      case GAD_H3D:
	i = _3D_IN;
	break;
      default:
	i = _3D_NONE;
	break;
      }
    }

    if (!(gadget->flags & GAD_NOFILL))
      XFillRectangle(app->display, mcw->window, app->gc[GC_SELECTED],
		     gadget->x, gadget->y, gadget->width, gadget->height);

#if 0
    printf("XFillRectangle(display, window, gc, %d, %d, %d, %d);\n",
	   gadget->x, gadget->y, gadget->width, gadget->height);
#endif

    if (gadget->specialInfo && gadget->specialInfo->borderProc) {
      (*(gadget->specialInfo->borderProc))(gadget, i);
    } else {
      McAppDrawbox(mcw, mcw->window,
		   gadget->x, gadget->y, gadget->width, gadget->height, i);
    }
  } else { /* Normal */
    if (!(gadget->flags & GAD_NOFILL))
      XFillRectangle(mcw->app->display, mcw->window, mcw->app->gc[GC_CLEAR],
		     gadget->x, gadget->y, gadget->width, gadget->height);

    if (gadget->flags & GAD_3D)
      if (gadget->specialInfo && gadget->specialInfo->borderProc) {
	(*(gadget->specialInfo->borderProc))(gadget, _3D_OUT);
      } else {
	McAppDrawbox(mcw, mcw->window, gadget->x, gadget->y,
		     gadget->width, gadget->height, _3D_OUT);
      }
  }
}

static void GadgetRedraw(McGadget *gadget, int busy) {
  McWindow *mcw=gadget->mcw;

  if (gadget->flags & GAD_NOSTIPPLE) busy=0;
  GadgetBorderRedraw(gadget);
  GadgetUpdate(gadget, busy, 1);
  if (busy || (gadget->flags & GAD_DISABLE)) {
    XFillRectangle(mcw->app->display, mcw->window, mcw->app->gc[GC_BUSY],
		   gadget->x, gadget->y, gadget->width, gadget->height);
  }
}

static void GadgetUpdate(McGadget *gadget, int busy, int all) {
  McText *txt;
  McBitmap *bmp;
  XRectangle rect;
  McWindow *mcw=gadget->mcw;
  McApp *app=mcw->app;

  if (gadget->flags & GAD_NOSTIPPLE) busy=0;

  if (gadget->flags & GAD_INVIS) return;
  if (gadget->flags & GAD_STATE) {  /* Highlighted */
    if (gadget->specialInfo && gadget->specialInfo->updateProc)
      (*(gadget->specialInfo->updateProc))(gadget, busy, all);

    if (gadget->flags & GAD_CLIP) {
      rect.x=gadget->x+BW; rect.y=gadget->y+BW;
      rect.width=gadget->width-BW-BW;
      rect.height=gadget->height-BW-BW;
      McSetClipRectangle(mcw, app->gc[GC_SELECTED_BITMAP], &rect);
    }
    if ((bmp=gadget->selectBitmap) || (bmp=gadget->normalBitmap))
      McPutBitmap(mcw, 0, bmp, app->gc[GC_SELECTED_BITMAP],
		  gadget->x, gadget->y);
    if ((txt=gadget->selectLabel) || (txt=gadget->normalLabel))
      McWriteText(mcw, 0, txt, app->gc[GC_SELECTED_BITMAP],
		  gadget->x, gadget->y);
    if (gadget->flags & GAD_CLIP) {
      McClearClipRectangle(mcw, app->gc[GC_SELECTED_BITMAP]);
    }
  } else {
    if (gadget->specialInfo && gadget->specialInfo->updateProc)
      (*(gadget->specialInfo->updateProc))(gadget, busy, all);

    if (gadget->flags & GAD_CLIP) {
      rect.x=gadget->x+BW; rect.y=gadget->y+BW;
      rect.width=gadget->width-BW-BW;
      rect.height=gadget->height-BW-BW;
      McSetClipRectangle(mcw, app->gc[GC_NORMAL], &rect);
    }
    if ((bmp=gadget->normalBitmap))
      McPutBitmap(mcw, 0, bmp, app->gc[GC_NORMAL], gadget->x, gadget->y);
    if ((txt=gadget->normalLabel))
      McWriteText(mcw, 0, txt, app->gc[GC_NORMAL], gadget->x, gadget->y);
    if (gadget->flags & GAD_CLIP) {
      McClearClipRectangle(mcw, app->gc[GC_NORMAL]);
    }
  }
}

static void set_state(McGadget *gadget) {
  if (gadget->flags&GAD_SELECTED)
    gadget->flags|=GAD_STATE;
  else
    gadget->flags&=~GAD_STATE;
}

void McGadgetRedraw(McGadget *gadget) {
  set_state(gadget);
  GadgetRedraw(gadget, 0);
}

void McGadgetBorderRedraw(McGadget *gadget, int busy) {
  McWindow *mcw=gadget->mcw;
  if (gadget->flags & GAD_NOSTIPPLE) busy=0;
  GadgetBorderRedraw(gadget);
  if (busy || (gadget->flags & GAD_DISABLE)) {
    XFillRectangle(mcw->app->display, mcw->window, mcw->app->gc[GC_BUSY],
		   gadget->x, gadget->y, gadget->width, gadget->height);
  }
}

void McGadgetUpdate(McGadget *gadget) {
  set_state(gadget);
  GadgetUpdate(gadget, 0, 0);
}

void McGadgetBusy(McGadget *gadget) {
  set_state(gadget);
  GadgetRedraw(gadget, 1);
}

void McReleaseGadget(McGadget *gadget, int inside) {
  McWindow *mcw=gadget->mcw;
  gadget->flags&=~GAD_PRESSED;
  if (inside || (gadget->flags & GAD_NORELVERIFY)) {
    if (gadget->callbackUp) {
      if ((gadget->flags & GAD_HMASK) != GAD_HNONE) {
	set_state(gadget);
	GadgetRedraw(gadget, 1);
	XFlush(mcw->app->display);
      }
      (*gadget->callbackUp)(gadget);
    }
    if ((gadget->flags & GAD_HMASK) != GAD_HNONE) {
      if ((!gadget->mutualExclude) && (!(gadget->flags&GAD_TOGGLE))) {
	gadget->flags^=GAD_SELECTED;
      }
    }
  } else {
    if ((gadget->flags & GAD_HMASK) != GAD_HNONE) {
      gadget->flags^=GAD_SELECTED;
    }
  }
  if ((gadget->flags & GAD_HMASK) != GAD_HNONE) {
    set_state(gadget);
    GadgetRedraw(gadget, 0);
  }
  gadget->flags&=~GAD_DOUBLE_CLICK;
}

int McGadgetEvent(McGadget *gadget, XEvent *event) {
  McWindow *mcw=gadget->mcw;
  unsigned long excl;
  int in,diff;

  if (!(gadget->flags&GAD_ACTIVE)) return 0;

  if (gadget->specialInfo && gadget->specialInfo->eventProc)
    return (*(gadget->specialInfo->eventProc))(gadget, event);

  if (event->type==Expose) return 0;

  /* Standard bool gadget */
  if (gadget->flags&GAD_PRESSED) {
    switch (event->type) {
    case ButtonRelease:
      if (event->xbutton.button==1) {
	McReleaseGadget(gadget,
			McInRectangle(event->xbutton.x, event->xbutton.y,
				      gadget->x, gadget->y,
				      gadget->width, gadget->height));
	return 1;
      }
      return 0;
    case MotionNotify:
      if (gadget->flags & GAD_NORELVERIFY) return 1;
      in = McInRectangle(event->xmotion.x, event->xmotion.y,
			 gadget->x, gadget->y,
			 gadget->width, gadget->height);
      
      if ((gadget->flags & GAD_HMASK) != GAD_HNONE) {
	if ((in && (gadget->flags&GAD_SELECTED)) ||
	    ((!in) && (!(gadget->flags&GAD_SELECTED)))) {
	  if (!(gadget->flags&GAD_STATE)) {
	    gadget->flags|=GAD_STATE;
	    GadgetRedraw(gadget, 0);
	  }
	} else {
	  if (gadget->flags&GAD_STATE) {
	    gadget->flags&=~GAD_STATE;
	    GadgetRedraw(gadget, 0);
	  }
	}
      }
      return 1;
    case ButtonPress:
      if (event->xbutton.button==1) return 1; else return 0;
    default:
      return 0;
    }
  } else { /* not currently pressed */
    switch (event->type) {
    case ButtonPress:
      if (event->xbutton.button==1) {
	if (McInRectangle(event->xbutton.x, event->xbutton.y,
			  gadget->x, gadget->y,
			  gadget->width, gadget->height)) {

	  diff=event->xbutton.time-gadget->time;
	  if (diff>0 && diff<500) {
	    gadget->flags|=GAD_DOUBLE_CLICK;
	  }
	  gadget->time=event->xbutton.time;
	  gadget->flags|=GAD_PRESSED;
	  if ((excl=gadget->mutualExclude)) {
	    McGadget *next;
	    gadget->flags|=GAD_SELECTED;
	    next = mcw->firstGadget;
	    while(next) {
	      if ((next!=gadget) && (next->mutualExclude & excl)) {
		if (next->flags&GAD_SELECTED) {
		  next->flags^=GAD_SELECTED;
		  set_state(next);
		  GadgetRedraw(next, 0);
		}
	      }
	      next=next->next;
	    }
	  } else {
	    gadget->flags^=GAD_SELECTED;
	  }
	  if (gadget->callbackDown) {
	    if ((gadget->flags & GAD_HMASK) != GAD_HNONE) {
	      set_state(gadget);
	      GadgetRedraw(gadget, 1);
	      XFlush(mcw->app->display);
	    }
	    (*gadget->callbackDown)(gadget);
	  }
	  if ((gadget->flags & GAD_HMASK) != GAD_HNONE) {
	    set_state(gadget);
	    GadgetRedraw(gadget, 0);
	  }
	  McMoveGadgetToStart(gadget);
	  return 1;
	}
	return 0;
      } else {
	return 0;
      }
    default:
      return 0;
    }
    }
  }

#ifdef DEBUG_CODE
void McGadgetInfo(McGadget *gadget) {
  fprintf(stderr,
	  "        x:%d\n"
	  "        y:%d\n"
	  "    width:%d\n"
	  "   height:%d\n"
	  "    flags:$%04x\n"
	  "     type:%d\n",
	  gadget->x, gadget->y, gadget->width, gadget->height,
	  gadget->flags, gadget->type);
}
#endif

void McFreeGadget(McGadget *gadget) {
  McGadgetExtension *ext, *next;
  McWindow *mcw=gadget->mcw;

  if (mcw->keyboardFocus==gadget) {
    if (gadget->specialInfo)
      mcw->keyboardFocus=gadget->specialInfo->next;
    else
      mcw->keyboardFocus=NULL;
  }

  if (mcw->selectionOwner==gadget) mcw->selectionOwner=NULL;

  McWipeGadget(gadget);

  ext=gadget->ext;
  while(ext) {
    next=ext->next; /* Extension may self-destruct while beeing cleaned up */
    if (ext->cleanupProc) (*ext->cleanupProc)(gadget);
    ext=next;
  }

  if (gadget->normalBitmap) McFreeBitmap(mcw->app, gadget->normalBitmap);
  if (gadget->selectBitmap) McFreeBitmap(mcw->app, gadget->selectBitmap);
  if (gadget->normalLabel) McFreeText(gadget->normalLabel);
  if (gadget->selectLabel) McFreeText(gadget->selectLabel);

  if (gadget->specialInfo) {
    if (gadget->specialInfo->cleanupProc)
      (*gadget->specialInfo->cleanupProc)(gadget);
    else
      McFree(gadget->specialInfo);
  }
  McRemoveGadgetFromList(gadget);
  McFree(gadget);
}

/****************************************************************************/

void McWipeGadget(McGadget *gadget) {
  McWindow *mcw=gadget->mcw;
  if (gadget->flags & (GAD_3D | GAD_H3D)) {
    XClearArea(mcw->app->display, mcw->window,
	       gadget->x-BW, gadget->y-BW,
	       gadget->width+BW+BW, gadget->height+BW+BW, True);
  } else {
    XClearArea(mcw->app->display, mcw->window,
	       gadget->x, gadget->y, gadget->width, gadget->height, True);
  }
}

/****************************************************************************/

void McSetFocus(McGadget *gadget) {
  McWindow *mcw=gadget->mcw;
  McGadget *old=mcw->keyboardFocus;
  if (old!=gadget) {
    mcw->keyboardFocus=gadget;
    if (old) {
      McGadgetUpdate(old);
      if (old->flags & GAD_CHANGED) {
	old->flags&=~GAD_CHANGED;
	if (old->callbackUp)
	  (*old->callbackUp)(old);
      }
    }
    McGadgetUpdate(gadget);
  }
}

