#undef DBG_UNUSED_EVENTS
/* Copyright (C) 1994 
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

 This library is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <X11/bitmaps/gray1>
#include <X11/bitmaps/gray3>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include "McAlloc.h"
#include <varargs.h>

#include "McColors.h"
#include "McApp.h"
#include "McGadget.h"
#include "McResource.h"
#ifdef DEBUG_CODE
#include "McDebug.h"
#endif

#undef SHOW_EXPOSE
#undef DBG_RECTANGLE
#undef DBG_EVENTS    

static int HandleExposeEvent(McApp *app, XExposeEvent *event);

char *default_colors[] = COLORS;
char *myname;     /* This is set in McResource.c */

/* STARTUP STARTUP STARTUP STARTUP STARTUP STARTUP STARTUP STARTUP STARTUP */

McApp *McAllocApp(int *ac, char *av[], char *title, char *class,
		  XrmOptionDescRec *DescTable, int DescSize) {
  McApp *app;
  int i;

  if (!(app = (McApp *)McAlloc(sizeof(McApp)))) {
    fprintf(stderr, "%s: out of memory.\n", myname);
    cleanup(1);
  }

  app->default_font_name = DEFAULT_FONT;
#ifdef FIXED_FONT
  app->fixed_font_name = FIXED_FONT;
#endif
  app->wb_font_name = WB_FONT;

  for (i=0;i<MAX_COLORS;i++) app->color_names[i]=strdup(default_colors[i]);

  app->display = McParseOpenDisp(app,ac,av,class,DescTable,DescSize);
  McMergeDatabases(app);

  if (title)
    app->window_title=title;
  else
    app->window_title=class;
  app->class=class;
  McReadStandards(app);

  if (app->flags & MCAPP_SYNCED) {
    fprintf(stderr,"Synchronizing server...\n");
    XSynchronize(app->display,1);
  }

  app->screen = DefaultScreen(app->display);

  return app;
}

McApp *McCreateApp(int *ac, char *av[], char *title, char *class,
		   XrmOptionDescRec *DescTable, int DescSize) {
  McApp *app;
  srandom(time(NULL));
  app=McAllocApp(ac, av, title, class, DescTable, DescSize);
  return McInitApp(app);
}

McApp *McInitApp(McApp *app) {
  int i;
  XGCValues values;
  XColor color, rcolor;
  Display *display = app->display;

  /*
   * Free everything
   */
  McFreeAppPrefs(app);

  app->defaultFont = McLoadFont(display,app->default_font_name);
  if (app->fixed_font_name)
    app->fixedFont = McLoadFont(display,app->fixed_font_name);
  else
    app->fixedFont = app->defaultFont;

  app->wbFont = McLoadFont(display,app->wb_font_name);

  app->firstWindow = app->lastWindow = NULL;

  app->stipple1 = XCreateBitmapFromData(display,
					RootWindow(display, app->screen),
					gray1_bits, gray1_width, gray1_height);

  app->stipple3 = XCreateBitmapFromData(display,
					RootWindow(display, app->screen),
					gray3_bits, gray3_width, gray3_height);

  values.function = GXand;
  values.font = app->defaultFont->fid;
  values.stipple = app->stipple3;
  values.graphics_exposures = False;
  values.fill_style = FillStippled;

  if (DefaultDepth(display, app->screen) > 1) {

    app->flags |= MCAPP_COLOR;

    app->colormap = DefaultColormap(app->display, app->screen);

    for (i = 0; i < MAX_COLORS; i++) {
      if (!XLookupColor(app->display, app->colormap, app->color_names[i],
			&color, &rcolor)) {
	fprintf(stderr,"%s: No color named '%s', using default '%s'.\n",
		myname,app->color_names[i],default_colors[i]);
	if (!XLookupColor(app->display, app->colormap, default_colors[i],
			    &color, &rcolor)) {
	  fprintf(stderr,
		  "%s: Ayieee, Can't allocate default color '%s',"
		  "commiting suicide.\n", myname,default_colors[i]);
	  cleanup(1);
	} else {
	  app->color_names[i]=default_colors[i];
	}
      }
      if (app->flags & MCAPP_REVERSE) {
	rcolor.red=~rcolor.red;
	rcolor.green=~rcolor.green;
	rcolor.blue=~rcolor.blue;
      }

      if(!XAllocColor(app->display, app->colormap, &rcolor)) {
	fprintf(stderr,"%s: Cannot allocate colormap entry for '%s'.\n",
		myname, app->color_names[i]);
	/* Hmm, lets use black for green<0x8000, white else */
	if (rcolor.green & 0x8000)
	  rcolor.pixel = WhitePixel(app->display, app->screen);
	else
	  rcolor.pixel = BlackPixel(app->display, app->screen);
      }

      app->colors[i] = rcolor.pixel;
      continue;

    }

    values.foreground = app->colors[COL_BACKGROUND];
    values.background = app->colors[COL_FOREGROUND];

    app->gc[GC_CLEAR] = XCreateGC(app->display,
				  RootWindow(app->display, app->screen),
				  GCForeground | GCBackground | GCFont |
				  GCGraphicsExposures, &values);

    values.foreground = app->colors[COL_FOREGROUND];

    values.background = app->colors[COL_BACKGROUND];

    app->gc[GC_NORMAL] = XCreateGC(app->display,
				   RootWindow(app->display, app->screen),
				   GCForeground | GCBackground | GCFont |
				   GCGraphicsExposures, &values);

    app->gc[GC_BUSY] = XCreateGC(app->display,
				 RootWindow(app->display, app->screen),
				 GCForeground | GCBackground | GCFont |
				 GCFillStyle | GCStipple | GCGraphicsExposures,
				 &values);
    
    for (i=GC_SELECTED;i<=GC_DARK;i++) {
      app->gc[i] = XCreateGC(app->display,
			     RootWindow(app->display, app->screen),
			     GCForeground | GCBackground |
			     GCFont | GCGraphicsExposures, &values);
    }

    XSetForeground(app->display, app->gc[GC_BRIGHT],
		   app->colors[COL_BRIGHT]);
    XSetForeground(app->display, app->gc[GC_DARK],
		   app->colors[COL_DARK]);

    XSetForeground(app->display, app->gc[GC_SELECTED],
		   app->colors[COL_SELECTED]);
    XSetBackground(app->display, app->gc[GC_SELECTED],
		   app->colors[COL_FOREGROUND]);

    XSetForeground(app->display, app->gc[GC_SELECTED_BITMAP],
		   app->colors[COL_FOREGROUND]);
    XSetBackground(app->display, app->gc[GC_SELECTED_BITMAP],
		   app->colors[COL_SELECTED]);

    XSetForeground(app->display, app->gc[GC_SET_SELECTED_BITMAP],
		   app->colors[COL_FOREGROUND]);
    XSetBackground(app->display, app->gc[GC_SET_SELECTED_BITMAP],
		   app->colors[COL_SELECTED]);
    XSetFunction(app->display, app->gc[GC_SET_SELECTED_BITMAP], GXcopy);

    values.function   = GXcopy;
    app->gc[GC_SET_NORMAL] = XCreateGC(app->display,
				   RootWindow(app->display, app->screen),
				   GCForeground | GCBackground | GCFont |
				   GCGraphicsExposures |
				   GCFunction, &values);
  } else { /* mono */
    app->flags &= ~MCAPP_COLOR;
    if (app->flags & MCAPP_REVERSE) {
      values.foreground = WhitePixel(app->display, app->screen);
      values.background = BlackPixel(app->display, app->screen);
    } else {
      values.foreground = BlackPixel(app->display, app->screen);
      values.background = WhitePixel(app->display, app->screen);
    }

    app->gc[GC_NORMAL] = XCreateGC(app->display,
				   RootWindow(app->display, app->screen),
				   GCForeground | GCBackground | GCFont |
				   GCGraphicsExposures, &values);

    app->gc[GC_SET_NORMAL] = XCreateGC(app->display,
				  RootWindow(app->display, app->screen),
				  GCForeground | GCBackground | GCFont |
				  GCFunction | GCStipple | GCGraphicsExposures,
				  &values);

    values.fill_style = FillStippled;
    app->gc[GC_BUSY] = XCreateGC(app->display,
				RootWindow(app->display, app->screen),
				GCForeground | GCBackground | GCFont |
				GCFillStyle | GCStipple | GCGraphicsExposures,
				&values);
    values.stipple = app->stipple1;
    values.fill_style = FillOpaqueStippled;
    app->gc[GC_BRIGHT] = XCreateGC(app->display,
				 RootWindow(app->display, app->screen),
				 GCForeground | GCBackground | GCFont |
				 GCFillStyle | GCStipple | GCGraphicsExposures,
				 &values);
    app->gc[GC_SELECTED_BITMAP]=app->gc[GC_DARK] = XCreateGC(app->display,
			RootWindow(app->display, app->screen),
			GCForeground | GCBackground | GCFont |
			GCGraphicsExposures,&values);
    app->gc[GC_SET_SELECTED_BITMAP]=app->gc[GC_DARK] = XCreateGC(app->display,
			RootWindow(app->display, app->screen),
			GCForeground | GCBackground | GCFont | GCFunction |
			GCGraphicsExposures,&values);

    app->gc[GC_SELECTED]=XCreateGC(app->display,
			    RootWindow(app->display, app->screen),
			    GCForeground | GCBackground | GCFont |
			    GCGraphicsExposures, &values);
    XSetForeground(app->display, app->gc[GC_SELECTED], values.background);
    XSetBackground(app->display, app->gc[GC_SELECTED], values.foreground);

    app->gc[GC_CLEAR] = XCreateGC(app->display,
				   RootWindow(app->display, app->screen),
				   GCForeground | GCBackground | GCFont |
				   GCGraphicsExposures, &values);
    XSetForeground(app->display, app->gc[GC_CLEAR], values.background);
    XSetBackground(app->display, app->gc[GC_CLEAR], values.foreground);
  }

  /* both, color & mono */
  app->flags &= ~MCAPP_REINIT;
  return app;
}

XFontStruct *McLoadFont(Display *display, char *name) {
  XFontStruct *font;
  
  if (!(font = XLoadQueryFont(display, name))) {
    fprintf(stderr,"%s: Can't open font '%s', using 'fixed'.\n",
	    myname, name);
    if (!(font = XLoadQueryFont(display, "fixed"))) {
      fprintf(stderr,"%s: Can't open font 'fixed' either!\n", myname);
      exit(1);
    }
  }
  return font;
}

McWindow *McCreateAppWindow(McApp *app, int x, int y, int w, int h,
		      void (*configureCallback)(struct McWindow *, int),
		      int (*eventCallback)(struct McWindow *, XEvent *event)) {
  Window win;
  McWindow *mcw;
  XSetWindowAttributes attr;

  if (x<0) x=app->x;
  if (y<0) y=app->y;
  if (w<0) w=app->w;
  if (h<0) h=app->h;

  mcw=(McWindow *)McAlloc(sizeof(McWindow));
  mcw->app=app;
  mcw->configureCallback = configureCallback;
  mcw->eventCallback = eventCallback;
  mcw->window_visible=0;
  mcw->firstGadget=mcw->lastGadget=NULL;
  mcw->x=x;  mcw->y=y;  mcw->w=w;  mcw->h=h;

  if (app->flags & MCAPP_COLOR) {
    win = XCreateSimpleWindow(app->display,
			      RootWindow(app->display, app->screen),
			      mcw->x, mcw->y, mcw->w, mcw->h, 0,
			      app->colors[COL_FOREGROUND],
			      app->colors[COL_BACKGROUND]);
  } else {
    win = XCreateSimpleWindow(app->display,
			      RootWindow(app->display, app->screen),
			      mcw->x, mcw->y, mcw->w, mcw->h, 0,
			      BlackPixel(app->display, app->screen),
			      WhitePixel(app->display, app->screen));
  }

  if (app->flags & MCAPP_BSTORE) { /* If you really like... */
    attr.backing_store=WhenMapped;
    XChangeWindowAttributes(app->display, win, CWBackingStore, &attr);
  }

  /*
   * Select the input events we will be wanting to deal with
   */
  mcw->event_mask=0;
  mcw->window=win;
  mcw->flags = 0;

  McAddInput(mcw, 
	     FocusChangeMask |
	     ExposureMask |
	     KeyPressMask |
	     VisibilityChangeMask |
	     ButtonPressMask |
	     ButtonReleaseMask |
	     ButtonMotionMask |
	     StructureNotifyMask);

  McAddWindowToList(app, mcw);
  return mcw;
}

void McAddInput(McWindow *mcw, long mask) {
  mcw->event_mask |= mask;
  XSelectInput(mcw->app->display, mcw->window, mcw->event_mask);
}

void McRemoveInput(McWindow *mcw, long mask) {
  mcw->event_mask &= ~mask;
  XSelectInput(mcw->app->display, mcw->window, mcw->event_mask);
}

/* 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D 3D */

void McAppDrawbox(McWindow *mcw, Window win,
		  int x, int y, int width, int height, int mode) {
  GC g1, g2;
  McApp *app=mcw->app;
  XPoint points[6];
  if (!win) win=mcw->window;

  switch (mode) {
  case _3D_OUT:
    g1=app->gc[GC_BRIGHT];
    g2=app->gc[GC_DARK];
    break;
  case _3D_IN:
    g2=app->gc[GC_BRIGHT];
    g1=app->gc[GC_DARK];
    break;
  case _3D_NONE:
    g1=g2=app->gc[GC_NORMAL];
    break;
  }

  points[0].x = x - BW;		points[0].y = y - BW;
  points[1].x = width + 2 * BW;	points[1].y = 0;
  points[2].x = -BW;		points[2].y = BW;
  points[3].x = -width;		points[3].y = 0;
  points[4].x = 0;		points[4].y = height;
  points[5].x = -BW;		points[5].y = BW;
  XFillPolygon(app->display, win, g1, points,6, Nonconvex, CoordModePrevious);
  points[0].y = y + height + BW;
  points[1].x = BW;		points[1].y = -BW;
  points[2].x = width;		points[2].y = 0;
  points[3].x = 0;		points[3].y = -height;
  points[4].x = BW;		points[4].y = -BW;
  points[5].x = 0;		points[5].y = height + 2 * BW;
  XFillPolygon(app->display, win, g2, points, 6, Nonconvex, CoordModePrevious);
}

int McInRectangle(int ex, int ey, int x, int y, int width, int height) {
#ifdef DBG_RECTANGLE
  int in;
  in= (ex >= x && ex < x + width && ey >= y && ey < y + height);

  printf("%d,%d is %s %d-%d,%d-%d\n",ex,ey,in?"inside":"outside",
	 x,x+width-1,y,y+height-1);

  return in;

#else
  if (ex >= x && ex < x + width &&
      ey >= y && ey < y + height)
    return 1;
  else
    return 0;
#endif
}

/* LISTS LISTS LISTS LISTS LISTS LISTS LISTS LISTS LISTS LISTS LISTS LISTS */

void McAddGadgetToList(McWindow *mcw, McGadget *gadget) {
  if (!(mcw->firstGadget))
    mcw->firstGadget = gadget;

  if (!(mcw->lastGadget)) {
    mcw->lastGadget = gadget;
    gadget->prev = gadget->next = NULL;
  } else {
    gadget->prev = mcw->lastGadget;
    gadget->next = NULL;
    mcw->lastGadget->next = gadget;
    mcw->lastGadget = gadget;
  }
}

void McRemoveGadgetFromList(McGadget *gadget) {
  McWindow *mcw=gadget->mcw;

  if (gadget->prev)
    gadget->prev->next=gadget->next;
  if (gadget->next)
    gadget->next->prev=gadget->prev;

  if (mcw->firstGadget == gadget)
    mcw->firstGadget = gadget->next;
  if (mcw->lastGadget == gadget)
    mcw->lastGadget = gadget->prev;

  gadget->prev = gadget->next = NULL;

}

void McMoveGadgetToStart(McGadget *gadget) {
  McWindow *mcw=gadget->mcw;

  McRemoveGadgetFromList(gadget);
  if (!(mcw->lastGadget))
    mcw->lastGadget = gadget;

  if (!(mcw->firstGadget)) {
    mcw->firstGadget = gadget;
    gadget->prev = gadget->next = NULL;
  } else {
    gadget->next = mcw->firstGadget;
    gadget->prev = NULL;
    mcw->firstGadget->prev = gadget;
    mcw->firstGadget = gadget;
  }

  McMoveWindowToStart(mcw->app, mcw);
}

void McAddWindowToList(McApp *app, McWindow *window) {
  if (!(app->firstWindow))
    app->firstWindow = window;

  if (!(app->lastWindow)) {
    app->lastWindow = window;
    window->prev = window->next = NULL;
  } else {
    window->prev = app->lastWindow;
    window->next = NULL;
    app->lastWindow->next = window;
    app->lastWindow = window;
  }
}

void McRemoveWindowFromList(McApp *app, McWindow *window) {
  if (window->prev)
    window->prev->next=window->next;
  if (window->next)
    window->next->prev=window->prev;

  if (app->firstWindow == window)
    app->firstWindow = window->next;
  if (app->lastWindow == window)
    app->lastWindow = window->prev;

  window->prev = window->next = NULL;

}    

void McMoveWindowToStart(McApp *app, McWindow *window) {

  McRemoveWindowFromList(app, window);
  if (!(app->lastWindow))
    app->lastWindow = window;

  if (!(app->firstWindow)) {
    app->firstWindow = window;
    window->prev = window->next = NULL;
  } else {
    window->next = app->firstWindow;
    window->prev = NULL;
    app->firstWindow->prev = window;
    app->firstWindow = window;
  }
}

/* EVENTS EVENTS EVENTS EVENTS EVENTS EVENTS EVENTS EVENTS EVENTS EVENTS */

static int set_window_visible(McApp *app, Window win, int flag) {
  McWindow *mcw=app->firstWindow;
  while(mcw) {
    if (win==mcw->window) {
      mcw->window_visible = flag;
      return 1;
    }
    mcw=mcw->next;
  }
  return 0;
}
    
static int set_window_focus(McApp *app, Window win, int flag) {
  McWindow *mcw=app->firstWindow;
  McGadget *gadget;

  while(mcw) {
    if (win==mcw->window) {
      mcw->window_has_focus = flag;
      if ((gadget=mcw->keyboardFocus))
	McGadgetUpdate(gadget);
      return 1;
    }
    mcw=mcw->next;
  }
  return 0;
}

/*
 * This is the smart part...
 * We scan the gadgets, and recompute their positions according to their
 * gravity settings. A nice example is the file requester...
 */
static int RecomputePositions(McWindow *mcw, int neww, int newh) {
  McGadget *next = mcw->firstGadget;
  int NeedToDrain = 0;
  int oldw, oldh;

  oldw=mcw->w; oldh=mcw->h;
  mcw->w=neww; mcw->h=newh;

#ifdef DBG_CONFIGURE_NOTIFY
  printf("x=%d y=%d w=%d h=%d\n", newx, newy, neww, newh);
#endif

  if ((oldw!=neww) || (oldh!=newh)) {
    while(next) {
      switch(next->topLeftGravity) {
      case ForgetGravity:
      case NorthGravity:
      case NorthWestGravity:
      case WestGravity:
	break;
      case SouthGravity:
      case SouthWestGravity:
	next->y = newh - (oldh - next->y);
	NeedToDrain = 1;
	break;
      case NorthEastGravity:
      case EastGravity:
	next->x = neww - (oldw - next->x);
	NeedToDrain = 1;
	break;
      case SouthEastGravity:
	next->y = newh - (oldh - next->y);
	next->x = neww - (oldw - next->x);
	NeedToDrain = 1;
	break;
#ifdef DEBUG_CODE
      default:
	fprintf(stderr,"%s: Funny, you used a weird gravity.\n", myname);
	break;
#endif
      }

      switch(next->bottomRightGravity) {
      case ForgetGravity:
      case NorthGravity:
      case NorthWestGravity:
      case WestGravity:
	break;
      case SouthGravity:
      case SouthWestGravity:
	next->height+= newh - oldh;
	NeedToDrain = 1;
	break;
      case NorthEastGravity:
      case EastGravity:
	next->width += neww - oldw;
	NeedToDrain = 1;
	break;
      case SouthEastGravity:
	next->height+= newh - oldh;
	next->width += neww - oldw;
	NeedToDrain = 1;
	break;
#ifdef DEBUG_CODE
    default:
	fprintf(stderr,"%s: Funny, you used a weird gravity.\n", myname);
	break;
#endif
      }

#if 0
      if (next->type == SLIDERGADGET) {
	printf("x=%d y=%d w=%d h=%d\n",
	       next->x, next->y, next->width, next->height);
      }
#endif

      next=next->next;
    }
  }

  if (mcw->configureCallback) {
    (*mcw->configureCallback)(mcw, NeedToDrain);
  }

  return NeedToDrain;
}

/* returns 1 if event was eaten */

int McAppDoEvent(McApp *app, XEvent *event) {
  McWindow *mcw, *nextwin;
  McGadget *next;

#ifdef DBG_EVENTS
  if (event->type!=Expose)
    printf("window=%d event=%s\n", (int)(event->xany.window),
	   McEventName(event->type));
#endif

  if ((event->xany.send_event==False) ||
      (app->flags & MCAPP_ALLOW_SENDEVENT) ||
      (event->type == ClientMessage) ||
      (event->type == SelectionNotify)) {
    switch (event->type) {
    case ConfigureNotify: /* Window has been resized */

      /* The goal of all this: If we get a configure notify and any gadget
       * changed its size or position, we have to redraw the complete window.
       */
      mcw=app->firstWindow;
      while(mcw) {
	nextwin=mcw->next;
	if (!(mcw->flags&MCW_CLOSEREQUEST)) {
	  if (event->xconfigure.window==mcw->window) {
	    if (RecomputePositions(mcw,
				   event->xconfigure.width,
				   event->xconfigure.height)) {
drain: /* Yuck, could be better... */
	      if (XCheckTypedWindowEvent(app->display, mcw->window,
					 ConfigureNotify, event)) goto drain;
	      if (XCheckTypedWindowEvent(app->display, mcw->window,
					 Expose, event)) goto drain;
	      XClearWindow(app->display, mcw->window);
	      next = mcw->firstGadget;
	      while(next) {
		McGadgetRedraw(next);
		next = next->next;
	      }
	    }
	    break;
	  }
	}
	if (mcw->flags&MCW_CLOSEREQUEST) McFreeWindow(app, mcw);
	mcw=nextwin;
      }
      return 1;

    case KeyPress:
      if ((event->xkey.state&Mod1Mask) && (app->hotkeyHandler)) {
	return (*(app->hotkeyHandler->callback))(app->hotkeyHandler->gadget,
						 event);
      }
      break;

    case MappingNotify:
      if (event->xmapping.request==MappingKeyboard)
	XRefreshKeyboardMapping((XMappingEvent *)event);
      return 1;
      
    case UnmapNotify:
      if (set_window_visible(app, event->xunmap.window, 0)) return 1;
      return 0;
    
    case MapNotify:
      if (set_window_visible(app, event->xmap.window, 1)) return 1;
      return 0;

    case FocusIn:
      if (set_window_focus(app, event->xfocus.window, 1)) return 1;
      return 0;

    case FocusOut:
      if (set_window_focus(app, event->xfocus.window, 0)) return 1;
      return 0;

    case VisibilityNotify:
      if (set_window_visible(app, event->xvisibility.window,
			    event->xvisibility.state!=VisibilityFullyObscured))
	return 1;
      return 0;

    case Expose:
#ifdef SHOW_EXPOSE
      printf("Expose %d: win=$%08X  x=%d y=%d w=%d h=%d\n",
	     event->xexpose.count,
	     (int)event->xexpose.window,event->xexpose.x,event->xexpose.y,
	     event->xexpose.width,event->xexpose.height);
#endif
      if (HandleExposeEvent(app, (XExposeEvent *)event)) return 1;
      break; /* Perhaps a gadget wants it? */
    }

    mcw=app->firstWindow;
    while(mcw) {
      nextwin=mcw->next;
      if (!(mcw->flags&MCW_CLOSEREQUEST)) {
	if (mcw->window == event->xany.window) {
	  next = mcw->firstGadget;
	  while(next) {
	    if (McGadgetEvent(next, event)) goto check_close;
	    next = next->next;
	  }
	  if (mcw->eventCallback) {
	    if ((*mcw->eventCallback)(mcw, event)) goto check_close;
	  }
#ifdef DBG_UNUSED_EVENTS
	  if (event->type!=Expose)
	    printf("unused event: %s\n",McEventName(event->type));
#endif
	  return 0;
	}
      }
      if (mcw->flags&MCW_CLOSEREQUEST) McFreeWindow(app, mcw);
      mcw=nextwin;
    }

    mcw=app->firstWindow;
    while(mcw) {
      nextwin=mcw->next;
      if (!(mcw->flags&MCW_CLOSEREQUEST)) {
	next = mcw->firstGadget;
	while(next) {
	  if (McGadgetEvent(next, event)) goto check_close;
	  next = next->next;
	}
      }
      if (mcw->flags&MCW_CLOSEREQUEST) McFreeWindow(app, mcw);
      mcw=nextwin;
    }

#ifdef DBG_UNUSED_EVENTS
    if (event->type!=Expose)
      printf("unused event: %s for window $%08X\n",
	     McEventName(event->type), (unsigned int)event->xany.window);
#endif
  }
  return 0;

check_close:
  if (mcw->flags&MCW_CLOSEREQUEST) McFreeWindow(app, mcw);
  return 1;
}

void McFreeWindow (McApp *app, McWindow *mcw) {
  McGadget *gadget, *next;

  gadget=mcw->firstGadget;
  while(gadget) {
    next=gadget->next;
    McFreeGadget(gadget);
    gadget=next;
  }
  if (mcw->selection) McFree(mcw->selection);
  if (mcw->region) XDestroyRegion(mcw->region);

  XDestroyWindow(app->display, mcw->window);

  McRemoveWindowFromList(app, mcw);
  McFree(mcw);
}

void McFreeApp(McApp *app) {
  McWindow *mcw, *nxtmcw;

  if (app) {
    McFreeAppPrefs(app);
    mcw=app->firstWindow;
    while(mcw) {
      nxtmcw=mcw->next;
      McFreeWindow(app, mcw);
      mcw=nxtmcw;
    }
    if (app->checkmark)
      McFreeBitmap(app, app->checkmark);
    if (app->altmark)
      McFreeBitmap(app, app->altmark);
    XCloseDisplay(app->display);
    if (app->wbcolors) McFree(app->wbcolors);
    McFree(app);
  }
}

void McFreeAppPrefs(McApp *app) {
  Display *dp=app->display;
  int i;

  if (app->defaultFont) {
    XFreeFont(dp,app->defaultFont);
    app->defaultFont=NULL;
  }

  if (app->wbFont) {
    XFreeFont(dp,app->wbFont);
    app->wbFont=NULL;
  }

  if (app->fixedFont) {
    if (app->fixed_font_name)
      XFreeFont(dp,app->fixedFont);
    app->fixedFont=NULL;
  }

  if (app->stipple1)    { XFreePixmap(dp, app->stipple1); app->stipple1=0; }
  if (app->stipple3)    { XFreePixmap(dp, app->stipple3); app->stipple3=0; }

  if (app->colormap) {
    XFreeColors(dp, app->colormap, app->colors, MAX_COLORS, 0);
    app->colormap = 0;
  }

  for (i=0; i<NUM_GC; i++)
    if (app->gc[i]) { XFreeGC(dp, app->gc[i]); app->gc[i]=0; }
}

unsigned char *McMakePathAbsolute(unsigned char *pa) {
  unsigned char *name;
  unsigned char *prefix = NULL;
  unsigned char *path = pa;

  int pflen;

  if (!path) return 0;

  if (strlen(path)) {
    if (*path != '~') {
      return strdup(path);
    }

    path++;
    if (*path== '/') path++;
    prefix = getenv("HOME");
    if (!prefix) prefix="/";
  }
 
  if (!prefix) prefix = "";

  name = (unsigned char *)McAlloc(strlen(path) + (pflen=strlen(prefix)) + 2);
  strcpy(name, prefix);
  if (name[pflen-1]!='/') name[pflen++]='/';
  strcpy(name+pflen, path);
  return name;
}

/*
 * Create a window ready for mapping, set all usual hints.
 * If x & y are non zero, make the window a transient and let it pop with
 * its x/y position right under the pointer.
 *
 */
McWindow *McCreateSimpleWindow(McApp *app, const unsigned char *title,
			       int width, int height,
			       int minwidth, int minheight,
			       int x, int y,
			       void (*configureCallback)
					(struct McWindow *, int),
			       int (*eventCallback)
					(struct McWindow *, XEvent *event)) {
  McWindow *mcw;
  XWMHints wm_hints;
  XSizeHints sizehints;
  int px, py, dummy;

  memset(&sizehints, 0, sizeof(XSizeHints));
  memset(&wm_hints, 0, sizeof(XWMHints));

  if (x && y) {
    XQueryPointer(app->display, RootWindow(app->display, app->screen),
		  (Window *)&dummy, (Window *)&dummy,
		  &px, &py, &dummy, &dummy, &dummy);
    px-=x; if (px<0) px=0;
    py-=y; if (py<0) py=0;
    sizehints.flags = PMinSize | USPosition;
    sizehints.x = px;
    sizehints.y = py;
  } else {
    sizehints.flags = PMinSize;
    px=-1; py=-1;
  }

  sizehints.min_width  = minwidth;
  sizehints.min_height = minheight;

  mcw=McCreateAppWindow(app, px, py, width, height,
			configureCallback, eventCallback);
  McSetHints(mcw, (char *)title, 0, NULL,  &sizehints, &wm_hints);
  if (x && y) XSetTransientForHint(app->display, mcw->window, mcw->window);

  return mcw;
}

/*
 * Default handler: Closes the window on request
 *
 */
int McAppDefaultEventHandler(McWindow *mcw, XEvent *event) {
  switch(event->type) {
  case ClientMessage:
    if ((event->xclient.format == 32) &&
	(event->xclient.data.l[0] == mcw->app->wmDelWin)) {
      mcw->flags|=MCW_CLOSEREQUEST;
      return 1;
    }
  }
  return 0;
}

/**************************************************************************/

void McResizeWindow(McWindow *mcw, int w, int h) {
  XSizeHints	 sizehints;

  if (w<0) w=mcw->w;
  if (h<0) h=mcw->h;

  memset(&sizehints, 0, sizeof(XSizeHints));
  sizehints.flags = USSize;
  sizehints.width=w; sizehints.height=h; 
  XSetWMProperties(mcw->app->display, mcw->window, NULL, NULL,
		   NULL, 0, &sizehints, NULL, NULL);
  XResizeWindow(mcw->app->display, mcw->window, w, h);

  RecomputePositions(mcw, w, h);
}

/**************************************************************************/

/*
 * A gadget wants to clip a gc down.
 * If the window has a global region (which means there have been expose
 * events), then the rectangle is clipped down and set.
 * Without a global region, just the rectangle is used.
 */
void McSetClipRectangle(McWindow *mcw, GC gc, XRectangle *rect) {
  if (mcw->region) {
    Region region=XCreateRegion();
    XUnionRectWithRegion(rect, region, region);
    XIntersectRegion(mcw->region, region, region);
    XSetRegion(mcw->app->display, gc, region);
    XDestroyRegion(region);
  } else {
    XSetClipRectangles(mcw->app->display, gc, 0, 0, rect, 1, Unsorted);
  }
}

/*
 * The clipping is no longer needed
 * If the window has a global region (which means there have been expose
 * events), then this region is set as clip mask.
 * If not, the clip mask is just cleared.
 */
void McClearClipRectangle(McWindow *mcw, GC gc) {
  if (mcw->region) {
    XSetRegion(mcw->app->display, gc, mcw->region);
  } else {
    XSetClipMask(mcw->app->display, gc, None);
  }
}

/**************************************************************************/

/*
 * Collects exposure events for a single window and triggers the
 * redraw mechanism when all exposures in a row are collected.
 */
static int HandleExposeEvent(McApp *app, XExposeEvent *event) {
  McWindow *mcw, *nextwin;
  McGadget *gadget;
  XRectangle rect;
  int i;

  mcw=app->firstWindow;
  while(mcw) {
    nextwin=mcw->next;
    if (mcw->flags&MCW_CLOSEREQUEST) {
      McFreeWindow(app, mcw);
    } else {
      if (mcw->window == event->window) break;
    }
    mcw=nextwin;
  }
  if (!mcw) return 0; /* Huh, not for me? */

  if (!mcw->firstGadget) return 0; /* user code must handle it */

  if (!mcw->region) mcw->region=XCreateRegion();

  rect.x=event->x; rect.y=event->y;
  rect.width=event->width; rect.height=event->height;
  XUnionRectWithRegion(&rect, mcw->region, mcw->region);
  if (event->count) return 1;

  /*
   * Now go and redraw what is needed after clipping down all gc's.
   * TODO: Only really used gc's should be clipped.
   */

  for (i=0; i<NUM_GC; i++)
    if (app->gc[i]) XSetRegion(mcw->app->display, app->gc[i], mcw->region);

  gadget=mcw->firstGadget;
  for (gadget=mcw->firstGadget; gadget; gadget=gadget->next) {
    if (XRectInRegion(mcw->region, gadget->x-BW,
		      gadget->y-BW, gadget->width+BW+BW,
		      gadget->height+BW+BW)!=RectangleOut) {
      McGadgetRedraw(gadget);
    }
  }

  for (i=0; i<NUM_GC; i++)
    if (app->gc[i]) XSetClipMask(mcw->app->display, app->gc[i], None);

  XDestroyRegion(mcw->region);
  mcw->region=NULL;
  return 1;
}

