/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: main.c,v 1.12 94/07/19 14:09:51 nau Exp $";

/* main program, initializes some stuff and handles user input
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#include "global.h"

#include "action.h"
#include "create.h"
#include "control.h"
#include "cursor.h"
#include "data.h"
#include "draw.h"
#include "error.h"
#include "file.h"
#include "menu.h"
#include "misc.h"
#include "remove.h"
#include "parse_l.h"

#include <X11/Shell.h> 
#include <X11/Xaw/AsciiText.h> 
#include <X11/Xaw/Command.h> 
#include <X11/Xaw/Form.h> 
#include <X11/Xaw/Label.h> 
#include <X11/Xaw/Scrollbar.h> 
#include <X11/Xaw/Simple.h> 
#include <X11/Xaw/Viewport.h> 

/* ---------------------------------------------------------------------------
 * icon data as created by 'bitmap'
 */
#include "icon.data"

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void			GetSizeOfDrawingArea(void);
static	XtEventHandler	OutputEvent(Widget, XtPointer, XEvent *, Boolean *);
static	XtEventHandler	ToplevelEvent(Widget, XtPointer, XEvent *, Boolean *);
static	XtEventHandler	ViewportEvent(Widget, XtPointer, XEvent *, Boolean *);
static	void			InitAccelerators(void);
static	void			InitShell(int *, char**);
static	void			InitStatusLine(Widget, Widget, Widget);
static	void			InitNameFields(Widget, Widget, Widget);
static	void			InitViewport(Widget, Widget, Widget);
static	void			InitWidgets(void);

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Atom			WMDeleteWindowAtom,
						WMProtocolsAtom;

/* ---------------------------------------------------------------------------
 * resources to query
 * the number of layers has to be adapted here and in 'global.h'
 */
static	XtResource		ToplevelResources[] = {
	{ "backupInterval", "BackupInterval", XtRInt, sizeof(long),
	  XtOffsetOf(SettingType, BackupInterval), XtRString, "300" },
	{ "pinoutFont", "Font", XtRFontStruct, sizeof(XFontStruct *),
	  XtOffsetOf(SettingType, PinoutFont), XtRFontStruct, XtDefaultFont },
	{ "layerGroups", "LayerGroups", XtRString, sizeof(String),
	  XtOffsetOf(SettingType, Groups), XtRString, "1:2:3:4:5:6:7:8" },
	{ "pinoutNameLength", "PinoutNameLength", XtRInt, sizeof(int),
	  XtOffsetOf(SettingType, PinoutNameLength), XtRString, "8" },
	{ "charactersPerLine", "CharactersPerLine", XtRInt, sizeof(int),
	  XtOffsetOf(SettingType, CharPerLine), XtRString, "80" },
	{ "pinoutZoom", "PinoutZoom", XtRInt, sizeof(int),
	  XtOffsetOf(SettingType, PinoutZoom), XtRString, "2" },
	{ "zoom", "Zoom", XtRInt, sizeof(int),
	  XtOffsetOf(SettingType, Zoom), XtRString, "3" },
	{ "grid", "Grid", XtRInt, sizeof(int),
	  XtOffsetOf(SettingType, Grid), XtRString, "100" },
	{ "fileCommand", "FileCommand", XtRString, sizeof(String),
	  XtOffsetOf(SettingType, FileCommand), XtRString, "cat %f" },
	{ "fontCommand", "FontCommand", XtRString, sizeof(String),
	  XtOffsetOf(SettingType, FontCommand), XtRString, "cat %f" },
	{ "elementCommand", "ElementCommand", XtRString, sizeof(String),
	  XtOffsetOf(SettingType, ElementCommand), XtRString, "M4PATH=\"%p\";export M4PATH;m4 %f" },
	{ "saveCommand", "SaveCommand", XtRString, sizeof(String),
	  XtOffsetOf(SettingType, SaveCommand), XtRString, "cat - > %f" },
	{ "usePolygonLines", "UsePolygonLines", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(SettingType, UsePolygonLines), XtRString, "False" },
	{ "useBackingStore", "UseBackingStore", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(SettingType, UseBackingStore), XtRString, "False" },
	{ "ringBellWhenFinished", "RingBellWhenFinished", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(SettingType, RingBellWhenFinished), XtRString, "True" },
	{ "saveLastCommand", "SaveLastCommand", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(SettingType, SaveLastCommand), XtRString, "False" },
	{ "saveInTMP", "SaveInTMP", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(SettingType, SaveInTMP), XtRString, "True" },
	{ "resetAfterElement", "ResetAfterElement", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(SettingType, ResetAfterElement), XtRString, "False" },
	{ "fontFile", "FontFile", XtRString, sizeof(String),
	  XtOffsetOf(SettingType, FontFile), XtRString, FONTFILENAME },

	{ "filePath", "FilePath", XtRString, sizeof(String),
	  XtOffsetOf(SettingType, FilePath), XtRString, "." },
	{ "fontPath", "FontPath", XtRString, sizeof(String),
	  XtOffsetOf(SettingType, FontPath), XtRString, PCBLIBDIR },
	{ "elementPath", "ElementPath", XtRString, sizeof(String),
	  XtOffsetOf(SettingType, ElementPath), XtRString, PCBLIBDIR },

	{ "maxPCBWidth", XtCWidth, XtRDimension, sizeof(Dimension),
	  XtOffsetOf(SettingType, MaxPCBWidth), XtRString, "20000" },
	{ "maxPCBHeight", XtCHeight, XtRDimension, sizeof(Dimension),
	  XtOffsetOf(SettingType, MaxPCBHeight), XtRString, "20000" },

	{ "viaThickness", XtCThickness, XtRDimension, sizeof(Dimension),
	  XtOffsetOf(SettingType, ViaThickness), XtRString, "40" },
	{ "lineThickness", XtCThickness, XtRDimension, sizeof(Dimension),
	  XtOffsetOf(SettingType, LineThickness), XtRString, "10" },

	{ "cursorColor", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, CursorColor), XtRString, XtDefaultForeground },
	{ "viaColor", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, ViaColor), XtRString, XtDefaultForeground },
	{ "pinColor", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, PinColor), XtRString, XtDefaultForeground },
	{ "elementColor", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, ElementColor), XtRString, XtDefaultForeground },
	{ "connectedColor", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, ConnectedColor), XtRString, XtDefaultForeground },

	{ "layerColor1", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, LayerColor[0]), XtRString, XtDefaultForeground },
	{ "layerColor2", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, LayerColor[1]), XtRString, XtDefaultForeground },
	{ "layerColor3", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, LayerColor[2]), XtRString, XtDefaultForeground },
	{ "layerColor4", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, LayerColor[3]), XtRString, XtDefaultForeground },
	{ "layerColor5", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, LayerColor[4]), XtRString, XtDefaultForeground },
	{ "layerColor6", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, LayerColor[5]), XtRString, XtDefaultForeground },
	{ "layerColor7", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, LayerColor[6]), XtRString, XtDefaultForeground },
	{ "layerColor8", XtCColor, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(SettingType, LayerColor[7]), XtRString, XtDefaultForeground }};

/* ---------------------------------------------------------------------------
 * fallback resources
 */
static	String			Fallback[] = {
	"*pinoutFont:       -*-courier-bold-r-*-*-12-*-*-*-*-*-*-*",
	"*font:             -*-courier-bold-r-*-*-12-*-*-*-*-*-*-*",
	"*zoom:                             2",
	"*maxPCBWidth:                      20000",
	"*maxPCBHeight:                     20000",
	"*pinoutMasterForm.viewport.width:  200",
	"*pinoutMasterForm.viewport.height: 150",
	"*outputMasterForm.viewport.width:  800",
	"*outputMasterForm.viewport.height: 600",
	NULL };

/* ---------------------------------------------------------------------------
 * additional command line arguments
 */
static	XrmOptionDescRec	CommandLineOptions[] = {
	{ "-b", "*useBackingStore", XrmoptionNoArg, (caddr_t) "True" },
	{ "+b", "*useBackingStore", XrmoptionNoArg, (caddr_t) "False" },
	{ "-backup", "*backupInterval", XrmoptionSepArg, (caddr_t) NULL },
	{ "-c", "*charactersPerLine", XrmoptionSepArg, (caddr_t) NULL },
	{ "-fontfile", "*fontFile", XrmoptionSepArg, (caddr_t) NULL },
	{ "-lelement", "*elementCommand", XrmoptionSepArg, (caddr_t) NULL },
	{ "-lfile", "*fileCommand", XrmoptionSepArg, (caddr_t) NULL },
	{ "-lfont", "*fontCommand", XrmoptionSepArg, (caddr_t) NULL },
	{ "-lg", "*layerGroups", XrmoptionSepArg, (caddr_t) NULL },
	{ "-mh", "*maxPCBHeight", XrmoptionSepArg, (caddr_t) NULL },
	{ "-mw", "*maxPCBWidth", XrmoptionSepArg, (caddr_t) NULL },
	{ "+p", "*useOwnLines", XrmoptionNoArg, (caddr_t) "False" },
	{ "-p", "*useOwnLines", XrmoptionNoArg, (caddr_t) "True" },
	{ "-pnl", "*pinoutNameLength", XrmoptionSepArg, (caddr_t) NULL },
	{ "-pp", "*ppCommand", XrmoptionSepArg, (caddr_t) NULL },
	{ "-pz", "*pinoutZoom", XrmoptionSepArg, (caddr_t) NULL },
	{ "-reset", "*resetAfterElement", XrmoptionNoArg, (caddr_t) "True" },
	{ "+reset", "*resetAfterElement", XrmoptionNoArg, (caddr_t) "False" },
	{ "-ring", "*ringBellWhenFinished", XrmoptionNoArg, (caddr_t) "True" },
	{ "+ring", "*ringBellWhenFinished", XrmoptionNoArg, (caddr_t) "False" },
	{ "-s", "*saveLastCommand", XrmoptionNoArg, (caddr_t) "True" },
	{ "+s", "*saveLastCommand", XrmoptionNoArg, (caddr_t) "False" },
	{ "-save", "*saveInTMP", XrmoptionNoArg, (caddr_t) "True" },
	{ "+save", "*saveInTMP", XrmoptionNoArg, (caddr_t) "False" },
	{ "-sfile", "*saveCommand", XrmoptionSepArg, (caddr_t) NULL }};

/* ---------------------------------------------------------------------------
 * default translations
 */
static	String			DefaultTranslations = "";

/* ---------------------------------------------------------------------------
 * actions
 */
static	XtActionsRec	Actions[] = {
	{ "SetSetting", ActionSetSetting},
	{ "MovePointer", ActionMovePointer},
	{ "SetCursor", ActionSetCursor},
	{ "Create", ActionCreate },
	{ "Remove", ActionRemove},
	{ "Mode", ActionMode},
	{ "Block", ActionBlock },
	{ "RotateElement", ActionRotateElement },
	{ "Change", ActionChange },
	{ "Redraw", ActionRedraw },
	{ "GrabElement", ActionGrabElement },
	{ "EndCommand", ActionEndCommand },
	{ "Name", ActionName },
	{ "Center", ActionCenter },
	{ "Text", ActionText },
	{ "Connection", ActionConnection },
	{ "ElementNameMode", ActionElementNameMode },
	{ "StartCommand", ActionStartCommand },
	{ "LineStack", ActionLineStack },
	{ "DisplayPinout", ActionDisplayPinout }};

/* ----------------------------------------------------------------------
 * sets the size of the drawing area
 * is called vy MapNotify events in output window or
 * by ConfigureNotify events of viewport window
 */
static void GetSizeOfDrawingArea(void)
{
	Dimension		width, height,
					scroll_width, scroll_height,
					border_left, border_bottom;

	XtVaGetValues(Output.Viewport,
		XtNwidth, &width,
		XtNheight, &height,
		NULL);
	XtVaGetValues(Output.ScrollbarBottom,
		XtNheight, &scroll_height,
		XtNborderWidth, &border_bottom,
		NULL);
	XtVaGetValues(Output.ScrollbarLeft,
		XtNwidth, &scroll_width,
		XtNborderWidth, &border_left,
		NULL);
	Output.Width = width -scroll_width -border_left;
	Output.Height = height -scroll_height -border_bottom;
}

/* ---------------------------------------------------------------------- 
 * handles all events from viewport window
 */
static XtEventHandler ViewportEvent(Widget W, XtPointer ClientData, XEvent *Event, Boolean *Flag)
{
	XConfigureEvent	*event = (XConfigureEvent *) Event;

	switch(event->type)
	{
		case ConfigureNotify:		/* get new size of drawing area */
			GetSizeOfDrawingArea();
			break;
	}
} 

/* ---------------------------------------------------------------------- 
 * handles all events from output window
 * the handling of MapNotify and ConfigureNotify could be done much
 * easier with X11R5 and later but...
 */
static XtEventHandler OutputEvent(Widget W, XtPointer ClientData, XEvent *Event, Boolean *Flag)
{
    XConfigureEvent *event = (XConfigureEvent *) Event;

	switch(event->type)
	{
		case Expose:				/* expose event means redraw */
			RedrawOutput();
			break;

		case EnterNotify:			/* enter output area */
			CursorOn();
			switch(RedrawOnEnter)	/* control panel was used -> redraw */
			{
				case REDRAW_ALL:	/* redraw complete board */
					RedrawOutput();
					break;

				case REDRAW_CURRENT:/* redraw only current layer */
					RedrawCurrentLayer();
					break;

				default:
					break;
			}
			RedrawOnEnter = REDRAW_NONE;
			break;

		case LeaveNotify:			/* leave output area */
			CursorOff();
			break;

		case MapNotify:				/* get initial size of drawing area */
			GetSizeOfDrawingArea();
			break;

		case ConfigureNotify:		/* get position of origin */
			Output.OffsetX = -event->x;
			Output.OffsetY = -event->y;
			break;
	}
} 

/* ---------------------------------------------------------------------------
 * init toplevel shell and get application resources
 */
static void InitShell(int *Argc, char **Argv)
{
		/* init application toplevel window, get resources
		 * and add event handler to catch messages from the window manager
		 */
	Output.Toplevel = XtAppInitialize(&Context, "Pcb",
		CommandLineOptions, XtNumber(CommandLineOptions),
		Argc, Argv, Fallback, NULL, 0);
	Dpy = XtDisplay(Output.Toplevel);
	XtAddEventHandler(Output.Toplevel,
		NoEventMask, True, (XtEventHandler) ToplevelEvent, NULL);

		/* clear structure and get resources */
	memset(&Settings, 0, sizeof(SettingType));
	XtGetApplicationResources(Output.Toplevel, &Settings,
		ToplevelResources, XtNumber(ToplevelResources), NULL, 0);
	if (Settings.LineThickness > MAX_LINESIZE || Settings.LineThickness < MIN_LINESIZE)
		Settings.LineThickness = 10;
	if (Settings.ViaThickness > MAX_PINORVIASIZE || Settings.ViaThickness < MIN_PINORVIASIZE)
		Settings.ViaThickness = 40;
	if (Settings.MaxPCBWidth > MAX_COORD)
		Settings.MaxPCBWidth = MAX_COORD;
	if (Settings.MaxPCBHeight > MAX_COORD)
		Settings.MaxPCBHeight = MAX_COORD;
	Settings.Width = Settings.MaxPCBWidth;
	Settings.Height = Settings.MaxPCBHeight;
	ParseGroupString(Settings.Groups, &Settings.LayerGroups);
}

/* ---------------------------------------------------------------------------
 * initializes name output fields
 */
static void InitNameFields(Widget Parent, Widget Top, Widget Left)
{
	Output.PCBname = XtVaCreateManagedWidget("pcbName", labelWidgetClass,
		Parent,
		XtNfromHoriz, Left,
		XtNfromVert, Top,
		XtNleft, XtChainLeft,
		XtNright, XtChainLeft,
		XtNtop, XtChainTop,
		XtNbottom, XtChainTop,
		NULL);

	Output.Elementname= XtVaCreateManagedWidget("elementName", labelWidgetClass,
		Parent,
		XtNfromHoriz, Left,
		XtNfromVert, Output.PCBname,
		XtNleft, XtChainLeft,
		XtNright, XtChainLeft,
		XtNtop, XtChainTop,
		XtNbottom, XtChainTop,
		NULL);
}

/* ---------------------------------------------------------------------------
 * initializes statusline, cursor position and input field
 */
static void InitStatusLine(Widget Parent, Widget Top, Widget Left)
{
	Output.StatusLine = XtVaCreateManagedWidget("statusLine", labelWidgetClass,
		Parent,
		XtNresizable, True,
		XtNfromHoriz, Left,
		XtNfromVert, Top,
		XtNleft, XtChainLeft,
		XtNright, XtChainLeft,
		XtNtop, XtChainTop,
		XtNbottom, XtChainTop,
		NULL);

	Output.CursorPosition = XtVaCreateManagedWidget("cursorPosition", labelWidgetClass,
		Parent,
		XtNlabel, "",
		XtNfromVert, Top,
		XtNfromHoriz, Output.StatusLine,
		XtNleft, XtChainLeft,
		XtNright, XtChainLeft,
		XtNtop, XtChainTop,
		XtNbottom, XtChainTop,
		NULL);

		/* the message label */
	Output.InputText = XtVaCreateManagedWidget("inputText", labelWidgetClass,
		Output.MasterForm,
		XtNresizable, True,
		XtNfromHoriz, Output.CursorPosition,
		XtNfromVert, Top,
		XtNmappedWhenManaged, False,
		XtNleft, XtChainLeft,
		XtNright, XtChainLeft,
		XtNtop, XtChainTop,
		XtNbottom, XtChainTop,
		NULL);
}

/* ---------------------------------------------------------------------------
 * initialize output and viewport widget
 */
static void InitViewport(Widget Parent, Widget Top, Widget Left)
{
		/* viewport widget to to handle scrolling */
	Output.Viewport = XtVaCreateManagedWidget("viewport", viewportWidgetClass,
		Parent,
		XtNfromHoriz, Left,
		XtNfromVert, Top,
		XtNleft, XtChainLeft,
		XtNright, XtChainRight,
		XtNtop, XtChainTop,
		XtNbottom, XtChainBottom,
		XtNallowHoriz, True,
		XtNallowVert, True,
		XtNuseBottom, True,
		XtNforceBars, True,
		NULL);

		/* create simple widget to which we draw to (handled by viewport) */
	Output.Output = XtVaCreateManagedWidget("output", simpleWidgetClass,
		Output.Viewport,
		XtNresizable, True,
		XtNheight, TO_SCREEN(Settings.Height),
		XtNwidth, TO_SCREEN(Settings.Width),
		NULL);

		/* add event handlers */
	XtAddEventHandler(Output.Output,
		ExposureMask | LeaveWindowMask | EnterWindowMask | StructureNotifyMask,
		True, (XtEventHandler) OutputEvent, NULL);
	XtAddEventHandler(Output.Viewport,
		StructureNotifyMask,
		True, (XtEventHandler) ViewportEvent, NULL);
}

/* ---------------------------------------------------------------------------
 * installs accelerators for scrollbars and a default translation table
 */
static void InitAccelerators(void)
{
		/* set translations */
	XtAugmentTranslations(Output.Output,
		XtParseTranslationTable(DefaultTranslations));

		/* install accelerators for scrollbars */
	if ((Output.ScrollbarBottom = XtNameToWidget(Output.Viewport, "horizontal")) != NULL)
	{
		XtInstallAccelerators(Output.Viewport, Output.ScrollbarBottom);
		XtInstallAccelerators(Output.Output, Output.ScrollbarBottom);
	}
	if ((Output.ScrollbarLeft = XtNameToWidget(Output.Viewport, "vertical")) != NULL)
	{
		XtInstallAccelerators(Output.Viewport, Output.ScrollbarLeft);
		XtInstallAccelerators(Output.Output, Output.ScrollbarLeft);
	}
}

/* ---------------------------------------------------------------------- 
 * initialize widgets
 */
static void InitWidgets(void)
{
	XSetWindowAttributes    attributes;
	Screen					*screen;

		/* this form widget is a container for most of the other widgets */
	Output.MasterForm = XtVaCreateManagedWidget("outputMasterForm", formWidgetClass,
		Output.Toplevel,
		XtNresizable, True,
		NULL);

		/* set actions;
		 * the menu must be initialized before the actions are added
		 * for XawPositionSimpleMenu() to be available
		 * InitMenu() returns the last positioned button
		 * the control panel is initialized too
		 */
	Output.Menu = InitMenu(Output.MasterForm, NULL, NULL);
	Output.Control = InitControlPanel(Output.MasterForm, Output.Menu, NULL);

	InitNameFields(Output.MasterForm, Output.Control, NULL);
	InitStatusLine(Output.MasterForm, NULL, Output.Control);
	XtAppAddActions(Context, Actions, XtNumber(Actions));
	InitViewport(Output.MasterForm, Output.InputText, Output.Control);

		/* realize the tree and get the IDs of the output window */
	XtRealizeWidget(Output.Toplevel);
	Output.OutputWindow = XtWindow(Output.Output);	

		/* enable backing store only im RAM limit is not reached */
	if (Settings.UseBackingStore)
	{
		attributes.backing_store = Always;
		attributes.backing_planes = AllPlanes;
		XChangeWindowAttributes(Dpy, Output.OutputWindow,
			CWBackingStore | CWBackingPlanes, &attributes);
	}

	InitAccelerators();

		/* create a public GC for drawing with
		 * background and foreground colors, the foreground color is
		 * set by some routines in draw.c
		 */
	XtVaGetValues(Output.Output,
		XtNbackground, &Settings.bgColor,
		NULL);
	if (!(Output.fgGC = XCreateGC(Dpy, Output.OutputWindow, 0, NULL)) ||
	    !(Output.bgGC = XCreateGC(Dpy, Output.OutputWindow, 0, NULL)))
		MyFatal("can't create default GC\n");

	XSetForeground(Dpy, Output.bgGC, Settings.bgColor);
	XSetFont(Dpy, Output.fgGC, Settings.PinoutFont->fid);

		/* initialize icon pixmap */
	screen = XtScreen(Output.Toplevel);
	XtVaSetValues(Output.Toplevel,
		XtNiconPixmap, XCreatePixmapFromBitmapData(Dpy,
			XtWindow(Output.Toplevel), icon_bits, icon_width, icon_height,
			BlackPixelOfScreen(screen), WhitePixelOfScreen(screen),
			DefaultDepthOfScreen(screen)),
		NULL);

	InitCursor();

		/* install a new error handler and catch terminating signals */
	XtAppSetErrorHandler(Context, X11ErrorHandler);
	signal(SIGHUP, CatchSignal);
	signal(SIGQUIT, CatchSignal);
	signal(SIGABRT, CatchSignal);
	signal(SIGSEGV, CatchSignal);
	signal(SIGTERM, CatchSignal);
	signal(SIGINT, CatchSignal);

		/* calling external program by popen() may cause a PIPE signal,
		 * so we ignore it
		 */
	signal(SIGPIPE, SIG_IGN);

		/* set window manager property to get 'delete window' messages */
	WMDeleteWindowAtom = XInternAtom(Dpy, "WM_DELETE_WINDOW", True);
	WMProtocolsAtom = XInternAtom(Dpy, "WM_PROTOCOLS", True);
	if (WMDeleteWindowAtom != None && WMProtocolsAtom != None)
		XSetWMProtocols(Dpy, XtWindow(Output.Toplevel), &WMDeleteWindowAtom, 1);
}

/* ---------------------------------------------------------------------------
 * handles all events from output window
 */
static XtEventHandler ToplevelEvent(Widget W, XtPointer ClientData, XEvent *Event, Boolean *Flag)
{
	XClientMessageEvent	*event = (XClientMessageEvent *) Event;

	switch(event->type)
	{
		case ClientMessage:		/* if 'delete window' message */
			if (event->message_type == WMProtocolsAtom &&
				event->data.l[0] == WMDeleteWindowAtom)
				QuitApplication();
			break;
	}
}

/* ---------------------------------------------------------------------- 
 * main program
 */
int main(int argc, char *argv[]) 
{
		/* init application:
		 * - make program name available for error handlers
		 * - evaluate special options
		 * - initialize toplevel shell, resources and font
		 * - update screen and get size of drawing area
		 * - evaluate command-line arguments
		 * - register 'call on exit()' function
		 */
	Progname = *argv;

		/* take care of special options */
	if (argc == 2)
	{
		if (!strcmp("-help", argv[1]))
			Usage();
		if (!strcmp("-copyright", argv[1]))
			Copyright();
		if (!strcmp("-version", argv[1]))
		{
			puts(RELEASE);
			exit(0);
		}
	}

	InitShell(&argc, argv);
	PCB = CreateNewPCB();
	CreateDefaultFont();
	InitWidgets();

		/* update zoom and grid... */
	UpdateSettingsOnScreen();
	GetSizeOfDrawingArea();
	switch(--argc)
	{
		case 0:		/* only program name */
			break;

		case 1:		/* load an initial layout and check for 'version' option */
					/* abort if filename looks like an option */
			++argv;
			if (**argv == '-')
				Usage();
			if (LoadPCB(*argv))
			{
				RemovePCB(PCB);
				PCB = CreateNewPCB();
				CreateDefaultFont();
			}
			break;

		default:		/* error */
			Usage();
			break;
	}

		/* Register a function to be called when the program terminates.
		 * This makes sure that data is save even if lex() aborts the program.
		 */
#ifdef HAS_ATEXIT
	atexit(EmergencySave);
#else
#ifdef HAS_ON_EXIT
	on_exit(GlueEmergencySave, NULL);
#endif
#endif

	XtAppMainLoop(Context);
	return(0);
} 
