------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                     S Y S T E M . I N T E R R U P T S                    --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.4 $                            --
--                                                                          --
--      Copyright (C) 1991,1992,1993,1994,1995 Florida State University     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This implemetation is a version where no tasking supports provided.

with Ada.Interrupts;
with Ada.Interrupts.Names;

with Interfaces.C;
with Interfaces.C.Extensions;
with Interfaces.C.Pthreads;

with System.Storage_Elements;

with Unchecked_Conversion;

package body System.Interrupts is

   package C        renames Interfaces.C;
   package Pthreads renames Interfaces.C.Pthreads;
   package INTNAM   renames Ada.Interrupts.Names;

   use type Ada.Interrupts.Interrupt_ID;
   use type Ada.Interrupts.Parameterless_Handler;
   use type C.Int;

   function TPH is new Unchecked_Conversion
     (Integer, Ada.Interrupts.Parameterless_Handler);
   function TPH is new Unchecked_Conversion
     (Pthreads.Isr_Address, Ada.Interrupts.Parameterless_Handler);
   function TISR is new Unchecked_Conversion
     (Ada.Interrupts.Parameterless_Handler, Pthreads.Isr_Address);

   SIG_ERR  : constant Ada.Interrupts.Parameterless_Handler := TPH (-1);
   SIG_IGN  : constant Ada.Interrupts.Parameterless_Handler := TPH (1);
   SIG_HOLD : constant Ada.Interrupts.Parameterless_Handler := TPH (2);
   SIG_DFL  : constant Ada.Interrupts.Parameterless_Handler := TPH (0);

   Usable_Interrupts : array (Ada.Interrupts.Interrupt_ID) of Boolean;

   Attached_Interrupts : array (Ada.Interrupts.Interrupt_ID) of Boolean
     := (others => False);

   procedure Unimplemented;
   procedure Unimplemented (Feature : String);
   --  This procedure writes out a message to the effect that tasking is
   --  not implemented, and then raises Program_Error.

   -----------------
   -- Is_Reserved --
   -----------------

   function Is_Reserved (Interrupt : Ada.Interrupts.Interrupt_ID)
     return Boolean is
   begin
      return not Usable_Interrupts (Interrupt);
   end Is_Reserved;

   -----------------
   -- Is_Attached --
   -----------------

   function Is_Attached (Interrupt : Ada.Interrupts.Interrupt_ID)
     return Boolean is
   begin
      if Is_Reserved (Interrupt) then
         raise Program_Error;
      end if;

      return Attached_Interrupts (Interrupt);
   end Is_Attached;

   ---------------------
   -- Current_Handler --
   ---------------------

   function Current_Handler (Interrupt : Ada.Interrupts.Interrupt_ID)
     return Ada.Interrupts.Parameterless_Handler is

      Isr    : aliased Pthreads.isr_address;

   begin
      if Is_Reserved (Interrupt) then
         raise Program_Error;
      end if;

      if Is_Attached (Interrupt) then
         if (Pthreads.intr_current_isr (C.Int (Interrupt), Isr'Access) =
             Pthreads.FUNC_ERR) then
            raise Program_Error; --  ???
         end if;
      end if;

      return TPH (Isr);

   end Current_Handler;

   --------------------
   -- Attach_Handler --
   --------------------

   procedure Attach_Handler
     (New_Handler : Ada.Interrupts.Parameterless_Handler;
      Interrupt   : Ada.Interrupts.Interrupt_ID;
      Static      : boolean := false) is
   begin
      if Is_Reserved (Interrupt) then
         raise Program_Error;
      end if;

      if (Pthreads.Intr_Attach (C.Int (Interrupt),
                               TISR (New_Handler)) = Pthreads.FUNC_ERR) then
         raise Program_Error;
      end if;
   end Attach_Handler;

   ----------------------
   -- Exchange_Handler --
   ----------------------

   procedure Exchange_Handler
     (Old_Handler : out Ada.Interrupts.Parameterless_Handler;
      New_Handler : Ada.Interrupts.Parameterless_Handler;
      Interrupt   : Ada.Interrupts.Interrupt_ID;
      Static      : boolean := false) is

      Old_Isr : aliased Pthreads.isr_address;

   begin
      if Is_Reserved (Interrupt) then
         raise Program_Error;
      end if;

      if Pthreads.intr_exchange
        (C.Int (Interrupt), TISR (New_Handler), Old_Isr'Access)
          = Pthreads.FUNC_ERR then
         raise Program_Error;
      end if;

      Old_Handler := TPH (Old_Isr);

   end Exchange_Handler;

   --------------------
   -- Detach_Handler --
   --------------------

   procedure Detach_Handler
     (Interrupt : Ada.Interrupts.Interrupt_ID;
      Static    : boolean := false) is
   begin
      --
      --  To "detach" the handler, we simply install SIG_DFL
      --

      Attach_Handler (SIG_DFL, Interrupt);

   end Detach_Handler;

   ---------------
   -- Reference --
   ---------------

   function Reference (Interrupt : Ada.Interrupts.Interrupt_ID)
     return System.Address is

      Signal : System.Address :=
        System.Storage_Elements.To_Address
          (System.Storage_Elements.Integer_Address (Interrupt));

   begin
      if Is_Reserved (Interrupt) then
      --  Only usable Interrupts can be used for binding it to an Entry.
         raise Program_Error;
      end if;
      return Signal;
   end Reference;

   --------------------------------
   -- Register_Interrupt_Handler --
   --------------------------------

   procedure Register_Interrupt_Handler
     (Handler : Ada.Interrupts.Parameterless_Handler) is
   begin
      Unimplemented ("Ada Signal Handlers");
   end Register_Interrupt_Handler;

   -----------------------------
   -- Bind_Interrupt_To_Entry --
   -----------------------------

   procedure Bind_Interrupt_To_Entry
     (T       : System.Tasking.Task_ID;
      E       : System.Tasking.Task_Entry_Index;
      Int_Ref : System.Address) is
   begin
      Unimplemented ("Ada Signal Handlers");
   end Bind_Interrupt_To_Entry;

   ------------------------------
   -- Detach_Interrupt_Entries --
   ------------------------------

   procedure Detach_Interrupt_Entries (T : Tasking.Task_ID) is
   begin
      Unimplemented ("Ada Signal Handlers");
   end Detach_Interrupt_Entries;

   ---------------------
   -- Block_Interrupt --
   ---------------------

   procedure Block_Interrupt (Interrupt : Ada.Interrupts.Interrupt_ID) is
   begin
      Unimplemented ("Ada Signal Handlers");
   end Block_Interrupt;

   -----------------------
   -- Unblock_Interrupt --
   -----------------------

   procedure Unblock_Interrupt (Interrupt : Ada.Interrupts.Interrupt_ID) is
   begin
      Unimplemented ("Ada Signal Handlers");
   end Unblock_Interrupt;

   ----------------
   -- Is_Blocked --
   ----------------

   function Is_Blocked (Interrupt : Ada.Interrupts.Interrupt_ID)
      return boolean is
   begin
      Unimplemented ("Ada Signal Handlers");
      return false;
   end Is_Blocked;

   procedure Put_Character (C : Integer);
   pragma Import (C, Put_Character, "putchar");

   procedure Prog_Exit (Status : Integer);
   pragma Import (C, Prog_Exit, "exit");

   procedure Write_Character (C : Character);
   procedure Write_EOL;
   procedure Write_String (S : String);
   --  Debugging procedures used for assertion output

   ---------------------
   -- Write_Character --
   ---------------------

   procedure Write_Character (C : Character) is
   begin
      Put_Character (Character'Pos (C));
   end Write_Character;

   ---------------
   -- Write_Eol --
   ---------------

   procedure Write_EOL is
   begin
      Write_Character (Ascii.LF);
   end Write_EOL;

   ------------------
   -- Write_String --
   ------------------

   procedure Write_String (S : String) is
   begin
      for J in S'Range loop
         Write_Character (S (J));
      end loop;
   end Write_String;

   -------------------
   -- Unimplemented --
   -------------------

   procedure Unimplemented (Feature : String) is
   begin
      Write_String (Feature);
      Write_String (" is unimplemented in this version of GNARL.");
      Write_EOL;
      Prog_Exit (1);
   end Unimplemented;

   procedure Unimplemented is
   begin
      Write_String ("tasking is not implemented in this version of GNARL.");
      Write_EOL;
      Prog_Exit (1);
   end Unimplemented;

begin

   Usable_Interrupts  :=
        (INTNAM.SIGILL  => False,  -- Maps to Program_Error
         INTNAM.SIGABRT => False,  -- ???
         INTNAM.SIGFPE  => False,  -- Maps to Constraint_Error
         INTNAM.SIGSEGV => False,  -- Maps to Constraint_Error or Storage_Error
         INTNAM.SIGKILL => False,  -- Can't be caught or ignored
         INTNAM.SIGSYS  => False,  -- Maps to Program_Error
         INTNAM.SIGBUS  => False,  -- Maps to Constraint_Error
         INTNAM.SIGALRM => False,  -- Used for ada delay operations
         INTNAM.SIG32 .. Ada.Interrupts.Interrupt_ID'Last => False,
--                                 -- Reserved for kernel usage
         others  => True);

end System.Interrupts;
