#! /bin/sh
# voice_script.sh --- Small voicemail system based on vgetty
# Marc SCHAEFER <schaefer@alphanet.ch>
#    with the help of: Marc@ThPhy.Uni-Duesseldorf.DE
# Creation: february 96
# Last change: 13/07/96
# V1.0 PV011 mSC96
# DISCLAIMER
#    No warranty, either expressed or implied.
# COPYRIGHT
#    Fully protected under Berne Convention, use authorized except
#    if you make money from this use (contact me before).
# *UNTESTED CHANGES*
#    - levelling
# DESCRIPTION
#    This software is launched when receiving a call. It will
#    wait upto two seconds for either:
#    - a DTMF tone, indicating the system should switch to voicemail
#    - a calling tone, indicating the system should immediately switch to
#      data-fax
# BUGS-SEEMS-SOLVED
#    - In the voicemail: What happens when someone DTMF's during a message
#      playback ? Perhaps we don't get a READY but a DTMF we have to handle.
#      This is better now
#    - Awful ways of exiting in case of errors. May hang vgetty. This is better now.
#    - What happens after the famous 20 seconds of the voicemail loop ? Nothing, it seems.
#    - in vgetty: child termination should always lead to a modem reset
#      (it happens sometimes that when voice_script.sh crashes, this
#      locks vgetty into an expect loop, but of course, if voice_script.sh
#      was perfect, this would not happen anyway)
#    - now, escaping from recording via a DTMF works  (except it is recorded
#      in the record file). What about hanguping ? Seems to work to.
#    - Two corrected bugs in ZyXEL_2864.c, reported to Mark, about
#      expecting VCON instead of OK (in PLAY and RECORD).
# BUGS
#    - using local vars, would prevent strange bugs (keyword local).
#    - some plays have been removed because of hangs.
#    - should not run under root.
#    - should protect the voice hierarchy
#    - should not exec user's programs.
#    - vgetty should use some sort of internal throughtput measurement
#      so we can send a DTMF to abort *immediately* (or hot-keys, if
#      that was implemented here).
#    - no locking against multiple instances, except there is some
#      trickery to avoid corruption (like using process IDs or copying
#      everything at a time in edit_menu).
# TODO
#    - see comments in functions.
# NOTES
#    - Must run under /bin/ksh for RISCos. sh is fine for Linux.
#    - not possible to receive vars from programs executed with `` (subshell)
#    - The current voicemail menu is quite small:
#         Modem answers. You must hit one of the DTMF keys before 2
#         seconds, else we switch to fax/modem. If you hit a DTMF,
#         you get played the INTRODUCTION_FILE. Then you may choose
#         different keys:
#            0 for quitting, acknowledged by GOODBYE_FILE
#            1 for reading private mail (* authentification needed)
#            2 for sending private mail (same)
#            3 for creating an account. When successful this is like *
#            4 for receiving a cookie
#            5 random sample
#            9 for operator functions (currently like a 1 with a separate password hardcoded)
#            * for identifying yourself
#            # for leaving a message to operator (like 2 to 0 except you can be not
#                                                 authentified).
#    - in playfile, currently, keys are not memorized (no hot keys) they are eaten
#      and interpreted like:
#         * aborts
#         all others: stops message, no abort
#    - ideas for edit_menu: either set a vriable such as LAST_MENU= or use a second parameter
#      this should be a normal menu. Now we implement this as hardcoded functions, which are
#      easier to handle
#    - Menu items are currently
#         key command min_level [parameter only for goto_menu]
# NEXT-TODO
#    - test changes (dynamic menus, menu editor)
#    - user edit
#    - public messaging
#    - general message selector and editor

# Sets the default to English
SELECTED_LANGUAGE=

# 0  not identified
# 1  identified
# 100 sub-operator
# NOTES
#    - Operator level is only for user 0 (CURRENT_USER==0)
#    - Cannot be set by dologin.
USER_LEVEL=0

VOICEMAIL=/tmp/voice

SELECT_RANDOM_FILE=/users/schaefer/bin/randomize

MAILBOX_DIR=$VOICEMAIL/mailboxes
LIBRARY_DIR=$VOICEMAIL/messages/library

TMPFILE=/tmp/fle_$$

INTRODUCTION_FILE=$VOICEMAIL/messages/welcome
NUMBER_FILE=$VOICEMAIL/messages/test
GOODBYE_FILE=$VOICEMAIL/messages/goodbye
LEAVE_MESSAGE_FILE=$VOICEMAIL/messages/leave_message
CONFIRM_FILE=$VOICEMAIL/messages/confirm

MAKE_ACCOUNT_FILE=$VOICEMAIL/messages/make_account
CHOOSE_PASSWORD_FILE=$VOICEMAIL/messages/choose_password
VERIFY_PASSWORD_FILE=$VOICEMAIL/messages/verify_password
PASSWORDS_DO_NOT_MATCH_FILE=$VOICEMAIL/messages/passwords_do_not_match
USER_NAME_FILE=$VOICEMAIL/messages/user_name
PERSONAL_INFO_FILE=$VOICEMAIL/messages/personal_info
FINDING_ID_FILE=$VOICEMAIL/messages/finding_id
ACCOUNT_MADE_FILE=$VOICEMAIL/messages/account_made

PASSWORD_ERROR_FILE=$VOICEMAIL/messages/password_error
PASSWORD_OK_FILE=$VOICEMAIL/messages/password_ok
ALREADY_LOGGED_FILE=$VOICEMAIL/messages/already_logged

ASK_CODE_FILE=$VOICEMAIL/messages/ask_code
ASK_PASSWORD_FILE=$VOICEMAIL/messages/ask_password

NO_MESSAGES_FILE=$VOICEMAIL/messages/no_messages
ONE_MESSAGE_FILE=$VOICEMAIL/messages/one_message
END_OF_MESSAGES_FILE=$VOICEMAIL/messages/end_of_messages
CONTROL_MESSAGE_FILE=$VOICEMAIL/messages/message_control

WHICH_RECIPIENT_FILE=$VOICEMAIL/messages/which_recipient
NO_SUCH_RECIPIENT_FILE=$VOICEMAIL/messages/no_such_recipient

SELECT_LANGUAGE_FILE=$VOICEMAIL/messages/select_language

EDIT_MENU_FILE=$VOICEMAIL/messages/edit_menu # tocreate
EDIT_MENU_ITEM=$VOICEMAIL/messages/edit_menu_which_item # tocreate
EDIT_MENU_ITEM_EXISTS=$VOICEMAIL/messages/edit_menu_item_exists # tocreate
EDIT_MENU_NO_SUCH_ITEM=$VOICEMAIL/messages/edit_menu_no_such_item # tocreate
EDIT_MENU_ITEM_TYPE=$VOICEMAIL/messages/edit_menu_item_type # tocreate
EDIT_MENU_FILE_TYPE=$VOICEMAIL/menu_items_types # tospeccreate
EDIT_MENU_ITEM_LEVEL=$VOICEMAIL/messages/edit_menu_item_level # tocreate

EDIT_USER_FILE=$VOICEMAIL/messages/edit_user # tocreate

function receive {
   read -r INPUT <&$VOICE_INPUT;
   echo "$INPUT";
}

function send {
   echo $1 >&$VOICE_OUTPUT;
   kill -PIPE $VOICE_PID
}

# SYNOPSIS
#    result = make_account(password, namefile, personalfile)
# RESULT
#       - account number
#       - NONE if no accounts available
# DESCRIPTION
#    This function creates a new account with the
#    password indicated and the two description files.
# MODIFIES
#    Variables: RESULT
# NOTE
#    - This function does not use vgetty
function makeaccount {
   RESULT=1 # Do not mess with account 0, the superuser.
   while :
   do
      if [ -e $MAILBOX_DIR/$RESULT ]; then
         RESULT=`expr $RESULT + 1`
      else
         mkdir $MAILBOX_DIR/$RESULT
         if [ $? != 0 ]; then
            RESULT=NONE # collision or fatal error ? we don't know, so ...
            break
         fi
         # no tests from here
         echo $1 > $MAILBOX_DIR/$RESULT/.password
         cat $2 >  $MAILBOX_DIR/$RESULT/.name
         cat $3 >  $MAILBOX_DIR/$RESULT/.personalinfo
         # ev. a level file.
         break
      fi
   done

   echo "$RESULT"
}

# SYNOPSIS
#    result = check_password(accountnumber, password)
# RESULT
#       - OK
#       - WRONG
#       - INEXISTANT
# DESCRIPTION
#    This function checks a password. Note that
#    it gives more information than the standard
#    UNIX login gives. However, this is not UNIX and
#    BTW, account numbers are linear so easy to guess.
# MODIFIES
#    Variables: RESULT
# NOTE
#    - This function does not use vgetty
function checkpassword {
   RESULT=INEXISTANT

   if [ -e $MAILBOX_DIR/$1/.password ]; then
      RESULT=`cat $MAILBOX_DIR/$1/.password`
      if [ "$RESULT" = "$2" ]; then
         RESULT=OK
      else
         RESULT=WRONG
      fi
   fi
   echo "$RESULT"
}

# SYNOPSIS
#    result = message_date(file)
# RESULT
#       - datestring
# DESCRIPTION
#    This functions outputs a date from a filename
# MODIFIES
#    Variables RESULT
# BUGS
#    In some cases won't produce good output.
# NOTE
#    - This function does not use vgetty
function messagedate {
   RESULT="`ls -la $1`"

   echo `echo "$RESULT" | cut -c43-48` at `echo "$RESULT" | cut -c50-51` `echo "$RESULT" | cut -c53-54`
}

# SYNOPSIS
#    result = random_file(dir)
# RESULT
#       - a random file name
# DESCRIPTION
#    This functions outputs a random file from a directory
# MODIFIES
# NOTE
#    - This function does not use vgetty
#    - it does not quite what it says, but it works.
function randomfile {
   $SELECT_RANDOM_FILE `ls $1`
}

# SYNOPSIS
#    result = say_something(text, file)
# RESULT
#       - OK
#       - ERROR
# DESCRIPTION
#    This function uses rsynth, sox and convert_au.sh
#    to convert a text to its file representation.
#    This is slow and should only be used for
#    texts which must be generated and spoken.
# MODIFIES
# BUGS
#    - too dependant on local policy
# NOTE
#    - This function does not use vgetty
#    - Outputs to stderr removed (caused problems)
function saysomething {
   (cd /users/schaefer/ported/rsynth
    rm -f tmp
    touch tmp
    say -r 9600 "$1" > /dev/null
    /users/schaefer/ported/sox10p11/sox -t raw -r 9600 -b -u tmp -t au -r 9600 tmp2
    (cd /users/schaefer/ported/mgetty/mgetty099-Mar10/voice/pvftools
     ./convert_au.sh) < tmp2 > $2
    rm -f tmp tmp2
   ) 2>/dev/null
   echo "OK"
}

# SYNOPSIS
#    result = tellid(id)
# RESULT
#    will be in the form
#       - id
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function communicates an account ID
# BUGS
#    What about stderr (eg, rm)
# MODIFIES
#    The ANSWER and RESULT variables.
#    $TMPFILE
#    IGNORE
function tellid {
   RESULT=`playfile $ACCOUNT_MADE_FILE`
   case $RESULT in
      READY) IGNORE=`saysomething "Your account number is $1" $TMPFILE`
             RESULT=`playfile $TMPFILE`
             rm -f $TMPFILE
             case $RESULT in
                READY) RESULT=$1;;
                *) ;;
             esac;;
      *) ;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = beep()
# RESULT
#    will be in the form
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function beeps.
# MODIFIES
#    The RESULT variable.

function beep {
   send "BEEP"
   RESULT=`receive`
   case $RESULT in
      BEEPING)       RESULT=`receive`
                     case $RESULT in
                        READY)         ;;
                        RECEIVED_DTMF) RESULT=`receive` # loop ?
                                       RESULT=ABORTED;;
                     esac;;
      *)             RESULT=END;;
      RECEIVED_DTMF) RESULT=`receive` # loop ?
                     RESULT=ABORTED;;
      *) RESULT=END;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = edit_user()
# RESULT
#    will be in the form
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function edits a specific user, which is asked
#    first. All operations are done here, except for
#    access level which must be done by calling procedure,
#    usually the edit_user menu item).
#    Current implemented items are:
#       1 show/change level
#       2 show/change name
#       3 show/change information
#       4 set new password
#       9 deletes the user
#       # terminates the editing (no fallback!)
# BUGS
#    - What about stderr (eg, rm)
# MODIFIES
#    The RESULT variable.
# NOTES
# TODO
#    - To implement

function edituser {
   RESULT=`playfile $EDIT_USER_FILE`
   echo "$RESULT"
}

# SYNOPSIS
#    result = ask_item(banner)
# RESULT
#    will be in the form
#       - either any DTMF (one single char)
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function plays the description, then
#    waits for a single DTMF. It then returns
#    it.
# MODIFIES
#    The RESULT variable.
# NOTES
# BUGS
# TODO
function askitem {
   RESULT=`playfile $1`
   case $RESULT in
      READY) RESULT=`askdtmf`;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = ask_menu_type(banner)
# RESULT
#    will be in the form
#       - [a-z]* for menu type
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function plays the description, then
#    waits for a DTMF code which is interpreted
#    as a menu item type.
# MODIFIES
#    The ANSWER, ANSWER2 and RESULT variables.
# NOTES
# BUGS
#    - no nonexistant item notification (returns ABORTED)
# TODO
function askmenutype {
   RESULT=`askcode $1`
   case $RESULT in
      [0-9]*) RESULT=`egrep "^$RESULT " $EDIT_MENU_FILE_TYPE`
              case $RESULT in
                 *_*) RESULT=`echo $RESULT | awk '{print $2}'`;;
                 *)   RESULT=ABORTED;;
              esac;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = edit_menu_search_item(menupath, item)
# RESULT
#    will be in the form
#       - READY
#       - NOT_FOUND
# DESCRIPTION
#    This functions searches for a specific menu item in a menu.
# BUGS
# MODIFIES
#    nothing
# NOTES
# TODO
function editmenusearchitem {
   egrep '^$2 ' $1/commands > /dev/null
   if [ $? = 0 ]; then
      echo "READY"
   else
      echo "NOT_FOUND"
   fi
}

# SYNOPSIS
#    result = edit_menu_new_item(menupath)
# RESULT
#    will be in the form
#       - READY
#       - or END
# DESCRIPTION
#    This function creates a new menu item. It first asks for
#    the menu identifier, verifies that this item is not yet
#    used, and then asks the type and the level, and, if needed,
#    a secondary argument. It then adds the item to the menu
#    and returns. We don't do any user level check.
# BUGS
#    - no secondary argument.
# MODIFIES
#    The RESULT, ANSWER variable.
# NOTES
# TODO
function editmenunewitem {
   RESULT=`askitem $EDIT_MENU_ITEM`
   case $RESULT in
      ?)      if [ "`editmenusearchitem $1 $RESULT`" = "NOT_FOUND" ]; then
                 ANSWER=$RESULT
                 RESULT=`askmenutype $EDIT_MENU_ITEM_TYPE`
                 case $RESULT in
                    [a-z]*)  ANSWER="$ANSWER $RESULT"
                             RESULT=`askcode $EDIT_MENU_ITEM_LEVEL`
                             case $RESULT in
                                [0-9]*)  echo "$ANSWER $RESULT" >> $1/commands;;
                                ABORTED) RESULT=READY;;
                             esac;;
                    ABORTED) RESULT=READY;;
                 esac
           else
           RESULT=`playfile $EDIT_MENU_ITEM_EXISTS`
           fi;;
       END)   ;;
       *)     RESULT=ABORTED;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = edit_menu_kill_item(menupath)
# RESULT
#    will be in the form
#       - READY
#       - or END
# DESCRIPTION
#    This function deletes an existing menu (which is introduced by
#    the user) item if it finds it.
#    We don't do any user level check.
# BUGS
#    - missing confirmation
# MODIFIES
#    The RESULT variable.
# NOTES
# TODO
function editmenukillitem {
   RESULT=`askitem $EDIT_MENU_ITEM`
   case $RESULT in
         ?) if [ "`editmenusearchitem $1 $RESULT`" = "READY" ]; then
           egrep -v '^$RESULT ' < $1/commands > $1/commands.new
           mv $1/commands.new $1/commands
           else
           RESULT=`playfile $EDIT_MENU_NO_SUCH_ITEM`
           fi;;
       END)   ;;
       *)     RESULT=ABORTED;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = edit_menu(menu_path)
# RESULT
#    will be in the form
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function edits a specific menu. It is not
#    itself implemented yet as a menu (see BUGS).
#    It will not check itself the access level (must
#    be done by calling procedure, usually the edit_menu
#    menu item).
#    Current implemented items are:
#       1 adds an item (can be a goto_menu, then asks owner,
#         or any type asked from a list)
#       2 removes an item (can be a goto_menu, the system will then
#         ask if removing the branch is asked)
#       3 tests the menu
#       * cancel changes
#       # accept changes and back
# BUGS
#    - What about stderr (eg, rm)
#    - What about implementing this also as a menu and
#      use menu items checking the level each ? This
#      would mean maintaining a LAST_MENU variable.
#      Editing the edit_menu would also be possible.
#    - It is possible to remove the edit menu function
#      from a menu. Oups.
#    - Moving menus and some operations can only be done
#      at the command line level.
# MODIFIES
#    The RESULT and TMPMENU variable.
# NOTES
#    - It does allow to add an item to a menu which is not allowed
#    to use by the current user. Checks will be done at run-time.
#    - Menu cannot be deleted from within the edit_menu. This means
#    that an arborescence starting from one user cannot be deleted
#    else by the user. (think of it). Thus we do not need a back_menu
#    result. And as menus are not cached, this is also OK.
# TODO
function editmenu {
   TMPMENU=/tmp/tmpmenu_$$
   mkdir $TMPMENU
   if [ -f $1/commands ]; then
      cp $1/* $TMPMENU
   else
      # default settings
      mkdir $1 # creates dir, will NOT be deleted
               # on abort --- BUG
      cat > $TMPMENU/commands <<FINGLOP
9 edit_menu 100
0 goodbye 0
# message_to_operator 0
* back_menu 0
FINGLOP
   fi

   RESULT=`playfile $EDIT_MENU_FILE`
   case $RESULT in
      READY) while :
             do
          RESULT=`askdtmf`
          case $RESULT in
             [0-9]) case $RESULT in
                    1) RESULT=`editmenunewitem $TMPMENU`
                                case $RESULT in
                                   READY) ;;
                                   *)     break;;
                                esac;;

                    2) RESULT=`editmenukillitem $TMPMENU`
                                case $RESULT in
                                   READY) ;;
                                   *)      break;;
                                esac;;
                    3) RESULT=`dovoicemail $TMPMENU`
                                case $RESULT in
                                   READY|goodbye) ;;
                                   *)             break;;
                                esac;;
                    \*) RESULT=READY
                                 rm -rf $TMPMENU
                                 break;;
                    \#) RESULT=READY
                                 cp $TMPMENU/* $1
                                 rm -rf $TMPMENU
                                 break;;
                    *)  break;;
                 esac;;
          esac
             done;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = do_login()
# RESULT
#    will be in the form
#       - account ID
#       - or ABORTED
#       - or END
#       - or NONE if account could not be accessed
# DESCRIPTION
#    This function logins to an existant account.
# BUGS
#    What about stderr (eg, rm)
# MODIFIES
#    The ANSWER, ANSWER2 and RESULT variables.
# TODO
function dologin {
   RESULT=`askcode $ASK_CODE_FILE`
   case $RESULT in
      [0-9]*) ANSWER=`askcode $ASK_PASSWORD_FILE`
              case $ANSWER in
                 [0-9]*) ANSWER2=$RESULT
                         RESULT=`checkpassword $RESULT $ANSWER`
                         case $RESULT in
                            INEXISTANT|WRONG) RESULT=NONE;;
                            OK)               RESULT=$ANSWER2;;
                         esac;;
                 *)      RESULT=$ANSWER;;
              esac;;
       END)   ;;
       *)     RESULT=ABORTED;; # We do not accept empty case.
   esac

   # we had a hang bug here. I enable it again to check it again.
   # seems to work now.
   if [ "1" = "1" ]; then
   case $RESULT in
      NONE)    ANSWER=`playfile $PASSWORD_ERROR_FILE`
               case $ANSWER in
                  READY) ;;
                  *)     RESULT=$ANSWER;;
               esac;;
      [0-9]*)  ANSWER=`playfile $PASSWORD_OK_FILE`
               case $ANSWER in
                  READY) ;;
                  *)     RESULT=$ANSWER;;
               esac;;
   esac
   fi

   echo "$RESULT"
}

# SYNOPSIS
#    result = send_mail(toaccountID, fromaccountID, askconfirmation)
# RESULT
#    will be in the form
#       - READY
#       - INEXISTANT
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function allows to send a voice mail message.
# BUGS
#    - What about stderr (eg, rm)
#    - risk of collision with sendmail(1) ? -> use send_mail instead
# MODIFIES
#    The RESULT variable.
# TODO
function sendmail {
   case $3 in
      YES) if [ -e $MAILBOX_DIR/$1/.name ]; then
           RESULT=`playfile $MAILBOX_DIR/$1/.name`
           case $RESULT in
           READY) RESULT=`confirm`
               case $RESULT in
                  YES) RESULT=READY;;
                           NO)  RESULT=ABORTED;;
               esac;;
           esac
        else
           RESULT=INEXISTANT
        fi;;
       *)  RESULT=READY;;
   esac
   case $RESULT in
      READY) RESULT="$MAILBOX_DIR/$1/msg_`date '+%y%m%d%H%M%S'`_from_$2_$$"
             RESULT=`recordfile $LEAVE_MESSAGE_FILE $RESULT`;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = do_send_mail()
# RESULT
#    will be in the form
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function allows to send a voice mail message
#    by asking for destinator, and doing all checks.
# BUGS
#    - What about stderr (eg, rm)
#    - What if CURRENT_USER is nothing ?
# TODO
# MODIFIES
#    The ANSWER, and RESULT variables.
function dosendmail {
   RESULT=`askcode $WHICH_RECIPIENT_FILE`
   case $RESULT in
      [0-9]*) ;;
      *)      RESULT=0;; # operator (note that [0-9]* does not accept empty)
   esac
   case $RESULT in
      [0-9]*) RESULT=`sendmail $RESULT $CURRENT_USER YES`
              case $RESULT in
                 INEXISTANT) RESULT=`playfile $NO_SUCH_RECIPIENT_FILE`;;
              esac ;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = read_mail(accountID)
# RESULT
#    will be in the form
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function allows to browse through voice messages
# BUGS
#    - What about stderr (eg, rm)
#    - FORWARD not implemented.
# TODO
#    - We could add the spoken name of the sender, ability
#      to reply, forward, and so on, remove capabilities, etc.
# MODIFIES
#    The ANSWER, ANSWER2, NUMBERM, and RESULT variables.
function readmail {
   NUMBERM=`ls -1 $MAILBOX_DIR/$1/ | wc -l`
   NUMBERM=`echo $NUMBERM`
   case $NUMBERM in
      0) RESULT=`playfile $NO_MESSAGES_FILE`;;
      1) RESULT=`playfile $ONE_MESSAGE_FILE`;;
      *) RESULT=`saysomething "You have $NUMBERM messages waiting." $TMPFILE`
         RESULT=`playfile $TMPFILE`
         rm -f $TMPFILE;;
   esac
   case $RESULT in
      READY) case $NUMBERM in
             0) ;;
             *) sleep 1
                for ANSWER2 in $MAILBOX_DIR/$1/*
                do
                   while :
                   do
                NUMBERM=`echo $ANSWER2 | awk -F_ '{print $4}'` # recycle vars !
                ANSWER=`messagedate $ANSWER2`
                ANSWER=`saysomething "Next message from account $NUMBERM is dated $ANSWER." $TMPFILE`
                RESULT=`playfile $TMPFILE`
                rm -f $TMPFILE
                case $RESULT in
                READY) sleep 1
                    RESULT=`playfile $ANSWER2`
                    case $RESULT in
                       READY) sleep 1
                                          RESULT=`messagecommand`
                                          case $RESULT in
                                             NEXT)   RESULT=READY; break;;
                                             AGAIN|FORWARD)  RESULT=READY; continue;;
                                             DELETE) RESULT=READY
                                                     rm -f $ANSWER2
                                                     break;;
                                             REPLY) RESULT=`sendmail $NUMBERM $CURRENT_USER NO`
                                                    case $RESULT in
                                                       READY) ;;
                                                       INEXISTANT) RESULT=`playfile $NO_SUCH_RECIPIENT_FILE`
                                                                   case $RESULT in
                                                                      READY) ;;
                                                                      *)     break;;
                                                                   esac;;
                                                       *)     break;;
                                                    esac
                                                    ;;
                                             *) break;;
                                          esac;;
                       *) break;;
                    esac;;
                *) break;;
                esac
                   done
                   case $RESULT in
                      READY) ;;
                      *)     break;;
                   esac
                done
                case $RESULT in
                   READY) RESULT=`playfile $END_OF_MESSAGES_FILE`;;
                esac;;
             esac;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = create_account()
# RESULT
#    will be in the form
#       - account ID
#       - or ABORTED
#       - or END
#       - or NONE if account could not be created
# DESCRIPTION
#    This function creates a new account.
# BUGS
#    - What about stderr (eg, rm)
#    - [0-9]* matches a non-empty sequence. Thus, an
#      empty sequence will make this function return nothing
#      which will be handled correctly by the main loop, but
#      this is a bug.
#    - use the confirm function instead of doing it by hand.
# MODIFIES
#    The ANSWER and RESULT variables.
#    The SELECTED_PASSWORD, USER_NAME_TMP, INFO_PERS_TMP variables.
function createaccount() {
   RESULT=`playfile $MAKE_ACCOUNT_FILE`
   case $RESULT in
      READY) ANSWER=`askdtmf`
             case $ANSWER in
                END)     RESULT=END;;
                ABORTED) RESULT=ABORTED;;
                1)       RESULT=`askcode $CHOOSE_PASSWORD_FILE`
                         case $RESULT in
                            [0-9]*) SELECTED_PASSWORD=$RESULT
                                    RESULT=`askcode $VERIFY_PASSWORD_FILE`
                                    case $RESULT in
                                       [0-9]*) if [ "$RESULT" = "$SELECTED_PASSWORD" ]; then
                                                  USER_NAME_TMP=/tmp/usern_$$
                                                  RESULT=`recordfile $USER_NAME_FILE $USER_NAME_TMP`
                                                  case $RESULT in
                                                     READY) INFO_PERS_TMP=/tmp/infopers_$$
                                                            RESULT=`recordfile $PERSONAL_INFO_FILE $INFO_PERS_TMP`
                                                            case $RESULT in
                                                               READY) RESULT=`playfile $FINDING_ID_FILE`
                                                                      case $RESULT in
                                                                         READY) RESULT=`makeaccount $SELECTED_PASSWORD $USER_NAME_TMP $INFO_PERS_TMP`
                                                                                rm -f $USER_NAME_TMP $INFO_PERS_TMP
                                                                                case $RESULT in
                                                                                   NONE) ;;
                                                                                   *) RESULT=`tellid $RESULT`
                                                                                      case $RESULT in
                                                                                         [0-9]*) ;;
                                                                                         *) # kill account
                                                                                            ;;
                                                                                      esac;;
                                                                                esac;;
                                                                         *) rm -f $USER_NAME_TMP $INFO_PERS_TMP;;
                                                                      esac;;
                                                               *) rm -f $USER_NAME_TMP;;
                                                            esac;;
                                                  esac
                                               else
                                                  RESULT=`playfile $PASSWORDS_DO_NOT_MATCH_FILE`
                                                  case $RESULT in
                                                     READY) RESULT=NONE;;
                                                  esac
                                               fi;;
                                    esac;;
                         esac;;
                *)       RESULT=ABORTED;;
             esac;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = ask_dtmf()
# RESULT
#    will be in the form
#       - either [0-9]
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function returns ONE DTMF digit.
#    Abort is when the input delays terminates.
# MODIFIES
#    The ANSWER, ANSWER2 and RESULT variables.
function askdtmf {
   RESULT=

   send "WAIT 20"
   ANSWER=`receive`
   case $ANSWER in
      WAITING)       ANSWER=`receive`
                     send "STOP"
                     ANSWER2=`receive`

                     case $ANSWER in
                        RECEIVED_DTMF) ANSWER=$ANSWER2
                                       ANSWER2=`receive`
                                       RESULT=$ANSWER;;
                        READY)         RESULT=ABORTED
                                       # Timeout
                                       ;;
                        *)             RESULT=END;;
                    esac

                    case $ANSWER2 in
                       READY)         ;;
                       RECEIVED_DTMF) # Eat all remaining ones
                                      while :
                                      do
                                         ANSWER=`receive`
                                         ANSWER=`receive`
                                         case $ANSWER in
                                            READY)         break ;;
                                            RECEIVED_DTMF) ;;
                                            *)             RESULT=END;;
                                         esac
                                      done
                                      # We do not need to set ABORTED again,
                                      # because END might be.
                                      ;;
                       *)             RESULT=END;;
                     esac;;
      RECEIVED_DTMF) ANSWER=`receive`
                     RESULT=ABORTED;;
      *)             RESULT=END;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = record_file(banner, tofile)
# RESULT
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This functions plays the description, then
#    records a file until any DTMF is received.
# MODIFIES
#    The ANSWER, RESULT variables.
# BUGS
#    - Does not support multiple DTMF
#    - stderr is used (rm)
#    - beep missing
function recordfile {
   RESULT=`playfile $1`
   case $RESULT in
      READY) RESULT=`beep`
             case $RESULT in
                READY) send "RECORD $2"
                 ANSWER=`receive`
                 case $ANSWER in
                 RECORDING)     RESULT=`receive`
                          case $RESULT in
                             READY)         RESULT=`beep`;;
                             RECEIVED_DTMF) ANSWER=`receive` # eat key
                                      send "STOP"
                                      RESULT=`receive` # may need an eat_dtmf loop here
                                                           RESULT=`beep`
                                      ;;
                             *)             RESULT=END
                                                           rm -f $2;;
                          esac;;
                 RECEIVED_DTMF) ANSWER=`receive` # eat key
                          send "STOP"
                          RESULT=`receive` # may need an eat_dtmf loop here;;
                          case $RESULT in
                             READY) RESULT=ABORTED;;
                          esac;;
                          *)             RESULT=END;;
                 esac;;
                *) ;;
             esac;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = confirm()
# RESULT
#    will be in the form
#       - YES
#       - NO
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This functions plays a standard confirm message,
#    and ask for confirmation (1), any other key returns NO.
# MODIFIES
#    The RESULT variable.
function confirm {
   RESULT=`playfile $CONFIRM_FILE`
   case $RESULT in
      READY) RESULT=`askdtmf`
             case $RESULT in
                END|ABORTED) ;;
                *)           if [ "$RESULT" = "1" ]; then
                                RESULT=YES
                             else
                                RESULT=NO
                             fi;;
             esac;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = message_command()
# RESULT
#    will be in the form
#       - NEXT
#       - DELETE
#       - AGAIN
#       - REPLY
#       - FORWARD
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function is used for private message
#    to control what is done with a received
#    message.
#       # next message
#       * quit
#       1 play again
#       2 forward message
#       3 reply
#       4 delete and next message
# BUGS
# MODIFIES
#    The RESULT variable.
function messagecommand {
  while :
  do
     RESULT=`playfile $CONTROL_MESSAGE_FILE`
     case $RESULT in
     READY) RESULT=`askdtmf`
            case $RESULT in
            \#) RESULT=NEXT; break;;
            \*) RESULT=ABORTED; break;;
            1)  RESULT=AGAIN; break;;
            2)  RESULT=FORWARD; break;;
            3)  RESULT=REPLY; break;;
            4)  RESULT=DELETE; break;;
                  END|ABORTED) break;;
            esac;;
     esac
  done
  echo "$RESULT"
}

# SYNOPSIS
#    result = set_language()
# RESULT
#    will be in the form
#       - a code
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function sets the new current language.
# BUGS
#    This does not handle the empty code case. See also dologin
# MODIFIES
#    The RESULT variable.
#    The SELECTED_LANGUAGE variable.
function setlanguage {
   RESULT=`askcode $SELECT_LANGUAGE_FILE`
   case $RESULT in
      [0-9]*) ;;
      \*)     RESULT=ABORTED
              ;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = ask_code(banner)
# RESULT
#    will be in the form
#       - either [0-9]*
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function plays the description, then
#    waits for a sequence of numbers terminated
#    by a # (which is not kept). It then returns
#    it. Aborting can be done with *, or by a
#    DTMF during the play. An empty
#    code is possible.
# MODIFIES
#    The ANSWER, ANSWER2 and RESULT variables.
function askcode {
   RESULT=`playfile $1`
   case $RESULT in
      READY) RESULT=
             while :
             do
                ANSWER=`askdtmf`
                case $ANSWER in
                   END)     RESULT=END
                            break;;
                   ABORTED) RESULT=ABORTED
                            break;;
                   *)       if [ "$ANSWER" = "#" ]; then
                               break
                            fi
                            if [ "$ANSWER" = "*" ]; then
                               RESULT=ABORTED
                               break
                            fi
                            RESULT="${RESULT}$ANSWER";;
                esac
             done;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = play_file(file)
# RESULT
#    will be in the form
#       - either READY (normal case)
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This functions plays file, then returns.
#    It sets ABORTED if a DTMF * has been played,
#    stops the message but returns READY if #
#    is used, any other key is used.
#    It sets END if something wrong happened.
# MODIFIES
#    The ANSWER and RESULT variables.
# BUGS
#    No way to say if the file was non existant.
#    It is perhaps understood as END anyway.
function playfile {
   RESULT=READY

   if [ "$SELECTED_LANGUAGE" != "" ]; then
      ANSWER=$1.${SELECTED_LANGUAGE}
      if [ ! -e $ANSWER ]; then
         ANSWER=$1
      fi
   else
      ANSWER=$1
   fi

   send "PLAY $ANSWER"
   ANSWER=`receive`
   case $ANSWER in
      PLAYING) ANSWER=`receive`
               case $ANSWER in
                  READY)         ;;
                  *)             case $ANSWER in
                                    RECEIVED_DTMF) ANSWER=`receive`
                                                   case $ANSWER in
                                                      \*) RESULT=ABORTED;;
                                                      *)  RESULT=READY;;
                                                   esac;;
                                    *)             RESULT=END;;
                                 esac

                                 send "STOP"
                                 ANSWER=`receive`
                                 case $ANSWER in
                                    READY) ;;
                                    RECEIVED_DTMF) # Eat all remaining ones
                                                   while :
                                                   do
                                                      ANSWER=`receive`
                                                      ANSWER=`receive`
                                                      case $ANSWER in
                                                         READY)         break ;;
                                                         RECEIVED_DTMF) ;;
                                                         *)             RESULT=END;;
                                                      esac
                                                   done
                                                   # We do not need to set ABORTED again,
                                                   # because END might be.
                                                   ;;
                                    *)             RESULT=END;;
                                 esac;;
               esac
               ;;
      *)
               case $ANSWER in
                  RECEIVED_DTMF) ANSWER=`receive`
                                 RESULT=ABORTED;;
                  *)             RESULT=END;;
               esac

               # No need to STOP because we haven't started yet.

               ;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = dovoicemail(menudir)
# RESULT
#    will be in the form
#       - READY
#       - goodbye
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function handles a submenu.
# MODIFIES
#    The RESULT variable.
function dovoicemail {
   CURRENT_MENU=$1

   while :
   do
      # Play this menu welcome message.
      if [ -f $CURRENT_MENU/welcome ]; then
         ANSWER=`cat $CURRENT_MENU/welcome`
         if [ -f $ANSWER ]; then
            ANSWER=`playfile $ANSWER`
         fi
      fi

      while :
      do
      ANSWER=`askdtmf`
      case $ANSWER in
         ABORTED) ;;
         END)     break;;
            *)       # Analyze the command
                     # DTMF command_name min_level args ... args
                     if [ -f $CURRENT_MENU/commands ]; then
                        # next line: changed from " to ' on PV010
                        ANSWER=`egrep '^$ANSWER ' $CURRENT_MENU/commands`
                        set - ANSWER
                        shift
                        # Aie, aie, no error check
                        # command_name min_level args ... args
                        ANSWER=$1
                        shift
                        # min_level args ... args
                        if [ $USER_LEVEL -lt $2 ]; then
                           # User level too low.
                           if [ "$CURRENT_USER" != "0" ]; then
                              continue
                           fi
                        fi
                        shift # forget the required level
                        # remaining: args ... args
                     else
                        # default menu
                        case $ANSWER in
                           0)  ANSWER=goodbye;;
                           1)  ANSWER=read_mail;;
                           2)  ANSWER=send_mail;;
                           3)  ANSWER=create_account;;
                           4)  ANSWER=cookie;;
                           5)  ANSWER=random_sample;;
                           9)  ANSWER=edit_menu;;
                           \*) ANSWER=identify;;
                           \#) ANSWER=message_to_operator;;
                        esac
                     fi
                     case $ANSWER in
               goodbye)   # goodbye
                          break;;
               read_mail) # read your private mail
                          if [ "$CURRENT_USER" != "" ]; then
                          ANSWER=`readmail $CURRENT_USER`
                          case $ANSWER in
                          ABORTED) ;;
                          END)     break;;
                          esac
                       fi
                       ;;
               send_mail) # send private mail
                       if [ "$CURRENT_USER" != "" ]; then
                          ANSWER=`dosendmail`
                          case $ANSWER in
                          ABORTED) ;;
                          END)     break;;
                          esac
                       fi
                       ;;
               create_account)
                                   # create an account
                       if [ "$CURRENT_USER" = "" ]; then
                          ANSWER=`createaccount`
                          case $ANSWER in
                          ABORTED) ;;
                          END)     break;;
                          [0-9]*)  CURRENT_USER=$ANSWER;;
                          esac
                       fi
                       ;;
               cookie)   # receive a cookie
                      IGNORE="`/usr/games/fortune -s`"
                      IGNORE=`saysomething "Here is a cookie for you. $IGNORE" $TMPFILE`
                      ANSWER=`playfile $TMPFILE`
                                  rm -f $TMPFILE
                      case $ANSWER in
                         ABORTED) ;;
                         END)     break;;
                      esac
                      ;;
               random_sample)
                                  # select a random file from the library
                      # and play it
                      IGNORE=`randomfile $LIBRARY_DIR`
                      # why do we need this prefix ?
                      ANSWER=`playfile $LIBRARY_DIR/$IGNORE`
                      case $ANSWER in
                         ABORTED) ;;
                         END)     break;;
                      esac
                      ;;
               edit_menu)
                      # menu editing for the owner or the 0
                                  # for the first version can be hardcoded,
                                  # then it should be a normal menu with
                                  # items (who control the access each time).
                                  if [ "$CURRENT_USER" != "" ]; then
                                     if [ -f $CURRENT_MENU/.owner ]; then
                                        ANSWER=`cat $CURRENT_MENU/.owner`
                                     else
                                        ANSWER=0
                                     fi
                                     if [ "$ANSWER" = "$CURRENT_USER" ]; then
                                        ANSWER=`edit_menu $CURRENT_MENU`
                         case $ANSWER in
                            READY)   ;;
                            ABORTED) ;;
                            *)       ;;
                         esac
                                     fi
                                  fi
                      ;;
                        edit_user)
                      # user editing for 0 only --- not implemented
                                  if [ "$CURRENT_USER" = "0" ]; then
                                     ANSWER=`edit_user`
                               case $ANSWER in
                            END)     break;;
                         ABORTED) ;;
                         *)       ;;
                            esac
                                  fi
                      ;;
               message_to_operator)
                      # leave message to operator
                      if [ "$CURRENT_USER" = "" ]; then
                         ANSWER=`sendmail 0 -1 NO`
                      else
                         ANSWER=`sendmail 0 $CURRENT_USER NO`
                      fi
                      case $ANSWER in
                         ABORT) ;;
                         END)   break;;
                      esac
                      ;;
               identify) # identify user
                      ANSWER=READY
                                  if [ "$CURRENT_USER" != "" ]; then
                                     ANSWER=`playfile $ALREADY_LOGGED_FILE`
                                  fi
                                  case $ANSWER in
                                     ABORT) ;;
                                     READY) ANSWER=`dologin`
                                case $ANSWER in
                                [0-9]*)  if [ -f $MAILBOX_DIR/$ANSWER/.level ]; then
                                                           USER_LEVEL=`cat $MAILBOX_DIR/$ANSWER/.level`
                                                        else
                                                           USER_LEVEL=1 # identified.
                                                        fi
                                                        CURRENT_USER=$ANSWER
                                      ANSWER=`readmail $CURRENT_USER`
                                      case $ANSWER in
                                         ABORTED) ;;
                                         END)     break;;
                                      esac;;
                                NONE)    ;;
                                ABORTED) ;;
                                END)     break;;
                                esac;;
                                     *)     break;;
                                  esac
                      ;;
                        back_menu) break;;
                        goto_menu) if [ $# = 1 ]; then
                                      ANSWER=`dovoicemail $1`
                                      case $ANSWER in
                             END|goodbye) break;;
                                      esac
                                   fi;;
               message_to_user) if [ $# = 1 ]; then
                                # leave message to specified user
                             if [ "$CURRENT_USER" = "" ]; then
                                ANSWER=`sendmail $1 -1 NO`
                             else
                                ANSWER=`sendmail $1 $CURRENT_USER NO`
                             fi
                             case $ANSWER in
                                ABORT) ;;
                                END)   break;;
                             esac
                          fi;;
                        # MISSING: see mips/private/mvm/SPEC and menu_items_types

                     esac;;
      esac
      done
      case $ANSWER in
         END|goodbye) break;;
         backmenu)    ANSWER=READY
                      break;;
      esac
   done
   echo "$ANSWER"
}

#
# Let's see if the voice library is talking to us
#

ANSWER=`receive`
if [ "$ANSWER" != "HELLO SHELL" ]; then
     echo "$0: voice library not answering" >&2
     exit 1
fi

#
# Let's answer the message
#

send "HELLO VOICE PROGRAM"

#
# Let's see if it worked
#

ANSWER=`receive`
if [ "$ANSWER" != "READY" ]; then
     echo "$0: initialization failed" >&2
     exit 1
fi


#
# Enable events
#

send "ENABLE EVENTS"
ANSWER=`receive`
if [ "$ANSWER" != "READY" ]; then
     echo "$0: could not enable events" >&2
     exit 1
fi

# FROM NOW ON, TRY TO HANDLE ALL POSSIBLE CASES CLEVERLY

# Let's wait for two seconds

send "WAIT 2"
ANSWER=`receive`
# BUG: what if we receive the DTMF event here and after the WAITING ?
if [ "$ANSWER" != "WAITING" ]; then
     echo "$0: could not start waiting" >&2
fi

while :
do
     ANSWER=`receive`

     case $ANSWER in
        DATA_CALLING_TONE|FAX_CALLING_TONE) MODEM=true
                                            break;;
        RECEIVED_DTMF) MODEM=false
                       break;;
        0|READY) MODEM=true # Timeout
               break;;
        # BUGS: if we receive a hangup event, we will ignore it and
        # timeout. The ATA will then fail and recover after some time.
     esac
done

if [ $MODEM = true ]; then
   send "STOP"
   ANSWER=`receive`
   # BUG: what if we receive a DTMF here ?
   if [ "$ANSWER" != "READY" ]; then
     echo "$0: could not stop waiting" >&2
   fi

   send "GOODBYE" # ENTER_DATA_FAX_MODE

   ANSWER=`receive`

   if [ "$ANSWER" != "GOODBYE SHELL" ]; then
     echo "$0: could not say goodbye to the voice library" >&2
   fi

   exit 3 # special value
else
   # Get the received DTMF

   ANSWER=`receive`
   echo "dtmf is $ANSWER" >> /tmp/voice_events

   send "STOP"
   ANSWER=`receive`
   # BUG: what if we receive another DTMF ?
   if [ "$ANSWER" != "READY" ]; then
     echo "$0: could not stop waiting" >&2
   fi

fi

send "DEVICE DIALUP_LINE"
ANSWER=`receive`
# no check =

ANSWER=`setlanguage`
case $ANSWER in
 [0-9]*|ABORTED) case $ANSWER in
                    [0-9]*) SELECTED_LANGUAGE=$ANSWER
                            if [ "$SELECTED_LANGUAGE" = "0" ]; then
                               SELECTED_LANGUAGE=
                            fi;;
                 esac

  case `playfile $INTRODUCTION_FILE` in
   READY|ABORTED)

   # The voicemail thing

   while :
   do
      ANSWER=`dovoicemail $VOICEMAIL/main`
      case $ANSWER in
         READY) ;;  # falling back_menu
         *)     break;;
      esac
   done

   if [ "$ANSWER" != "END" ]; then
      IGNORE=`playfile $GOODBYE_FILE`
   fi;;
 esac;;
esac

# Let's say goodbye

send "GOODBYE"
IGNORE=`receive`

exit 0

