/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPSession.h

	Contains:	RTPSession represents an, well, an RTP session. The server creates
				one of these when a new client connects, and it lives for the duration
				of an RTP presentation. It contains all the resources associated with
				that presentation, such as RTPStream objects. It also provides an
				API for manipulating a session (playing it, pausing it, stopping it, etc) 
				
				It is also the active element, ie. it is the object that gets scheduled
				to send out & receive RTP & RTCP packets
				
	Change History (most recent first):

	$Log: RTPSession.h,v $
	Revision 1.2  1999/02/19 23:08:26  ds
	Created
	

*/


#ifndef _RTPSESSION_H_
#define _RTPSESSION_H_

#include "RTSPProtocol.h"
#include "RTPServerInterface.h"

#include "OSQueue.h"
#include "QTSS.h"
#include "IdleTask.h"
#include "TimeoutTask.h"
#include "SessionDictionary.h"

#include "RTPModule.h"
#include "RTSPServerInterface.h"
#include "atomic.h"

#include "SessionDictionary.h"

#define RTPSESSION_DEBUGGING 0 	//enable to produce verbose printfs confirming correctness 
#define RTPSESSION_LOGGING 1 	//turn on code to log timestamps / seq #'s for each session.
								//this code only gets executed if a command-line pref is specified

//RTP sessions are tasks simply so that they can be timed out
class RTPSession : public Task 
{
	public:
	
		RTPSession();
		virtual ~RTPSession();
		
		//This puts this session into the session map (called by the server! not the module!)
		//If this function fails (it can return QTSS_DupName), it means that there is already
		//a session with this session ID in the map.
		QTSS_ErrorCode	Activate(const StrPtrLen& inSessionID);
		
		//STATE. A session can be playing or paused currently
		enum
		{
			kPausedState = 0,	//UInt32s
			kPlayingState = 1
		};
		
		//ACCESS FUNCTIONS
		
		OSQueue*	GetStreamQueue() 	{ return &fStreamQueue; }
		OSMutex*	GetSessionMutex()	{ return &fSessionMutex; }
		RTPModule*	GetModule()			{ return fModule; }
		bool	IsBound() 				{ return fModule != NULL; }
		bool	IsFirstPlay()			{ return fIsFirstPlay; }
		//Time (msec) first play was issued
		SInt64	GetFirstPlayTime()		{ return fFirstPlayTime; }
		//Time (msec) most recent play was issued
		SInt64	GetPlayTime()			{ return fPlayTime; }
		SInt64	GetNTPPlayTime()		{ return fNTPPlayTime; }
		//Time (msec) most recent play, adjusted for start time of the movie
		//ex: PlayTime() == 20,000. Client said start 10 sec into the movie,
		//so AdjustedPlayTime() == 10,000
		SInt64	GetAdjustedPlayTime()	{ return fAdjustedPlayTime; }
		UInt32	GetState()				{ return fState; }
		QTSS_MediaSrcRef GetMediaSrcRef(){ return fMediaRef; }
		UInt32	GetBytesSent()			{ return fBytesSent; }
		UInt32	GetPacketsSent()			{ return fPacketsSent; }
		char*	GetRTCPSR()				{ return fSenderReportBuffer; }
		UInt32	GetRTCPSRLen()			{ return fSenderReportSize; }
		UInt32	GetNumQualityLevels()	{ return fNumQualityLevels; }
		SInt64	GetSessionCreateTime()	{ return fSessionCreateTime; }
		FILE*	GetDebugFile()			{ return fDebugFile; }
				
		SessionDictionary* GetSessionDictionary(){return &fSessionDictionary;}
		
		//Timeouts. This allows clients to disarm and re-arm the timeout on this session
		void			DisarmTimeout() { fTimeoutTask.SetTimeoutMilSecs(0); }
		void			RearmTimeout()  { fTimeoutTask.SetTimeoutMilSecs(fTimeout); }
		
		//The session tracks the total number of bytes sent during the session.
		//Streams can update that value by calling this function
		void			UpdateBytesSent(UInt32 bytesSent)
										{ (void)atomic_add(&fBytesSent, bytesSent); }
						
		//The session tracks the total number of packets sent during the session.
		//Streams can update that value by calling this function				
		void  			UpdatePacketsSent(UInt32 packetsSent)  
										{ (void)atomic_add(&fPacketsSent, packetsSent); }

		//Binding a session.
		//A session exists on behalf of a module. Establishing this relationship
		//requires the module bind itself to the session. It does so by passing a
		//MediaSrcRef to the Bind function.
		void	Bind(RTPModule* inModule, QTSS_MediaSrcRef inRef, UInt32 inNumQualityLevels)
						{ 	fModule = inModule; fMediaRef = inRef;
							fNumQualityLevels = inNumQualityLevels; }
		
		//Once the session is bound, a module can add streams to it.
		//It must pass in a streamID that uniquely identifies this stream.
		//This call can only be made during an RTSP Setup request, and the
		//RTSPRequestInterface must be provided.
		//You may also opt to attach a codec name and type to this stream.
		RTSPProtocol::RTSPStatusCode	AddStream(UInt32 inStreamID, StrPtrLen* inCodecName,
										UInt32 inCodecType,
										RTSPRequestInterface* request,
										RTPStream** outStream);
		
		//Begins playing all streams. Currently must be associated with an RTSP Play
		//request, and the request interface must be provided.
		void	Play(RTSPRequestInterface* request, bool inSendRTCP,
							bool inShouldWriteHeader);
		
		//Pauses all streams.
		void			Pause() { fState = kPausedState; }

		//Utility functions. Modules aren't required to use these, but can be useful
		void			SendPlayResponse(RTSPRequestInterface* request);
		void			SendPauseResponse(RTSPRequestInterface* request)
																{ request->SendHeader(); }
		void			SendTeardownResponse(RTSPRequestInterface* request)
									{ request->SetKeepAlive(false); request->SendHeader(); }

		
	private:
	
		//where timeouts, deletion conditions get processed
		virtual SInt64	Run();
	
		//time to wait between issuing the play and actually playing
		enum
		{
			kPlayDelayTimeInMilSecs = 500 //UInt32
		};

		OSQueue				fStreamQueue;//queue of associated streams

		//stores the session ID
		OSRef			 	fRTPMapElem;
		StrPtrLen			fRTSPSessionID;
		char				fRTSPSessionIDBuf[RTPModuleInterface::kMaxSessionIDLength + 2];
		
		QTSS_MediaSrcRef	fMediaRef;
		RTPModule*			fModule;
		
		//Some stream related information that is shared amongst all streams
		bool		fIsFirstPlay;
		SInt64		fFirstPlayTime;//in milliseconds
		SInt64		fPlayTime;
		SInt64		fAdjustedPlayTime;
		SInt64		fNTPPlayTime;
		SInt64		fSessionCreateTime;
		
		//Start and stop times for this play spurt
		Float64		fStartTime;
		Float64		fStopTime;
		
		//for timing out this session
		TimeoutTask	fTimeoutTask;
		UInt32		fTimeout;
		
		//keeps track of whether we are playing or not
		UInt32		fState;
		
		//Packet priority levels. The session has a current level, and
		//the module that owns this session sets what the number of levels is.
		UInt32		fNumQualityLevels;
		
		//Used for storing various miscellanea about the session (RTCP stats, etc.)
		SessionDictionary fSessionDictionary;
		
		//Statistics
		unsigned int fBytesSent;
		unsigned int fPacketsSent;
		
		//Session mutex. This mutex should be grabbed before invoking the module
		//responsible for managing this session. This allows the module to be
		//non-preemptive-safe with respect to a session
		OSMutex		fSessionMutex;
		
		//stuff for RTCP sender reports
		enum
		{
			kSenderReportSizeInBytes = 36	//UInt32
		};
		char		fSenderReportBuffer[kSenderReportSizeInBytes + RTPServerInterface::kMaxCNameLen];
		UInt32		fSenderReportSize;
		
		//A fd for writing session debugging information
		FILE*		fDebugFile;
};

#endif //_RTPSESSION_H_