/****************************************************************************/
/*                                                                          */
/* Program:   i_tools.c                                                     */
/* Author:    Simon A.J. Winder                                             */
/* Date:      Wed Feb 19 21:41:56 1992                                      */
/* Function:  Image manipulation tools.                                     */
/* Copyright (C) 1994 Simon A.J. Winder                                     */
/*                                                                          */
/****************************************************************************/

#include "lib.h"
#include <math.h>
#include <values.h>

/*#define debug(m) DEBUG("<tools> " m)*/
#define debug(m)

/*****************************************************************************/

#define ROOTTWOPI (sqrt(2.0*M_PI))

it_image *i_create_operator(int width,int height,double *op)
{
  it_image *im;
  int x,y;

  debug("Entering I_CREATE_OPERATOR");
  im=i_create_image(width,height,IT_DOUBLE,IM_CONTIG);
  if(im!=NULL)
    {
      for(y=0;y<height;y++)
	for(x=0;x<width;x++)
	  im_double_value(im,x,y)= *op++;
    }
  return(im);
}

int i_convolve(it_image *im1,it_image *im2,it_image *op)
{
  register int xx;
  register it_double *opptr;
  register it_float *srcptr;
  register double sum;
  int src_x,src_y,src_width,src_height,op_width,op_height;
  int dest_x,dest_y,dest_width,dest_height;
  int yd_end,xs_end,yy_start,yy_end,yd,xs,yy;
  it_double *opptrinit;
  it_float *destptr;
  double maxsum,minsum;

  /* reject silly values */
  debug("Entering I_CONVOLVE");
  if(im1==NULL || im2==NULL || op==NULL)
    return(1);
  if(im1->width!=im2->width || im1->height!=im2->height)
    return(1);
  if(op->width<1 || op->height<1 || op->mode!=IM_CONTIG)
    return(1);
  if(im1->type!=IT_FLOAT || im2->type!=IT_FLOAT || op->type!=IT_DOUBLE)
    return(1);

  /* calculate parameters */
  src_x=im1->valid_x;
  src_y=im1->valid_y;
  src_width=im1->valid_width;
  src_height=im1->valid_height;
  op_width=op->width;
  op_height=op->height;
  dest_x=src_x+(op_width-1)/2;
  dest_y=src_y+(op_height-1)/2;
  dest_width=src_width-op_width+1;
  dest_height=src_height-op_height+1;

  /* check there will be an output area */
  if(dest_width<1 || dest_height<1)
    return(1);

  /* set the oputput area in image 2 */
  im2->valid_x=dest_x;
  im2->valid_y=dest_y;
  im2->valid_width=dest_width;
  im2->valid_height=dest_height;

  /* initialize prior to looping */
  opptrinit=im_double_row(op,op_height-1)+op_width-1;
  yd_end=dest_y+dest_height;
  xs_end=src_x+dest_width;
  yy_start=src_y;
  yy_end=src_y+op_height;

  /* optimized convolution */
  debug("convolving");
  minsum=MAXFLOAT;
  maxsum= -MAXFLOAT;
  for(yd=dest_y;yd<yd_end;yd++)
    {
      destptr=im_float_row(im2,yd)+dest_x;
      for(xs=src_x;xs<xs_end;xs++)
	{
	  opptr=opptrinit;
	  sum=0.0;
	  for(yy=yy_start;yy<yy_end;yy++)
	    {
	      srcptr=im_float_row(im1,yy)+xs;
	      for(xx=0;xx<op_width;xx++)
		sum+=(*opptr--)*((double) (*srcptr++));
	    }
	  *destptr++ =(it_float) sum;
	  if(sum>maxsum) maxsum=sum;
	  if(sum<minsum) minsum=sum;
	}
      yy_start++;
      yy_end++;
    }

  im2->max_value=maxsum;
  im2->min_value=minsum;
  return(0);
}

int i_smooth(it_image *im1,it_image *im2,double sigma)
{
  register int xx,yy;
  register double *kernptr;
  it_image *im3;
  double *kernel,mult,fact,sum,maxsum,minsum;
  int src_x,src_y,src_width,src_height,dest_x,dest_y,dest_width,dest_height;
  int op_rad,op_diam,r,x,y,yyend;
  it_float *fptr1,*fptr2;
  it_double *dptr1,**dfield;
  
  /* check for bad parameter values */
  debug("Entering I_SMOOTH");
  if(im1==NULL || im2==NULL)
    return(1);
  if(im1->width!=im2->width || im1->height!=im2->height)
    return(1);
  if(im1->type!=IT_FLOAT || im2->type!=IT_FLOAT)
    return(1);
  
  /* compute regions and the operator size */
  op_rad=(int) (sigma*3.0+0.5);
  op_diam=op_rad*2+1;
  src_x=im1->valid_x;
  src_y=im1->valid_y;
  src_width=im1->valid_width;
  src_height=im1->valid_height;
  dest_x=src_x+op_rad;
  dest_y=src_y+op_rad;
  dest_width=src_width-op_diam+1;
  dest_height=src_height-op_diam+1;

  /* create image and malloc operator */
  if(dest_height<1 || dest_width<1)
    return(1);
  debug("creating intermediate image");
  if((kernel=(double *) malloc(op_diam*sizeof(double)))==NULL)
    return(-1);
  if((im3=i_create_image(dest_width,src_height,IT_DOUBLE,IM_FRAGMENT))==NULL)
    return(-1);

  /* set resultant area */
  im2->valid_x=dest_x;
  im2->valid_y=dest_y;
  im2->valid_width=dest_width;
  im2->valid_height=dest_height;

  /* make the kernel */
  kernptr=kernel;
  mult=1.0/(sigma*ROOTTWOPI);
  fact= -1.0/(2.0*sigma*sigma);
  sum=0.0;
  for(r= -op_rad;r<=op_rad;r++)
    {
      *kernptr = (mult*exp(fact*((double) r)*((double) r)));
      sum+= *kernptr++;
    }
  /* normalise to area of 1.0 */
  kernptr=kernel;
  for(r= -op_rad;r<=op_rad;r++)
    *kernptr++ /=sum;

  /* smooth horizontally */
  debug("smoothing horizontally");
  for(y=0;y<src_height;y++)
    {
      dptr1=im_double_row(im3,y);
      fptr1=im_float_row(im1,y+src_y)+src_x;
      for(x=0;x<dest_width;x++)
	{
	  sum=0.0;
	  kernptr=kernel;
	  fptr2=fptr1++;
	  for(xx=0;xx<op_diam;xx++)
	    sum+=(*kernptr++)*((double) *fptr2++);
	  *dptr1++ =sum;
	}
    }

  /* smooth vertically */
  debug("smoothing vertically");
  maxsum= -MAXFLOAT;
  minsum=MAXFLOAT;
  dfield=im_double_field(im3);
  for(y=0;y<dest_height;y++)
    {
      yyend=y+op_diam;
      fptr1=im_float_row(im2,y+dest_y)+dest_x;
      for(x=0;x<dest_width;x++)
	{
	  sum=0.0;
	  kernptr=kernel;
	  for(yy=y;yy<yyend;yy++)
	    sum+=(*kernptr++)*dfield[yy][x];
	  *fptr1++ =(it_float) sum;
	  if(sum>maxsum) maxsum=sum;
	  if(sum<minsum) minsum=sum;
	}
    }

  /* set max and min and destroy storage */
  im2->max_value=maxsum;
  im2->min_value=minsum;
  debug("destroying intermediate image");
  i_destroy_image(im3);
  free((char *) kernel);
  return(0);
}

#define to_float(n) ((float)(n))
#define to_double(f) ((double)(f))
#define pi2          to_double(6.28318530717959)
#define index(i)     (((i) - 1) >> 1)
#define swap(x,y,t)  ((t) = (x), (x) = (y), (y) = (t))

int i_fourier_transform_2d(it_image *matrix,int type)
{
  long z_size;
  int size[2];
  int dim,n,nprev;
  int nrem,ip1,ip2,ip3,i2rev;
  int i1,i2,i3,i3rev,ibit;
  it_complex *z,*z1,*z2,tmp;
  int ifp1,ifp2;
  double wpr,wpi,wr,wi;
  double theta,wtemp;
  float Re,Im;

  if(matrix->type!=IT_COMPLEX || matrix->mode!=IM_CONTIG)
    return(1);

  /* compute total number of complex values */
  z_size=(long) matrix->width * (long) matrix->height;
  size[0]=matrix->width;
  size[1]=matrix->height;
  z=im_complex_row(matrix,0);

  for(nprev=1, dim=1; dim>=0; nprev*=n, dim--)
    {
      /* initialisation */
      n=size[dim];
      nrem=z_size/(n*nprev);
      ip1= nprev<<1;
      ip2= ip1*n;
      ip3=ip2*nrem;
      i2rev=1;
      
      /* bit reversal section */
      for(i2=1;i2<=ip2;i2rev+=ibit,i2+=ip1)
	{
	  if(i2<i2rev)
	    {
	      for(i1=i2;i1<=(i2+ip1-2);i1+=2)
		{
		  for(i3=i1;i3<=ip3;i3+=ip2)
		    {
		      i3rev=i2rev+i3-i2;
		      z1= &z[index(i3)];
		      z2= &z[index(i3rev)];
		      swap(*z1,*z2,tmp);
		    }
		}
	    }
	  for(ibit=(ip2>>1);(ibit>=ip1)&&(i2rev>ibit);i2rev-=ibit,ibit>>=1);
	}

      /* Danielson-Lanczos section */
      for(ifp1=ip1;ifp1<ip2;ifp1=ifp2)
	{
	  /* initialise trig values to avoid repeated calls */
	  ifp2=(ifp1<<1);
	  theta=(type*pi2)/(ifp2/to_double(ip1));
	  wtemp=sin(theta*0.5);
	  wpr=wtemp*wtemp*(-2.0);
          wpi=sin(theta);
          wr=1.0;
          wi=0.0;
	  
	  for(i3=1;i3<=ifp1;i3+=ip1)
	    {
	      for(i1=i3;i1<=(i3+ip1-2);i1+=2)
		{
		  for(i2=i1;i2<=ip3;i2+=ifp2)
		    {
		      /* Danielson-Lanczos Formula */
		      z1= &z[index(i2)];
		      z2= &z[index(i2+ifp1)];
		      Re = to_float((wr * z2->Re) - (wi * z2->Im));
		      Im = to_float((wr * z2->Im) + (wi * z2->Re));
		      z2->Re = (z1->Re) - Re;
		      z2->Im = (z1->Im) - Im;
		      z1->Re += Re;
		      z1->Im += Im;
		    }
		}
	      wr=((wtemp=wr)*wpr)-(wi*wpi)+wr;
	      wi=(wi*wpr)+(wtemp*wpi)+wi;
	    }
	}
    }
  return(0);
}

int i_fourier_transform_1d(it_image *matrix,int type)
{
  int i,j,k,n2,n;
  int max_k,istep;
  double wr,wi,wpr,wpi;
  double theta,wtmp;
  it_complex *z,*z_i,*z_j,tmp;
  float Re,Im;
  double wtemp;

  if(matrix->type!=IT_COMPLEX)
    return(1);

  z=im_complex_row(matrix,0);
  n=matrix->width;

  /* Bit Reversal Section */
  for(n2=n<<1,j=i=1; i<n2; j+=k,i+=2)
    {
      if(j>i)
	{
	  z_i= &z[index(i)];
	  z_j= &z[index(j)];
	  swap(*z_i,*z_j,tmp);
	}
      for(k=n; (k>=2) && (j>k); j-=k,k>>=1);
    }
  
  /* Danielson-Lanczos Section */
  for(n2=n<<1,max_k=2; n2>max_k; max_k=istep)
    {
      /* Initialise for trigonometric recurrence */
      istep = max_k<<1;
      theta = pi2/(type * to_double(max_k));
      wtmp  = sin(theta * to_double(0.5));
      wpr   = wtmp * wtmp * to_double(-2.0);
      wpi   = sin(theta);
      wr    = 1.0;
      wi    = 0.0;
      
      for(k=1; k<max_k; k+=2)
	{
	  for(i=k; i<=n2; i+=istep)
	    {
	      /* Danielson-Lanczos Formula */
	      z_i= &z[index(i)];
	      z_j= &z[index(i+max_k)];

	      Re=to_float((wr * (z_j->Re)) - (wi * (z_j->Im)));
	      Im=to_float((wr * (z_j->Im)) + (wi * (z_j->Re)));

	      z_j->Re = (z_i->Re) - Re;
	      z_j->Im = (z_i->Im) - Im;
	      z_i->Re += Re;
	      z_i->Im += Im;
	    }
	  /* Trigonometric Recurrence */
	  wr = ((wtemp=wr) * wpr) - (wi * wpi) + wr;
	  wi = (wi * wpr) + (wtemp * wpi) + wi;
	}
    }
  return(0);
}
/* Version 1.0 (Oct 1994) */
/* Version 1.1 (Nov 1994) */
