/* -*- C++ -*- */
// $Id: TAO.h,v 1.27 1999/10/25 05:27:57 coryan Exp $

// ============================================================================
//
// = LIBRARY
//    tao
//
// = FILENAME
//   TAO.h
//
// = DESCRIPTION
//   Externally visible TAO extensions.
//
// = AUTHOR
//    Chris Cleeland
//
// ============================================================================

#ifndef TAO_H
#define TAO_H

#include "tao/corbafwd.h"

#if !defined (ACE_LACKS_PRAGMA_ONCE)
# pragma once
#endif /* ACE_LACKS_PRAGMA_ONCE */

#include "tao/POAC.h"
#include "tao/ORB.h"

class TAO_Export TAO_ORB_Manager
{
  // = TITLE
  //     Helper class for simple ORB/POA initialization and
  //     registering servants with the POA.
  //
  // = DESCRIPTION
  //     This class is a TAO extension that makes it easier to write
  //     CORBA applications.  It's just a wrapper and doesn't do
  //     anything special within the ORB itself.
public:
  // = Initialization and termination methods.
  TAO_ORB_Manager (CORBA::ORB_ptr orb = 0,
                   PortableServer::POA_ptr poa = 0,
                   PortableServer::POAManager_ptr poa_manager = 0);
  // Constructor.

  ~TAO_ORB_Manager (void);
  // Destructor.

  int init (int &argc,
            char *argv[],
            CORBA_Environment &ACE_TRY_ENV = TAO_default_environment ());
  // Initialize the ORB/root POA, using the supplied command line
  // arguments or the default ORB components.  Returns -1 on failure.

  int init (int &argc,
            char *argv[],
            const char *orb_name,
            CORBA_Environment &ACE_TRY_ENV = TAO_default_environment ());
  // Initialize the ORB/root POA, using the supplied command line
  // arguments or the default ORB components.  Returns -1 on failure.

  int init_child_poa (int &argc,
                      char *argv[],
                      const char *poa_name,
                      CORBA_Environment &ACE_TRY_ENV =
                          TAO_default_environment ());
  // Creates a child poa under the root poa with PERSISTENT and
  // USER_ID policies.  Call this if you want a <child_poa> with the
  // above policies, otherwise call init.  Returns -1 on failure.

  int init_child_poa (int &argc,
                      char *argv[],
                      const char *poa_name,
                      const char *orb_name,
                      CORBA_Environment &ACE_TRY_ENV =
                          TAO_default_environment ());
  // Creates a child poa under the root poa with PERSISTENT and
  // USER_ID policies.  Call this if you want a <child_poa> with the
  // above policies, otherwise call init.  Returns -1 on failure.

  int activate_poa_manager (CORBA_Environment &ACE_TRY_ENV =
                                TAO_default_environment ());
  //  Put POA manager into the <Active> state, so that incoming corba
  //  requests are processed.  This method is useful for clients,
  //  which are not going to enter "orb->run" loop, yet may want to
  //  service incoming requests while waiting for a result of CORBA
  //  call on a server.  Returns -1 on failure.

  char* activate (PortableServer::Servant servant,
                  CORBA_Environment &ACE_TRY_ENV =
                      TAO_default_environment ());
  // Activate <servant>, using the POA <activate_object> call.  Users
  // can call this method multiple times to activate multiple objects.
  // Returns 0 on failure.

  void deactivate (const char *id,
                   CORBA_Environment &ACE_TRY_ENV =
                       TAO_default_environment ());
  // Deactivate object in RootPOA.

  char * activate_under_child_poa (const char *servant_name,
                                   PortableServer::Servant servant,
                                   CORBA_Environment &ACE_TRY_ENV =
                                       TAO_default_environment ());
  // Precondition: init_child_poa has been called.  Activate <servant>
  // using the POA <activate_object_with_id> created from the string
  // servant_name. Users should call this to activate objects under
  // the child_poa.

  void deactivate_under_child_poa (const char *id,
                                   CORBA_Environment &ACE_TRY_ENV =
                                       TAO_default_environment ());
  // Deactivate object in child POA.

  int run (CORBA_Environment &ACE_TRY_ENV,
           ACE_Time_Value *tv);
  int run (ACE_Time_Value &tv,
           CORBA_Environment &ACE_TRY_ENV = TAO_default_environment ());
  int run (CORBA_Environment &ACE_TRY_ENV = TAO_default_environment ());
  // Run the ORB event loop with the specified <tv> time value Returns
  // -1 on failure.

  CORBA::ORB_ptr orb (void);
  // Accessor which returns the ORB pointer.  Following the normal
  // CORBA memory management rules of return values from functions,
  // this function duplicates the orb return value before returning
  // it.

  PortableServer::POA_ptr root_poa (void);
  // Accessor which returns the root poa. Following the normal CORBA
  // memory management rules of return values from functions, this
  // function duplicates the poa return value before returning it.

  PortableServer::POA_ptr child_poa (void);
  // Accessor which returns the child poa. Following the normal CORBA
  // memory management rules of return values from functions, this
  // function duplicates the poa return value before returning it.

  PortableServer::POAManager_ptr poa_manager (void);
  // Accessor which returns the poa manager. Following the normal
  // CORBA memory management rules of return values from functions,
  // this function duplicates the poa manager return value before
  // returning it.

protected:
  CORBA::ORB_var orb_;
  // The ORB.

  PortableServer::POA_var poa_;
  // The POA for this ORB.

  PortableServer::POA_var child_poa_;
  // Child poa under the root POA.

  PortableServer::POAManager_var poa_manager_;
  // The POA manager of poa_.
};


#endif /* TAO_H */
