/* tpconfig --- Command-line TouchPad configuration utility.
 * Copyright (C) 1997  C. Scott Ananian <cananian@alumni.princeton.edu>
 * 
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *    
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *    
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   $Id: tpconfig.c,v 1.4 1997/12/16 12:33:40 cananian Exp $
 */

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <assert.h>

#define fatal(string) \
{ fprintf(stderr, "fatal: " string); exit(1); } 

/* Prototypes for PS/2 utility functions */
typedef unsigned char byte;

/* write a byte to the ps/2 port, handling ACK */
void putbyte(int fd, byte b);
/* read a byte from the ps/2 port */
byte getbyte(int fd);
/* use the Synaptics extended ps/2 syntax to write a special command byte */
void send_cmd(int fd, byte cmd);
/* write 'cmd' to mode byte 1 */
void set_mode1(int fd, byte cmd);
/* write 'cmd' to mode byte 2 */
void set_mode2(int fd, byte cmd);
/* read three byte status ('a','b','c') corresponding to register 'cmd' */
void status_rqst(int fd, byte cmd, byte *byte1, byte *byte2, byte *byte3);
/* get mode byte 1 */
byte get_mode1(int fd);
/* get mode byte 2 */
byte get_mode2(int fd);
/* initialize and determine whether this is a synaptics touchpad */
int is_touchpad(int fd);

/*--------------------------------------------------------------------*/

int silent = 0; /* should we print anything to the console? */

/* Other prototypes */

/* print copyright information */
void copyright(void);
/* print usage information */
void usage(char *progname);
/* print info about the touchpad to file 'info' */
void touchpad_info(int fd, FILE *info);

/* initialize file description if hasn't already been done. */
void init_fd(int *fd) {
  if (*fd<0) {
    *fd = open("/dev/psaux", O_RDWR|O_SYNC);	
    if (*fd<0) fatal("Could not open /dev/psaux");	
    if (!is_touchpad(*fd)) fatal("Not a touchpad device.\n");
    if (!silent) 
      fprintf(stdout, 
	   "Synaptics Touchpad configurator. (c) 1997 C. Scott Ananian\n\n");
  }
}

#include <getopt.h>

int main(int argc, char **argv) {
  int fd = -1;
  int c;
  int digit_optind = 0;

  if (argc<2) usage(argv[0]); /* no command line options */

  opterr=1;
  while (1) {
    int this_option_optind = optind ? optind : 1;
    int option_index = 0;
    static struct option long_options[] = 
    {
      {"help",		no_argument, NULL, 'h'},
      {"silent",	no_argument, NULL, 'q'}, /* -q */
      {"quiet",		no_argument, NULL, 'q'}, /* -q */
      {"info",		no_argument, NULL, 'i'}, /* -i */
      {"version", 	no_argument, NULL, 'v'}, /* -v */
      /* Mode 1 options */
      {"zthreshold",	optional_argument, NULL, 'z'}, /* -z */
      {"threshold",	optional_argument, NULL, 'z'}, /* -z */
      {"corner",	optional_argument, NULL, 'c'}, /* -c */
      {"tapmode",	optional_argument, NULL, 't'}, /* -t */
      {"edgemode",	optional_argument, NULL, 'e'}, /* -e */
      /* Mode 2 options */
      {"absolute",	no_argument, NULL, 'A'},       /* similar to -a */
      {"relative",	no_argument, NULL, 'R'},       /* similar to -a */
      {"rate",		optional_argument, NULL, 'r'}, /* -r */
      {"two-button",	no_argument, NULL, '2'},       /* -2 */
      {"three-button",	no_argument, NULL, '3'},       /* -3 */
      {"middle-button", no_argument, NULL, '<'},       /* similar to -m */
      {"right-button",	no_argument, NULL, '>'},       /* similar to -m */
      {NULL, 		no_argument, NULL, 0}
    };
    c = getopt_long (argc, argv, "qivz::c::t::e::a::r::23m::",
		     long_options, &option_index);
    if (c == -1)
      break;
    
    switch (c) {
    case 'h': /* --help */
      usage(argv[0]);
      break;
    case 'q': /* --quiet */
      silent = 1;
      break;
    case 'v': /* --version */
      copyright();
      break;
    case 'i': /* --info */
      init_fd(&fd);
      if (!silent) touchpad_info(fd, stderr);
      break;
    case 'z':       /* Mode 1 modifications */
    case 'c':
    case 't':
    case 'e':
      { 
	byte mode1;
	init_fd(&fd);
	mode1 = get_mode1(fd);
	switch(c) {
	case 'z': /* --zthreshold */
	  if (optarg) {
	    if ((optarg[0]<'0') || (optarg[0]>'7')) 
	      fprintf(stderr, "Invalid Z threshold. [Use 0-7]\n");
	    else {
	      mode1 = (mode1&0x8F)|((optarg[0]-'0')<<4);
	      set_mode1(fd, mode1);
	    }
	  }
	  if (!silent) 
	    fprintf(stderr, "Z threshold is %d of 7\n", 
		    (unsigned)(mode1>>4)&0x7);
	  break;
	case 't': /* --tapmode */
	  if (optarg) {
	    if ((optarg[0]<'0') || (optarg[0]>'3'))
	      fprintf(stderr, "Invalid tap mode. [Use 0-3]\n");
	    else {
	      mode1 = (mode1&0xF3)|((optarg[0]-'0')<<2);
	      set_mode1(fd, mode1);
	    }
	  }
	  if (!silent)
	    fprintf(stderr, "%s\n", 
		    (((mode1>>2)&0x3)==0)?"No tap gestures.":
		    (((mode1>>2)&0x3)==1)?"Tap-to-click only.":
		    (((mode1>>2)&0x3)==2)?"Tap and non-locking drag.":
		    (((mode1>>2)&0x3)==3)?"Tap and locking drag.":
		    "internal error");
	  break;
	case 'e': /* --edgemode */
	  if (optarg) {
	    if ((optarg[0]<'0') || (optarg[0]>'3') || (optarg[0]=='2'))
	      fprintf(stderr, "Invalid edge mode. [Use 0,1,3]\n");
	    else {
	      mode1 = (mode1&0xFC)|((optarg[0]-'0')<<0);
	      set_mode1(fd, mode1);
	    }
	  }
	  if (!silent)
	    fprintf(stderr, "%s\n", 
		    ((mode1&0x3)==0)?"No edge motion.":
		    ((mode1&0x3)==1)?"Edge motion always.":
		    ((mode1&0x3)==3)?"Edge motion during drag.":
		    "internal error");
	  break;
	case 'c': /* --corner */
	  if (optarg) {
	    if ((optarg[0]<'0') || (optarg[0]>'1'))
	      fprintf(stderr, "Invalid corner setting. [Use 0-1]\n");
	    else {
	      mode1 = (mode1&0x7F)|((optarg[0]-'0')<<7);
	      set_mode1(fd, mode1);
	    }
	  }
	  if (!silent)
	    fprintf(stderr, "Corner taps %sabled\n", (mode1>>7)?"en":"dis");
	  break;
	default:
	  fatal("Internal error.");
	  break;
	}
	break;
      }
    case 'A':
    case 'R':
    case 'a':       /* Mode 2 modifications */
    case 'r':
    case '2':
    case '3':
    case '<':
    case '>':
    case 'm':
      { 
	byte mode2;
	init_fd(&fd);
	mode2 = get_mode2(fd);
	switch(c) {
	case 'A': /* --absolute */
	case 'R': /* --relative */
	  if (c=='A') mode2=mode2|0x80;
	  if (c=='R') mode2=mode2&0x7F;
	  set_mode2(fd, mode2);
	case 'a': 
	  if (optarg) {
	    if ((optarg[0]<'0') || (optarg[0]>'1'))
	      fprintf(stderr, "Invalid Reporting mode. [Use 0-1]\n");
	    else { 
	      mode2 = (mode2&0x7F)|((optarg[0]-'0')<<7);
	      set_mode2(fd, mode2);
	    }
	  }
	  if (!silent)
	    fprintf(stderr, "%s\n", 
		    (mode2>>7)?"Absolute XYZ mode":"Relative mode");
	  break;
	case 'r': /* --rate */
	  if (optarg) {
	    if ((optarg[0]<'0') || (optarg[0]>'1'))
	      fprintf(stderr, "Invalid rate. [Use 0-1]\n");
	    else {
	      mode2 = (mode2&0xBF)|((optarg[0]-'0')<<6);
	      set_mode2(fd, mode2);
	    }
	  }
	  if (!silent)
	    fprintf(stderr, "%s sample rate.\n", 
		    ((mode2>>6)&1)?"High":"Low");
	  break;
	case '2': /* --two-button */
	case '3': /* --three-button */
	  mode2 = (mode2&0xF3)|((c=='2')?0:0x4);
	  set_mode2(fd, mode2);
	  if (!silent)
	    fprintf(stderr,"%s-button mode.\n",
		    ((mode2>>2)&1)?"3":"2");
	  break;
	case '<': /* --middle-button */
	case '>': /* --right-button  */
	  if (c=='<') mode2 = mode2|0x02;
	  if (c=='>') mode2 = mode2&0xFD;
	  set_mode2(fd, mode2);
	case 'm':
	  if (optarg) {
	    if ((optarg[0]<'0') || (optarg[0]>'1'))
	      fprintf(stderr, "Invalid middle-button setting. [Use 0-1]\n");
	    else {
	      mode2 = (mode2&0xFD)|((optarg[0]-'0')<<1);
	      set_mode2(fd, mode2);
	    }
	  }
	  if (!silent)
	    fprintf(stderr, "Corner taps simulate %s-button taps.\n", 
		    ((mode2>>1)&1)?"middle":"right");
	  break;
	default:
	  fatal("Internal error.");
	  break;
	}
	break;
      }
    case '?':
      break;
    default:
      fprintf(stderr, "Unknown option %c.\n", (char)c);
      break;
    }
  }
  if (optind < argc) {
    fprintf(stderr, "Extra options: ");
    while (optind < argc)
      fprintf(stderr, "%s ", argv[optind++]);
    fprintf(stderr, "\n");
  }
  if (!(fd<0)) close(fd);
  return 0;
}

void copyright(void) {
  printf(
      "Synaptics Touchpad configuration tool (tpconfig), version " VERSION "\n"
      "Copyright (C) 1997 C. Scott Ananian <cananian@alumni.princeton.edu>\n"
      "tpconfig comes with ABSOLUTELY NO WARRANTY.  This is free software,\n"
      "and you are welcome to redistribute it under the terms of the GPL.\n"
      "\n");
}

void usage(char *progname) {
  copyright();

  printf("Usage: %s [OPTION]...\n", progname?progname:"tpconfig");
  printf
("Configure a Synaptics TouchPad.\n"
 "\n"
 "  -i, --info                    display current TouchPad configuration\n"
 "  -q, --quiet, --silent         suppress verbose output\n"
 "\n"
 "  -2, --two-button              set two-button mode\n"
 "  -3, --three-button            set three-button mode\n" 
 "  -a                            display absolute/relative mode\n"
 "      --absolute, --relative    set absolute/relative mode\n"
 "  -c, --corner=[0-1]            display/enable corner taps:\n"
 "                                   0=disable, 1=enable\n"
 "  -e, --edgemode=[0,1,3]        display/set edge motion:\n"
 "                                   0=never, 1=always, 3=only during drag\n"
 "  -m                            display which button corner taps simulate\n"
 "      --middle-button           make corner taps simulate middle button\n"
 "      --right-button            make corner taps simulate right button\n"
 "  -r, --rate=[0,1]              display/set reporting rate\n"
 "                                   0=normal, 1=high\n"
 "  -t, --tapmode=[0-3]           display/set tapping mode:\n"
 "                                   0 = no tap gestures\n"
 "                                   1 = tap-to-click only\n"
 "                                   2 = tap and non-locking drag\n"
 "                                   3 = tap and locking drag\n"
 "  -z                            display z threshold setting\n"
 "      --zthreshold=[0-7]        set z threshold (tap sensitivity; 0=light)\n"
 "      --threshold=[0-7]         same as --zthreshold\n"
 "\n"
 "      --help                    display this help and exit\n"
 "      --version                 output version information\n"
 "\n"
 "Report bugs to <cananian@alumni.princeton.edu>\n");
  exit(1);
}

void touchpad_info(int fd, FILE *info) {
  byte a,b,c;
  status_rqst(fd, 0x00, &a, &b, &c);
  fprintf(info, "Synaptics Touchpad, firmware rev. %d.%d [%s]\n",
	  (unsigned) c&0xF, (unsigned) a,
	  ((c>>4)==1)?"Standard OEM":
	  ((c>>4)==2)?"180-degree reversed":
	  ((c>>4)==3)?"Serial/PS-2 combo":
	  "unknown model ID");
  status_rqst(fd, 0x03, &a, &b, &c);
  if (b&1!=0) {
    if (b==0x47) fprintf(info,"Firmware revision < 3.2\n");
    if (b==0x03) fprintf(info,"Not a touchpad.\n");
  } else {
    fprintf(info, "Sensor: %s\n",
	    ((a&0x3F)==0)?"Unknown/unspecified":
	    ((a&0x3F)==1)?"Standard-size module":
	    ((a&0x3F)==2)?"\"Mini\" module":
	    "Reserved sensor type");
    fprintf(info, "Geometry: %s\n",
	    ((c&0x0F)==0)?"Unknown/unspecified":
	    ((c&0x0F)==1)?"Standard module":
	    "Reserved geometry type");
    fprintf(info, "%s/%s\t\t",
	    ((a>>7)&1)?"Upside-down":"Rightside-up",
	    ((a>>6)&1)?"Portrait":"Landscape");
    fprintf(info, "%s-style abs-mode packets.\n",
	    (c>>7)?"New":"Old");
  }
  status_rqst(fd, 0x01, &a, &b, &c);
  fprintf(info,"Corner taps %s\t\tTap mode: %s\n",
	  (a>>7)?"enabled. ":"disabled.",
	  (((a>>2)&3)==0)?"No tap gestures":
	  (((a>>2)&3)==1)?"Tap-to-click only":
	  (((a>>2)&3)==2)?"Tap and non-locking drag":
	  (((a>>2)&3)==3)?"Tap and locking drag":"internal error");
  fprintf(info,"%s\tZ-threshold %d of 7\n",
	  ((a&3)==0)?"No edge motion.              ":
	  ((a&3)==1)?"Edge motion always.          ":
	  ((a&3)==3)?"Edge motion only during drag.":"internal error",
	  (unsigned)((a>>4)&7));
  fprintf(info, "%s\t\t%s sample rate.\n",
	  (c>>7)?"Absolute XYZ mode":"Relative mode    ",
	  ((c>>6)&1)?"High":"Low ");
  fprintf(info, "%s button mode\t\t\tCorner tap is %s button click\n",
	  ((c>>2)&1)?"3":"2",((c>>1)&1)?"middle":"right");
}

/*------------------------------------------------------------------------*/
/*   PS/2 Utility functions.                                              */
/*------------------------------------------------------------------------*/

/* write a byte to the ps/2 port, handling ACK */
void putbyte(int fd, byte b) {
  byte ack;
  write(fd, &b, 1);
  read(fd, &ack, 1);
  assert(ack==0xFA);
}
/* read a byte from the ps/2 port */
byte getbyte(int fd) {
  byte b;
  read(fd, &b, 1);
  return b;
}
/* use the Synaptics extended ps/2 syntax to write a special command byte */
void send_cmd(int fd, byte cmd) {
  int i;
  /* initialize with 'inert' command */
  putbyte(fd, 0xE6);
  for (i=0; i<4; i++) {
    putbyte(fd, 0xE8);
    putbyte(fd, (cmd>>6)&0x3);
    cmd<<=2;
  }
}
/* write 'cmd' to mode byte 1 */
void set_mode1(int fd, byte cmd) {
  send_cmd(fd, cmd);
  putbyte(fd, 0xF3);
  putbyte(fd, 0x0A);
}
/* write 'cmd' to mode byte 2 */
void set_mode2(int fd, byte cmd) {
  send_cmd(fd, cmd);
  putbyte(fd, 0xF3);
  putbyte(fd, 0x14);
}
/* read three byte status ('a','b','c') corresponding to register 'cmd' */
void status_rqst(int fd, byte cmd, byte *byte1, byte *byte2, byte *byte3) {
  byte chk;
  send_cmd(fd, cmd);
  putbyte(fd, 0xE9);
  *byte1=getbyte(fd);
  *byte2=getbyte(fd);
  *byte3=getbyte(fd);
}
byte get_mode1(int fd) {
  byte a,b,c;
  status_rqst(fd, 0x01, &a, &b, &c);
  return a;
}
byte get_mode2(int fd) {
  byte a,b,c;
  status_rqst(fd, 0x01, &a, &b, &c);
  return c;
}
/* initialize and determine whether this is a synaptics touchpad */
int is_touchpad(int fd) {
  byte a,b,c;
  status_rqst(fd, 0x00, &a, &b, &c);
  return (b==0x47);
}
