/*
    ras - Redundant Archive System
    Copyright (C) 1999  Nick Cleaton

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    Nick Cleaton <nick@cleaton.net>
*/
/*

utils.h

Misc utilities

*/


/*************************************************************************
**
**  Exiting (and ensuring that all output files are removed) if an error
**  occurs.
*/

enum raserr { INVALID_INPUT, NOT_ENOUGH, 
              EXTERNAL_FAILURE, INTERNAL_FAILURE
	    };

void exit_because_of_error(int errtype);
/*  Speaks for itself really, but see open_output_file below  */


FILE *open_output_file(char *name);
/*  Opens an output file (exits with an error message if the open fails) and
    adds the filename to a list so that it will be automatically unlinked if 
    exit_because_of_error is called.  This avoids leaving invalid or
    partial output files behind if something goes wrong.
*/

void init_unlink_list();
/*  Called once at the start of the prog, initialises the list of files to
    be unlinked on error to empty.
*/



/**************************************************************************
**
** Testing for error conditions
*/

void myassert(int cond, char *msg);
/*  exit with an error message if cond is 0  */



/**************************************************************************
**
**  Keeping track of how many segments there are in the archive, and giving
**  an error if 2 things that fix how many segments there must be disagree.
*/

int get_segment_count();  /*  -1 means not fixed yet  */

void register_segment_count(int count, char *source);
/*  To be called whenever something happens that dictates the number of
    segments in the archive.  Source should describe the source of the
    information (used for error messages if there's a disagreement)
*/




/**************************************************************************
**
**  String manipulation
*/

unsigned char fit_int_in_uchar(int i);
/*  i must be in the range 0 to 255. Exit with an error message if it
    won't fit.
*/

int countchar(char *str, char c);
/*  return the number of occurrences of c in str  */

char *popfirstword(char **s, char sep);
/*  Modify both the string pointed to by *s and the pointer *s itself, so 
    that *s is advanced one word (where words are a blocks of 0 or 
    more chars separated by occurrences of sep) in the string, and return
    a pointer to the word advanced over.  For all but the last word this
    entails writing a null over the separator, hence the need to modify
    the string pointed to by *s as well as *s itself.  Returns NULL if
    there are no chars left in *s.
*/



/**************************************************************************
**
**  Memory allocation
*/

void *Malloc(size_t bytes);
char *Strdup(char *s);
/*  malloc/strdup + exit with error message if out of memory  */

void *nofree(void *ptr);
/*  Declares that there will be no explicit free call for malloc'ed pointer
    ptr.  After the call to nofree ptr may no-longer be valid, the value 
    returned by nofree should be used instead.  For each malloc in the prog
    there should be either 1 free call or 1 nofree call.  Currently has no
    effect (i.e ras leeks memory), but one day it may store all the pointers
    somewhere for freeing later.
*/


