/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2000  Andrew Clausen, Lennert Buytenhek and Red Hat Inc.

	Andrew Clausen			<clausen@gnu.org>
	Lennert Buytenhek		<buytenh@gnu.org>
	Matt Wilson, Red Hat Inc.	<msw@redhat.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* It's a bit silly calling a swap partition a filesystem.  Oh well...  */

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) gettext (String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <unistd.h>
#include <asm/page.h>
#include <string.h>

#define SWAP_SPECIFIC(fs) ((SwapSpecific*) (fs->type_specific))
#define BUFFER_SIZE 32

typedef struct {
	char		page_map[1];
} SwapOldHeader;

/* ripped from mkswap */
typedef struct {
	char		bootbits[1024];    /* Space for disklabel etc. */
	unsigned int	version;
	unsigned int	last_page;
	unsigned int	nr_badpages;
	unsigned int	padding[125];
	unsigned int	badpages[1];
} SwapNewHeader;

typedef struct {
	union {
		SwapNewHeader	new;
		SwapOldHeader	old;
	}* header;
	char*		sig;		/* will point inside header */

	void*		buffer;
	int		buffer_size;

	PedSector	page_sectors;
	unsigned int	page_count;
	unsigned int	version;
	unsigned int	max_bad_pages;
} SwapSpecific;

static int swap_probe (const PedGeometry* geom);
static PedFileSystem* swap_open (const PedGeometry* geom);
static PedFileSystem* swap_create (const PedGeometry* geom);
static int swap_close (PedFileSystem* fs);
static int swap_resize (PedFileSystem* fs, PedGeometry* geom);
static int swap_copy (PedFileSystem* fs, PedGeometry* geom);
static int swap_check (PedFileSystem* fs);
static PedSector swap_get_min_size (const PedFileSystem* fs);
static char swap_get_system (const PedFileSystem* fs, const PedPartition* part,
			     PedDiskType* disk_type);

static PedFileSystemOps swap_ops = {
	probe:		swap_probe,
	open:		swap_open,
	create:		swap_create,
	close:		swap_close,
	check:		swap_check,
	copy:		swap_copy,
	resize:		swap_resize,
	get_min_size:	swap_get_min_size,
	get_system:	swap_get_system
};

static PedFileSystemType swap_type = {
	next:	NULL,
	ops:	&swap_ops,
	name:	"linux-swap"
};

void
ped_file_system_linux_swap_init ()
{
	ped_file_system_type_register (&swap_type);
}

void
ped_file_system_linux_swap_done ()
{
	ped_file_system_type_unregister (&swap_type);
}

static int
swap_probe (const PedGeometry* geom)
{
	PedFileSystem*	fs;
	fs = swap_open (geom);
	if (fs) {
		swap_close (fs);
		return 1;
	}
	return 0;
}

static PedFileSystem*
swap_alloc (const PedGeometry* geom)
{
	PedFileSystem*		fs;
	SwapSpecific*		fs_info;

	fs = (PedFileSystem*) ped_malloc (sizeof (PedFileSystem));
	if (!fs)
		goto error;

	fs->type_specific = (SwapSpecific*) ped_malloc (sizeof (SwapSpecific));
	if (!fs->type_specific)
		goto error_free_fs;

	fs_info = SWAP_SPECIFIC (fs);
	fs_info->header = ped_malloc (getpagesize());
	if (!fs_info->header)
		goto error_free_type_specific;

	fs_info = SWAP_SPECIFIC (fs);
	fs_info->buffer_size = getpagesize() * BUFFER_SIZE;
	fs_info->buffer = ped_malloc (fs_info->buffer_size);
	if (!fs_info->buffer)
		goto error_free_header;

	fs->geom = ped_geometry_duplicate (geom);
	if (!fs->geom)
		goto error_free_buffer;

	fs->type = &swap_type;

	return fs;

error_free_buffer:
	ped_free (fs_info->buffer);
error_free_header:
	ped_free (fs_info->header);
error_free_type_specific:
	ped_free (fs->type_specific);
error_free_fs:
	ped_free (fs);
error:
	return NULL;
}

static void
swap_free (PedFileSystem* fs)
{
	SwapSpecific*	fs_info = SWAP_SPECIFIC (fs);

	ped_free (fs_info->buffer);
	ped_free (fs_info->header);
	ped_free (fs->type_specific);

	ped_geometry_destroy (fs->geom);
	ped_free (fs);
}

static PedFileSystem*
swap_open (const PedGeometry* geom)
{
	PedFileSystem*		fs;
	SwapSpecific*		fs_info;

	fs = swap_alloc (geom);
	if (!fs)
		goto error;

	fs_info = SWAP_SPECIFIC (fs);
	fs_info->page_sectors = getpagesize () / 512;
	if (!ped_geometry_read (fs->geom, fs_info->header, 0,
				fs_info->page_sectors))
		goto error_free_fs;
	fs_info->sig = ((char*) fs_info->header) + getpagesize() - 10;

	if (strncmp (fs_info->sig, "SWAP-SPACE", 10) == 0) {
		fs_info->version = 0;
	} else if (strncmp (fs_info->sig, "SWAPSPACE2", 10) == 0) {
		fs_info->version = 1;
	} else {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Unrecognised linux swap signature %10s."),
			fs_info->sig);
		goto error_free_fs;
	}

	return fs;

error_free_fs:
	swap_free (fs);
error:
	return NULL;
}

static int
swap_new_find_bad_page (PedFileSystem* fs, unsigned int page)
{
	SwapSpecific*	fs_info = SWAP_SPECIFIC (fs);
	unsigned int	i;

	for (i=0; i < fs_info->header->new.nr_badpages; i++) {
		if (fs_info->header->new.badpages [i] == page)
			return i;
	}

	return 0;
}

static int
swap_new_remove_bad_page (PedFileSystem* fs, unsigned int page)
{
	SwapSpecific*	fs_info = SWAP_SPECIFIC (fs);
	unsigned int	pos;

	pos = swap_new_find_bad_page (fs, page);
	if (!pos)
		return 0;

	for (; pos < fs_info->header->new.nr_badpages; pos++) {
		fs_info->header->new.badpages [pos - 1]
			= fs_info->header->new.badpages [pos];
	}

	return 1;
}

static int
swap_mark_page (PedFileSystem* fs, unsigned int page, int ok)
{
	SwapSpecific*	fs_info = SWAP_SPECIFIC (fs);
	char*		ptr;
	unsigned int	mask;

	if (fs_info->version == 0) {
		ptr = &fs_info->header->old.page_map [page/8];
		mask = 1 << (page%8);
		*ptr = (*ptr & ~mask) + ok * mask;
	} else {
		if (ok) {
			if (swap_new_remove_bad_page (fs, page))
				fs_info->header->new.nr_badpages--;
		} else {
			if (swap_new_find_bad_page (fs, page))
				return 1;

			if (fs_info->header->new.nr_badpages
					> fs_info->max_bad_pages) {
				ped_exception_throw (PED_EXCEPTION_ERROR,
					PED_EXCEPTION_CANCEL,
					_("Too many bad pages."));
				return 0;
			}

			fs_info->header->new.badpages
				[fs_info->header->new.nr_badpages] = page;
			fs_info->header->new.nr_badpages++;
		}
	}

	return 1;
}

static void
swap_clear_pages (PedFileSystem* fs)
{
	SwapSpecific*	fs_info = SWAP_SPECIFIC (fs);
	unsigned int	i;

	for (i = 1; i < fs_info->page_count; i++) {
		swap_mark_page (fs, i, 1);
	}

	if (fs_info->version == 0) {
		for (; i < 1024; i++) {
			swap_mark_page (fs, i, 0);
		}
	}
}

static int
swap_check_pages (PedFileSystem* fs)
{
	SwapSpecific*	fs_info = SWAP_SPECIFIC (fs);
	PedSector	result;
	int		first_page = 1;
	int		stop_page = 0;
	int		last_page = fs_info->page_count - 1;

	swap_clear_pages (fs);
	while (first_page <= last_page) {
		result = ped_geometry_check (
				fs->geom,
				fs_info->buffer,
				fs_info->buffer_size / 512,
				first_page * fs_info->page_sectors,
				fs_info->page_sectors,
				(last_page - first_page + 1)
					* fs_info->page_sectors);
		if (!result)
			return 1;
		stop_page = result / fs_info->page_sectors;
		if (!swap_mark_page (fs, stop_page, 0))
			return 0;
		first_page = stop_page + 1;
	}
	return 1;
}

/* if rewrite is set, then no warning messages about compatibility are
 * thrown, and the boot-code for new Linux swap partitions is preserved
 * (for LILO and friends).
 */
static int
swap_write (PedFileSystem* fs, int rewrite, PedGeometry* old_geom)
{
	SwapSpecific*	fs_info = SWAP_SPECIFIC (fs);

	fs_info->page_sectors = getpagesize () / 512;
	memset (fs_info->header, 0, fs_info->page_sectors * 512);
	fs_info->sig = (char*) fs_info->header
			+ fs_info->page_sectors * 512 - 10;

	fs_info->page_count = fs->geom->length / fs_info->page_sectors;
	if (fs_info->page_count > 8 * (fs_info->page_sectors * 512 - 10) ) {
		if (rewrite) {
			if (!ped_geometry_read (old_geom, fs_info->header,
						0, 2))
				return 0;
		} else {
			ped_exception_throw (PED_EXCEPTION_WARNING,
				PED_EXCEPTION_OK_CANCEL,
				_("This swap partition will not be compatible "
				"with Linux version 2.1.117 or earlier.  Use a "
				"smaller partition (maximum size 128mb) if you "
				"want to use old versions of Linux."));
		}

		fs_info->version = 1;
		fs_info->header->new.version = 1;
		fs_info->header->new.last_page = fs_info->page_count - 1;
		fs_info->header->new.nr_badpages = 0;
		fs_info->max_bad_pages = (getpagesize()
						- sizeof (SwapNewHeader)) / 4;
		memcpy (fs_info->sig, "SWAPSPACE2", 10);
	} else {
		fs_info->version = 0;
		fs_info->max_bad_pages = fs_info->page_count;
		memcpy (fs_info->sig, "SWAP-SPACE", 10);
	}
	if (!swap_check_pages (fs))
		return 0;
	return ped_geometry_write (fs->geom, fs_info->header, 0,
				   fs_info->page_sectors);
}

static PedFileSystem*
swap_create (const PedGeometry* geom)
{
	PedFileSystem*		fs;

	fs = swap_alloc (geom);
	if (!fs)
		goto error;
	if (!swap_write (fs, 0, NULL))
		goto error_free_fs;
	return fs;

error_free_fs:
	swap_free (fs);
error:
	return NULL;
}

static int
swap_close (PedFileSystem* fs)
{
	swap_free (fs);
	return 1;
}

static int
swap_resize (PedFileSystem* fs, PedGeometry* geom)
{
	PedGeometry*	old_geom;
	
	old_geom = fs->geom;
	fs->geom = ped_geometry_duplicate (geom);
	if (!swap_write (fs, 0, old_geom))
		goto error;
	ped_geometry_destroy (old_geom);
	return 1;

error:
	ped_geometry_destroy (fs->geom);
	fs->geom = old_geom;
	return 0;
}

static int
swap_copy (PedFileSystem* fs, PedGeometry* geom)
{
	PedFileSystem*	new_fs;

	new_fs = swap_create (geom);
	if (!new_fs)
		goto error;
	swap_close (new_fs);
	return 1;

error:
	return 0;
}

static int
swap_check (PedFileSystem* fs)
{
	swap_write (fs, 1, fs->geom);
	return 1;
}

static PedSector
swap_get_min_size (const PedFileSystem* fs)
{
	return getpagesize() / 512;
}

static char
swap_get_system (const PedFileSystem* fs, const PedPartition* part,
		 PedDiskType* disk_type)
{
	if (part->hidden) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Linux-swap partitions can not be hidden."));
		return 0;
	}
	return PED_LINUX_SWAP;
}

