/*
    libparted
    Copyright (C) 1998-2000  Andrew Clausen  <clausen@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>

#include "fat.h"
#include "calc.h"

static PedFileSystemOps fat_ops = {
	probe:		fat_probe,
	open:		fat_open,
	create:		fat_create,
	close:		fat_close,
	check:		fat_check,
	resize:		fat_resize,
	copy:		fat_copy,
	get_min_size:	fat_get_min_size,
	get_system:	fat_get_system
};

static PedFileSystemType fat_type = {
	next:	NULL,
	ops:	&fat_ops,
	name:	"FAT"
};

void
ped_file_system_fat_init ()
{
	if (sizeof (FatBootSector) != 512)
		ped_exception_throw (PED_EXCEPTION_BUG, PED_EXCEPTION_CANCEL,
			_("GNU parted was miscompiled: the FAT boot sector "
			"should be 512 bytes.  FAT support will be disabled."));
	else
		ped_file_system_type_register (&fat_type);
}

void
ped_file_system_fat_done ()
{
	ped_file_system_type_unregister (&fat_type);
}

PedFileSystem*
fat_file_system_alloc (const PedGeometry* geom)
{
	PedFileSystem*		fs;

	fs = (PedFileSystem*) ped_malloc (sizeof (PedFileSystem));
	if (!fs)
		goto error;

	fs->type_specific = (FatSpecific*) ped_malloc (sizeof (FatSpecific));
	if (!fs->type_specific)
		goto error_free_fs;

	fs->geom = ped_geometry_duplicate (geom);
	if (!fs->geom)
		goto error_free_type_specific;
	
	return fs;

error_free_type_specific:
	ped_free (fs->type_specific);
error_free_fs:
	ped_free (fs);
error:
	return NULL;
}

/* Requires the boot sector to be analysed, and the fat to be loaded */
int
fat_file_system_alloc_buffers (PedFileSystem* fs)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);

	fs_info->buffer_size = BUFFER_SIZE;
        fs_info->buffer = ped_malloc (fs_info->buffer_size
					* fs_info->cluster_size);
        if (!fs_info->buffer)
		goto error;

	fs_info->fat_flag_map = ped_malloc (fs_info->fat->cluster_count + 2);
	if (!fs_info->fat_flag_map)
		goto error_free_buffer;

	return 1;

error_free_buffer:
	ped_free (fs_info->buffer);
error:
	return 0;
};

void
fat_file_system_free_buffers (PedFileSystem* fs)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);

	ped_free (fs_info->fat_flag_map);
	ped_free (fs_info->buffer);
}

void
fat_file_system_free (PedFileSystem* fs)
{
	ped_geometry_destroy (fs->geom);
	ped_free (fs->type_specific);
	ped_free (fs);
}

int
fat_probe (const PedGeometry* geom)
{
	PedFileSystem*		fs;
	FatSpecific*		fs_info;

	fs = fat_file_system_alloc ((PedGeometry*) geom);
	if (!fs)
		goto error;
	fs_info = (FatSpecific*) fs->type_specific;

	fs->type = &fat_type;

	if (!fat_boot_sector_read (&fs_info->boot_sector, fs->geom))
		goto error_free_fs;

	fat_file_system_free (fs);
	return 1;

error_free_fs:
	fat_file_system_free (fs);
error:
	return 0;
}

static int
fat_file_system_init_fats (PedFileSystem* fs)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);
	int		i;
	FatCluster	table_size;
	FatTable*	fat_copy;

	table_size = fs_info->fat_sectors * 512
		     / fat_table_entry_size (fs_info->fat_type);
	fs_info->fat = fat_table_new (fs_info->fat_type, table_size);
	if (!fs_info->fat) goto error;

	fat_copy = fat_table_new (fs_info->fat_type, table_size);
	if (!fat_copy) goto error_free_fat;

	if (!fat_table_read (fs_info->fat, fs, 0))
		goto error_free_copy;

	for (i = 1; i < fs_info->fat_table_count; i++) {
		if (!fat_table_read (fat_copy, fs, i))
			goto error_free_copy;
		if (!fat_table_compare (fs_info->fat, fat_copy)) {
			if (ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("The FATs don't match.  If you don't know "
				"what this means, then select cancel, run "
				"scandisk on the file system, and then come "
				"back."))
			    == PED_EXCEPTION_CANCEL)
				goto error_free_copy;
		}
	}

	fat_table_destroy (fat_copy);
	return 1;

error_free_copy:
	fat_table_destroy (fat_copy);
error_free_fat:
	ped_free (fs_info->fat);
error:
	return 0;
}

/*
	Note: the FAT tables must be initialised before all of the buffers
	are allocated, because buffer sizes depend on FAT statistics
	(ie how many clusters)
*/
PedFileSystem*
fat_open (const PedGeometry* geom)
{
	PedFileSystem*		fs;
	FatSpecific*		fs_info;

	fs = fat_file_system_alloc (geom);
	if (!fs)
		goto error;
	fs_info = (FatSpecific*) fs->type_specific;

	fs->type = &fat_type;

	if (!fat_boot_sector_read (&fs_info->boot_sector, geom))
		goto error_free_fs;
	if (!fat_boot_sector_analyse (&fs_info->boot_sector, fs))
		goto error_free_fs;
	if (fs_info->fat_type == FAT_TYPE_FAT32) {
		if (!fat_info_sector_read (&fs_info->info_sector, fs))
			goto error_free_fs;
	}

	if (!fat_file_system_init_fats (fs))
		goto error_free_fs;
	if (!fat_file_system_alloc_buffers (fs)) 
		goto error_free_fat_table;
	if (!fat_flag_clusters (fs))
		goto error_free_buffers;

	return fs;

error_free_buffers:
	fat_file_system_free_buffers (fs);
error_free_fat_table:
	fat_table_destroy (fs_info->fat);
error_free_fs:
	fat_file_system_free (fs);
error:
	return NULL;
}

static int
fat_root_dir_clear (PedFileSystem* fs)
{
	FatSpecific*		fs_info = FAT_SPECIFIC (fs);
	memset (fs_info->buffer, 0, 512 * fs_info->root_dir_sector_count);
	return ped_geometry_write (fs->geom, fs_info->buffer,
				   fs_info->root_dir_offset,
				   fs_info->root_dir_sector_count);
}

PedFileSystem*
fat_create (const PedGeometry* geom)
{
	PedFileSystem*		fs;
	FatSpecific*		fs_info;

	fs = fat_file_system_alloc (geom);
	if (!fs)
		goto error;
	fs_info = (FatSpecific*) fs->type_specific;

	fs->type = &fat_type;

	fs_info->logical_sector_size = 1;
	fs_info->sector_count = fs->geom->length;
	fs_info->fat_table_count = 2;
/* some initial values, to be changed later */
	fs_info->root_dir_sector_count = FAT_ROOT_DIR_ENTRY_COUNT
					  / (512 / sizeof (FatDirEntry));
	fs_info->root_dir_entry_count = FAT_ROOT_DIR_ENTRY_COUNT;

	fs_info->fat_type = FAT_TYPE_FAT16;
	if (!fat_calc_sizes (fs->geom, 0,
			FAT_TYPE_FAT16,
			fs_info->root_dir_sector_count,
			&fs_info->cluster_size,
			&fs_info->cluster_count,
			&fs_info->fat_sectors)) {
		if (!fat_calc_sizes (fs->geom, 0,
				     FAT_TYPE_FAT32,
				     0,
				     &fs_info->cluster_size,
				     &fs_info->cluster_count,
				     &fs_info->fat_sectors)) {
			ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Partition too small for a FAT file system"));
			goto error_free_fs;
		}
		if (ped_exception_throw (PED_EXCEPTION_WARNING,
			PED_EXCEPTION_OK_CANCEL,
			_("The filesystem is going to be too big for FAT16, so "
			  "FAT32 will be used.  This is not compatible with "
			  "MS-DOS, early versions of MS-Windows 95 and "
			  "Windows NT.  If you use these operating systems, "
			  "then select cancel, and create a smaller "
			  "partition.  If you only use Linux, BSD, MS "
			  "Windows 98 and/or MS Windows 95 B, then select OK."))
				== PED_EXCEPTION_CANCEL)
			goto error_free_fs;
		if (!fat_calc_sizes (fs->geom, 0,
				     FAT_TYPE_FAT32,
				     0,
				     &fs_info->cluster_size,
				     &fs_info->cluster_count,
				     &fs_info->fat_sectors)) {
			ped_exception_throw (PED_EXCEPTION_BUG,
				PED_EXCEPTION_CANCEL,
				_("Weird: fat_calc_sizes() failed "
				"for FAT32!"));
			goto error_free_fs;
		}
		fs_info->fat_type = FAT_TYPE_FAT32;
	} else if (fs_info->cluster_size > 8192) {
		switch (ped_exception_throw (PED_EXCEPTION_WARNING,
			PED_EXCEPTION_YES_NO_CANCEL,
			_("Would you like to use FAT32 for this filesystem?  "
			"It is much more efficient with your disk space, but "
			"is not compatible with early versions of Windows 95 "
			"and Windows NT.  Only select yes if you only "
			"use Linux, BSD, MS Windows 98 and/or MS Windows 95 "
			"B."))) {
		case PED_EXCEPTION_CANCEL:
			goto error_free_fs;

		case PED_EXCEPTION_YES:
			if (!fat_calc_sizes (fs->geom, 0,
					     FAT_TYPE_FAT32,
					     0,
					     &fs_info->cluster_size,
					     &fs_info->cluster_count,
					     &fs_info->fat_sectors)) {
				ped_exception_throw (PED_EXCEPTION_BUG,
					PED_EXCEPTION_CANCEL,
					_("Weird: fat_calc_sizes() failed "
					"for FAT32!"));
				goto error_free_fs;
			}
			fs_info->fat_type = FAT_TYPE_FAT32;
			break;

		case PED_EXCEPTION_NO:
		default:
			break;
		}
	}

	fs_info->cluster_sectors = fs_info->cluster_size / 512;

	fs_info->fat_offset = fat_min_reserved_sector_count (fs_info->fat_type);
	fs_info->dir_entries_per_cluster
		= fs_info->cluster_size / sizeof (FatDirEntry);

	if (fs_info->fat_type == FAT_TYPE_FAT16) {
		/* FAT16 */
		if (fs_info->cluster_count
			> fat_max_cluster_count (fs_info->fat_type)) {
			fs_info->cluster_count
				= fat_max_cluster_count (fs_info->fat_type);
		}

		fs_info->root_dir_sector_count
			= FAT_ROOT_DIR_ENTRY_COUNT
				/ (512 / sizeof (FatDirEntry));
		fs_info->root_dir_entry_count = FAT_ROOT_DIR_ENTRY_COUNT;
                fs_info->root_dir_offset
			= fs_info->fat_offset
			+ fs_info->fat_sectors * fs_info->fat_table_count;
		fs_info->cluster_offset
			= fs_info->root_dir_offset
			  + fs_info->root_dir_sector_count
			  - 2 * fs_info->cluster_sectors;
	} else {
		/* FAT32 */
		fs_info->info_sector_offset = 1;
		fs_info->boot_sector_backup_offset = 6;

		fs_info->root_cluster = 2;
		fs_info->root_dir_sector_count = 0;
		fs_info->root_dir_entry_count = 0;
		fs_info->root_dir_offset = 0;

		fs_info->cluster_offset
			= fs_info->fat_offset
			  + fs_info->fat_sectors * fs_info->fat_table_count
			  - 2 * fs_info->cluster_sectors;
	}

	fs_info->fat = fat_table_new (fs_info->fat_type,
				      fs_info->cluster_count + 2);
	if (!fs_info->fat)
		goto error_free_fs;
	if (!fat_file_system_alloc_buffers (fs)) 
		goto error_free_fat_table;

	if (fs_info->fat_type == FAT_TYPE_FAT32) {
		fs_info->root_cluster
			= fat_table_alloc_cluster (fs_info->fat);
		fat_table_set_eof (fs_info->fat, fs_info->root_cluster);
		memset (fs_info->buffer, 0, fs_info->cluster_size);
		if (!fat_write_cluster (fs, fs_info->buffer,
					fs_info->root_cluster))
			return 0;
	}

	if (!fat_boot_sector_set_boot_code (&fs_info->boot_sector))
		goto error_free_buffers;
	if (!fat_boot_sector_generate (&fs_info->boot_sector, fs))
		goto error_free_buffers;
	if (!fat_boot_sector_write (&fs_info->boot_sector, fs))
		goto error_free_buffers;
	if (fs_info->fat_type == FAT_TYPE_FAT32) {
		if (!fat_info_sector_generate (&fs_info->info_sector, fs))
			goto error_free_buffers;
		if (!fat_info_sector_write (&fs_info->info_sector, fs))
			goto error_free_buffers;
	}

	if (!fat_table_write_all (fs_info->fat, fs))
		goto error_free_buffers;

	if (fs_info->fat_type == FAT_TYPE_FAT16) {
		if (!fat_root_dir_clear (fs))
			goto error_free_buffers;
	}

#ifdef VERBOSE
	fat_print (fs);
#endif

	return fs;

error_free_buffers:
	fat_file_system_free_buffers (fs);
error_free_fat_table:
	fat_table_destroy (fs_info->fat);
error_free_fs:
	fat_file_system_free (fs);
error:
	return NULL;
}

int
fat_close (PedFileSystem* fs)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);

	fat_file_system_free_buffers (fs);
	fat_table_destroy (fs_info->fat);
	fat_file_system_free (fs);
	return 1;
}

/* Hack: just resize the filesystem outside of it's boundaries! */
int
fat_copy (PedFileSystem* fs, PedGeometry* geom)
{
	PedFileSystem*		new_fs;

	new_fs = fat_open (fs->geom);
	if (!new_fs)
		goto error;
	if (!fat_resize (new_fs, geom))
		goto error_close_new_fs;
	fat_close (new_fs);
	return 1;

error_close_new_fs:
	fat_close (new_fs);
error:
	return 0;
}

int
fat_check (PedFileSystem* fs)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);
	int		cluster_size;
	FatCluster	cluster_count;
	PedSector	fat_sectors;
	FatType		fat_type;
	PedSector	align_sectors;
	FatCluster	info_free_clusters;

#ifdef VERBOSE
	fat_print (fs);
#endif

	align_sectors = fs_info->fat_offset
			- fat_min_reserved_sector_count (fs_info->fat_type);

	if (!fat_calc_sizes (fs->geom, align_sectors,
			     fs_info->fat_type,
			     fs_info->root_dir_sector_count,
			     &cluster_size,
			     &cluster_count,
			     &fat_sectors)) {
		if (ped_exception_throw (PED_EXCEPTION_BUG,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("There are no possible configurations for this FAT "
			  "type."))
				!= PED_EXCEPTION_IGNORE)
			return 0;
	}

	if (fs_info->fat_type == FAT_TYPE_FAT16) {
		if (cluster_size != fs_info->cluster_size
		    || cluster_count != fs_info->cluster_count
		    || fat_sectors != fs_info->fat_sectors) {
			if (ped_exception_throw (PED_EXCEPTION_WARNING,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("File system doesn't have expected sizes for "
				"Windows to like it.  "
				"Cluster size is %dk (%dk expected); "
				"number of clusters is %d (%d expected); "
				"size of FATs is %d sectors (%d expected)."),
				fs_info->cluster_size / 1024,
					cluster_size / 1024,
				(int) fs_info->cluster_count,
					(int) cluster_count,
				(int) fs_info->fat_sectors,
					(int) fat_sectors)
					!= PED_EXCEPTION_IGNORE)
				return 0;
		}
	} else {
		if (!fat_calc_resize_sizes (fs->geom, align_sectors,
				     fs_info->cluster_size,
				     fs_info->root_dir_sector_count,
				     &cluster_count,
				     &fat_sectors,
				     &fat_type)) {
			if (ped_exception_throw (PED_EXCEPTION_BUG,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("There are no possible configurations for "
				"this cluster size."))
					!= PED_EXCEPTION_IGNORE)
				return 0;
		}

		if (cluster_count != fs_info->cluster_count
		    || fat_sectors != fs_info->fat_sectors) {
			if (ped_exception_throw (PED_EXCEPTION_WARNING,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("File system doesn't have expected sizes for "
				"Windows to like it.  "
				"Number of clusters is %d (%d expected); "
				"size of FATs is %d sectors (%d expected)."),
				(int) fs_info->cluster_count,
					(int) cluster_count,
				(int) fs_info->fat_sectors,
					(int) fat_sectors)
					!= PED_EXCEPTION_IGNORE)
				return 0;
		}
	}

	if (fs_info->fat_type == FAT_TYPE_FAT32) {
		info_free_clusters
			= PED_LE32_TO_CPU (fs_info->info_sector.free_clusters);
		if (info_free_clusters != fs_info->fat->free_cluster_count) {
			if (ped_exception_throw (PED_EXCEPTION_WARNING,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("File system is reporting the free space as "
				"%d clusters, not %d clusters."),
				info_free_clusters,
				fs_info->fat->free_cluster_count)
					!= PED_EXCEPTION_IGNORE)
				return 0;
		}
	}

	return 1;	/* existence of fs implies consistency ;-) */
}

PedSector
fat_get_min_size (const PedFileSystem* fs)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);
	FatCluster	min_cluster_count;

	min_cluster_count = fs_info->fat->cluster_count
			    - fs_info->fat->free_cluster_count
			    + fs_info->total_dir_clusters
			    + 32;

	return fs_info->cluster_offset
	       + fs_info->cluster_sectors * min_cluster_count;
}

char
fat_get_system (const PedFileSystem* fs, const PedPartition* part,
		PedDiskType* disk_type)
{
	PedDevice*	dev;
	int		is_lba;

	dev = part->geom.disk->dev;
	is_lba = (part->geom.end >= 1024 * dev->heads * dev->sectors);

	if (strcasecmp (disk_type->name, "msdos")) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_BUG,
			_("FAT filesystems don't have an id for non-msdos "
			"partition tables"));
		return 0;
	}

	if (!fs) {
		if (part->hidden) {
			if (is_lba)
				return PED_WIN95_FAT16_LBA;
			else
				return PED_DOS_FAT16_BIG;
		} else {
			if (is_lba)
				return PED_WIN95_FAT16_LBA;
			else
				return PED_DOS_FAT16_BIG;
		}
	}

	if (FAT_SPECIFIC (fs)->fat_type == FAT_TYPE_FAT32) {
		if (part->hidden) {
			if (is_lba)
				return PED_WIN95_HIDDEN_FAT32_LBA;
			else
				return PED_WIN95_HIDDEN_FAT32;
		} else {
			if (is_lba)
				return PED_WIN95_FAT32_LBA;
			else
				return PED_WIN95_FAT32;
		}
	} else {
		if (part->hidden) {
			if (is_lba)
				return PED_WIN95_HIDDEN_FAT16_LBA;
			else
				return PED_DOS_HIDDEN_FAT16_BIG;
		} else {
			if (is_lba)
				return PED_WIN95_FAT16_LBA;
			else
				return PED_DOS_FAT16_BIG;
		}
	}
}

void
fat_print (const PedFileSystem* fs)
{
	FatSpecific*		fs_info = FAT_SPECIFIC (fs);

	printf ("%-8d     clusters\n", (int) fs_info->cluster_count);
	printf ("%-8d     bytes per cluster\n",
		(int) fs_info->cluster_sectors * 512);
	printf ("%-8d     sectors per FAT\n", (int) fs_info->fat_sectors);
	printf ("%-8d     free clusters\n",
		(int) fs_info->fat->free_cluster_count);
	printf ("%-8d     sectors available for clusters and FATs\n",
		(int) (fs->geom->length - fs_info->fat_offset));
}
