/*
    ext2.c -- generic ext2 stuff
    Copyright (C) 1998,99 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_c[] = "$Id: ext2.c,v 1.54 1999/10/08 06:20:16 buytenh Exp $";

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "ext2.h"

/* ext2 stuff ****************************************************************/

unsigned char _bitmap[8] = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80};

int ext2_copy_block(struct ext2_fs *fs, blk_t from, blk_t to)
{
	unsigned char buf[fs->blocksize];

	if (!ext2_bcache_flush(fs, from)) return 0;
	if (!ext2_bcache_flush(fs, to)) return 0;

	if (!ext2_read_blocks(fs, buf, from, 1)) return 0;
	if (!ext2_write_blocks(fs, buf, to, 1)) return 0;

	return 1;
}

int ext2_get_block_state(struct ext2_fs *fs, blk_t block)
{
	struct ext2_buffer_head *bh;
	int group;
	int offset;
	int state;

	block -= fs->sb.s_first_data_block;
	group = block / fs->sb.s_blocks_per_group;
	offset = block % fs->sb.s_blocks_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);
	state = bh->data[offset>>3] & _bitmap[offset&7];
	ext2_brelse(bh, 0);

	return state;
}

blk_t ext2_find_free_block(struct ext2_fs *fs)
{
	int i;

	for (i=0;i<fs->numgroups;i++)
		if (fs->gd[i].bg_free_blocks_count)
		{
			blk_t j;
			blk_t offset;

			offset = i * fs->sb.s_blocks_per_group + fs->sb.s_first_data_block;
			for (j=fs->adminblocks;j<fs->sb.s_blocks_per_group;j++)
				if (ext2_is_data_block(fs, offset + j) &&
				    !ext2_get_block_state(fs, offset + j))
					return offset + j;

			ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Inconsistent group descriptors!"));
		}

	ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			     _("Filesystem full!"));
	return 0;
}

ino_t ext2_find_free_inode(struct ext2_fs *fs)
{
	int i;

	for (i=0;i<fs->numgroups;i++)
		if (fs->gd[i].bg_free_inodes_count)
		{
			ino_t j;
			ino_t offset;

			offset = i * fs->sb.s_inodes_per_group + 1;
			for (j=0;j<fs->sb.s_inodes_per_group;j++)
				if (!ext2_get_inode_state(fs, offset + j))
					return offset + j;

			ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Inconsistent group descriptors!"));
		}

	ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			     _("Filesystem full!"));
	return 0;
}

int ext2_move_blocks(struct ext2_fs *fs, blk_t src, blk_t num, blk_t dest)
{
	unsigned char *buf;
	blk_t i;

	ped_exception_fetch_all();
	if ((buf = ped_malloc(num << fs->logsize)) != NULL)
	{
		ped_exception_leave_all();

		if (!ext2_bcache_flush_range(fs, src, num)) return 0;
		if (!ext2_bcache_flush_range(fs, dest, num)) return 0;

		if (!ext2_read_blocks(fs, buf, src, num)) return 0;
		if (!ext2_write_blocks(fs, buf, dest, num)) return 0;

		ped_free(buf);
		return 1;
	}
	ped_exception_catch();
	ped_exception_leave_all();

	if (src > dest)
	{
		for (i=0;i<num;i++)
			if (!ext2_copy_block(fs, src+i, dest+i))
				return 0;
	}
	else
	{
		for (i=num;i>0;i--)
			if (!ext2_copy_block(fs, src+i, dest+i))
				return 0;
	}
	return 1;
}

int ext2_read_blocks(struct ext2_fs *fs, void *ptr, blk_t block, blk_t num)
{
	return fs->devhandle->ops->read(fs->devhandle->cookie, ptr, block, num);
}

int ext2_set_block_state(struct ext2_fs *fs, blk_t block, int state, int updatemetadata)
{
	struct ext2_buffer_head *bh;
	int                      group;
	int                      offset;

	block -= fs->sb.s_first_data_block;
	group = block/fs->sb.s_blocks_per_group;
	offset = block%fs->sb.s_blocks_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);
	bh->dirty = 1;
	if (state)
		bh->data[offset>>3] |= _bitmap[offset&7];
	else
		bh->data[offset>>3] &= ~_bitmap[offset&7];
	ext2_brelse(bh, 0);

	if (updatemetadata)
	{
		int diff;

		diff = state ? -1 : 1;

		fs->gd[group].bg_free_blocks_count += diff;
		fs->sb.s_free_blocks_count += diff;
		fs->metadirty |= EXT2_META_SB | EXT2_META_GD;
	}
	return 1;
}

int ext2_write_blocks(struct ext2_fs *fs, void *ptr, blk_t block, blk_t num)
{
	return fs->devhandle->ops->write(fs->devhandle->cookie, ptr, block, num);
}

int ext2_zero_blocks(struct ext2_fs *fs, blk_t block, blk_t num)
{
	unsigned char *buf;
	blk_t i;

	ped_exception_fetch_all();
	buf = ped_malloc (num << fs->logsize);
	if (buf)
	{
		ped_exception_leave_all();

		memset(buf, 0, num << fs->logsize);
		if (!ext2_bcache_flush_range(fs, block, num))
			goto error_free_buf;
		if (!ext2_write_blocks(fs, buf, block, num))
			goto error_free_buf;
		ped_free(buf);
		return 1;
	}
	ped_exception_catch();

	buf = ped_malloc (fs->blocksize);
	if (buf)
	{
		ped_exception_leave_all();

		memset(buf, 0, fs->blocksize);

		for (i=0;i<num;i++)
		{
			if (!ext2_bcache_flush(fs, block+i))
				goto error_free_buf;
			if (!ext2_write_blocks(fs, buf, block+i, 1))
				goto error_free_buf;
		}

		ped_free(buf);
		return 1;
	}
	ped_exception_catch();
	ped_exception_leave_all();

	for (i=0;i<num;i++)
	{
		struct ext2_buffer_head *bh;

		bh = ext2_bcreate(fs, block+i);
		if (!bh)
			goto error;
		bh->dirty = 1;
		if (!ext2_brelse(bh, 1))
			goto error;
	}
	return 1;

error_free_buf:
	ped_free(buf);
error:
	return 0;
}





















off_t ext2_get_inode_offset(struct ext2_fs *fs, ino_t inode, blk_t *block)
{
	int group;
	int offset;

	inode--;

	group = inode / fs->sb.s_inodes_per_group;
	offset = (inode % fs->sb.s_inodes_per_group) * sizeof(struct ext2_inode);

	*block = fs->gd[group].bg_inode_table + (offset >> fs->logsize);

	return offset & (fs->blocksize - 1);
}

int ext2_get_inode_state(struct ext2_fs *fs, ino_t inode)
{
	struct ext2_buffer_head *bh;
	int                      group;
	int                      offset;
	int                      ret;

	inode--;
	group = inode / fs->sb.s_inodes_per_group;
	offset = inode % fs->sb.s_inodes_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_inode_bitmap);
	ret = bh->data[offset>>3] & _bitmap[offset&7];
	ext2_brelse(bh, 0);

	return ret;
}

int ext2_read_inode(struct ext2_fs *fs, ino_t inode, struct ext2_inode *data)
{
	struct ext2_buffer_head *bh;
	blk_t			 blk;
	off_t			 off;

	off = ext2_get_inode_offset(fs, inode, &blk);

	bh = ext2_bread(fs, blk);
	if (!bh)
		return 0;

	memcpy(data, bh->data + off, sizeof(struct ext2_inode));
	ext2_brelse(bh, 0);
	return 1;
}

int ext2_set_inode_state(struct ext2_fs *fs, ino_t inode, int state, int updatemetadata)
{
	struct ext2_buffer_head *bh;
	int                      group;
	int                      offset;

	inode--;
	group = inode / fs->sb.s_inodes_per_group;
	offset = inode % fs->sb.s_inodes_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_inode_bitmap);
	if (!bh)
		return 0;
	bh->dirty = 1;
	if (state)
		bh->data[offset>>3] |= _bitmap[offset&7];
	else
		bh->data[offset>>3] &= ~_bitmap[offset&7];
	ext2_brelse(bh, 0);

	if (updatemetadata)
	{
		int diff;

		diff = state ? -1 : 1;

		fs->gd[group].bg_free_inodes_count += diff;
		fs->sb.s_free_inodes_count += diff;
		fs->metadirty = EXT2_META_SB | EXT2_META_GD;
	}
	return 1;
}

int ext2_do_inode(struct ext2_fs *fs, struct ext2_inode *inode, blk_t block,
		   int action)
{
	struct ext2_buffer_head *bh;
	__u32			*udata;
	int			 count = 0;
	int			 i;
	int			 u32perblock = fs->blocksize >> 2;
	int			 i512perblock = 1 << (fs->logsize - 9);

	if (block == 0 || inode->i_mode == 0)
		return -1;

	if (fs->opt_debug)
		switch (action)
		{
			case EXT2_ACTION_ADD:
				fprintf(stderr,"adding 0x%04x to inode\n",block);
				break;
			case EXT2_ACTION_DELETE:
				fprintf(stderr,"deleting 0x%04x from inode\n",
					block);
				break;
			case EXT2_ACTION_FIND:
				fprintf(stderr,"finding 0x%04x in inode\n",block);
				break;
		}

	/* Direct blocks for first 12 blocks */
	for (i = 0; i < EXT2_NDIR_BLOCKS; i++)
	{
		if (action == EXT2_ACTION_ADD && !inode->i_block[i])
		{
			inode->i_block[i] = block;
			/* i_blocks is in 512 byte blocks */
			inode->i_blocks += i512perblock;
			inode->i_size += fs->blocksize;
			inode->i_mtime = time(NULL);
			ext2_set_block_state(fs, block, 1, 1);
			return i;
		}
		if (inode->i_block[i] == block)
		{
			if (action == EXT2_ACTION_DELETE)
			{
				inode->i_block[i] = 0;
				inode->i_blocks -= i512perblock;
				inode->i_size -= fs->blocksize;
				inode->i_mtime = time(NULL);
				ext2_set_block_state(fs, block, 0, 1);
			}
			return i;
		}
		if (inode->i_block[i])
			count += i512perblock;
	}

	count += inode->i_block[EXT2_IND_BLOCK] ? i512perblock : 0;
	count += inode->i_block[EXT2_DIND_BLOCK] ? i512perblock : 0;
	count += inode->i_block[EXT2_TIND_BLOCK] ? i512perblock : 0;

	if (!inode->i_block[EXT2_IND_BLOCK] ||
	    (count >= inode->i_blocks && action != EXT2_ACTION_ADD))
		return -1;

	bh = ext2_bread(fs, inode->i_block[EXT2_IND_BLOCK]);
	udata = (__u32 *)bh->data;

	/* Indirect blocks for next 256/512/1024 blocks (for 1k/2k/4k blocks) */
	for (i = 0; i < u32perblock; i++) {
		if (action == EXT2_ACTION_ADD && !udata[i]) {
			bh->dirty = 1;
			udata[i] = block;
			inode->i_blocks += i512perblock;
			inode->i_size += fs->blocksize;
			inode->i_mtime = time(NULL);
			ext2_set_block_state(fs, block, 1, 1);
			ext2_brelse(bh, 0);
			return EXT2_NDIR_BLOCKS + i;
		}
		if (udata[i] == block) {
			if (action == EXT2_ACTION_DELETE) {
				bh->dirty = 1;
				udata[i] = 0;
				inode->i_blocks -= i512perblock;
				inode->i_size -= fs->blocksize;
				inode->i_mtime = time(NULL);
				ext2_set_block_state(fs, block, 0, 1);
			}
			ext2_brelse(bh, 0);
			return EXT2_NDIR_BLOCKS + i;
		}
		if (udata[i])
		{
			count += i512perblock;
			if (count >= inode->i_blocks &&
			    action != EXT2_ACTION_ADD)
				return -1;
		}
	}

	ext2_brelse(bh, 0);

	if (!inode->i_block[EXT2_DIND_BLOCK] ||
	    (count >= inode->i_blocks && action != EXT2_ACTION_ADD))
		return -1;
	bh = ext2_bread(fs, inode->i_block[EXT2_DIND_BLOCK]);
	udata = (__u32 *)bh->data;

	/* Double indirect blocks for next 2^16/2^18/2^20 1k/2k/4k blocks */
	for (i = 0; i < u32perblock; i++) {
		struct ext2_buffer_head	*bh2;
		__u32			*udata2;
		int			 j;

		if (!udata[i]) {
			ext2_brelse(bh, 0);
			return -1;
		}
		bh2 = ext2_bread(fs, udata[i]);
		udata2 = (__u32 *)bh2->data;
		count += i512perblock;

		for (j = 0; j < u32perblock; j++) {
			if (action == EXT2_ACTION_ADD && !udata2[j]) {
				bh2->dirty = 1;
				udata2[j] = block;
				inode->i_blocks += i512perblock;
				inode->i_size += fs->blocksize;
				inode->i_mtime = time(NULL);
				ext2_set_block_state(fs, block, 1, 1);
				ext2_brelse(bh, 0);
				ext2_brelse(bh2, 0);
				return EXT2_NDIR_BLOCKS + i * u32perblock + j;
			}
			if (udata2[j] == block) {
				if (action == EXT2_ACTION_DELETE) {
					bh2->dirty = 1;
					udata2[j] = 0;
					inode->i_blocks -= i512perblock;
					inode->i_size -= fs->blocksize;
					inode->i_mtime = time(NULL);
					ext2_set_block_state(fs, block, 0, 1);
				}
				ext2_brelse(bh, 0);
				ext2_brelse(bh2, 0);
				return EXT2_NDIR_BLOCKS + i * u32perblock + j;
			}
			if (udata2[j])
			{
				count += i512perblock;
				if (count >= inode->i_blocks &&
				    action != EXT2_ACTION_ADD)
					return -1;
			}
		}
		ext2_brelse(bh2, 0);
	}
	ext2_brelse(bh, 0);

	/* FIXME: we should check for triple-indirect blocks here, but it
	 * would be nice to have a better routine to traverse blocks, and
	 * filesystems that need triple-indirect blocks for the resize
	 * inode are too big to worry about yet.
	 */

	return -1;
}

int ext2_write_inode(struct ext2_fs *fs, ino_t inode, const struct ext2_inode *data)
{
	struct ext2_buffer_head *bh;
	blk_t			 blk;
	off_t			 off;

	off = ext2_get_inode_offset(fs, inode, &blk);

	bh = ext2_bread(fs, blk);
	if (!bh)
		return 0;
	bh->dirty = 1;
	memcpy(bh->data + off, data, sizeof(struct ext2_inode));
	ext2_brelse(bh, 0);

	return 1;
}

int ext2_zero_inode(struct ext2_fs *fs, ino_t inode)
{
	struct ext2_inode buf;

	memset(&buf, 0, sizeof(struct ext2_inode));
	return ext2_write_inode(fs, inode, &buf);
}





/* formula grabbed from linux ext2 kernel source */
static int is_root(int x, int y)
{
	if (!x)
		return 1;

	while (1)
	{
		if (x == 1)
			return 1;

		if (x % y)
			return 0;

		x /= y;
	}
}

int ext2_is_group_sparse(struct ext2_fs *fs, int group)
{
	if (!fs->sparse)
		return 1;

	if (is_root(group, 3) || is_root(group, 5) || is_root(group, 7))
		return 1;

	return 0;
}









void ext2_close(struct ext2_fs *fs)
{
#if 0
	ext2_journal_deinit(fs);
#endif

	ext2_commit_metadata(fs, EXT2_META_PRIMARY | EXT2_META_BACKUP);
	ext2_sync(fs);

	ext2_bcache_deinit(fs);

	fs->devhandle->ops->close(fs->devhandle->cookie);

	ped_free(fs->gd);
	ped_free(fs);
}

int ext2_commit_metadata(struct ext2_fs *fs, int copies)
{
	int		i;
	int		num;
	int		wmeta = fs->metadirty & copies;
	unsigned char	sb[fs->blocksize];
	int		sb_block;

	/* See if there is even anything to write... */
	if (wmeta == EXT2_META_CLEAN)
		return 1;

	fs->sb.s_r_blocks_count = (fs->r_frac * (loff_t)fs->sb.s_blocks_count)
				  / 100;

	memset(sb, 0, fs->blocksize);

	if (fs->sb.s_first_data_block)
		memcpy(sb, &fs->sb, 1024);
	else
		memcpy(sb+1024, &fs->sb, 1024);

	num = copies & EXT2_META_BACKUP ? fs->numgroups : 1;

	for (i = 0, sb_block = fs->sb.s_first_data_block; i < num;
	     i++, sb_block += fs->sb.s_blocks_per_group)
	{

		if (!ext2_is_group_sparse(fs, i))
			continue;

		if ((i == 0 && wmeta & EXT2_META_PRIMARY_SB) ||
		    (i != 0 && wmeta & EXT2_META_SB))
		{
			if (!ext2_bcache_flush_range(fs, sb_block, 1))
				return 0;
			if (!ext2_write_blocks(fs, sb, sb_block, 1))
				return 0;
		}
		if ((i == 0 && wmeta & EXT2_META_PRIMARY_GD) ||
		    (i != 0 && wmeta & EXT2_META_GD))
		{
			if (!ext2_bcache_flush_range(fs, sb_block + 1,
						     fs->gdblocks))
				return 0;
			if (!ext2_write_blocks(fs, fs->gd, sb_block + 1,
					       fs->gdblocks))
				return 0;
		}
	}

	/* Clear the flags of the components we just finished writing. */
	fs->metadirty &= ~copies;

	return 1;
}

static int ext2_determine_itoffset(struct ext2_fs *fs)
{
	int i;

	fs->itoffset = fs->gd[0].bg_inode_table - fs->sb.s_first_data_block;

	for (i=0;i<fs->numgroups;i++)
	{
		blk_t start;
		blk_t bb;
		blk_t ib;
		blk_t it;

		start = fs->sb.s_first_data_block
			+ (i * fs->sb.s_blocks_per_group);
		it = start + fs->itoffset;

		if (ext2_is_group_sparse(fs, i))
		{
			bb = it - 2;
			ib = it - 1;
		}
		else
		{
			bb = start;
			ib = start + 1;
		}

		if (fs->gd[i].bg_block_bitmap != bb ||
		    fs->gd[i].bg_inode_bitmap != ib ||
		    fs->gd[i].bg_inode_table != it)
		{
			ped_exception_throw (PED_EXCEPTION_BUG,
				PED_EXCEPTION_CANCEL,
			_("This ext2 filesystem has a rather strange layout!  "
			  "Please use dumpe2fs on this filesystem and send it "
			  "to <buytenh@gnu.org>. I won't resize it, sorry."));
			return 0;
		}
	}

	return 1;
}

int ext2_sync(struct ext2_fs *fs)
{
	if (!ext2_commit_metadata(fs, EXT2_META_PRIMARY)) return 0;
	if (!ext2_bcache_sync(fs)) return 0;
	if (!fs->devhandle->ops->sync(fs->devhandle->cookie)) return 0;
	return 1;
}

struct ext2_fs *ext2_open(struct ext2_dev_handle *handle, int state)
{
	struct ext2_fs *fs;

	if ((fs = (struct ext2_fs *) ped_malloc(sizeof(struct ext2_fs)))
		== NULL)
		goto error;

	if ((fs->gd = (struct ext2_group_desc *) ped_malloc(ext2_max_groups * sizeof(struct ext2_group_desc))) == NULL)
		goto error_free_fs;

	handle->ops->set_blocksize(handle->cookie, 10);

	if (!handle->ops->read(handle->cookie, &fs->sb, 1, 1)
	    || fs->sb.s_magic != EXT2_SUPER_MAGIC)
	{
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Invalid superblock.  Are you sure this is an ext2 "
			  "filesystem?"));
		goto error_free_gd;
	}

	fs->opt_debug = 1;
	fs->opt_safe = 1;
	fs->opt_verbose = 0;

	if (fs->sb.s_state & EXT2_ERROR_FS & ~(state & EXT2_ERROR_FS))
	{
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
		     _("Filesystem has errors!  Run e2fsck first!"));
		goto error_free_gd;
	}

	if (!((fs->sb.s_state | state) & EXT2_VALID_FS))
	{
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
		     _("Filesystem was not cleanly unmounted!  "
		       "Run e2fsck first!"));
		goto error_free_gd;
	}

	if (fs->sb.s_feature_compat ||
	    (fs->sb.s_feature_incompat & ~EXT2_FEATURE_INCOMPAT_FILETYPE) ||
	    (fs->sb.s_feature_ro_compat & ~EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER))
	{
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
		     _("Filesystem has incompatible feature enabled"));
		goto error_free_gd;
	}

	fs->devhandle = handle;
	fs->logsize = fs->sb.s_log_block_size + 10;
	handle->ops->set_blocksize(handle->cookie, fs->logsize);

	if (!ext2_bcache_init(fs))
	{
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
				     _("Error allocating buffer cache."));
		goto error_free_gd;
	}

	fs->blocksize = 1 << fs->logsize;

	fs->numgroups = howmany(fs->sb.s_blocks_count - fs->sb.s_first_data_block, fs->sb.s_blocks_per_group);
	fs->gdblocks = howmany(fs->numgroups * sizeof(struct ext2_group_desc), fs->blocksize);
	fs->inodeblocks = howmany(fs->sb.s_inodes_per_group * sizeof(struct ext2_inode), fs->blocksize);
	fs->r_frac = howmany(100 * (loff_t)fs->sb.s_r_blocks_count, fs->sb.s_blocks_count);
	fs->adminblocks = 3 + fs->gdblocks + fs->inodeblocks;

	fs->sparse = 0;
	if (fs->sb.s_feature_ro_compat & EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER)
		fs->sparse = 1;

	ext2_read_blocks(fs, fs->gd, fs->sb.s_first_data_block + 1, fs->gdblocks);

	if (!ext2_determine_itoffset(fs))
		goto error_deinit_bcache;

	fs->metadirty = 0;

#if 0
	ext2_journal_init(fs);
#endif

	return fs;

error_deinit_bcache:
	ext2_bcache_deinit(fs);
error_free_gd:
	ped_free(fs->gd);
error_free_fs:
	ped_free(fs);
error:
	return NULL;
}
