/*
 * FILE:
 * Sheet.c
 *
 * FUNCTION:
 * create reports 
 *
 * DESIGN:
 * Filters phtml files through an ePerl (Embedded Perl)
 * interpreter.  Sets up variables before filtering.
 *
 * Invokes ePerl interpreter to handle perl embedded in html
 * files. The resulting final html is returned.  Communications
 * with eperl are through fork/exec/pipes.
 *
 * More work needs to be done:
 *  o We need some way of defining which routine gets invoked 
 *    depending on which link gets pressed.  But first, we need
 *    a url-encoding decoder.
 *
 * The StartUpFile () routine creates a hidden file and returns
 *    a file handle to it.  Handy for IPC.
 *
 * HISTORY:
 * Created by Linas Vepstas October 1998
 * Copyright (c) 1998, 1999 Linas Vepstas
 */

/********************************************************************\
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
\********************************************************************/


#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include "File.h"
#include "FileDialog.h"
#include "FileIO.h"
#include "Group.h"
#include "util.h"
#include "pathconfig.h"

static pid_t interp_pid = 0;

/* ======================================================= */

static FILE *
StartUpFile (void) 
{
  int fd;
  char tmpfilename[200];
  FILE *fh;
  
  /* create a tmp file with a bit of privacy */
  strcpy (tmpfilename, "/tmp/gncXXXXXX");
  mktemp (tmpfilename);
  strcat (tmpfilename, ".html");
  fd = open (tmpfilename, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
  if (0 > fd) {
    ERROR ();
    fprintf (stderr, "Error: can't open file %s\n", tmpfilename);
    return NULL;
  }

  fh = fdopen (fd, "w+");
  if (!fh) {
    ERROR ();
    fprintf (stderr, "Error: can't open file %s\n", tmpfilename);
    return NULL;
  }

  /* The unlink makes sure that file is deleted even if we crash.
   * It also affords a bit of privacy; no-one can snoop on our
   * temp file, because without the directory entry, it becomes
   * much harder to find and open.
   */
  unlink (tmpfilename);  

  return fh;
}
  
/* ======================================================= */

/* hack alert -- the paths to the interpreters should probably
 * be taken from a config file somewhere.  Note however that
 * we must be *very very* careful wth these paths, as they
 * potentially open a flagarent security hole.
 */

/* 
 * XXX - ifdef'd out by rgmerk who is using configure to search for these
 * and #defining them in gnucash.h instead.
 * Should be properly fixed up in the scheme startup code later?? (rgmerk)
 */

# if 0
#define EPERL_PATH "/usr/bin/eperl"
#define EPERL_NAME "eperl"
#endif

static FILE *
StartUpInterp (FILE * out_fh, int *bs_fd_ret) 
{
  int out_fd;
  int in_fd[2];
  int bs_fd[2];
  FILE *to_perl;
  struct stat statbuf;
  int rc;

  /* don't bother if the interpreter is not present */
  rc = stat (EPERL_PATH, &statbuf);
  if (rc) {
    printf ("Error: StartUpInterp(): Can't find interpreter %s\n", EPERL_PATH);
    return NULL;
  }

  /* create the pipe that will be duped to stdin of the interpreter. */
  pipe (in_fd);

  /* create the pipe that will carry the byte stream 
   * from here to the interpreter (will be duped to fd=3) */
  pipe (bs_fd);

  /* stdout will be dumped into a file */
  out_fd = fileno (out_fh);

  /* interpreter will run in a sub-process */
  interp_pid = fork ();
  if (!interp_pid) {
    /* attach the pipes to stdin and stdout */
    dup2 (in_fd[0], 0);
    close (in_fd[1]);
    dup2 (out_fd, 1);
    dup2 (bs_fd[0], 3);
    close (bs_fd[1]);

#ifdef ECHO_DEBUG_STDIN_OUT
    do {
       int c, rc;
       rc = feof (stdin);
       printf ("rc=%d\n", rc);
       c = fgetc (stdin);
       if (EOF == c) {printf ("bye\n"); fflush (stdout); exit (0); }
       printf ("%c", c);
       fflush (stdout);
    } while (1);
#endif /* ECHO_DEBUG_STDIN_OUT */

    execl (EPERL_PATH, EPERL_NAME, "-", NULL);
    {
       int norr = errno;
       fprintf (stderr, "Error: couldn't exec: \n");
       fprintf (stderr, "\t%s (%d) %s\n", EPERL_PATH, norr, strerror (norr));
       exit (1);
    }
  }

  /* assign FILE structures to the fds, close the duped fds */
  to_perl = fdopen (in_fd[1], "w");
  close (in_fd[0]);
  close (bs_fd[0]);
  if (bs_fd_ret) *bs_fd_ret = bs_fd[1];

  return to_perl;
}

/* ======================================================= */

static char * 
FinishUp (FILE *fh) 
{
  int fd, rc;
  char * text = NULL;
  size_t size;

  /* cleanup zombie process, make sure that interpreter has written
   * to the output file, and properly flushed its buffers, etc.  
   * If we don't wait, then the output buffer might be empty or only
   * partially written.
   */
  waitpid (interp_pid, NULL, 0);

  /* make sure that file buffers are fluished ... Although, 
   * technically these should be empty since we aren't writing to them */
  fflush (fh);

  /* Find size: */
  fd = fileno (fh);
  size = lseek( fd, 0, SEEK_END );
  lseek( fd, 0, SEEK_SET );

  text = (char *) malloc ((size+1)*sizeof (char));

  /* read in file */
  rc = read (fd,text,size);
  if (-1 == rc) {
    free (text);
    text = NULL;
  }
  text[size] = 0x0;
  fclose (fh);
  return text;
}

/* ======================================================= */
/* hack alert -- maybe we should move this codee out of here?
 * Having the use here might be nice for install config, so that
 * we don't have to hack the path in each and every 
 * phtml eperl''ed report file ...
 */


static void 
PrtHeader (FILE *to_perl)
{
   char *fullpath;

   /* slightly less of a hack but still not perfect */
   fullpath = GNC_PERL_LIB_PREFIX;

   fprintf (to_perl,
       "<: \n"
       "use lib '%s';  \n"
       "use gnucash;   \n"
       "$topgroup = &gnucash::xaccReadAccountGroup(3); \n"
       ":>\n",
       fullpath);
}

/* ======================================================= */

char * 
gncReport (char * const source_file) 
{
  char * text;
  FILE *from_perl, *to_perl;
  int byte_stream_fd = -1;
  AccountGroup *topmostgroup = gncGetCurrentGroup();

  if (!source_file) return NULL; 
  text = gncReadFile (source_file);
  if (!text) return NULL;

  /* open temp staging file for our html output */
  from_perl = StartUpFile ();
  if (!from_perl) return NULL;

  /* initilaize the interpreter, attach its output 
   * to the temp file */
  to_perl = StartUpInterp (from_perl, &byte_stream_fd);
  if (!to_perl || 0 > byte_stream_fd) {
    fclose (from_perl);
    close (byte_stream_fd);
    return NULL;
  }

  PrtHeader (to_perl);
  fflush (to_perl);

  /* feed in the parsed-html template */
  fprintf (to_perl, "%s", text);

  fflush (to_perl);
  fclose (to_perl);

  /* write out the full byte stream */
  /* Note that eperl doesn't start processing any of the data until 
   * its stdn has been closed (our toperl file handle).  Therefore,
   * in order to avoid a deadlock, we must NOT write out the byte 
   * stream until after we've closed to_perl.  Otherwise, the
   * write will block because perl won't be draining the pipe.
   */
  xaccWriteAccountGroup (byte_stream_fd, topmostgroup);
  close (byte_stream_fd);

  text = FinishUp (from_perl);
  return text;
}

/* =============================  END OF FILE ===================== */
