/* -*-gnucash-c-*-
   gnucash.c -- gnucash startup function.
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
                                                                   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
                                                                   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   (C) 1998 Rob Browning <rlb@cs.utexas.edu>
   (C) 1999-2000 Linas Vepstas
*/

#define _GNU_SOURCE

#include <stdio.h>

#ifdef HAVE_GETTEXT
#include <libintl.h>
#endif

#include <locale.h>
#include <guile/gh.h>

#include "FileDialog.h"
#include "top-level.h"
#include "util.h"
#include "gnucash.h"
#include "enriched-messages.h"
#include "i18n.h"
#include "g-wrap.h"
#include "gnc.h"

static int guile_is_initialized = GNC_F;
static int gargc;
static char **gargv;

/* This static indicates the debugging module this .o belongs to.  */
static short module = MOD_GUILE;


/* ============================================================== */

void
gnc_set_double_entry_restriction(const int state) {

  if(state == 0) {
    gh_eval_str("(gnc:config-var-set! gnc:*double-entry-restriction* #f)");
  } else if(state == 1) {
    gh_eval_str("(gnc:config-var-set! gnc:*double-entry-restriction* 'force)");
  } else if(state == 2) {
    gh_eval_str("(gnc:config-var-set! gnc:*double-entry-restriction* 'collect)");
  } else {
    char *tmp = NULL;
    asprintf(&tmp, "(gnc:config-var-set! gnc:*double-entry-restriction* %d)", state);
    assert(tmp);
    gh_eval_str(tmp);
    free(tmp);
  }
}

/* ============================================================== */

static void 
gnucash_main_helper(int argc, char *argv[]) {

  char *bootstrap_scm = getenv("GNC_BOOTSTRAP_SCM");
  char *run_as_shell = getenv("GNC_RUN_AS_SHELL");

  guile_is_initialized = GNC_T;

  init_gnc();

  if(!bootstrap_scm) {
    if(setenv("GNC_BOOTSTRAP_SCM", GNC_BOOTSTRAP_SCM, 1) != 0) {
      PERR("gnucash: setenv(%s, %s, 1) failed.  Aborting.\n",
           "GNC_BOOTSTRAP_SCM", GNC_BOOTSTRAP_SCM);
      exit(1);
    }
    bootstrap_scm = GNC_BOOTSTRAP_SCM;
  }

  PINFO("gnucash: bootstrap file is %s\n", bootstrap_scm);

  if(run_as_shell) {
    /* Add the "--" to keep gh_repl from interpreting the gnucash args */
    
    int scm_argc = argc + 1;
    char **scm_argv = malloc(scm_argc * sizeof(char *));
    assert(scm_argv);

    scm_argv[0] = (char *) argv[0];
    scm_argv[1] = "--";
    memcpy(&(scm_argv[2]), &(argv[1]), (argc - 1) * sizeof(char *));
    gh_repl(scm_argc, scm_argv);
    
  } else {

    SCM func;

    /* Load the bootstrap file. */
    if(gh_eval_file(bootstrap_scm) == SCM_BOOL_F) {
      PERR("gnucash: (load %s) failed\n", bootstrap_scm);
      exit(1);
    }

    /* Now finally load the startup file. */
    func = gh_eval_str("gnc:load");
    
    if(gh_procedure_p(func)) {
      SCM init_scm = gh_str02scm("startup.scm");
      gh_call1(func, init_scm);
    } else {
      PERR("gnucash: gnc:load doesn't exist.  Aborting.\n");
      exit(1);
    }
    
    /* and then run gnc:main. */
    func = gh_eval_str("gnc:main");
    
    if(gh_procedure_p(func)) {
      gh_call0(func);
    } else {
      PERR("gnucash: (gnc:main) doesn't exist.  Aborting.\n");
      exit(1);
    }
  }
  
  exit(0);
}

/* ============================================================== */

int
gnucash_lowlev_app_init()
{
#if DEBUG_MEMORY
  char *blk;
  DEBUG("Initializing memory");
  blk = (char *)_malloc(8192);
  _free(blk);
  DEBUG(" coresize = %d\n",_coresize());
  DEBUG("Done initializing memory");
#endif

  printf ("\n\n"
          "This is a development version.  It may or may not work. \n"
          "Report bugs and other problems to http://www.gnucash.org/ \n"
          "The last stable version was gnucash-1.2.5 \n"
          "The next stable version will be gnucash-1.4.x \n"
          "\n\n");

  /* We need to initialize the UI early ... */
  gnucash_ui_init();
  return 0;
}

/* ============================================================== */

void
gnc_shutdown (int exit_status) {

  if(guile_is_initialized) {
    /* This should be gh_lookup, but that appears to be broken */
    SCM scm_shutdown = gh_eval_str("gnc:shutdown");

    if(gh_procedure_p(scm_shutdown)) {
      SCM scm_exit_code = gh_long2scm(exit_status);

      gh_call1(scm_shutdown, scm_exit_code);

      return;
    }
  }

  /* Either guile is not running, or for some reason we
     can't find gnc:shutdown. Either way, just exit. */
  exit(exit_status);
}

/* ============================================================== */

int
main(int argc, char *argv[])
{
  gargc = argc;
  gargv = argv;

#ifdef HAVE_GETTEXT
  bindtextdomain (TEXT_DOMAIN, LOCALE_DIR);
  textdomain (TEXT_DOMAIN);
#endif

  /* It seems we need this so guile can find the locale. */
  setlocale(LC_ALL, "");

  buildEnrichedMessages();

  gh_enter(argc, argv, gnucash_main_helper);
  /* This will never return. */
  
  assert(0);  /* Just to be anal */
  return 0;   /* Just to shut gcc up. */
}

/* ==================== END OF FILE ============================== */
